//************************************************************************************//
// Module       : gtbuffer.cpp
// Date         : 7/9/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a buffer template  object
// Derived From :
// Modifications:
//************************************************************************************//
#include <memory.h>
#include "gtbuffer.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
template<class TBUFF> GBuffer<TBUFF>::GBuffer<TBUFF>()
:
n                     (0),
data                  (NULL)
{
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
//************************************************************************************
template<class TBUFF> GBuffer<TBUFF>::GBuffer<TBUFF>(GINT  size)
:
n                     (MAX(size,0)),
data                  (NULL)
{
  if ( n > 0 ) {
    data = new TBUFF [n];
  }

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
//************************************************************************************
template<class TBUFF> GBuffer<TBUFF>::GBuffer<TBUFF>(TBUFF *buff, GINT  size)
:
n                     (MAX(size,0)),
data                  (NULL)
{
  if ( n > 0 )
  {
    data = new TBUFF [n];
    if ( buff != NULL )
      memcpy(data, buff, n*sizeof(TBUFF));
    else
      memset(data, '\0', n*sizeof(TBUFF));;
  }

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Copy constructor method
//************************************************************************************
template<class TBUFF> GBuffer<TBUFF>::GBuffer<TBUFF>(const GBuffer<TBUFF> &a)
{
   // copy member data:
   n    = a.n;
   if ( n > 0 ) data = new TBUFF  [a.n];

   //  copy array data:
// if ( data != NULL ) delete []  data;
// data = NULL;
   if ( a.data != NULL ) 
   {
     memcpy(data,a.data,n*sizeof(TBUFF));
   }   

} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
template<class TBUFF> GBuffer<TBUFF>::~GBuffer<TBUFF>()
{
   DeleteDynamic();
}


//************************************************************************************
//************************************************************************************
// Assignment operator method 
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::operator=(GBuffer<TBUFF> &a)
{
  if ( this != &a ) 
  {
    if ( a.n == 0 || a.data == NULL ) return;
    if ( a.n != n ) 
    {
      cout << "GBuffer::=: incompatible assignment" << endl;
      return;
    }


    //  copy array data:
    if ( a.data != NULL ) 
    {
      memcpy(data,a.data,n*sizeof(TBUFF));
     }   
  }

} // end = operator


//************************************************************************************
//************************************************************************************
// Assignment operator method
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::operator=(TBUFF  a)
{
  GINT  i;

  if ( data == NULL && n > 0 )
  {
    cout << "GBuffer::=: illegal assignment" << endl;
    exit(1);
  }

  //  copy array data:
  for ( i=0; i<n; i++ )
  {
    data[i] = a;
  }

} // end = operator


#if 0
//************************************************************************************
//************************************************************************************
// () operator method 
template<class TBUFF> TBUFF &GBuffer<TBUFF>::operator()(GINT  i)
{
  //if ( i >= n || i < 0 ) throw "GBuffer<TBUFF>::(): access error";
  if ( i >= n || i < 0 ) 
  {
    cout << "GBuffer<TBUFF>::(): access error: index=" << i << "; max=" <<  n << endl;
    exit(1);
  }

  return *(data+i); 
} // end of () operator

// () operator method
template<class TBUFF> TBUFF GBuffer<TBUFF>::operator()(GINT  i) const
{ 
// if ( i >= n || i < 0 ) throw "GBuffer<TBUFF>::(): access error";
  if ( i >= n || i < 0 )
  {
    cout << "GBuffer<TBUFF>::(): access error: index=" << i << "; max=" <<  n << endl;
    exit(1);
  }

   TBUFF fret = *(data+i); 
  return fret;
} // end of () operator
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : dim
// DESCRIPTION:
// RETURNS    :  size of vector
//************************************************************************************
template<class TBUFF> GINT  GBuffer<TBUFF>::dim() const 
{
  return n ;
} // end of method dim


//************************************************************************************
//************************************************************************************
// METHOD     : Data
// DESCRIPTION: 
// RETURNS    :  pointer to the array data area
//************************************************************************************
template<class TBUFF> TBUFF *GBuffer<TBUFF>::Data()
{
  return data;
} // end of method Data


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
template<class TBUFF> GBOOL GBuffer<TBUFF>::Resize(GINT  newSize)
{
  GBOOL bRet = FALSE;

  if ( newSize == n && n > 0 ) return TRUE;

  if ( data != NULL ) {
    delete [] data;
    data = NULL;
  }
  if ( newSize == 0 ) {
    n = newSize;
    return TRUE;
  }
  if ( newSize <  0 ) {
    cout << "GBuffer<TBUFF>::Resize: invalid dimension: " << newSize << endl;
    exit(1);
  }
  
  data = new TBUFF [newSize];
  bRet = FALSE;
  if ( data != NULL ) {
    memset(data, 0   , newSize*sizeof(TBUFF));
    n = newSize;
    bRet = TRUE;
  }
  return bRet;

} // end of method Resize


//************************************************************************************
//************************************************************************************
// METHOD     : Set
// DESCRIPTION: Sets const input value in data array
// RETURNS    : 
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::Set(const TBUFF tset)
{

  GINT  i;

  if ( data != NULL )
  {
    for ( i=0; i<n; i++ )
      *(data+i) = tset;
  }

} // end of method Set


//************************************************************************************
//************************************************************************************
// METHOD     : contains (1)
// DESCRIPTION: Determines if candidate member is in the buffer, and if so,
//              sets the index where it is found. Search continues until index
//              is found s.t. data <= floor.
// RETURNS    :  TRUE if member is in list, else FALSE. On TRUE, set index to
//               index where member is found first; else set index = -1.
//************************************************************************************
template<class TBUFF> GBOOL GBuffer<TBUFF>::contains(TBUFF imember, GINT  &index, TBUFF floor)
{
  GINT  i=0;

  if ( data == NULL ) return FALSE;

  index = -1;
  while ( i < n && data[i] > floor && data[i] != imember ) i++;

  if ( i >= n || data[i] <= floor ) return FALSE;

  index = i;

  return TRUE;

} // end of method contains (1)


//************************************************************************************
//************************************************************************************
// METHOD     : contains (2)
// DESCRIPTION: Determines if candidate member is in the buffer, and if so,
//              sets the index where it is found. Search continues until index
//              is found s.t. data <= floor. Search begins at istart, and continues
//              at most num indices.
// RETURNS    :  TRUE if member is in list, else FALSE. On TRUE, set index to
//               index where member is found first; else set index = -1.
//************************************************************************************
template<class TBUFF> GBOOL GBuffer<TBUFF>::contains(TBUFF imember, GINT  istart, GINT  num, 
                                                      GINT  &index, TBUFF floor)
{
  GINT  i=istart;

  if ( data == NULL ) return FALSE;

  index = -1;
  while ( i < (istart+num) && i < n && data[i] > floor && data[i] != imember ) i++;

  if ( i >= (istart+num) || i >= n ) return FALSE;

  if ( data[i] <= floor  ) return FALSE;

  index = i;

  return TRUE;

} // end of method contains (2)


//************************************************************************************
//************************************************************************************
// METHOD     : contains (3)
// DESCRIPTION: Determines if candidate member is in the buffer, and if so,
//              sets the index where it is found. Search continues until index
//              is found s.t. data <= floor. Determination is made on the basis 
//              of comparing member with LOWORD or HIWORD of data element, 
//              set by flag ihilo = 0 or 1 (>0, actually), respectively.
//              NOTE: use of this method is recommended only for TBUFF==GDWORD types.
// RETURNS    :  TRUE if member is in list, else FALSE. On TRUE, set index to
//               index where member is found first; else set index = -1.
//************************************************************************************
template<class TBUFF> GBOOL GBuffer<TBUFF>::contains(GWORD imember, GINT  &index, 
                                                       GINT  ihilo, TBUFF floor)
{
  GINT  i=0;

  if ( data == NULL ) return FALSE;

  index = -1;
  if ( ihilo == 0 ) {
    while ( i < n && LOWORD(data+i) > floor && LOWORD(data+i) != imember ) i++;
    if ( i >= n ) return FALSE;
    if ( LOWORD(data+i) <= floor  ) return FALSE;
  }
  else {
    while ( i < n && HIWORD(data+i) > floor && HIWORD(data+i) != imember ) i++;
    if ( i >= n ) return FALSE;
    if ( HIWORD(data+i) <= floor  ) return FALSE;
  }

  index = i;

  return TRUE;

} // end of method contains (3)


//************************************************************************************
//************************************************************************************
// METHOD     : contains (4)
// DESCRIPTION: Determines if candidate member is in the buffer, and if so,
//              sets the index where it is found. 
// RETURNS    :  TRUE if member is in list, else FALSE. On TRUE, set index to
//               index where member is found first; else set index = -1.
//************************************************************************************
template<class TBUFF> GBOOL GBuffer<TBUFF>::contains(TBUFF imember, GINT  &index)
{
  GINT  i=0;

  if ( data == NULL ) return FALSE;

  index = -1;
  while ( i < n && data[i] != imember ) i++;

  if ( i >= n ) return FALSE;

  index = i;

  return TRUE;

} // end of method contains (4)


//************************************************************************************
//************************************************************************************
// METHOD     : distinct (1)
// DESCRIPTION: From buffer, get indices where there are distinct values. This
//              is meant to be used for meaningful TBUFFs (e.g., not ELEMTYPE).
// ARGUMENTS  : indices : array containing indices into the buffer which contain
//                        the distinct values
//              nd      : number of distinct values. nd=0 is not considered an error
// RETURNS    :  TRUE on success; else FALSE 
//************************************************************************************
template<class TBUFF> GBOOL GBuffer<TBUFF>::distinct(GINT  *&indices, GINT  &nd, TBUFF tfloor)
{
  GINT  i, k, ind, *itmp;
  GBOOL b_in;
  TBUFF elem;
  GBuffer<TBUFF> btmp;

  nd = 0;
  if ( data == NULL || n == 0 ) return FALSE;

  if ( indices != NULL ) delete [] indices; indices = NULL;

  itmp = new GINT  [n];
  btmp.Resize(n); btmp = tfloor;
  for ( i=0, nd=0; i<n && nd<n; i++ ) {
    if ( !btmp.contains(data[i],ind) ) {
      btmp[nd] = data[i];
      itmp[nd] = i;
      nd++;
    }
  }
#if 0
  for ( i=0; i<n && nd<n; i++ ) {
    elem = data[i] ;
    for ( k=0,b_in=FALSE; k<nd && !b_in; k++ ) b_in = b_in || ( elem==(*(data+itmp[k])) ); 
    if ( !b_in ) itmp[nd++] = i;
  }
#endif

  if ( nd > 0 ) {
    indices = new GINT  [nd];
    for ( i=0; i<nd; i++ ) indices[i] = itmp[i];
  }
  delete [] itmp;

  return TRUE;

} // end of method distinct (1) 


#if 1
//************************************************************************************
//************************************************************************************
// METHOD     : sortdecreasing
// DESCRIPTION: sorts buffer in decreasing order
// RETURNS    :  none
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::sortdecreasing()
{
  GINT  i, j;
  TBUFF tmp;

  // Perhaps a more efficient algorithm (e.g., heapsort) 
  // would be better, but for now...
  for ( j=0; j<n; j++ ) {
    for ( i=j; i<n; i++ ) {
      if ( data[i] > data[j] ) {
        tmp     = data[j];
        data[j] = data[i];
        data[i] = tmp;
      }
    }
  }

} // sortdecreasing


//************************************************************************************
//************************************************************************************
// METHOD     : sortincreasing
// DESCRIPTION: sorts buffer in increasing order
// RETURNS    :  none
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::sortincreasing()
{
  GINT  i, j;
  TBUFF tmp;

  // Perhaps a more efficient algorithm (e.g., heapsort) 
  // would be better, but for now...
  for ( j=0; j<n; j++ ) {
    for ( i=j; i<n; i++ ) {
      if ( data[i] < data[j] ) {
        tmp     = data[j];
        data[j] = data[i];
        data[i] = tmp;
      }
    }
  }

} // sortincreasing
#endif

//************************************************************************************
//************************************************************************************
// METHOD     : GetSection (1)
// DESCRIPTION: Gets buffer section using argument indices
// RETURNS    : reference to SEBiffer<TBUFF> buffer created from specified indices
//************************************************************************************
template<class TBUFF> GBuffer<TBUFF> &GBuffer<TBUFF>::GetSection(GINT  *ilist, GINT  nlist)
{
  GBuffer<TBUFF> *tbuff;

  if ( data != NULL ) delete [] data ;
//ldim = ilist == NULL ? n : nlist;
  tbuff = new GBuffer<TBUFF>(0);
  GetSection(*tbuff,ilist,nlist);

  return *tbuff;

} // end of method GetSection (1)


//************************************************************************************ 
//************************************************************************************
// METHOD     : GetSection (2)
// DESCRIPTION: Gets buffer section using argument indices
// RETURNS    : none.
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::GetSection(GBuffer<TBUFF> &tbuff, GINT  *ilist, GINT  nlist)
{
  GINT  i, ldim;

  if ( data != NULL ) delete [] data ;
  ldim = ilist == NULL ? n : nlist; 

  tbuff.Resize(ldim); 
  if ( ilist == NULL ) {
    for ( i=0; i<ldim; i++ ) tbuff[i] = data[i];
  }
  else {
    for ( i=0; i<ldim; i++ ) tbuff[i] = data[ilist[i]];
  }

} // end of method GetSection (2)


//************************************************************************************ 
//************************************************************************************
// METHOD     : sortlohi
// DESCRIPTION: sorts data from lo to hi using heapsort method, taken from
//              Numerical Recipes in C++, 2nd Ed.
// RETURNS    : none.
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::sortlohi()
{
#if 1
  GINT  i;
  TBUFF dum;
  
  for ( i=n/2-1; i>=0; i-- ) {
    sift_down(i, n-1);
  }
  for ( i=n-1; i>0; i-- ) {
    dum = data[0]; data[0] = data[i]; data[i]=dum; // swap data[0], data[i]
    sift_down(0, i-1);
  }
#else
  GINT  k, nn=n;
  TBUFF x;

  for ( k=nn/2-1; k>=0; k-- ) sift_down(k,0);
  while ( --nn > 0 ) {
    x = data[0]; data[0] = data[nn]; data[nn] = x;
    sift_down(0,0);
  }
#endif

} // end of method sortlohi



//************************************************************************************ 
//************************************************************************************
// METHOD     : sift_down
// DESCRIPTION: 'sift-down' method from Numerical Recipes in C++, 2nd Ed. Part of
//              heap sort method
// RETURNS    : none.
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::sift_down(const GINT l, const GINT r)
{
#if 1
  GINT  j, jold;
  TBUFF a;

  a = data[l];
  jold = l;
  j = l + 1;
  while ( j <= r ) {
    if ( j < r && data[j] < data[j+1] ) j++;
    if ( a >= data[j] ) break;
    data[jold] = data[j];
    jold = j;
    j = 2*j + 1;
  }
  data[jold] = a;
#else
  GINT  i, j;
  TBUFF x;

  i = l; x = ra[i];
  while ( (j=2*i+1) < na ) {
    if ( j < n-1 && ra[j] < ra[j+1] ) j++;
    if ( x >= ra[j] ) break;
    ra[i] = ra[j]; i = j;
  }
  ra[i] = x;
  
#endif

} // end of method sift_down


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// RETURNS    :  none
//************************************************************************************
template<class TBUFF> void GBuffer<TBUFF>::DeleteDynamic()
{

  if ( data != NULL ) delete [] data ;
  data = NULL;

} // end of method DeleteDynamic


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer elements
// RETURNS    :
//************************************************************************************
template<class TBUFF> ostream &operator<<(ostream &str, GBuffer<TBUFF> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a[i] << "  ";

  return str;
} // end of << operator 
#else
#if defined (GBUFF_DEF_GINT )
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <GINT >
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GINT >&a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GINT >
#endif


#if defined (GBUFF_DEF_GDWORD)
//************************************************************************************
//************************************************************************************
// METHOD     : operator <<
// DESCRIPTION: output buffer of <GDWORD>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GDWORD>&a)
{ 
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str <<  a(i) << "  ";
//  str << "(" << HIWORD(a.Data()+i) << "," << LOWORD(a.Data()+i) <<  "," << a(i) << ")  ";

  return str; 
} // end of << operator <GDWORD>
#endif

#if defined (GBUFF_DEF_GSHORT)
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <GSHORT>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GSHORT> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GSHORT>
#endif


#if defined (GBUFF_DEF_GUSHORT)
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <GUSHORT>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GUSHORT> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GUSHORT>
#endif


#if defined (GBUFF_DEF_BYTE)
//************************************************************************************
//************************************************************************************
// METHOD     : operator <<
// DESCRIPTION: output buffer of <GBYTE>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GBYTE> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GBYTE>
#endif


#if defined (GBUFF_DEF_GDOUBLE)
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <GDOUBLE>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GDOUBLE> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GDOUBLE>
#endif


#if defined (GBUFF_DEF_BDYTYPE)
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <BDYTYPE>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<BDYTYPE> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <BDYTYPE>
#endif


#if defined (GBUFF_DEF_ELEMTYPE)
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <ELEMTYPE>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<ELEMTYPE> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <ELEMTYPE>
#endif


#if defined (GBUFF_DEF_GLONG )
//************************************************************************************
//************************************************************************************
// METHOD     : operator <<
// DESCRIPTION: output buffer of <GLONG >
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GLONG > &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GLONG >
#endif


#if defined (GBUFF_DEF_GKEY)
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <GKEY>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GKEY> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GKEY>
#endif

#if defined (GBUFF_DEF_GCHANDLE)
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <GCHandle>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GCHandle> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GCHandle>
#endif

#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <CHandle>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<CHandle> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <CHandle>
#endif


#if defined (GBUFF_DEF_GNODEID)
//************************************************************************************
//************************************************************************************
// METHOD     : operator << 
// DESCRIPTION: output buffer of <GNODEID>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GBuffer<GNODEID> &a)
{
  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << a(i) << "  ";

  return str;
} // end of << operator <GNODEID>
#endif
#endif

