//************************************************************************************//
// Module       : setmatrix.cpp
// Date         : 7/9/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a template matrix object, whose data is composed of
//                variable length vectors.
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include <typeinfo>
#include "setvmatrix.hpp"
#include "gcomm.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// constructor (1)
template<class T> SETVMatrix<T>::SETVMatrix<T>()
:
n1                    (0),
n2                    (0),
bDistributed          (FALSE),
iProcMap              (SERIAL),
dtype                 (G_GDOUBLE),
data                  (NULL)
{
  ISUB  i;

  gindex1(0, 0, 0, 0, 1, 0);
  gindex2(0, 0, 0, 0, 1, 0);
  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];
  for ( i=0; i<n1; i++ )
  {
    jMap[0][i] = 0;
    jMap[1][i] = n2-1;
  }

  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;


} // end of constructor 1 



//************************************************************************************
//************************************************************************************
// constructor (2)
template<class T>  SETVMatrix<T>::SETVMatrix<T>(const ISUB  size1, const ISUB  size2)
:
n1                    (size1),
n2                    (size2),
bDistributed          (FALSE),
iProcMap              (SERIAL),
dtype                 (G_GDOUBLE),
data                  (NULL)
{
  ISUB  i;

  gindex1(n1, n1, 0, n1-1, 1,  0);
  gindex2(n2, n2, 0, n2-1, 1,  0);
  data = new SETVector<T> * [n1];
  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];
  for ( i=0; i<n1; i++ )
  {
    jMap[0][i] = 0;
    jMap[1][i] = n2-1;
    data[i] = new SETVector<T> (jMap[1][i]-jMap[0][i]+1);
  }

  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;


  Zero();
} // end of constructor 2


//************************************************************************************
//************************************************************************************
// constructor (3)
template<class T> SETVMatrix<T>::SETVMatrix<T>(GIndex &sei1, GIndex &sei2)
:
n1                    (sei1.szlocal()),
n2                    (sei2.szlocal()),
bDistributed          (TRUE),
iProcMap              (M_PANEL),
dtype                 (G_GDOUBLE),
data                  (NULL)
{
  ISUB  i;
  GIndex si;

  gindex1 = sei1;
  gindex2 = sei2;
  if ( gindex1.szglobal() == n1 && gindex2.szglobal()== n2 ) bDistributed = FALSE;
  if ( bDistributed )
  {
    if      ( n1 <   gindex1.szglobal() && n2 == gindex2.szglobal() ) iProcMap=M_PANEL;
    else if ( n1 <=  gindex1.szglobal() && n2 <  gindex2.szglobal() ) iProcMap=M_CHECKERBRD;
    else                                       iProcMap=PM_INVALID;
    
  }
  if ( bDistributed && n1 <= gindex1.szglobal() && n2 <  gindex2.szglobal()) iProcMap=M_CHECKERBRD;

  data = new SETVector<T> * [n1];
  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];
  for ( i=0; i<n1; i++ )
  {
    jMap[0][i] = 0;
    jMap[1][i] = n2-1;
    si(gindex2.szglobal(), gindex2.szlocal(),jMap[0][i], jMap[1][i], 1, gindex2.pad());
    data[i] = new  SETVector<T> (si);
  }

  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;


  Zero();

} // end of constructor 3

//************************************************************************************
//************************************************************************************
// constructor (4)
template<class T> SETVMatrix<T>::SETVMatrix<T>(GIndex &sei1, GIndex &sei2, ISUB  **jjm)
:
n1                    (sei1.szlocal()),
n2                    (sei2.szlocal()),
bDistributed          (TRUE),
iProcMap              (SERIAL),
dtype                 (G_GDOUBLE),
data                  (NULL)
{
  ISUB  i;
  GIndex si;

  gindex1 = sei1;
  gindex2 = sei2;
  if ( gindex1.szglobal() == n1 && gindex2.szglobal()== n2 ) bDistributed = FALSE;
  if ( bDistributed )
  {
    if      ( n1 <   gindex1.szglobal() && n2 == gindex2.szglobal() ) iProcMap=M_PANEL;
    else if ( n1 <=  gindex1.szglobal() && n2 <  gindex2.szglobal() ) iProcMap=M_CHECKERBRD;
    else                                       iProcMap=PM_INVALID;

  }
  if ( bDistributed && n1 <= gindex1.szglobal() && n2 <  gindex2.szglobal()) iProcMap=M_CHECKERBRD;

  data = new SETVector<T> * [n1];
  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];
  for ( i=0; i<n1; i++ )
  {
    jMap[0][i] = jjm[0][i];
    jMap[1][i] = jjm[1][i];
    si(gindex2.szglobal(), gindex2.szlocal(),jMap[0][i], jMap[1][i], 1, gindex2.pad());
    data[i] = new  SETVector<T> (si);
  }

  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;


  Zero();

} // end of constructor 4



//************************************************************************************
//************************************************************************************
// constructor (5)
template<class T>  SETVMatrix<T>::SETVMatrix<T>(GIndex &sei1, GIndex &sei2, ISUB  nUpper, ISUB  nLower)
:
n1                    (sei1.szlocal()),
n2                    (sei2.szlocal()),
bDistributed          (TRUE),
iProcMap              (SERIAL),
dtype                 (G_GDOUBLE),
data                  (NULL)
{
  ISUB  i,j;
  GIndex si;

  gindex1 = sei1;
  gindex2 = sei2;
  if ( gindex1.szglobal() == n1 && gindex2.szglobal()== n2 ) bDistributed = FALSE;
  if ( bDistributed )
  {
    if      ( n1 <   gindex1.szglobal() && n2 == gindex2.szglobal() ) iProcMap=M_PANEL;
    else if ( n1 <=  gindex1.szglobal() && n2 <  gindex2.szglobal() ) iProcMap=M_CHECKERBRD;
    else                                       iProcMap=PM_INVALID;

  }
  if ( bDistributed && n1 <= gindex1.szglobal() && n2 <  gindex2.szglobal()) iProcMap=M_CHECKERBRD;

  // Determine column--mapping:

  data = new SETVector<T> * [n1];
  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];
  for ( i=0; i<n1; i++ )
  {
    jMap[0][i] = MAX(i+gindex1.beg()-nLower,0);
    jMap[1][i] = MIN(i+gindex1.beg()+nUpper,n2-1);
  }

  // build matrix data structure:
  for ( i=0; i<n1; i++ )
  {
    si(gindex2.szglobal(), gindex2.szlocal(), jMap[0][i], jMap[1][i], 1, gindex2.pad());
    data[i] = new  SETVector<T> (si);
  }

  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;


  Zero();

} // end of constructor 5

//************************************************************************************
//************************************************************************************
// constructor (6)
template<class T>  SETVMatrix<T>::SETVMatrix<T>(T *array, ISUB  m1, ISUB  m2)
:
n1                    (m1),
n2                    (m2),
bDistributed          (FALSE),
iProcMap              (SERIAL),
dtype                 (G_GDOUBLE),
data                  (NULL)
{
  ISUB  i,j;
  GIndex si;

  gindex1(0, 0, 0, 0, 1, 0);
  gindex2(0, 0, 0, 0, 1, 0);
  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];
  for ( i=0; i<n1; i++ )
  {
    jMap[0][i] = 0;
    jMap[1][i] = n2-1;
  }


  data = new SETVector<T> * [n1];
  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];

  for ( i=0; i<n1; i++ )
  { 
    jMap[0][i] = 0;
    jMap[1][i] = n2-1;
    data[i] = new SETVector<T> (jMap[1][i]-jMap[0][i]+1);
  }

    // build matrix data structure:

  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;

  for ( i=0; i<n1; i++ )
  {
    si(gindex2.szglobal(), gindex2.szlocal(), jMap[0][i], jMap[1][i], 1, gindex2.pad());
    data[i] = new  SETVector<T> (si);
    for ( j=0; j<n2; j++ )
      memcpy(data[i]->Data(), array+i*n1, n2*G_TYPESZ[dtype]);
  }

  Zero();

} // end of constructor 6


//************************************************************************************
//************************************************************************************
// Copy constructor:
template<class T> SETVMatrix<T>::SETVMatrix<T>(const SETVMatrix<T> &m)
{

  ISUB  i;
  GIndex si;

  // copy member data:
  n1      = m.n1;
  n2      = m.n2;
  bDistributed = m.bDistributed;
  iProcMap     = m.iProcMap;
  dtype        = m.dtype;
  gindex1 = m.gindex1;
  gindex2 = m.gindex2;
  

  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];
  memcpy(jMap[0],m.jMap[0],n1*sizeof(ISUB ));
  memcpy(jMap[1],m.jMap[1],n1*sizeof(ISUB ));

  data = new SETVector<T> * [n1];
  for ( i=0; i<n1; i++ )
  {
    si(gindex2.szglobal(), gindex2.szlocal(),jMap[0][i], jMap[1][i], 1, gindex2.pad());
    data[i] = new  SETVector<T> (si);
  }

  //  copy array data:
  if ( m.data != NULL ) 
  {
      for ( i=0; i<n1; i++ )
        *data[i] = *(m.data[i]);
  }
} // end of copy constructor


//************************************************************************************
//************************************************************************************
// Destructor
template<class T> SETVMatrix<T>::~SETVMatrix<T>()
{
   DeleteDynamic();
}

// Assignment operator method
template<class T> SETVMatrix<T> SETVMatrix<T>::operator=(const SETVMatrix<T> &m)
{
  if ( data == NULL ) return *this;


  ISUB  i;

  if ( &m != this ) 
  {
    if ( m.n1 != n1 || m.n2 != n2 )
    {
      cout << "SETVMatrix<T>::=: incompatible matrices" << endl;
      exit(1);
    }
    // copy member data:
    n1      = m.n1;
    n2      = m.n2;
    gindex1 = m.gindex1;
    gindex2 = m.gindex2;
    bDistributed = m.bDistributed;
    iProcMap     = m.iProcMap;
    dtype        = m.dtype;


    //  copy array data:
    if ( m.data != NULL )
    {
      for ( i=0; i<n1; i++ )
      {
//      (*data[i])(gindex2.beg(),gindex2.end(),gindex2.stride());
        *data[i] = *(m.data[i]);
      }
    }
    memcpy(jMap[0],m.jMap[0],n1*sizeof(ISUB ));
    memcpy(jMap[1],m.jMap[1],n1*sizeof(ISUB ));
  
  }
  return *this;

} // end = operator

// Assignment operator method
template<class T> void  SETVMatrix<T>::operator=(T m)
{
  if ( data == NULL )
  {
     cout << "SETVMatrix<T>::operator=(T m): illegal assignment" << endl;
     exit(1);
//   return *this;
  }


  ISUB  i;


  for ( i=0; i<n1; i+=gindex1.stride() )
  {
      (*data[i]) = m;
  }


} // end = operator


#if 0
//************************************************************************************
//************************************************************************************
// () operator method-- reference (for writing) 
template<class T> T &SETVMatrix<T>::operator()(const ISUB  i, const ISUB  j)
{
  if ( i-beg(1) >= dim(1) || i < 0 ||
       j-beg(2) >= dim(2) || j < 0  ) 
  {
    cout << "SETVMatrix<T>::&(): access error"<< endl;
    exit(1);
  }
  SETVector<T> *v;
  T            *p;

  v = data[i-beg(1)];
  if ( i < beg(1) || i > end(1) ) 
     return ( fNULL=0.0 );
  if ( j < v->GetIndex().beg() || j > v->GetIndex().end() ) 
     return ( fNULL=0.0 );

  return *(v->Data() + j - v->GetIndex().beg());

//return (*(data[i-beg(1)]))(j) ;
} // end of () operator


//************************************************************************************
//************************************************************************************
// () operator method--reading only
template<class T> T SETVMatrix<T>::operator()(const ISUB  i, const ISUB  j) const
{

  if ( i-beg(1) >= dim(1) || i < 0 ||
       j-beg(2) >= dim(2) || j < 0  ) 
  {
    cout << "SETVMatrix<T>::(): access error"<< endl;
    exit(1);
  }
  SETVector<T> *v;
  T            *p;

  v = data[i-beg(1)];
  if ( i < beg(1) || i > end(1) ) 
     return ( 0.0 );
  if ( j < v->GetIndex().beg() || j > v->GetIndex().end() ) 
     return ( 0.0 );

  return *(v->Data() + j - v->GetIndex().beg());
  
//return (*(data[i-beg(1)]))(j) ;
} // end of () operator
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : operator *
// DESCRIPTION: multiplies this by constant, and returns
//              result
// RETURNS    : product matrix
template<class T> SETVMatrix<T> SETVMatrix<T>::operator*(T a) 
{
  ISUB         i, ix=0, incx=gindex1.stride();
  SETVector<T> **p, *v;
  GIndex sei1=gindex1, sei2=gindex2;
  SETVMatrix<T>  aprod(sei1,sei2);

  aprod = *this;

  p = aprod.Data();
  for ( i=0; i<n1; i++ )
  {
    v       = data[ix];   
    *(p[i]) = ( (*v) * a );
    ix     += incx;
  }

  return aprod;

} // end of * operator (for constant ref.)


//************************************************************************************
//************************************************************************************
// Matrix-vector product:
template<class T> SETVector<T> SETVMatrix<T>::operator*(SETVector<T>  local_a)
{

  GIndex si=local_a.GetIndex();
  ISUB          i, ix=0, nd, incx=gindex1.stride();
  T             *pl, *pg, *adat, prod;
  SETVector<T>  *aret, *v;


#if 0
  // Only allow 'panel' matrix distribution for this option:
  if ( this->n2 != local_a.GetIndex().szglobal()) 
  {
    cout << "SETVMatrix<T>::*: incompatible vector"<< endl;
    exit(1);
  }
#endif

  // If array factor is not global, then gather all components; 
  // else, go ahead and multiply:

  if ( local_a.GetIndex().szlocal() >= n2 )
    si = gindex1;

  aret = new SETVector<T> (si);
  adat = aret->Data() + si.beg() + si.base();
  pl   = local_a.Data() + local_a.GetIndex().beg() + local_a.GetIndex().base();
  nd = end(1) - beg(1) + 1;
  if ( local_a.GetIndex().szlocal() < n2 )
  {
    SETVector<T> glob_a(si);

    pg = glob_a.Data() + si.beg() + si.base();
    GComm::Allgather(pl, (int)local_a.tsize(), dtype, pg, (int)(local_a.tsize()), dtype);
    for ( i=0; i<nd; i++ )
    {
      v        = data[ix];
      prod     = (*v) * glob_a;
      adat[ix] = prod;
      ix      += incx;
    }
  }
  else
  {
    switch (dtype)
    {
      case G_GDOUBLE:
        MTK::fmatvec_prod((SETVMatrix<GDOUBLE>&)*this,(SETVector<GDOUBLE>&)local_a,(SETVector<GDOUBLE>&)*aret);
        break;
      case G_GQUAD:
        MTK::qmatvec_prod((SETVMatrix<GQUAD>&)*this,(SETVector<GQUAD>&)local_a,(SETVector<GQUAD>&)*aret);
        break;
      default:
        cout << " SETVMatrix<T>::operator*: invalid data type" << endl;
        exit(1);
    }
  }


  return *aret;

} // end of operator *


//************************************************************************************
//************************************************************************************
// METHOD     : operator *
// DESCRIPTION: multiplies this by m1, and returns
//              result
// RETURNS    : product matrix
template<class T> SETVMatrix<T> SETVMatrix<T>::operator*(SETVMatrix<T> m1) 
{
  if ( this->n2 != m1.dim(1) )
  {
    cout << "SETVMatrix<T>::*: (Matrix) incompatible matrix"<< endl;
    exit(1);
  }

  ISUB  i, j, k;
  T     sum;
  GIndex  sei1=gindex1;
  GIndex sei2=m1.GetIndex(2);
  SETVMatrix<T> mprod(sei1,sei2);

  for ( i=beg(1); i<end(1)+1; i++ )
  {
    for ( j=sei2.beg(); j<sei2.end()+1; j++ )
    {
      sum = 0.0;
      for ( k=beg(1); k<end(1)+1; k++ )
        sum += (*this)(i,k) * m1(k,j);
      mprod(i,j) = sum;
    }
  }

  return mprod;

} // end of operator * (SETVMatrix<T>)


//************************************************************************************
//************************************************************************************
template<class T> SETVMatrix<T> SETVMatrix<T>::operator+(SETVMatrix<T>  a) 
{

  if ( this->n1 != a.dim(1) || this->n2 !=a.dim(2) )
  {
    cout << "SETVMatrix<T>::+: incompatible matrices"<< endl;
    exit(1);
  }


  ISUB  i, j;
  SETVMatrix<T>  asum(this->n1,this->n2);

  for ( i=0; i<dim(1); i++ )
  {
    for ( j=0; j<dim(2); j++ )
      asum(i,j) = (*this)(i,j) + a(i,j);
  }

  return asum;

}


//************************************************************************************
//************************************************************************************
template<class T> SETVMatrix<T> SETVMatrix<T>::operator-(SETVMatrix<T>  a) 
{

  if ( this->n1 != a.dim(1) || this->n2 !=a.dim(2) )
  {
    cout << "SETVMatrix<T>::+: incompatible matrices"<< endl;
    exit(1);
  }

  ISUB  i, j;
  SETVMatrix<T>  asum(this->n1,this->n2);

  for ( i=0; i<dim(1); i++ )
  {
    for ( j=0; j<dim(2); j++ )
      asum(i,j) = (*this)(i,j) - a(i,j);
  }

  return asum;

}

#if 1
//************************************************************************************
//************************************************************************************
// << operator method (1)
ostream &operator<<(ostream &str, SETVMatrix<GDOUBLE> a)
{

//str << "\n" << "size(0):" << a.dim(1) << " size(1): " << a.dim(2) <<  "\n" << "\n";
  str << "{ ";

  ISUB  i, j;
  for ( i=a.GetIndex(1).beg(); i<a.GetIndex(1).beg()+a.dim(1); i++ )
  {
    str << "{ ";
    for ( j=a.GetIndex(2).beg(); j<a.GetIndex(2).beg()+a.dim(2)-1; j++ )
      str 
//        << setiosflags(ios::scientific)
//        << setw(11)
//        << setprecision(4)
//        << setiosflags(ios::fixed)
          << a(i,j)
//        << setw(1)
          << ", ";
 
      str 
//        << setiosflags(ios::scientific)
//        << setw(11)
//        << setprecision(4)
//        << setiosflags(ios::fixed)
          << a(i,j) ;
//        << setw(1) ;
    if ( i < a.GetIndex(1).beg()+a.dim(1)-1 ) str << " }, ";
    else str << " }";
  }
    str << " } \n ";

  return str;
} // end of << operator(1)


// << operator method(2)
ostream &operator<<(ostream &str, SETVMatrix<GQUAD> a)
{

//str << "\n" << "size(0):" << a.dim(1) << " size(1): " << a.dim(2) <<  "\n" << "\n";
  str << "{ ";

  ISUB  i, j;
  for ( i=a.GetIndex(1).beg(); i<a.GetIndex(1).beg()+a.dim(1); i++ )
  {
    str << "{ ";
    for ( j=a.GetIndex(2).beg(); j<a.GetIndex(2).beg()+a.dim(2)-1; j++ )
      str
//        << setiosflags(ios::scientific)
//        << setw(11)
//        << setprecision(4)
//        << setiosflags(ios::fixed)
          << a(i,j)
//        << setw(1)
          << ", ";

      str
//        << setiosflags(ios::scientific)
//        << setw(11)
//        << setprecision(4)
//        << setiosflags(ios::fixed)
          << a(i,j) ;
//        << setw(1) ;
    if ( i < a.GetIndex(1).beg()+a.dim(1)-1 ) str << " }, ";
    else str << " }";
  }
    str << " } \n ";

  return str;
} // end of << operator (2)
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : Data
// DESCRIPTION: 
// RETURNS    :  pointer to the data area
template<class T> SETVector<T> **SETVMatrix<T>::Data() const
{
  return data;
} // end of method Data


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// RETURNS    :  none
template<class T> void SETVMatrix<T>::DeleteDynamic()
{
  ISUB  i;

  for ( i=0; i<n1; i++ )
  {
    delete data[i];
    data[i] = NULL;
  }
  delete [] data;
  data = NULL;
  if ( jMap != NULL )
  {
    delete [] jMap[1];
    delete [] jMap[0];
  }

} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// RETURNS    :  TRUE on success, else FALSE
template<class T> GBOOL SETVMatrix<T>::Resize(ISUB  new1, ISUB  new2)
{
  // Do a resize in all cases

  ISUB  i;
  GBOOL bRet = TRUE;
  T     *copy=NULL;

  for ( i=0; i<n1; i++ )
  {
    delete data[i];
    data[i] = NULL;
  }
  data = NULL;

  n1 = new1;
  n2 = new2;
  gindex1(n1, n1, 0, n1-1, 1, 0); 
  gindex2(n2, n2, 0, n2-1, 1, 0); 
  jMap[0] = new ISUB  [n1];
  jMap[1] = new ISUB  [n1];
  for ( i=0; i<n1; i++ )
  {
    jMap[0][i] = 0;
    jMap[1][i] = n2-1;
  }

  data = new SETVector<T> * [n1];
  for ( i=0; i<n1; i++ )
  {
    data[i] = new SETVector<T>(gindex2);
    if ( data[i] == NULL ) 
    {  
      bRet = FALSE;
      break;
    }
  }

  return bRet;

} // end of method Resize


//************************************************************************************
//************************************************************************************
// METHOD     : GetIndex()
// DESCRIPTION: returns pointer to GIndex member
// ARGUMENTS  : ISUB  idir: coordinate direction (1 or 2)
// RETURNS    : GIndex data member; else NULL if coord
//              direction invalid
template<class T> GIndex   SETVMatrix<T>::GetIndex(ISUB  idir)  const
{
  GIndex si;
  if ( idir == 1 )
    si = gindex1;
  else if ( idir == 2 ) 
    si = gindex2;

  return si;;
} // end of method GetIndex


//************************************************************************************
//************************************************************************************
// METHOD     : tsize
// DESCRIPTION: total size of data storage area
//              in direction idir (= n + npad_)
// RETURNS    : ISUB  size
template<class T> ISUB  SETVMatrix<T>::tsize(ISUB  idir)
{

  if      ( idir == 1 )
    return (n1 + gindex1.pad());
  else if ( idir == 2 )
    return ( n2 + gindex2.pad()); 
  else
    return 0;
} // end of method tsize


//************************************************************************************
//************************************************************************************
// METHOD     : dim
// DESCRIPTION: array dimension (usable)
//              in direction idir 
// RETURNS    : ISUB  size
template<class T> ISUB  SETVMatrix<T>::dim(ISUB  idir) const
{

  if      ( idir == 1 )
    return (n1);
  else if ( idir == 2 )
    return (n2);
  else
    return 0;
} // end of method dim


//************************************************************************************
//************************************************************************************
// METHOD     : beg
// DESCRIPTION: beginning index in the idir direction
// RETURNS    : ISUB  size
template<class T> ISUB  SETVMatrix<T>::beg(ISUB  idir) const
{

  if      ( idir == 1 )
    return (gindex1.beg());
  else if ( idir == 2 )
    return (gindex2.beg());
  else
    return 0;
} // end of method beg


//************************************************************************************
//************************************************************************************
// METHOD     : end
// DESCRIPTION: end index in the idir direction
// RETURNS    : ISUB  size
template<class T> ISUB  SETVMatrix<T>::end(ISUB  idir) const
{

  if      ( idir == 1 )
    return (gindex1.end());
  else if ( idir == 2 )
    return (gindex2.end());
  else
    return 0;
} // end of method end


//************************************************************************************
//************************************************************************************
// METHOD     : Zero
// DESCRIPTION: Zeros out data elemnts
// RETURNS    : none
template<class T> void SETVMatrix<T>::Zero()
{ 
  ISUB  i;

  for ( i=0; i<n1; i++ ) data[i]->Zero();
}



//************************************************************************************
//************************************************************************************
// METHOD     : Transpose (1)
// DESCRIPTION: computes transpose of this, but
//              does not destroy data.
// RETURNS    : transpose of this
template<class T> GBOOL  SETVMatrix<T>::Transpose(SETVMatrix<T> &trans)
{

  if ( trans.dim(2) !=  n1 || trans.dim(1) != n2 ) 
  {
    cout << "SETVMatrix<T>::Transpose: incompatible matrix"<< endl;
    exit(1);

  }

  ISUB  i, j;

  for ( i=0; i<trans.dim(1); i++ )
  {
    for ( j=0; j<trans.dim(2); j++ )
    {
       trans(i,j) = (*this)(j,i);
    }
  }
  return TRUE;
 
} // end of method Transpose (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Transpose (2)
// DESCRIPTION: computes transpose of this, but
//              does not destroy data. A copy is made and
//              returned.
// RETURNS    : transpose of this
template<class T> SETVMatrix<T>  SETVMatrix<T>::Transpose()
{

  SETVMatrix<T> t(n1,n2);

  if ( !Transpose(t) )
  {
    cout << "SETVMatrix<T>::Transpose(2): failed" << endl;
    exit(1);
  }

  return t;

} // end of method Transpose (2)


//************************************************************************************
//************************************************************************************
// METHOD     : Inverse (1)
// DESCRIPTION: computes inverse of this, copying the
//              result to mret
// RETURNS    : inverse of this
template<class T> GBOOL  SETVMatrix<T>::Inverse(SETVMatrix<T> &mret)
{

  if ( mret.dim(1) !=  n1 || mret.dim(2) != n2 )
  {
    cout << "SETVMatrix<T>::Inverse: incompatible matrix"<< endl;
    exit(1);
  }
  if ( n1 != n2 )
  {
    cout << "SETVMatrix<T>::Inverse: matrix not square"<< endl;
    exit(1);
  }

  ISUB  i, j, *indx;
  GBOOL bRet=TRUE;
  T     **A, *col,  d;
  SETVector<T> *row;

  A = new T   *[n1];
  col = new T [n1];

  for ( i=0; i<n1; i++ )
    A[i] = new T [n1];
  indx = new ISUB  [n1];
 
  mret = 0.0;

  for ( i=0; i<n1; i++ )
  {
    row = data[i];
    for ( j=0; j<n1; j++ )
    {
      A[i][j] = (*row)(j);
    }
    mret(i,i) = 1.0;
  }

  if ( !ludcmp(A, n1, indx, &d) )
  {
     cout << "SETVMatrix::Inverse: ludcmp failed" << endl;
     bRet = FALSE;
  }

#if 0
  SEMatrix LU(n1,n1), L(n1,n1), U(n1,n1);
  for ( i=0; i<n1; i++ )
    for ( j=0; j<n1; j++ )
      LU(i,j) = A[i][j];
  for ( i=0; i<n1; i++ )
    for ( j=i; j<n1; j++ )
      U(i,j) = A[i][j] ;
  for ( i=0; i<n1; i++ )
  {
    for ( j=0; j< i; j++ )
      L(i,j) = A[i][j];
    L(i,i) = 1.0;
  }

  cout << "SETVMatrix: A= " << LU << endl;
  cout << "SETVMatrix: L= " << L << endl;
  cout << "SETVMatrix: U= " << U << endl;
  cout << "SETVMatrix: L*U= " << (L*U) << endl;
  cout << "SETVMatrix: d = " << d << "  indx= " <<  endl;
  for ( i=0; i<n1 && bRet; i++ )
    cout << indx[i] << " ";
  cout << endl;
#endif

  for ( j=0; j<n1 && bRet; j++ )
  {
    for ( i=0; i<n1; i++ )
    {
      col[i] = mret(i,j);
    }
    if ( !(bRet=lubksb(A, n1, indx, col)) ) 
    {
       cout << "SETVMatrix::Inverse: lubjsb failed" << endl;
       bRet = FALSE;
       break;
    }
    for ( i=0; i<n1; i++ )
    {
      mret(i,j) = col[i];
    }
  }

  for ( i=0; i<n1; i++ )
    delete [] A[i];
  delete [] A;
  delete [] col;
  delete [] indx;

  return bRet;

} // end of method Inverse


//************************************************************************************
//************************************************************************************
// METHOD     : Inverse (2)
// DESCRIPTION: computes inverse of this, but
//              does not destroy data. A copy is made and
//              returned.
// RETURNS    : inverse of this
template<class T> SETVMatrix<T>  SETVMatrix<T>::Inverse()
{

  SETVMatrix<T> mi(n1,n2);

  if ( !Inverse(mi) )
  {
    cout << "SETVMatrix<T>::Inverse(2): failed" << endl;
    exit(1);
  }

  return mi;

} // end of method Inverse (2)


//************************************************************************************
//************************************************************************************
// METHOD     : isSymmetric 
// DESCRIPTION: determines if matrix is symmetric
// RETURNS    : TRUE or FALSE 
template<class T> GBOOL  SETVMatrix<T>::isSymmetric()
{

  if ( n1 != n2 ) return FALSE;

  ISUB  i, j;

  // NOTE: should be symmetric w.r.t some tolerance!!!
  for ( i=1; i<n1-1; i++ )
  {
    for ( j=i+1; j<n2; j++ )
    {
       if ( *(data+i*n2+j) != *(data+j*n2+i) ) return FALSE; 
    }
  }
  return TRUE;
 
} // end of method isSymmetric


//************************************************************************************
//************************************************************************************
// METHOD     : GetProcMap
// DESCRIPTION: gets the processor distriubtion mapping type
// RETURNS    : ProcMapping type
template<class T> ProcMapping SETVMatrix<T>::GetProcMapping()
{

  return iProcMap;

} // end of method GetProcMap


//************************************************************************************
//************************************************************************************
// METHOD     : iDistributed
// DESCRIPTION: gets the bDistributed flag
// RETURNS    : GBOOL flag
template<class T> GBOOL  SETVMatrix<T>::isDistributed()
{

  return bDistributed ;

} // end of method isDistributed


//************************************************************************************
//************************************************************************************
// METHOD     : ludcmp
// DESCRIPTION: Taken largely from Numerical Recipes
// RETURNS    : GBOOL flag
template<class T> GBOOL  SETVMatrix<T>::wludcmp(T **&a, ISUB  n, ISUB  *&indx, T *d)
{
  if ( a == NULL || indx == NULL ) return FALSE;

  ISUB  i, imax=n-1, j, k;
  GBOOL bRet=TRUE;
  T     big, dum, sum, temp;
  T     *vv;

  vv = new T [n]; 
  *d=1.0;
  for (i=0;i<n && bRet;i++) {
    big=0.0;
    for (j=0;j<n;j++)
      if ((temp=fabs(a[i][j])) > big) big=temp;
    if (big == 0.0){
      bRet = FALSE; 
      break;
    }
    vv[i]=1.0/big;
  }

  for (j=0;j<n && bRet;j++) {
    for (i=0;i<j;i++) {
      sum=a[i][j];
      for (k=0;k<i;k++) sum -= a[i][k]*a[k][j];
      a[i][j]=sum;
    }
    big=0.0;
    for (i=j;i<n;i++) {
      sum=a[i][j];
      for (k=0;k<j;k++) sum -= a[i][k]*a[k][j];
      a[i][j]=sum;
      if ( (dum=vv[i]*fabs(sum)) >= big) {
        big=dum;
        imax=i;
      }
    }
    if (j != imax) {
      for (k=0;k<n;k++) {
        dum=a[imax][k];
        a[imax][k]=a[j][k];
        a[j][k]=dum;
      }
     *d = -(*d);
     vv[imax]=vv[j];
    }
    indx[j]=imax;
    if (a[j][j] == 0.0) a[j][j]=TINYTINY;
    if (j != n-1) {
      dum=1.0/(a[j][j]);
      for (i=j+1;i<n;i++) a[i][j] *= dum;
    }
  }
  delete [] vv;


  return bRet;

} // end of method wludcmp


//************************************************************************************
//************************************************************************************
// METHOD     : lubksb
// DESCRIPTION: Taken largely from Numerical Recipes
// RETURNS    : GBOOL flag
template<class T> GBOOL  SETVMatrix<T>::lubksb(T **&a, ISUB  nd, ISUB  *&indx, T b[])
{

    ISUB n=nd-1; 
    ISUB i, ii = -1, ip, j;
    T   sum;

    for ( i=0; i <= n; i++ ) {
        ip = indx[i];
        sum = b[ip];
        b[ip] = b[i];
        if (ii > -1) {
            for (j = ii; j < i; j++) sum -= a[i][j]*b[j];
        }
        else { 
            if (sum) ii = i;
        }
        b[i] = sum;
    }
    for ( i=n; i>=0; i-- ) {
        sum=b[i];
        if ( i < n )
        for (j = i+1; j <= n; j++) sum -= a[i][j]*b[j];
        b[i] = sum /(a[i][i]);
    }

  return TRUE;

} // end of method lubksb


//************************************************************************************
//************************************************************************************
// METHOD     : ludcmp
// DESCRIPTION: LU decomposition method provided by
//              Warren Jasper
// RETURNS    : GBOOL flag
template<class T> GBOOL  SETVMatrix<T>::ludcmp(T **&a, ISUB  nd, ISUB  *&indx, T *d)
{
    ISUB  i, imax, j, k;
    ISUB  n=nd-1;
    GBOOL bRet = TRUE;
    T     big, dum, sum, temp;
    T     *vv = new T [nd];

    *d = 1.0;
    imax = -1;

    for ( i = 0; i <= n; i++ ) {
        big = 0.0;
        for ( j = 0; j <= n; j++ )
            if ((temp = fabs( a[i][j] )) > big) big = temp;
            if (big == 0.0) {
              bRet = FALSE;  
              break;
            }
        vv[i] = 1.0 / big;
    }
    for ( j = 0; j <= n; j++ ) {
        for ( i = 0; i < j; i++ ) {
            sum = a[i][j];
                for (k = 0; k < i; k++) sum -= a[i][k]*a[k][j];
                a[i][j] = sum;
        }
        big = 0.0;
        for ( i = j; i <= n; i++ ) {
            sum = a[i][j];
            for ( k = 0; k < j; k++ )
                sum -= a[i][k]*a[k][j];
            a[i][j] = sum;
            if ( (dum = vv[i]*fabs(sum)) >= big) {
                big = dum;
                imax = i;
            }
        }
        if (j != imax) {
            for ( k = 0; k <= n; k++ ) {
                dum = a[imax][k];
                a[imax][k] = a[j][k];
                a[j][k] = dum;
            }
            *d = -(*d);
            vv[imax] = vv[j];
        }
        indx[j] = imax;
        if (a[j][j] == 0.0) a[j][j] = TINYTINY;
        if (j != n) {
            dum = 1.0 / (a[j][j]);
            for (i = j+1; i <= n; i++) a[i][j] *= dum;
        }
    }
    delete vv;

    return bRet;

} // end of method ludcmp

//************************************************************************************
//************************************************************************************
// METHOD     : isamax
// DESCRIPTION: BLAS routine that finds the index of element having max.
//              absolute value.
// ARGUMENTS  : n   : Number of elements to check.
//              sx  : Vector to be checked.
//              incx: Every incx-th element is checked.

// RETURNS    : int
ISUB isamax( ISUB n, GDOUBLE *sx, ISUB incx )
{
  GDOUBLE smax = 0.0e0;
  ISUB  i, istmp = 0;

  if( n <= 1 ) return( istmp );
  if( incx != 1 ) {
    /* Code for increment not equal to 1. */
    if( incx < 0 ) sx = sx + ((-n+1)*incx + 1);
    istmp = 0;
    smax  = fabs( *sx );
    sx += incx;
    for( i=1; i<n; i++, sx+=incx )
      if( fabs( *sx ) > smax ) {
        istmp = i;
        smax  = fabs( *sx );
      }
    return( istmp );
  }
  /* Code for increment equal to 1. */
  istmp = 0;
  smax  = fabs(*sx);
  sx++;
  for( i=1; i<n; i++, sx++ )
    if( fabs( *sx ) > smax ) {
      istmp = i;
      smax  = fabs( *sx );
    }
  return( istmp );
}  // end of method isamax


