//************************************************************************************//
// Module       : gasparFile.cpp
// Date         : 6/11/02 (RMP)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Base class encapsulating the methods and data associated with
//                a Gaspar file.
// Derived From : none.
// Modifications:
//************************************************************************************//  

#include "gasparFile.hpp"

const char* GasparFile::THIS = "GasprFile";                 //Name of this class; used for error message
const char* GasparFile::VSIZE_META_NAME = "VECTOR_SIZE";    //Default name tag for array size data stored in file
const char* GasparFile::MDIM_1_NAME = "MATRIX_DIM_1";       //Default name tag for matrix row size stored in file
const char* GasparFile::MDIM_2_NAME = "MATRIX_DIM_2";       //Default name tag for matrix column size stored in file
const ISUB GasparFile::FILE_DEPTH_DEFAULT = 10;             //Default group nesting depth for the file
const GUSHORT  GasparFile::GROUP_HINT = 100;                 //Default total size hint (in bytes) for all objects to be
                                                            //...be included in a particular group


//************************************************************************************
//************************************************************************************
// METHOD     : setFileName()
// DESCRIPTION: records the name of the file opened by this GasparFile object. 
// ARGUMENTS  : fileNameIn  : The name of the file to store.
// RETURNS    : TRUE on success; else FALSE, if the file name to store is null.
//************************************************************************************
GBOOL GasparFile::setFileName(const char* fileNameIn){ 
  if(!fileNameIn){
    return FALSE;
  }
  fileName = fileNameIn; 
  return TRUE;
}
//end of method setFileName()


//************************************************************************************
//************************************************************************************
// METHOD     : setComms() 1
// DESCRIPTION: Creates two dedicated MPI communicators by duplicating the argument
//              communicator: One for use with HDF functions, and one for use with MPI 
//              functions called by this GasparFile object. 
// ARGUMENTS  : commIn  : The communicator to duplicate.
// RETURNS    : TRUE if both duplications succeed; FALSE otherwise.
//************************************************************************************
GBOOL GasparFile:: setComms(MPI_Comm commIn){
  return (MPI_Comm_dup(commIn, &hdfComm) == MPI_SUCCESS &&
	  MPI_Comm_dup(commIn, &mpiComm) == MPI_SUCCESS);
}
//end of method setComms() 1 


//************************************************************************************
//************************************************************************************
// METHOD     : setComms() 2
// DESCRIPTION: Creates two dedicated MPI communicators: One for use with HDF functions, 
//              and one for use with MPI functions called by this GasparFile object.
//              This method creates nLists MPI communicators, each including those
//              process IDs listed in procIDLists[n] (n from 0 - nLists).  The HDF and
//              MPI communicators mentioned above are then created by duplicating the
//              communicator that includes the current process.
// ARGUMENTS  : parentComm  : The communicator from which to derive the new set of 
//                            communicators (one of which duplicated as this objects HDF 
//                            and MPI Communicators).
//              procIDLists : A 2D array of process IDs in the context of parentComm, 
//                            where procIDLists[n] is a list of the processes to include in 
//                            the nth communicator (n from 0 - nLists). NOTE: procLists must 
//                            exactly contain all processes in parentComm, including each 
//                            only once.
//             nLists       : The number of process lists from which to create a communicator
//             listSizes    : An array of lengths, where listSizes[n] is the length of
//                            procIDLists[n].
// PRECONDS  : parentComm must be a valid MPI communicator and procIDLists must include
//             all processes in parentComm exactly once.       
// RETURNS   : TRUE on success; FALSE otherwise.
//************************************************************************************
GBOOL GasparFile::setComms(MPI_Comm parentComm, ISUB** procIDLists, ISUB nLists, ISUB* listSizes){

  ISUB rank, parentNProcs, i, j, procsAssigned = 0, status;
  MPI_Group parentGroup, newGroup;

  MPI_Comm_rank(parentComm, &rank);             //Get the rank of this process in the parent comm
  MPI_Comm_size(parentComm, &parentNProcs);     //Get the number of processes in the parent comm

  MPI_Comm_group(MPI_COMM_WORLD, &parentGroup); //Get the group of processes in the parent comm

  for(i=0; i<nLists; i++){                     //For each process list...
    for(j=0; j<listSizes[i]; j++){             //For each process in the current list...
      if(procIDLists[i][j] > parentNProcs){
	cerr << rank << ": " << THIS << "::setMPIComm() failed.  Specified proc i.d # " << 
	  procIDLists[i][j] << "is not a valid proc. id in the argument communicator" << endl;
	return FALSE;
      }
      if(rank == procIDLists[i][j]){          //If the current process in procIDLists is this process...
	
	if(MPI_Group_incl(parentGroup, listSizes[i], procIDLists[i], &newGroup) != MPI_SUCCESS){
	  cerr << rank << ": " << THIS << "::setMPIComm()::MPI_Group_incl() failed " << endl;
	  return FALSE;
	}                                     //Then include it in the communicator to create 
      }
      procsAssigned++;
    }
  }
  
                                             //Check to be sure that procListIDs contained exactly, only
  if(procsAssigned != parentNProcs){         //...the processes in parentComm
    cerr << rank << ": " << THIS << "::setMPIComm() failed." <<
      "  Number of procs in requested communicator doesn't equal those in argument communicator" << endl;
    return FALSE;
  }
                                            //Duplicate the new communicator to create the HDF and MPI
                                            //...communicators for this GasparFile object
  return (MPI_Comm_create(parentComm, newGroup, &hdfComm) == MPI_SUCCESS &&
	  MPI_Comm_dup(hdfComm, &mpiComm) == MPI_SUCCESS);
  
}
//end of method setComms() 2
