//************************************************************************************//
// Module       : gasparFileWriter.cpp
// Date         : 6/11/02 (RMP)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                Gaspar file output.
// Derived From : GasparFileIO.
// Modifications:
//************************************************************************************//

#include "gasparFileWriter.hpp"

const char* GasparFileWriter::THIS = "GasparFileWriter";


//************************************************************************************
//************************************************************************************
// METHOD     : GasparFileWriter Constructor
// DESCRIPTION: Initializes a new GasparFileWriter object with a reference to the MPI
//              communicator.
//************************************************************************************
 
GasparFileWriter::GasparFileWriter(){
  setOpenStatus(FALSE);
}
//end of GasparFileWriter Constructor


//************************************************************************************
//************************************************************************************
// METHOD     : GasparFileWriter Destructor
// DESCRIPTION: Releases all resources associated with this writer.. 
//************************************************************************************

GasparFileWriter::~GasparFileWriter(){
  if(isOpen()){
    closeFile();
  }

  if(getFile() != NULL){
    delete getFile();
    setFile(NULL);
  }
}
//end GasparFileWriter Destructor method


//************************************************************************************
//************************************************************************************
// METHOD     : createFile()
// DESCRIPTION: Creates a new HDF file and opens it for writing. 
// ARGUMENTS  : fileNameIn  : The name of the file to open.
//              overWrite   : Flag indicating whether to overwrite existing files of
//                            the same name in the current directory or to fail in their
//                            presence.
// RETURNS    : TRUE on success; else FALSE, if a file with the same name as the file to
//              open exists in the current directory and the overWrite flag is set to FALSE.
//************************************************************************************

GBOOL GasparFileWriter::createFile(GasparFile* inFile, GBOOL overWrite){
  setFile(inFile);
  inFile = NULL;
  if(!getFile()->openCreate(overWrite)){
    cerr << THIS << "::createFile() failed." << endl;
    return FALSE;
  }

  setOpenStatus(TRUE);
  return TRUE;
}
//end of method createFile()

//************************************************************************************
//************************************************************************************
// METHOD     : createHDF()
// DESCRIPTION: Creates a new HDF file and opens it for writing. 
// ARGUMENTS  : fileNameIn  : The name of the file to open.
//              overWrite   : Flag indicating whether to overwrite existing files of
//                            the same name in the current directory or to fail in their
//                            presence.
// RETURNS    : TRUE on success; else FALSE, if a file with the same name as the file to
//              open exists in the current directory and the overWrite flag is set to FALSE.
//************************************************************************************

GBOOL GasparFileWriter::createHDF(const char* fileNameIn, GBOOL overWrite, IO_TYPE fType, IO_TYPE dType, MPI_Comm commIn){
  setFile(new HDFFile(fileNameIn, fType, dType, commIn));
  
  if(!getFile()->openCreate(overWrite)){
    cerr << THIS << "::createHDF() failed." << endl;
    return FALSE;
  }

  setOpenStatus(TRUE);
  return TRUE;
}
//end of method createHDF()


//************************************************************************************
//************************************************************************************
// METHOD     : write() 1
// DESCRIPTION: Extracts the appropriate data from the SEVector argument and writes it
//              along with supporting metadata to the file opened/created with a call
//              to openXXX() or createXXX().
// ARGUMENTS  : vector     : The SEVector object from which to extract data to be written.
//              vectorName : The name of the SEVector data used to identify it in the
//                           file (written as metadata).
// RETURNS    : TRUE on success; else FALSE, if the data could not be written to the file
//              (ex. if vector is NULL).
// PRECOND    : A file has been opened or created with a successful call to one of the
//                openXXX() or createXXX() methods.
//************************************************************************************

GBOOL GasparFileWriter::write(const SEVector& vector, const char* vectorName){
  errorMsg = "::write() failed.";

  if(!isOpen()){
    cerr << THIS << errorMsg  << endl;
    return FALSE;
  }

  GDOUBLE* data = vector.Data();  //Extract the vector data array and
  ISUB dim = vector.dim();      //...array size.
    
  if(!getFile()->writeVector(vectorName, data, dim)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }      

  return TRUE;
}
//end of write() method 1


//************************************************************************************
//************************************************************************************
// METHOD     : write() 2
// DESCRIPTION: Extracts the appropriate data from the SEVector argument and writes it
//              along with supporting metadata to the file opened/created with a call
//              to openXXX() or createXXX().
// ARGUMENTS  : vector     : The SEVector object from which to extract data to be written.
//              vectorName : The name of the SEVector data used to identify it in the
//                           file (written as metadata).
// RETURNS    : TRUE on success; else FALSE, if the data could not be written to the file
//              (ex. if vector is NULL).
// PRECOND    : A file has been opened or created with a successful call to one of the
//                openXXX() or createXXX() methods.
//************************************************************************************

GBOOL GasparFileWriter::write(const SEMatrix& matrix, const char* matrixName){
  errorMsg = "::write(SEMatrix...) failed.";

  if(!isOpen()){
    cerr << THIS << errorMsg << "  This reader object is not open." << endl;
    return FALSE;
  }

  SEVector** vecs = matrix.Data();         //Extract the matrix SEVector array 
  ISUB dim1 = matrix.dim(1);               //and matrix dimensions
  ISUB dim2 = matrix.dim(2);
  ISUB i, maxVDim = -1;

  GDOUBLE* data = (GDOUBLE*)malloc(sizeof(GDOUBLE)*dim1*dim2);
  for(i=0; i<dim1; i++){
    memcpy(data+(i*dim2), vecs[i]->Data(), dim2*sizeof(GDOUBLE));
  }

  pass = TRUE;

  if(!getFile()->writeMatrix(matrixName, data, dim1, dim2)){
    cerr << THIS << errorMsg << endl;
    pass =  FALSE;
  } 

  delete[] data;
  data = NULL;

  return pass;
}

//end of method write() 2








