//************************************************************************************//
// Module       : hdfFile.cpp
// Date         : 8/04/02 (RMP)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Derived class encapsulating the methods and data associated with
//                an HDF-format Gaspar file.
// Derived From : GasparFile (gasparFile.hpp)
// Modifications:
//************************************************************************************//

#include "hdfFile.hpp"
#include <typeinfo>

const ISUB HDFFile::TOP = -1;                //The cursorD start position indicating the top of the file
const char* HDFFile::THIS = "HDFFile";       //The name of this class; used in error reporting


//************************************************************************************
//************************************************************************************
// METHOD     : HDFFile Constructor 1
// DESCRIPTION: Initializes a new HDFFile object with the fileName argument, data
// I/O type (dType), and HDF & MPI communicators generated using procLists, nLists, 
// listsSizes & commIn (SEE the documentation for setComms() 2 to see how procLists, 
// nLists, listSizes, and commIn are used to generate the 2 communicators for this object.
// ARGUMENTS  : fileNameIn  : The name of the file to open/create.
//              procLists   : Passed to setComms() 2.  See setComms() 2 documentation.
//              nListst     : Passed to setComms() 2.  See setComms() 2 documentation.  
//              listSizes   : Passed to setComms() 2.  See setComms() 2 documentation.
//              dType       : The data transfer I/O type to use with this file (either
//                            collective - all procs writing common data, or independent - 
//                            each proc writing independent data.
//              commIn      : Passed to setComms() 2.  See setComms() 2 documentation.
//************************************************************************************
HDFFile::HDFFile(const char* fileNameIn, ISUB** procLists, ISUB nLists, ISUB* listSizes, IO_TYPE dType, MPI_Comm commIn){
  errorMsg = "::HDFFile() failed().";

  if(!setFileName(fileNameIn)){
    cerr << THIS << errorMsg << " No file name specified" << endl;    
  }  

  dataIO = dType;

  if(dataIO != COLLECTIVE_IO && dataIO != INDEPENDENT_IO){
    cerr << THIS << errorMsg << "  Invalid data I/O type argument. Using Independent" << endl;
    dataIO = INDEPENDENT_IO;
  }

  depth = 0;
  setOpenStatus(FALSE);
  setComms(commIn, procLists, nLists, listSizes);
  MPI_Comm_size(getHDFComm(), &nProcs);
  MPI_Comm_rank(getHDFComm(), &rank);
  breadth = (dataIO == INDEPENDENT_IO ? nProcs : ONED);

  //H5Eset_auto(NULL, NULL);       //Disable HDF5 error handling
}
//end of constructor 1


//************************************************************************************
//************************************************************************************
// METHOD     : HDFFile Constructor 2
// DESCRIPTION: Initializes a new HDFFile object with the fileName, file I/O
// type, data I/O type and communicator arguments.  
// ARGUMENTS  : fileNameIn  : The name of the file to open/create.
//              fType       : The file access I/O type to use with this file (either
//                            collective - all processes access this file, or 
//                            independent - each process accesses its own version of 
//                            this file.
//              dType       : The data transfer I/O type to use with this file (either
//                            collective - all procs writing common data, or independent - 
//                            each proc writing independent data.
//              commIn      : The communicator from which this objects HDF and MPI
//                            communicators are created (exact duplicates).
// PRECONDS:    commIn must be a valid MPI communicator.
//************************************************************************************
HDFFile::HDFFile(const char* fileNameIn, IO_TYPE fType, IO_TYPE dType, MPI_Comm commIn){
  errorMsg = "::HDFFile() failed.";

  if(!setFileName(fileNameIn)){
    cerr << THIS << errorMsg << " No file name specified." << endl;    
  }  

  setOpenStatus(FALSE); 

  if(dType != COLLECTIVE_IO && dType != INDEPENDENT_IO){
    cerr << THIS << errorMsg << "  Invalid data I/O type argument. Using Independent" << endl;
    dataIO = INDEPENDENT_IO;
  }

  depth = 0;
  ISUB* listSizes;
  ISUB** procLists;
  ISUB next = 0;
  
  switch(fType){
  case INDEPENDENT_IO:          //If file access I/O type is independent, create a 1-process
                                //communicator for each process
    dataIO = COLLECTIVE_IO;
    ISUB commSize, i;
    MPI_Comm_size(commIn, &commSize);

    procLists = new ISUB* [commSize];
    listSizes = new ISUB[commSize];

    for(i=0; i<commSize; i++){
      procLists[i] = new ISUB[INDEPENDENT_IO];
      procLists[i][0] = next++;
      listSizes[i] = INDEPENDENT_IO;
    }

    pass = setComms(commIn, procLists, commSize, listSizes);

    for(i=0; i<commSize; i++){
      delete [] procLists[i];
    }

    delete [] procLists;
    procLists = NULL;
    
    delete [] listSizes;
    listSizes = NULL;

    if(pass == FALSE){
      cout << THIS << "::" << THIS << "()::setMPICommunicator() failed. Using all available processes to open file." << endl;
      setComms(commIn);
    }
    break;

  case COLLECTIVE_IO:
    dataIO = dType;

    if(dataIO != COLLECTIVE_IO && dataIO != INDEPENDENT_IO){
      cerr << THIS << "::" << THIS << "() - Invalid data I/O type argument. Using Independent" << endl;
      dataIO = INDEPENDENT_IO;
    }

    setComms(commIn);
    break;
  }
  MPI_Comm_size(getHDFComm(), &nProcs);                   //Record the number of processes and pid of the current
  MPI_Comm_rank(getHDFComm(), &rank);                     //process
  breadth = (dataIO == INDEPENDENT_IO ? nProcs : ONED);

  //H5Eset_auto(NULL, NULL);       //Disable HDF5 error handling
}                                         
//end of HDFFile() constructor 2


//************************************************************************************
//************************************************************************************
// METHOD     : HDFFile Destructor
// DESCRIPTION: Releases dynamic resources associated with this HDFFile object.
//************************************************************************************
HDFFile::~HDFFile(){
  if(isOpen()){
    if(!close()){
      cerr << THIS << "::~HDFFile()::close() failed." << endl;
    }
  }

  if(path){
    cleanPath();
  }
  
}
//end of HDFFile Destructor


//************************************************************************************
//************************************************************************************
// METHOD     : open()
// DESCRIPTION: opens an existing HDF file for reading &/or writing. 
// ARGUMENTS  : accessIn      : The access type for this file.  Choices are:
//                              1)READ_ONLY - only permit this file to be read
//       		        2)READ_WRITE - allow both reads and writes to this file
//              fileNestDepth : The depth of group nesting within this file. Exceeding
//                              depth when usin this file results in the extra overhead
//                              of path array reallocation.
// RETURNS    : TRUE on success; else FALSE, if the file to open does not exit in the
//              current directorya or cannot be opened for any reason.
//************************************************************************************
GBOOL HDFFile::open(FILE_ACCESS accessIn, ISUB fileNestDepth){
  errorMsg = "::open failed.";

  if(isOpen()){
    cerr << THIS << errorMsg << " This File object is already open" << endl;
    return FALSE;
  }

  if(!checkHDF(H5Fis_hdf5(getFileName()), errorMsg)){
    return FALSE;
  }

  depth = fileNestDepth;
  if(depth < 2){
    cerr << THIS << errorMsg << " Insufficient file depth."  << endl;
    return FALSE;
  }

  if(!setFilePlist()){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }

  cursorD = TOP;

  setFileAccess(accessIn);

  switch(accessIn){
  case READ_ONLY:
    fileID = H5Fopen(getFileName(), H5F_ACC_RDONLY, filePlistID);
    break;
  case READ_WRITE:
    fileID = H5Fopen(getFileName(), H5F_ACC_RDWR, filePlistID);
    break;               
  default:
    cerr << THIS << errorMsg << " Uknown file access argument." << endl;
    return FALSE;
  }

  if(!checkHDF(fileID)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }

  status = H5Pclose(filePlistID);
  if(!checkHDF(status, errorMsg)){
    return FALSE;
  }

  path = new HDFID*[depth];
  for(ISUB i=0; i<depth; i++){
    path[i] = new HDFID[breadth];
  }  

  setOpenStatus(TRUE);

  return TRUE;
}
//end of method  open()



//************************************************************************************
//************************************************************************************
// METHOD     : openCreate()
// DESCRIPTION: Creates a new HDF file and opens it in READ_WRITE mode.
// ARGUMENTS  : overWrite     : Flag indicating whether to overwrite existing files of
//                              the same name in the current director.
//            : fileNestDepth : The depth of group nesting within this file. Exceeding
//                              depth when usin this file results in the extra overhead
//                              of path array reallocation.
// RETURNS    : TRUE on success; else FALSE if the overWrite flag is set to false and
//              a file exists with the name fileNameIn in the current directory.
//************************************************************************************

GBOOL HDFFile::openCreate(GBOOL overWriteIn, ISUB fileNestDepth){
  errorMsg = "::openCreate failed().";

  if(isOpen()){
    cerr << THIS << errorMsg << " This File object is already open" << endl;
    return FALSE;
  }

  depth = fileNestDepth;
  if(depth < 2){
    cerr << THIS << errorMsg << " Insufficient file nest depth" << endl;
    return FALSE;
  }

  if(!setFilePlist()){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }
 
  cursorD = TOP;

  setFileAccess(READ_WRITE);

  //Set HDF flag to either overwrite existing files, or fail in their presence.
  GUSHORT  overWrite = (overWriteIn ? H5F_ACC_TRUNC : H5F_ACC_EXCL);

  fileID = H5Fcreate(getFileName(), overWrite, H5P_FILE_CREATE, filePlistID); 

  if(!checkHDF(fileID, errorMsg)){
    return FALSE;
  }

  status = H5Pclose(filePlistID);
  if(!checkHDF(status, errorMsg)){
    return FALSE;
  }

  path = new HDFID*[depth];        //Initialize the group nesting object ID path
  for(ISUB i=0; i<depth; i++){     //for this file
    path[i] = new HDFID[breadth];
  }

  setOpenStatus(TRUE);

  return TRUE;
}
//end of method openCreate();
 

//************************************************************************************
//************************************************************************************
// METHOD     : openGroup()
// DESCRIPTION: Opens an existing HDF group identified by 'groupName' within an existing
//              and open HDF file.
// ARGUMENTS  : groupName  : The name identifying the group within the file. 
// RETURNS    : TRUE on success; else FALSE if the group does not exist within the
//              file or cannot be opened for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate().
//************************************************************************************

GBOOL HDFFile::openGroup(const char* groupName){
  errorMsg = "::openGroup failed.";

  if(!isOpen()){
    cerr << THIS << errorMsg << "The file is not open." << endl;
    return FALSE;
  }

  nameBuffer = NULL;

  if(dataIO == INDEPENDENT_IO && breadth > 1){

    if(!gatherNames(groupName, nameBuffer, nbOffset)){
      cerr << THIS << errorMsg << endl;
      return FALSE;
    }
  }

  const char* name;
  pass = TRUE;

  for(ISUB i=0; i<breadth; i++){         

    name = (dataIO == INDEPENDENT_IO && breadth > 1 ? nameBuffer+(i*nbOffset) : groupName);
    if(cursorD == TOP){
      path[cursorD+1][i] = H5Gopen(fileID, name);
      pass = pass && checkHDF(path[cursorD+1][i], errorMsg); 
    }
    else{
      path[cursorD+1][i] = H5Gopen(path[cursorD][i], name);    
      pass = pass && checkHDF(path[cursorD+1][i], errorMsg);
    }
    if(!pass){
      cerr << THIS << errorMsg << endl;
    }
  }

  if(pass){
    cursorD++;
  }
  else{
    cerr << THIS << errorMsg << endl;
  }

  if(nameBuffer){
    free(nameBuffer);
    nameBuffer = NULL;
  }

  return pass;
}
//end of method openGroup()


//************************************************************************************
//************************************************************************************
// METHOD     : closeGroup()
// DESCRIPTION: Closes an HDF group which has been opened at the current position in
//              the file (path[cursor]) with a call to openGroup() or createGroup()
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE if the group does not exist at the current
//              position in the file or cannot be closed for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate()
//              and the group to be closed has been opened with a call to openGroup()
//              or createGroup().
//************************************************************************************

GBOOL HDFFile::closeGroup(){
  errorMsg = "::closeGroup() failed.";

  if(!isOpen()){
    cerr << THIS << errorMsg << " This file object is not open" << endl;
    return FALSE;
  }

  if(cursorD == TOP){
    cerr << THIS << errorMsg << " Already at top of file." << endl;
    return FALSE;
  }
  
  pass = TRUE;
  for(ISUB i=0; i<breadth; i++){
    pass = pass && checkHDF(H5Gclose(path[cursorD][i]));
  }
  if(!pass){
    cerr << THIS << errorMsg << endl;
  }
  cursorD--;
  return pass;
}
//end of method closeGroup()


//************************************************************************************
//************************************************************************************
// METHOD     : up()
// DESCRIPTION: Closes an HDF group which has been opened at the current position in
//              the file (path[cursor]) with a call to openGroup() or createGroup()
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE if the group does not exist at the current
//              position in the file or cannot be closed for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate()
//              and the group to be closed has been opened with a call to openGroup()
//              or createGroup().
//************************************************************************************
GBOOL HDFFile::up(){
  errorMsg = "::up() failed.";

  if(cursorD == TOP){
    cerr << THIS << errorMsg << "  Already at top of file." << endl;
    return FALSE;
  }

  if(!closeGroup()){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }
  return TRUE;
}
//end of method up()


//************************************************************************************
//************************************************************************************
// METHOD     : top()
// DESCRIPTION: Closes an HDF group which has been opened at the current position in
//              the file (path[cursor]) with a call to openGroup() or createGroup()
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE if the group does not exist at the current
//              position in the file or cannot be closed for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate()
//              and the group to be closed has been opened with a call to openGroup()
//              or createGroup().
//************************************************************************************

GBOOL HDFFile::top(){
  if(cursorD == TOP){
    return TRUE;
  }
 
  pass = TRUE;
  for(ISUB i=cursorD; i>TOP; i--){
    pass = pass && closeGroup();
  }

  if(!pass){
    cerr << THIS << "::top() failed." << endl;
  }
  return pass;
}
//end of method top()


//************************************************************************************
//************************************************************************************
// METHOD     : createGroup()
// DESCRIPTION: Closes an HDF group which has been opened at the current position in
//              the file (path[cursor]) with a call to openGroup() or createGroup()
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE if the group does not exist at the current
//              position in the file or cannot be closed for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate()
//              and the group to be closed has been opened with a call to openGroup()
//              or createGroup().
//************************************************************************************

GBOOL HDFFile::createGroup(const char* groupName, GUSHORT  hint){
  errorMsg = "::createGroup() failed.";

  if(!isOpen()){
    cerr << THIS << errorMsg << " File not open." << endl;
    return FALSE;
  }

  if(getFileAccess() != READ_WRITE){
    cerr << THIS << errorMsg << " read-only." << endl;
    return FALSE;
  }

  nameBuffer = NULL;

  if(dataIO == INDEPENDENT_IO && breadth > 1){
    if(!gatherNames(groupName, nameBuffer, nbOffset)){
      cerr << THIS << errorMsg << endl;
      return FALSE;
    }
  }

  HDFID parent = fileID;
  const char* name;
  
  for(ISUB i=0; i<breadth; i++){

    if(cursorD != TOP){
      parent = path[cursorD][i];
    }
    name = (dataIO == INDEPENDENT_IO ? nameBuffer+(i*nbOffset) : groupName);
    //if((path[cursorD][i] = H5Gopen(parent, localGroupName)) >= HDF_PASS){ 
      if(!checkHDF(path[cursorD+1][i] = H5Gcreate(parent, name, hint), errorMsg)){
	if(nameBuffer){
	  free(nameBuffer);
	  nameBuffer = NULL;
	}
	return FALSE;
      }
      //}
  }
  cursorD++;

  if(nameBuffer){
    free(nameBuffer);
    nameBuffer = NULL;
  }
  return TRUE;
}
//end of method createGroup()

 
//************************************************************************************
//************************************************************************************
// METHOD     : close()
// DESCRIPTION: closes and releases the resources associated with this HDF file object.
// RETURNS    : TRUE on success; else FALSE if the file cannot be closed.
//************************************************************************************

GBOOL HDFFile::close(){
  errorMsg = "::close() failed.";

  if(!isOpen()){
    return TRUE;
  }
  
  pass = TRUE; 

  if(!top()){
    cerr << THIS << errorMsg << endl;
    pass = FALSE;
  }

  cursorD = TOP;
  
  cleanPath();

  pass = pass && checkHDF(H5Fclose(fileID), errorMsg);
  setOpenStatus(!pass);

  return pass;
}
//end of method close()


//************************************************************************************
//************************************************************************************
// METHOD     : setFilePlist()
// DESCRIPTION: Initializes the HDF io properties list for parallel data file transfer.
// RETURNS    : TRUE on success; else FALSE if the HDF parallel file IO
//              properties cannot be initialized.
//************************************************************************************

GBOOL HDFFile::setFilePlist(){
  errorMsg = "::setFilePlist() failed.";

  //Set up file access property list with parallel I/O access
  filePlistID = H5Pcreate(H5P_FILE_ACCESS);
  if(!checkHDF(filePlistID, errorMsg)){
    return FALSE;
  }
  
  status = H5Pset_fapl_mpio(filePlistID, getMPIComm(), MPI_INFO_NULL);
  if(!checkHDF(status, errorMsg)){
    return FALSE;
  }
  return TRUE;
}
//end of method setFilePlist() 


//************************************************************************************
//************************************************************************************
// METHOD     : writMeta() 
// DESCRIPTION: Writes an ISUB data type meta data attribute to the current group in
//              the file
// ARGUMENTS  : metaName  : The name that will be used to identify the meta data 
//                          attribute in the file.
//              metaData  : The ISUB data attribute to write to the current group.
// RETURNS    : TRUE on success; else FALSE if the current position in the file is 
//              not a group or if the meta data cannot be written for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate()
//              and the group to write to has been opened with a call to openGroup()
//              or createGroup().
//************************************************************************************

GBOOL HDFFile::writeMeta(const char* metaName, HDFID location, GDOUBLE meta){

  HDFID metaSpaceID = H5Screate(H5S_SCALAR);
  HDFID metaID = H5Acreate(location, metaName, H5T_NATIVE_DOUBLE, metaSpaceID, H5P_DEFAULT);
  pass = checkHDF(metaID, "::writeMeta() failed.");

  pass = pass && checkHDF(H5Awrite(metaID, H5T_NATIVE_DOUBLE, &meta));
  
  H5Aclose(metaID);
  
  return pass;
}
//end of method writeMeta() 1


//************************************************************************************
//************************************************************************************
// METHOD     : writMeta() 2
// DESCRIPTION: Writes an GDOUBLE * data type meta data attribute to the current group in
//              the file
// ARGUMENTS  : metaName  : The name that will be used to identify the meta data
//                          attribute in the file.
//              location  : H5 ID of the object to which to attach meta data.
//              meta      : The GDOUBLE array data attribute to write to the current group.
//              len       : Length of the 'meta' array
// RETURNS    : TRUE on success; else FALSE if the current position in the file is
//              not a group or if the meta data cannot be written for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate()
//              and the group to write to has been opened with a call to openGroup()
//              or createGroup().
//************************************************************************************

GBOOL HDFFile::writeMeta(const char* metaName, HDFID location, GDOUBLE *meta, GSHORT  len){

  HDFID metaSpaceID = H5Screate_simple(1,&len,&len);
  HDFID metaID = H5Acreate(location, metaName, H5T_NATIVE_DOUBLE, metaSpaceID, H5P_DEFAULT);
  pass = checkHDF(metaID, "::writeMeta() failed.");

  pass = pass && checkHDF(H5Awrite(metaID, H5T_NATIVE_DOUBLE, meta));

  H5Aclose(metaID);

  return pass;
}
//end of method writeMeta() 2


GBOOL HDFFile::writeMeta(const char* metaName, HDFID location, ISUB meta){
  return writeMeta(metaName, location, GDOUBLE(meta));
}
//end of method writeMeta() 3


GBOOL HDFFile::readMeta(const char* metaName, HDFID location, GDOUBLE& meta){
  
  errorMsg = "::readMeta() failed.";

  HDFID metaID = H5Aopen_name(location, metaName);
  if(!checkHDF(metaID, errorMsg)){
    return FALSE;
  }

  if(!checkHDF(H5Aread(metaID, H5T_NATIVE_DOUBLE, &meta), errorMsg)){
    return FALSE;
  }

  H5Aclose(metaID);
  
  return TRUE;
}
//end of method readMeta() 1


GBOOL HDFFile::readMeta(const char* metaName, HDFID location, ISUB& meta){
  GDOUBLE metaTemp;
  pass = readMeta(metaName, location, metaTemp);
  meta = ISUB(metaTemp);
  return pass;
}
//end of method readMeta() 2


//************************************************************************************
//************************************************************************************
// METHOD     : writeTVector()
// DESCRIPTION: Writes an T typed array to the current group in the file.
// NOTE       : **************T MUST EITHER BE GDOUBLE OR ISUB TYPE*********************
// ARGUMENTS  : dataName  : The name that will be used to identify the array in 
//                          the file.
//              data      : The GDOUBLE array to write to the current group.
//              dataSize  : The length of the array to write to the current group.
// RETURNS    : TRUE on success; else FALSE if the current position in the file is 
//              not a group or if the array cannot be written for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate()
//              and the group to write to has been opened with a call to openGroup()
//              or createGroup().
//************************************************************************************

template<class T>
GBOOL HDFFile::writeTVector(const char* dataName, const T* data, ISUB dataSize){
  
  errorMsg = "::writeTVector() failed.";
  GBOOL atTop = FALSE;
  nameBuffer = NULL;

  if(!isOpen()){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  } 

  if(dataSize <= 0){
    cerr << THIS << errorMsg << " Vector dimension argument must be greater than 0" << endl;
    return FALSE;
  }
   
  if(!createGroup(dataName)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }

  HDFID fileSpaceID, memSpaceID, dataID;
  HDFID* dataIDs = NULL;
  HDFID* fileCursor = NULL;
  HDFHSIZE dims[ONED];
  ISUB remainder = 0, length = 0, degree = ONED, i;
  ISUB* lengths;

  HDFHSSIZE offset[ONED];

  //decode T to GDOUBLE or ISUB
  HDFID tType = (typeid(T) == typeid(GDOUBLE) ? H5T_NATIVE_DOUBLE : H5T_NATIVE_INT);

  if(!setDataPlist(dataIO, CREATE)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }

  switch(dataIO){

  case INDEPENDENT_IO: 

    lengths = new ISUB[nProcs];
    pass = (MPI_Allgather(&dataSize, 1, MPI_INT, lengths, 1, MPI_INT, getMPIComm()) == MPI_SUCCESS);

    dataIDs = new HDFID[breadth]; 
    fileCursor = path[cursorD];

    dims[0] = HDFHSIZE(max(lengths, nProcs));

    fileSpaceID = H5Screate_simple(degree, dims, NULL);
    pass = pass && checkHDF(fileSpaceID, errorMsg);
    
    pass = pass && gatherNames(dataName, nameBuffer, nbOffset);

    for(i=0; i<breadth; i++){  
      dataIDs[i] = H5Dcreate(fileCursor[i], nameBuffer+(i*nbOffset), tType, fileSpaceID, dataPlistID);
      if(!checkHDF(dataIDs[i], errorMsg)){
	for(ISUB j=0; j<i; j++){
	  H5Dclose(dataIDs[j]);
	}
	delete[] lengths;
	delete[] dataIDs;
	free(nameBuffer);
	return FALSE;
      }
      pass = pass && writeMeta(VSIZE_META_NAME, dataIDs[i], lengths[i]);
    }  
    dataID = dataIDs[rank];

    H5Sclose(fileSpaceID);

    memSpaceID = H5Screate_simple(degree, dims, NULL);
    fileSpaceID = H5Dget_space(dataID);
    break;

  case COLLECTIVE_IO:

    dims[0] = HDFHSIZE(dataSize);

    fileSpaceID = H5Screate_simple(degree, dims, NULL);
    if(!checkHDF(fileSpaceID, errorMsg)){
      return FALSE;
    }

    dataID = H5Dcreate(path[cursorD][0], dataName, tType, fileSpaceID, dataPlistID);
    pass = pass && checkHDF(dataID, errorMsg);

    pass = pass && writeMeta(VSIZE_META_NAME, dataID, dataSize);

    H5Sclose(fileSpaceID);

    length = dataSize / nProcs;
    remainder = dataSize % nProcs;

    if(length == 0){
      dims[0] = HDFHSIZE(1);
      offset[0] = HDFHSSIZE(rank%dataSize);
    }
    else if(remainder == 0){
      dims[0] = HDFHSIZE(length);  
      offset[0] = HDFHSSIZE(rank*dims[0]);
    }
    else if(rank < remainder){
      dims[0] = HDFHSIZE(length+1); 
      offset[0] = HDFHSSIZE(rank*dims[0]);
    }
    else{
      dims[0] = HDFHSIZE(length);      
      offset[0] = HDFHSSIZE(remainder*(length+1) + (rank-remainder)*dims[0]);
    }

    fileSpaceID = H5Dget_space(dataID);
    memSpaceID = H5Screate_simple(degree, dims, NULL);
    H5Sselect_hyperslab(fileSpaceID, H5S_SELECT_SET, offset, NULL, dims, NULL);

    break;

  default:
    cerr << THIS << errorMsg << endl;
    return FALSE;
    break;
  }

  pass = checkHDF(H5Pclose(dataPlistID), errorMsg);

  if(!setDataPlist(dataIO, XFER)){
    cerr << THIS << errorMsg << endl;
    pass = FALSE;
  }

  status = H5Dwrite(dataID, tType, memSpaceID, fileSpaceID, dataPlistID, data);
  pass = pass && checkHDF(status, errorMsg); 
  
  pass = pass && checkHDF(H5Sclose(memSpaceID), errorMsg);
  pass = pass && checkHDF(H5Sclose(fileSpaceID), errorMsg);

  if(dataIDs){
    for(i=0; i<breadth; i++){
      pass = pass && checkHDF(H5Dclose(dataIDs[i]), errorMsg);
    }
    delete[] dataIDs;
  }
  else{
    pass = pass && checkHDF(H5Dclose(dataID), errorMsg);
  }
  
  pass = pass && checkHDF(H5Pclose(dataPlistID), errorMsg);
  pass = pass && closeGroup();

  if(lengths){
    delete[] lengths;
  }

  if(nameBuffer){
    free(nameBuffer);
    nameBuffer = NULL;
  }

  return pass;
}
//end of method writeTVector() 


template<class T>
GBOOL HDFFile::readTVector(const char* dataName, T*& data, ISUB& dataSize){

  errorMsg = "::reaTVector() failed.";
  GBOOL atTop = FALSE;
  nameBuffer = NULL;           

  if(!isOpen()){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  } 

  if(!openGroup(dataName)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }

  HDFID fileSpaceID, dataID, memSpaceID;
  HDFID* dataIDs = NULL;
  HDFID* fileCursor = NULL;
  HDFHSIZE dims[ONED];
  ISUB remainder = 0, length = 0, degree = ONED, i;

  HDFHSSIZE offset[ONED];
  pass = TRUE;

  //decode T to GDOUBLE or ISUB
  HDFID tType = (typeid(T) == typeid(GDOUBLE) ? H5T_NATIVE_DOUBLE : H5T_NATIVE_INT);

  switch(dataIO){

  case INDEPENDENT_IO:
    
    pass = gatherNames(dataName, nameBuffer, nbOffset);

    dataIDs = new HDFID[breadth];
    fileCursor = path[cursorD];
    for(i=0; i<breadth; i++){
      dataIDs[i] = H5Dopen(fileCursor[i], nameBuffer+(i*nbOffset));
      if(!checkHDF(dataIDs[i], errorMsg)){
	for(ISUB j=0; j<i; j++){
	  H5Dclose(dataIDs[j]);
	  delete[] dataIDs;
	  if(nameBuffer){ free(nameBuffer); }
	  return FALSE;
	}
      }
    }
    dataID = dataIDs[rank];

    pass = pass && readMeta(VSIZE_META_NAME, dataID, dataSize);
    
    dims[0] = HDFHSIZE(dataSize);
    offset[0] = 0;
    fileSpaceID = H5Dget_space(dataID);   
    memSpaceID = H5Screate_simple(degree, dims, NULL);
    H5Sselect_hyperslab(fileSpaceID, H5S_SELECT_SET, offset, NULL, dims, NULL);

    break;

  case COLLECTIVE_IO:
    dataID = H5Dopen(path[cursorD][0], dataName);
    pass = pass && checkHDF(dataID, errorMsg);

    pass = pass && readMeta(VSIZE_META_NAME, dataID, dataSize);
    
    //fileSpaceID = H5Screate_simple(degree, dims, NULL);
    fileSpaceID = H5Dget_space(dataID);
    if(!checkHDF(fileSpaceID, errorMsg)){
      return FALSE;
    }

    //dims[0] = HDFHSIZE(dataSize);
    memSpaceID = H5S_ALL;
    break;
   
  default:
    cerr << THIS << errorMsg << endl;
    return FALSE;
    break;
  }    

  if(!setDataPlist(dataIO, XFER)){
    cerr << THIS << errorMsg << endl;
    pass = FALSE;
  }
  
  data = new T[dataSize];

  pass = pass && checkHDF(H5Dread(dataID, tType, memSpaceID, fileSpaceID, dataPlistID, data), errorMsg);

  if(dataIDs){
    for(i=0; i<breadth; i++){
      pass = pass && checkHDF(H5Dclose(dataIDs[i]), errorMsg);
    }
    delete[] dataIDs;
    dataIDs = NULL;
  }
  else{
    pass = pass && checkHDF(H5Dclose(dataID), errorMsg);
  }
      
  if(memSpaceID != H5S_ALL){
    pass = pass && checkHDF(H5Sclose(memSpaceID), errorMsg);
  }  
  pass = pass && checkHDF(H5Sclose(fileSpaceID), errorMsg);
  pass = pass && checkHDF(H5Pclose(dataPlistID), errorMsg);
  pass = pass && closeGroup();

  if(nameBuffer){
    free(nameBuffer);
    nameBuffer = NULL;
  }

  return pass;  
}
//end of method readTVector()


//************************************************************************************
//************************************************************************************
// METHOD     : writeMatrix()
// DESCRIPTION: Writes an GDOUBLE typed 2D array to the current group in the file.
// ARGUMENTS  : dataName  : The name that will be used to identify the array in 
//                          the file.
//              data      : The GDOUBLE array to write to the current group.
//              dims      : A 2-element array containing first and second order dimension
//                          for the 2D data array
// RETURNS    : TRUE on success; else FALSE if the current position in the file is 
//              not a group or if the array cannot be written for any reason.
// PRECOND    : This file object has been opened with a call to open() or openCreate()
//              and the group to write to has been opened with a call to openGroup()
//              or createGroup().
//************************************************************************************

GBOOL HDFFile::writeMatrix(const char* dataName, GDOUBLE* data, ISUB n1, ISUB n2){
  errorMsg = "::writeMatrix() failed.";

  if(!isOpen()){
    cerr << THIS << errorMsg << " The file isn't open." << endl;
    return FALSE;
  } 

  if(n1 <= 0 || n2 <= 0){
    cerr << THIS << errorMsg << " Matrix dimension arguments must be greater than 0" << endl;
    return FALSE;
  }

  nameBuffer = NULL;
  HDFID fileSpaceID, memSpaceID, dataID;
  HDFID* dataIDs = NULL;
  HDFID* fileCursor = NULL;
  HDFHSIZE dims[TWOD];
  HDFHSSIZE offset[TWOD];
  ISUB remainder = 0, nRows = 0, degree = TWOD, i;
  ISUB *lengths1, *lengths2;

  if(!createGroup(dataName)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }
  
  if(!setDataPlist(dataIO, CREATE)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }

  pass = TRUE;

  switch(dataIO){

  case INDEPENDENT_IO: 

    dataIDs = new HDFID[breadth]; 
    fileCursor = path[cursorD];

    lengths1 = new ISUB[nProcs];
    lengths2 = new ISUB[nProcs];

    pass = (MPI_Allgather(&n1, 1, MPI_INT, lengths1, 1, MPI_INT, getMPIComm()) == MPI_SUCCESS);
    pass = (MPI_Allgather(&n2, 1, MPI_INT, lengths2, 1, MPI_INT, getMPIComm()) == MPI_SUCCESS);

    dims[0] = HDFHSIZE(max(lengths1, nProcs));
    dims[1] = HDFHSIZE(max(lengths2, nProcs));

    fileSpaceID = H5Screate_simple(degree, dims, NULL);
    pass = pass && checkHDF(fileSpaceID, errorMsg);
    const char* name;

    pass = pass && gatherNames(dataName, nameBuffer, nbOffset);

    for(i=0; i<breadth; i++){
      dataIDs[i] = H5Dcreate(fileCursor[i], nameBuffer+(i*nbOffset), H5T_NATIVE_DOUBLE, fileSpaceID, dataPlistID);
      if(!checkHDF(dataIDs[i], errorMsg)){
	for(ISUB j=0; j<i; j++){
	  H5Dclose(dataIDs[j]);
	}
	delete[] dataIDs;
	delete[] lengths1;
	delete[] lengths2;
	return FALSE;
      }
      pass = pass && writeMeta(MDIM_1_NAME, dataIDs[i], lengths1[i]);
      pass = pass && writeMeta(MDIM_2_NAME, dataIDs[i], lengths2[i]);
    }  
    dataID = dataIDs[rank];
    H5Sclose(fileSpaceID);

    memSpaceID = H5Screate_simple(degree, dims, NULL);
    fileSpaceID = H5Dget_space(dataID);
    break;

  case COLLECTIVE_IO:

    dims[0] = HDFHSIZE(n1);
    dims[1] = HDFHSIZE(n2);

    fileSpaceID = H5Screate_simple(degree, dims, NULL);
    if(!checkHDF(fileSpaceID, errorMsg)){
      return FALSE;
    }

    dataID = H5Dcreate(path[cursorD][0], dataName, H5T_NATIVE_DOUBLE, fileSpaceID, dataPlistID);
    pass = pass && checkHDF(dataID, errorMsg);

    pass = pass && writeMeta(MDIM_1_NAME, dataID, n1);
    pass = pass && writeMeta(MDIM_2_NAME, dataID, n2);

    H5Sclose(fileSpaceID);

    nRows = n1/nProcs;
    remainder = n1%nProcs;

    if(nRows == 0){
      dims[0] = 1;
      offset[0] = HDFHSSIZE(rank%n1);
    }
    else if(remainder == 0){
      dims[0] = HDFHSIZE(nRows) ;
      offset[0] = HDFHSSIZE(rank*dims[0]);;
    }
    else if(rank < remainder){
      dims[0] = HDFHSIZE(nRows+1); 
      offset[0] = HDFHSSIZE(rank*dims[0]);
      
    }
    else{
      dims[0] = HDFHSIZE(nRows);  
      offset[0] = HDFHSSIZE(remainder*(nRows+1) + (rank-remainder)*dims[0]);
    }

    dims[1] = HDFHSIZE(n2);
    offset[1] = 0;

    fileSpaceID = H5Dget_space(dataID);
    memSpaceID = H5Screate_simple(degree, dims, NULL);
    H5Sselect_hyperslab(fileSpaceID, H5S_SELECT_SET, offset, NULL, dims, NULL);
    break;

  default:
    cerr << THIS << errorMsg << endl;
    return FALSE;
    break;
  }

  pass = checkHDF(H5Pclose(dataPlistID), errorMsg);

  if(!setDataPlist(dataIO, XFER)){
    cerr << THIS << errorMsg << endl;
    pass = FALSE;
  }

  status = H5Dwrite(dataID, H5T_NATIVE_DOUBLE, memSpaceID, fileSpaceID, dataPlistID, data);
  pass = pass && checkHDF(status, errorMsg); 

  if(memSpaceID != H5S_ALL){
    pass = pass && checkHDF(H5Sclose(memSpaceID), errorMsg);
  }

  pass = pass && checkHDF(H5Sclose(fileSpaceID), errorMsg);
  pass = pass && checkHDF(H5Pclose(dataPlistID), errorMsg);

  if(dataIDs){
    for(i=0; i<breadth; i++){
      pass = pass && checkHDF(H5Dclose(dataIDs[i]), errorMsg);
    }
    delete[] dataIDs;
  }
  else{
    pass = pass && checkHDF(H5Dclose(dataID), errorMsg);
  }

  pass = pass && closeGroup();

  if(nameBuffer){
    free(nameBuffer);
  }

  if(lengths1){
    delete[] lengths1;
  }

  if(lengths2){
    delete[] lengths2;
  }    

  return pass;
}
//end of method writeMatrix()
 

GBOOL HDFFile::readMatrix(const char* dataName, GDOUBLE*& data, ISUB& n1, ISUB& n2){

  errorMsg = "::readMatrix() failed.";

  if(!isOpen()){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  } 

  nameBuffer = NULL;

  HDFID fileSpaceID, memSpaceID, dataID;
  HDFID* dataIDs = NULL;
  HDFID* fileCursor = NULL;
  HDFHSIZE dims[TWOD];
  HDFHSSIZE offset[TWOD];
  ISUB remainder = 0, nRows = 0, degree = TWOD, i;
  ISUB maxDims[TWOD];

  if(!openGroup(dataName)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }

  switch(dataIO){

  case INDEPENDENT_IO:
    
    dataIDs = new HDFID[breadth];
    fileCursor = path[cursorD];

    pass = gatherNames(dataName, nameBuffer, nbOffset);

    for(i=0; i<breadth; i++){
      dataIDs[i] = H5Dopen(fileCursor[i], nameBuffer+(i*nbOffset));
      if(!checkHDF(dataIDs[i], errorMsg)){
	for(ISUB j=0; j<i; j++){
	  H5Dclose(dataIDs[j]);
	}
	delete[] dataIDs;
	if(nameBuffer){ free(nameBuffer); }
	return FALSE;
      }
    }

    dataID = dataIDs[rank];

    pass = pass && readMeta(MDIM_1_NAME, dataID, n1);
    pass = pass && readMeta(MDIM_2_NAME, dataID, n2);

    if(n1 <= 0 || n2 <= 0){
      cerr << THIS << errorMsg << " Invalid matrix dimensions read from file" << endl;
      pass = FALSE;
    }

    //dims[0] = HDFHSIZE(n1);
    //dims[1] = HDFHSIZE(n2);

    fileSpaceID = H5Dget_space(dataID);
    memSpaceID = H5S_ALL;

    break;

  case COLLECTIVE_IO:

    dataID = H5Dopen(path[cursorD][0], dataName);
    pass = pass && checkHDF(dataID, errorMsg);

    fileSpaceID = H5Dget_space(dataID);
    //memSpaceID = H5Screate_simple(degree, dims, NULL);
    memSpaceID = H5S_ALL;
    
    pass = pass && readMeta(MDIM_1_NAME, dataID, n1);
    pass = pass && readMeta(MDIM_2_NAME, dataID, n2);

    if(n1 <= 0 || n2 <= 0){
      cerr << THIS << errorMsg << " Invalid matrix dimensions read from file" << endl;
      pass = FALSE;
    }

    break;

  default:
    cerr << THIS << errorMsg << endl;
    return FALSE;
    break;
  }    

  if(data){
    delete[] data;
    data = NULL;
  }

  if(!setDataPlist(dataIO, XFER)){
    cerr << THIS << errorMsg << endl;
    pass = FALSE;
  }

  data = new GDOUBLE[H5Sget_simple_extent_npoints(fileSpaceID)];

  pass = pass && checkHDF(H5Dread(dataID, H5T_NATIVE_DOUBLE, memSpaceID, fileSpaceID, dataPlistID, data), errorMsg);

  if(dataIDs){
    for(i=0; i<breadth; i++){
      pass = pass && checkHDF(H5Dclose(dataIDs[i]), errorMsg);
    }
    delete[] dataIDs;
    dataIDs = NULL;
  }
  else{
    pass = pass && checkHDF(H5Dclose(dataID), errorMsg);
  }

  if(memSpaceID != H5S_ALL){
    pass = pass && checkHDF(H5Sclose(memSpaceID), errorMsg);
  }
  pass = pass && checkHDF(H5Sclose(fileSpaceID), errorMsg);
  pass = pass && checkHDF(H5Pclose(dataPlistID), errorMsg);

  pass = pass && closeGroup();

  if(nameBuffer){
    free(nameBuffer);
    nameBuffer = NULL;
  }

  return pass;  
}
//end of method readMatrix()


//************************************************************************************
// METHOD     : setDataPlist()
// DESCRIPTION: Initializes the HDF data transfer property list used for collective 
//              read/writes of data sets and attributes from/to HDF files.
// ARGUMENTS  : 
// RETURNS    : TRUE if the HDF property list has been successfully configured for 
//              collective data transfer;  FALSE otherwise.
// PRECONDITIONS:
//************************************************************************************

GBOOL HDFFile::setDataPlist(IO_TYPE ioType, DATA_OPERATION op){
  errorMsg = "::setDataPlist() failed.";

  switch(op){
  case CREATE:
    dataPlistID = H5Pcreate(H5P_DATASET_CREATE);
    if(!checkHDF(dataPlistID, errorMsg)){
      return FALSE;
    }
    break;
  
  case XFER:
    dataPlistID = H5Pcreate(H5P_DATASET_XFER);
    
    if(!checkHDF(dataPlistID, errorMsg)){
      return FALSE;
    }
       
    status = H5Pset_dxpl_mpio(dataPlistID, (ioType == COLLECTIVE_IO ? H5FD_MPIO_COLLECTIVE : H5FD_MPIO_INDEPENDENT));
    if(!checkHDF(status, errorMsg)){  
      return FALSE;
    }
    break;

  default:
    cerr << THIS << errorMsg << "  Invalid data operation type" << endl;
    return FALSE;
    break;
  }

  return TRUE;
}  
//end of method setDataPlist()

void HDFFile::cleanPath(){

  if(path){
    for(ISUB i=0; i<depth; i++){
      if(path[i]){
	delete[] path[i];
	path[i] = NULL;
      }
    }
    delete[] path;
    path = NULL;
  }
}

GBOOL HDFFile::checkHDF(HDFID statusIn, char* message) const{
  if(statusIn < HDF_PASS && message != NULL){
    cerr << THIS << message << endl;
  }
  //H5Eprint(stderr);
  //H5Eclear();
  return statusIn >= HDF_PASS;
}

template<class T> 
T HDFFile::max(T* array, ISUB length){
  T max = array[0];
  for(ISUB i=1; i<length; i++){
    if(array[i] > max){
      max = array[i];
    }
  }
  return max;
}

GBOOL HDFFile::gatherNames(const char* name, char*& nameBuffer, ISUB& offset){

  ISUB nameSize, maxNameSize = 0;

  nameSize = strlen(name)+1;
      
  if(MPI_Allreduce(&nameSize, &maxNameSize, sizeof(ISUB), MPI_INT, MPI_MAX, getMPIComm()) != MPI_SUCCESS){
    cerr << THIS << "::gatherNames() failed." << endl;
    return FALSE;
  }
  
  char* localName = new char[maxNameSize];
  sprintf(localName, "%s\0", name);

  nameBuffer = (char*)malloc(sizeof(char)*maxNameSize*breadth);

  pass = (MPI_Allgather(localName, maxNameSize, MPI_CHAR, nameBuffer, maxNameSize, MPI_CHAR, getMPIComm()) == MPI_SUCCESS);

  offset = maxNameSize;

  delete[] localName;
  localName = NULL;

  return pass;
}

//************************************************************************************
// METHOD     : getCurrentHDFObj()
// DESCRIPTION: gets id of the current HDF object
// ARGUMENTS  : 
// RETURNS    : id
//              
// PRECONDITIONS:
//************************************************************************************
HDFID getCurrentHDFObj()
{
  return path[cursorD][rank];
} // end of method getCurrentHDFObj
