//************************************************************************************//
// Module       : apost_error.cpp
// Date         : 6/4/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the object providing a-posteriori error estimates for
//                the state (field). The object uses primarily the methods detailed
//                in C. Mavriplis' Ph.D. dissertation.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "apost_error.hpp"
#include "gradop.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
//************************************************************************************
APostError::APostError(GERR_NORM norm, Field2D *fld)
:
bAvg_            (TRUE),
nfit_min_        (4),
itot_            (0),
nCoeffs_         (1),
ctiny_           (1.0e-4),
y_               (NULL),
x_               (NULL),
inorm_type_      (EN_L2),
ifit_func_       (FF_EXP),
field_           (NULL)
{
  GINT  j, r;

  memset(NN_         , 0, GDIM*sizeof(GINT ));
  memset(num_fit_pts_, nfit_min_, GDIM*sizeof(GINT ));
  memset(error_est_  , 0, GDIM*sizeof(GDOUBLE));
  memset(error_trunc_, 0, GDIM*sizeof(GDOUBLE));
  memset(error_extrp_, 0, GDIM*sizeof(GDOUBLE));
  for ( r=0; r<GDIM; r++ ) {
    basis_ [r] = NULL;
    du_    [r] = NULL;
    int_   [r] = NULL;
  }

  if ( norm == EN_L2 ) {
    nCoeffs_ = 1;
  }
  else if ( norm == EN_H1_SEMI ) {
    nCoeffs_ = GDIM;
  }
  inorm_type_ = norm;

  for ( r=0; r<nCoeffs_; r++ ) {
    du_    [r] = new GVector();
    int_   [r] = new GVector();
  }

  if ( fld != NULL ) SetField(fld);

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
APostError::~APostError()
{
  GINT  i, j;

  for ( i=0; i<GDIM; i++ ) {
    if ( du_     && du_    [i] != NULL ) delete du_    [i]; 
    if ( int_    && int_   [i] != NULL ) delete int_   [i]; 
  }
  if ( x_     != NULL ) delete [] x_ ; 
  if ( y_     != NULL ) delete [] y_ ; 
}


//************************************************************************************
//************************************************************************************
// METHOD     : SetField
// DESCRIPTION: Set field whose error estimate we are interested in.
// ARGUMENTS  : Field2D *fld--field to point to
//
// RETURNS    : none.
//************************************************************************************
void  APostError::SetField(Field2D *fld)
{
  GINT  i, j;

  if ( fld == NULL ) {
    cout << "APostError::SetField: NULL field" << endl;
    exit(1);
  }
  
  for ( i=0,itot_=1; i<GDIM; i++ ) {
    basis_[i] = fld->GetElement()->GetBasisObj(i+1); 
    NN_   [i] = basis_[i]->GetOrder() + 1;
    itot_    *= NN_[i];
    Lp_[i].Resize(NN_[i],NN_[i]); 
  }
  for ( i=0; i<nCoeffs_; i++ ) {
    sigma_[i].Resize(GDIM);
    intcp_[i].Resize(GDIM);
    qfit_ [i].Resize(GDIM);
    for ( j=0; j<GDIM; j++ ) {
      Coeffs_[i][j].Resize(NN_[j]);
    }
    du_    [i]->Resize(itot_);
    int_   [i]->Resize(itot_);
  }
  if ( x_ ) delete [] x_;
  if ( y_ ) delete [] y_;
  x_ = new GDOUBLE [itot_];
  y_ = new GDOUBLE [itot_];
  field_ = fld;
} // end of SetField

//************************************************************************************
//************************************************************************************
// METHOD     : SetNumFitPoints (1)
// DESCRIPTION: Sets number of points to fit to extrap. function
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void  APostError::SetNumFitPoints(GINT  idir, GINT  n)
{ 
  if ( idir < 1 || idir > GDIM ) {
    cout << "APostError::SetNumFitPoints: invalid coordinate direction" << endl;
    exit(1);
  }
  num_fit_pts_[idir-1] = n;
} // end of method SetNumFitPoints (1)


//************************************************************************************
//************************************************************************************
// METHOD     : SetNumFitPoints (2)
// DESCRIPTION: Sets number of points to fit to extrap. function
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void  APostError::SetNumFitPoints(GINT  n)
{
  GINT  i;
  
  for ( i=0; i<GDIM; i++ ) num_fit_pts_[i] = n;

} // end of method SetNumFitPoints (2)


//************************************************************************************
//************************************************************************************
// METHOD     : SetMinNumFitPoints 
// DESCRIPTION: Sets min. number of points for a good fit
// ARGUMENTS  :
// RETURNS    : none.
//************************************************************************************
void  APostError::SetMinNumFitPoints(GINT  n)
{
  nfit_min_ = n ;
} // end of method SetMinNumFitPoints 


//************************************************************************************
//************************************************************************************
// METHOD     : SetTiny
// DESCRIPTION: Sets size of min. significant ratio of Leg coeff/coeff_max
// ARGUMENTS  :
// RETURNS    : none.
//************************************************************************************
void  APostError::SetTiny(GDOUBLE c)
{
  ctiny_ = c;
} // end of method SetTiny


//************************************************************************************
//************************************************************************************
// METHOD     : SetExtrapFunc
// DESCRIPTION: Sets extrap function to use
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void APostError::SetExtrapFunc(GEXT_FUNC ifunc)
{
  return;  // function not used currently
  ifit_func_ = ifunc;
} // end of method SetExtrapFunc


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeErrEst 
// DESCRIPTION: Computes error estimate, which is a quadrature of truncatoin error 
//              & extrapolation error estimates.
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE;
//************************************************************************************
GBOOL APostError::ComputeErrEst()
{
  GINT  r;
  
  if ( !ComputeLegendreCoeffs() ) return FALSE;
  if ( !DoFit                () ) return FALSE;
  if ( !ComputeTruncError    () ) return FALSE;
  if ( !ComputeExtrapError   () ) return FALSE;

  for (r=0; r<nCoeffs_; r++ ) { 
    error_est_[r] = error_trunc_[r] + error_extrp_[r];
  }

  return TRUE;

} // end of method ComputeErrEst


//************************************************************************************
//************************************************************************************
// METHOD     : GetErrEst (1)
// DESCRIPTION: Gets computed error estimate
// ARGUMENTS  :  none.
// RETURNS    :  error est.
//************************************************************************************
GDOUBLE APostError::GetErrEst()
{
  GINT  i;
  GDOUBLE emax=-SEHUGE;

  if ( inorm_type_ == EN_L2 ) {
   emax = error_est_[0];
  }
  else if ( inorm_type_ == EN_H1_SEMI ) {
    for ( i=0; i<GDIM; i++ ) emax = MAX(emax,error_est_[i]);
  }
  return sqrt(emax);
} // end of method GetErrEst (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetErrEst (2)
// DESCRIPTION: Gets computed error estimate 
// ARGUMENTS  :  none.
// RETURNS    :  error est.
//************************************************************************************
GDOUBLE APostError::GetErrEst(GINT  idir)
{
  GDOUBLE emax=-SEHUGE;


  if ( inorm_type_ == EN_L2 ) {
    emax = error_est_[0];
  }
  else if ( inorm_type_ == EN_H1_SEMI ) {
    if ( idir < 1 || idir > GDIM ) {
      cout << "APostError::GetErrEst: invalid coordinate direction or wrong method for norm" << endl;
      exit(1);
    }
    emax = error_est_[idir-1];
  }
  return sqrt(emax);
} // end of method GetErrEst (1)


//************************************************************************************
//************************************************************************************
// METHOD     : isDecaying
// DESCRIPTION: gets flag indicating if spectrum is decaying
// ARGUMENTS  : 
// RETURNS    : GBOOL 
//************************************************************************************
GBOOL APostError::isDecaying(GINT  idir)
{
  GBOOL bRet;
  GINT  j, r;

  if ( idir < 1 || idir > GDIM ) {
    cout << "APostError::isDecaying: invalid coordindate direction" << endl;
    exit(1);
  }

  for ( r=0, bRet=TRUE; r<nCoeffs_; r++ ) {
    for ( j=0; j<GDIM; j++ ) bRet = bRet && sigma_[r][j] < 0.0;
  }
  return bRet;

} // end of method isDecaying


//************************************************************************************
//************************************************************************************
// METHOD     : NumberNormComps
// DESCRIPTION: Get number of norm components: if L2, n=1;
//              if H1-semi, n = GDIM
// ARGUMENTS  : none.
// RETURNS    : GVector *, one vector for each coeffs
//************************************************************************************
GINT APostError::NumberNormComps()
{
  return nCoeffs_;
} // end of method NumberNormComps


//************************************************************************************
//************************************************************************************
// METHOD     : GetDecayRate
// DESCRIPTION: Gets slopes of fits
// ARGUMENTS  : none.
// RETURNS    : GVector *, one vector for each coeffs
//************************************************************************************
GVector *APostError::GetDecayRate()
{
  return sigma_;
} // end of method GetDecayRate


//************************************************************************************
//************************************************************************************
// METHOD     : GetMaxDecayRate
// DESCRIPTION: Gets minimum decay rate over all norm components.
// ARGUMENTS  : 
// RETURNS    : GDOUBLE 
//************************************************************************************
GDOUBLE APostError::GetMaxDecayRate()
{
  GINT    r, j;
  GDOUBLE rate=-SEHUGE ;

  for ( r=0; r<nCoeffs_; r++ ) {
    rate = MAX(sigma_[r].Max(),rate);
  }
  return rate;
} // end of method GetMaxDecayRate


//************************************************************************************
//************************************************************************************
// METHOD     : GetQFit (1)
// DESCRIPTION: Gets quality of fit mesaure
// ARGUMENTS  : none.
// RETURNS    : GVector *
//************************************************************************************
GVector *APostError::GetQFit()
{
  return qfit_;
} // end of method GetQFit (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetLegendre
// DESCRIPTION: Gets Legendre exp. coeffs
// ARGUMENTS  : none.
// RETURNS    : GVector **
//************************************************************************************
GVector *APostError::GetLegendre(GINT icomp) 
{
  if ( icomp < 0 || icomp >= nCoeffs_ ) {
    cout << "APostError::GetLegendre: invalid norm component:" << icomp << endl;
    exit(1);
  }

  return Coeffs_[icomp];
} // end of method GetLegendre


//************************************************************************************
//************************************************************************************
// METHOD     : GetIntercept
// DESCRIPTION: Gets intercepts of fits
// ARGUMENTS  : none.
// RETURNS    : GVector *
//************************************************************************************
GVector *APostError::GetIntercept()
{
  return intcp_;
} // end of method GetIntercept


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeLegendreCoeffs
// DESCRIPTION: Computes Legendre coeffs for state 
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL APostError::ComputeLegendreCoeffs()
{
  GINT      i, j, k, m, n, r;
  GDOUBLE   gnorm;
  GVector   *u;
  GradOp    *grad=NULL;
  Elem2D    *elem;

  if ( field_ == NULL ) return FALSE;
  for ( i=0; i<GDIM; i++ ) {
    if ( !basis_[i]->GetLegMatrix(Lp_+i) ) return FALSE;
  }
  
  elem = field_->GetElement();
  u = field_->GetExpCoeffs(0); // get exp. coeffs for 0th (current) time level
  if ( u == NULL ) return FALSE;

  if ( inorm_type_ == EN_L2 ) {  // in L2 norm
    for ( j=0; j<du_[0]->dim(); j++ ) (*du_[0])[j] = fabs((*u)[j]);
  } 
  else {                         // in H1-semi norm (just the derivs)
    grad = new GradOp(field_->GetElement(), u);
    for ( n=0; n<nCoeffs_; n++ ) {
      grad->Grad(n+1,*du_[n]);
      for ( j=0; j<du_[n]->dim(); j++ ) (*du_[n])[j] = fabs((*du_[n])[j]);
    }
    delete grad;
  }

  for ( r=0; r<nCoeffs_; r++ ) {        // cycle over the coeffs
   for ( j=0; j<GDIM; j++ ) Coeffs_[r][j] = 0.0;
    // Avg over y to find 'avg' coeffs in x:
    for ( m=0; m<NN_[0]; m++ ) {        // for 1-index
      for ( n=0; n<NN_[1]; n++ ) {      // for 2-index
        for ( j=0; j<NN_[1]; j++ ) {
          for ( i=0; i<NN_[0]; i++ ) {  // construct 2d integrand:
            k  = i + NN_[0]*j;
            (*int_[r])(k) = (*du_[r])(k) * Lp_[0](m,i) * Lp_[1](n,j);
          }
        }
//      gnorm = 1.0/ ( (2.0*m +1.0) * (2.0*n +1.0) ); 
        gnorm = 1.0/ (2.0*m + 1.0);
        Coeffs_[r][0][m] += gnorm * fabs( elem->PIntegrate(int_[r]) );
      }
      Coeffs_[r][0][m] /= (GDOUBLE)NN_[1];
    }
    // Avg over x to find 'avg' coeffs in y:
    for ( n=0; n<NN_[1]; n++ ) {        // for 2-index
      for ( m=0; m<NN_[0]; m++ ) {      // for 1-index
        for ( j=0; j<NN_[1]; j++ ) {
          for ( i=0; i<NN_[0]; i++ ) {  // construct 2d integrand:
            k  = i + NN_[0]*j;
            (*int_[r])(k) = (*du_[r])(k) * Lp_[0](m,i) * Lp_[1](n,j);
          }
        }
//      gnorm = 1.0 / (2.0*n +1.0); 
        Coeffs_[r][1][n] += gnorm * fabs( elem->PIntegrate(int_[r]) );
      }
      Coeffs_[r][1][n] /= (GDOUBLE)NN_[0];
    }
  }
//cout << "Coeffs_x=" << Coeffs_[0][0] << endl;
//cout << "Coeffs_y=" << Coeffs_[0][1] << endl;

  return TRUE;
} // end of method ComputeLegendreCoeffs


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeTruncError
// DESCRIPTION: Computes square of truncation error portion of total error estimate
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL APostError::ComputeTruncError()
{
  GINT    i, j, r;
  GBOOL   bRet = FALSE;

//for ( j=0; j<GDIM; j++ ) gnorm[j] = 2.0/(2.0*NN_[j]-1.0);
  for ( r=0; r<nCoeffs_; r++ ) {
    error_trunc_[r] = ctiny_;
    for ( j=0; j<GDIM; j++ ) {
      error_trunc_[r] += pow(Coeffs_[r][j][NN_[j]-1],2);
    }
  }
  bRet = TRUE;

  return bRet;
} // end of method ComputeTruncError


//************************************************************************************
//************************************************************************************
// METHOD     : DoFit
// DESCRIPTION: Performs LLSq fit using exp. fcn on Leg. coeffs. Do these fits
//              line-by-line using an exponential and power law decay profile.
//              The spectral coeffs that are too small are not included in the
//              fits; if one of the required points is too small, then the entire
//              'raster' is neglected.
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL APostError::DoFit()
{
  GINT   i, j, mfit, nfit, r;
  GBOOL  bSig;
  GDOUBLE  slope, intcp, qfit, cmax, icmax, yy;

  for ( j=0; j<GDIM; j++ ) isDecaying_[j] = TRUE;
  SetParams();
  for ( r=0,cmax=0.0; r<nCoeffs_; r++ ) {    // find max coeff
    for ( j=0; j<GDIM; j++ ) 
      cmax = MAX(fabs(Coeffs_[r][j].MaxA()), cmax);
  }
  icmax = 1.0 / cmax;

  for ( r=0; r<nCoeffs_; r++ ) {   // over each norm component
    for ( i=0; i<GDIM; i++ ) {     // fit in i-dir (1-fit for each k!=i)
      nfit   = num_fit_pts_[i] > NN_[i] ? NN_[i] : num_fit_pts_[i];
      for ( j=NN_[i]-nfit, bSig=TRUE, mfit=0; j<NN_[i] && bSig; j++ ) {
        yy       = fabs(Coeffs_[r][i][j]);
        bSig     = (bSig && yy*icmax) > ctiny_;
        mfit     = bSig ? mfit+1: mfit;
        yy       = log(yy);
        x_[mfit] = ifit_func_==FF_EXP ? (GDOUBLE)(j+1) : log((GDOUBLE)(j+1));
        y_[mfit] = yy;
      }
      LinFit(slope, intcp, qfit, x_, y_, mfit);
      sigma_[r][i] = slope;
      intcp_[r][i] = exp(intcp);
      qfit_ [r][i] = qfit;
    }
//cout << "DoFit: sigma=" << sigma_[0] << endl;
  }

  return TRUE;
} // end of method DoFit


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeExtrapError
// DESCRIPTION: Computes extrapolation error portion of total error estimate.
//              This is approximated by considering fit of coeffs to a function, a(n),
//              and then integrating line-by-line s.t.:
//              err_ext^2 = Int_(N+1)(inf) 2 a(n)^2 / (2n+1)  dn
//                        <=
//                         1/[2(N+1)+1] Int_(N+1)(inf) a(n)^2 dn
//              Note: the decay rates must be valid prior to entry. ( i.e., negative).
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE. 
//              along a line.
//************************************************************************************
GBOOL APostError::ComputeExtrapError()
{
  GINT  i, j, r, km=0;
  GDOUBLE errext, errext1, gnorm, csq, sig, isig, arg;
  
  if ( ifit_func_ == FF_EXP ) {
    for ( r=0; r<nCoeffs_; r++ ) {
      error_extrp_[r] = 1.0;
      for ( j=0; j<GDIM; j++ ) {       // sum j-extrapolation errors:
        gnorm            = 1.0/(2.0*NN_[j]+1.0);
        sig              = fabs(sigma_[r][j]);
        isig             = sig < TINY ? 0.0 : 0.5/sig;
        csq              = intcp_[r][j] * intcp_[r][j] * isig;
        arg              = -2.0*sig*(NN_[j]+1.0);
        errext           = gnorm*csq*exp(arg);
        error_extrp_[r] *= errext;
      }
      error_extrp_[r] *= gnorm;
    }
    return TRUE;
  }
  else if ( ifit_func_ == FF_POWER_LAW ) {
    for ( r=0; r<nCoeffs_; r++ ) {
      error_extrp_[r] = 1.0;
      for ( j=0; j<GDIM; j++ ) {       // sum j-extrap errors
        gnorm            = 1.0/(2.0*NN_[j]-1.0);
        sig              = fabs(sigma_[r][j]);
        csq              = intcp_[r][j]*intcp_[r][j]/(2.0*sig-1.0);
        errext           = gnorm*csq*pow(NN_[0]+1,-2.0*sig+1);
        error_extrp_[r] *= errext/(fabs(sig)+TINY); 
      }
    }
    return TRUE;
  }

  return FALSE;
} // end of method ComputeExtrapError


//************************************************************************************
//************************************************************************************
// METHOD     : SetParams
// DESCRIPTION: Initializes parameters
// ARGUMENTS  : none.
// RETURNS    : none/
//************************************************************************************
void APostError::SetParams()
{
  GINT  i, j;

  for ( i=0; i<nCoeffs_; i++ ) {
    for ( j=0; j<GDIM; j++ ) {
      sigma_[i][j] = -SEHUGE;
      intcp_[i][j] = 0.0;
      qfit_ [i][j] = 0.0;
    }
  }
} // end of method SetParams 


//************************************************************************************
//************************************************************************************
// METHOD     : LinFit
// DESCRIPTION: performs linear fit on input quantities
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void APostError::LinFit(GDOUBLE &slope, GDOUBLE &intcp, GDOUBLE &qfit, 
                        GDOUBLE x[], GDOUBLE y[], GINT  nxy )
{
  GINT     i;
  GDOUBLE    Sxx, Syy, Sxy, iSxx, iSS, xavg, yavg, infit;

  Sxx = Syy = Sxy = xavg = yavg = 0.0;
  for ( i=0; i<nxy ; i++ ) {
    Sxx  += (x[i]*x[i]);
    Syy  += (y[i]*y[i]);
    Sxy  += (x[i]*y[i]);
    xavg += x[i];
    yavg += y[i];
  }
  infit  = nxy == 0 ? 0.0 : 1.0/((GDOUBLE)nxy);
  xavg  *= infit;
  yavg  *= infit;
  Sxx   -= (nxy*xavg*xavg);
  Syy   -= (nxy*yavg*yavg);
  Sxy   -= (nxy*xavg*yavg);
  iSxx   = Sxx == 0.0 ? 0.0 : 1.0/(Sxx);
  iSS    = Sxx*Syy == 1.0 ? 0.0 : 1.0/(Sxx*Syy);
  slope  = Sxy * iSxx;
  intcp  = yavg - slope*xavg; 
  qfit   = (Sxy*Sxy) * iSS;  // ( = r^2 )
//qfit   = fi < ctiny_ ? 0.0 : fi/(sqrt(St)*sqrt(Sr));  

} // end of method LinFit


