//************************************************************************************//
// Module       : ns.cpp
// Date         : 8/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the solver for the incompressible Navier-Stokes equations. 
//                The problem solved is
//                du/dt + u.Del u + 1/rho Del p -  nu_ Del^2 u = 0
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "ns.hpp"
#include "pcpointjac_helm.hpp"
#include "pcblockjac_helm.hpp"
#include "pcblockjac_lap.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
//************************************************************************************
NavierStokes::NavierStokes(GFieldList *ulist[], GINT nf, GElemList *uelems, GElemList *pelems, GINT t)
:
hDSOp            (NULL_HANDLE),
itime_type_      (TE_EXBDF),
iStokes_type_    ((STOKES_TYPE)t),
bInitReqd_       (TRUE),
bDoAdvect_       (TRUE),
irank_           (GDIM),
nelems_          (0),
icycle_          (0),
maxCFL_          (0.5),
dt_curr_         (0.0),
dt_last_         (0.0),
time_            (0.0),
rho_             (1.0),
uelems_          (uelems),
pelems_          (pelems),
stokes_          (NULL),
gsop             (NULL)
{
  GINT  j;

  if ( uelems_ == NULL || pelems_ == NULL ) {
    cout << "NavierStokes::NavierStokes: NULL element list" << endl;
    exit(1);
  }

  if ( nf < irank_+1 ) {
    cout << "NavierStokes::NavierStokes: insufficient number of fields computed" << endl;
    exit(1);
  }

  // Last field is assumed to be pressure:
  for ( j=0; j<=irank_; j++ ) {
    if ( ulist[j] == NULL ) {
      cout << "NavierStokes::NavierStokes: NULL input field" << endl;
      exit(1);
    }
    var_list_[j] = ulist[j];
  }
  if ( uelems_->size() != pelems_->size() ) {
    cout << "NavierStokes::NavierStokes: incompatible grids" << endl;
    exit(1);
  }

  nelems_  = uelems_->size();
  stokes_ = new StokesSolver(var_list_, irank_+1, uelems_, pelems_, iStokes_type_);
  advect_ = new GAdvect(var_list_, irank_, uelems);

  SetAdvOrder (2);
  SetBDFOrder(2);
  SetEvolType(itime_type_);

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
NavierStokes::~NavierStokes()
{
  DeleteDynamic();
}


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION:
// RETURNS    :
//************************************************************************************
void  NavierStokes::DeleteDynamic()
{
  GINT j;

  if ( stokes_   != NULL ) delete stokes_;
  if ( advect_   != NULL ) delete advect_;

  for ( j=0; j<irank_; j++ ) {
    pu_     [j].empty();
    Nj_     [j].empty();
  }


} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : ResetExpandables
// DESCRIPTION: Set quantities that depend on the number and size of elements in 
//              field lists. The member data, nelems_, repres. the current
//              number of elements *is* set here.
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GBOOL NavierStokes::ResetExpandables()
{
  GINT     i, j, num, ncurr, NN;
  GVector *cu[3]={NULL,NULL,NULL};
  Elem2D   *ue;

  if ( uelems_ == NULL || pelems_ == NULL ) {
    cout << "NavierStokes::ResetExpandables: NULL element list" << endl;
    exit(1);
  }

  if ( stokes_ == NULL || !stokes_->ResetExpandables() ) {
    cout << "NavierStokes::ResetExpandables: Could not reset Stokes operator" << endl;
    exit(1);
  }

  num   = uelems_->size();
  for ( j=0; j<irank_; j++ ) {
    pu_     [j].empty();
    Nj_     [j].empty();
  }
  ncurr = pu_[0].size();

  // Add elements as required:
  for ( i=ncurr; i<num; i++ ) {
    NN = var_list_[0]->member(i)->GetExpCoeffs(0)->dim();
    for ( j=0; j<irank_; j++ ) 
    cu[j] = var_list_[j] && var_list_[j]->member(i) ? var_list_[j]->member(i)->GetExpCoeffs(0) : NULL;
    ue    = (*uelems_)[i];
//  pe    = (*pelems_)[i];

    // Pointers
    for ( j=0; j<irank_; j++ ) {
      pu_     [j].add(NULL,FALSE);
      Nj_     [j].add(NULL,FALSE);
    }

    // Fully-instantiated data:
  }

  // Delete unnecessary elements from lists:
  for ( i=num; i<ncurr; i++ ) {
    for ( j=0; j<irank_; j++ ) {
      pu_     [j].del(i);
      Nj_     [j].del(i);
    }
  }

  // Must now renumber/re-order the reference ids for lists:
  for ( j=0; j<irank_; j++ ) {
    pu_     [j].renumber();
    Nj_     [j].renumber();
  }

  nelems_ = num;

  bInitReqd_ = TRUE;

  return TRUE;

} // end of method ResetExpandables


//************************************************************************************
//************************************************************************************
// METHOD     : SetEvolType
// DESCRIPTION: Sets time evolution method type by setting appropriate 
//              coefficient--sets
//
//              Caller must set fields with appropriate
//              number of time levels set.
// RETURNS    :  
//************************************************************************************
void  NavierStokes::SetEvolType(TIME_EVOLTYPE i)
{
  char  *serr = "NavierStokes::SetEvolType: ";
  GINT  j;

  itime_type_ = i;
  switch (itime_type_) {
    case TE_OIFS:
      advect_->SetEvolType(GAdvect::TE_OIFS);
    case TE_ABBDF: 
      advect_->SetEvolType(GAdvect::TE_ABBDF);
    case TE_EXBDF: 
      advect_->SetEvolType(GAdvect::TE_EXBDF);
      break;
    default:
      cout << serr << "invalid NavierStokes time stepping method specified" << endl;
      exit(1);
  }
  stokes_->Setc0(advect_->GetBDFSum());

} // end of method SetEvolType


//************************************************************************************
//************************************************************************************
// METHOD     : SetComm
// DESCRIPTION: Sets global gather scatter operator
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void  NavierStokes::SetComm(GS *ggs)
{
  gsop = ggs;

  if ( stokes_ == NULL )  return;

  stokes_->SetComm(gsop);

} // end of method SetComm


//************************************************************************************
//************************************************************************************
// METHOD     : SetCommHandle
// DESCRIPTION: Sets handle to active gather/scatter Init call 
// ARGUMENTS  : hIn :  existing handle
// RETURNS    : GCHandle
//************************************************************************************
GCHandle NavierStokes::SetCommHandle(GCHandle hIn)
{
  if ( !gsop  || !stokes_ ) return NULL_HANDLE;
  if ( hDSOp == NULL_HANDLE ) hDSOp=stokes_->SetCommHandle(hIn);

  return hDSOp;

} // end of method SetCommHandle


//************************************************************************************
//************************************************************************************
// METHOD     : SetVisc
// DESCRIPTION: Sets nu_
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void NavierStokes::SetVisc(const GDOUBLE f)
{
  stokes_->SetVisc(f);
} // end of method SetVisc


//************************************************************************************
//************************************************************************************
// METHOD     : SetVisc
// DESCRIPTION: Sets nu_
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void NavierStokes::SetVisc(const GDOUBLE f, GINT idir)
{
  if ( stokes_ == NULL ) return;
  stokes_->SetVisc(f, idir);
} // end of method SetVisc


//************************************************************************************
//************************************************************************************
// METHOD     : SetDen
// DESCRIPTION: Sets rho_
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void NavierStokes::SetDen(const GDOUBLE f)
{
  rho_ = f;
  if ( stokes_ == NULL ) return;
  stokes_->SetDen(rho_);
} // end of method SetDen


//************************************************************************************
//************************************************************************************
// METHOD     : SetAdvOrder
// DESCRIPTION: Sets order of advection term approximated on n+1 timestep
//
//              If this method is called, then SetEvolType must be called
//              to select coefficient set. If SetEvolType is not called, then
//              this call will have no effect.
//
//              NOTE: caller must be certain that field lists are set
//                    with an appropriate number of time levels for the
//                    order chosen here and in SetBDFOrder, or solve will fail.
// ARGUMENTS  : GINT  iorder: 1, 2, 3,...
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
void NavierStokes::SetAdvOrder(GINT  iorder)
{

  advect_->SetAdvOrder(iorder);

} // end of method SetAdvOrder


//************************************************************************************
//************************************************************************************
// METHOD     : SetBDFOrder
// DESCRIPTION: Sets order of BDF (backward time diff.) used to approximate the
//              explicit treatment of the time derivative.
//
//              If this method is called, then SetEvolType must be called
//              to select coefficient set. If SetEvolType is not called, then
//              this call will have no effect.
//
//              NOTE: caller must be certain that field lists are set
//                    with an appropriate number of time levels for the
//                    order chosen here, and in SetAdvOrder or solve will fail.
// ARGUMENTS  : GINT  iorder: 1, 2, 3, ...
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
void NavierStokes::SetBDFOrder(GINT  iorder)
{
  advect_->SetBDFOrder(iorder);

} // end of method SetBDFOrder


//************************************************************************************
//************************************************************************************
// METHOD     : Step
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GBOOL NavierStokes::Step(GDOUBLE dt)
{
  char  *serr = "NavierStokes::Step ";
  GINT  i, j;
  GBOOL bRet;

  dt_last_ = dt_curr_;
  dt_curr_ = dt;

  // If element list has changed, must resize member data:
  if ( ElemListChange() || bInitReqd_ ) {
    if ( !ResetExpandables() ) {
      cout << serr << "allocation error" << endl;
      exit(1);
    }
    stokes_->UpdateSolver();
    bInitReqd_ = FALSE;
  }

  switch (itime_type_) {
    case TE_OIFS:
    case TE_ABBDF: 
    case TE_EXBDF: 
      bRet = DoNormalStep(dt);
      break;
    default:
      cout << serr << "invalid NavierStokes time stepping method specified" << endl;
      bRet = FALSE;
      break;
  }

  if ( bRet ) {
    time_ += dt;
    icycle_ ++;
  }

  // Now, update field's time:
  for ( j=0; j<=irank_; j++ ) var_list_[j]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<=irank_; j++ ) {
      var_list_[j]->member()->SetTime(0,time_);
    }
    for ( j=0; j<=irank_; j++ ) var_list_[j]->next();
  }
 
  return bRet;

} //end of operator Step


//************************************************************************************
//************************************************************************************
// METHOD     : Advect
// DESCRIPTION: Perform advection partial update for NS equations.
// ARGUMENTS  : GDOUBLE dt: time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NavierStokes::Advect(GDOUBLE dt, GINT &iLevelTemp)
{ 
  GINT    i, j;
  GBOOL   bRet; 
 
  iLevelTemp = var_list_[0]->member(0)->GetTempMgr()->GetFirstAvail();
  for ( j=0; j<irank_; j++ ) var_list_[j]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<irank_; j++ ) {
      pu_[j][i] = var_list_[j]->member()->GetTemp(iLevelTemp);
      Nj_[j][i] = var_list_[j]->member()->GetTemp();        // temp space to hold advect update
      var_list_[j]->next();
    }
  }

  // Use GAdvect operator: first arg is a pointer to the
  // explicit portion of the terms -M(dudt +u.Grad u) : the
  // '-' sign indicates that this term is on the RHS of
  // an invertible equation.
  bRet = advect_->Advect(pu_, Nj_, time_, dt);

  // Unlock temp memory so that we can re-use it later:
  for ( j=0; j<irank_; j++ ) var_list_[j]->start(NULL);
  for ( j=0; j<irank_; j++ ) {
    for ( i=0; i<nelems_; i++ ) {
      var_list_[j]->member()->TempUnlock(pu_[j][i]);
      var_list_[j]->member()->TempUnlock(Nj_[j][i]);
      var_list_[j]->next();
    }
  }
    
  return bRet;
} // end of operator Advect


//************************************************************************************
//************************************************************************************
// METHOD     : DoNormalStep
// DESCRIPTION: Time step using methods set in GAdvect operator 
// ARGUMENTS  : GDOUBLE dt: time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NavierStokes::DoNormalStep(GDOUBLE dt)
{
  GINT    iTempLevel;
  GBOOL   bRet;
  
  if ( !Advect(dt, iTempLevel) ) return FALSE;
  
  bRet = stokes_->Solve(dt, iTempLevel, 0);

  return bRet;
} // end of operator DoNormalStep



//************************************************************************************
//************************************************************************************
// METHOD     : SetVBdyData
// DESCRIPTION: Set boundary conditions
// ARGUEMENTS : idir       : v-component whose bdy vals are being set
//              bdy_vals   : if present, gives the boundary values; each
//                           member of list must be of the same dimensionality
//                           as required by element's bdy_indices, even though
//                           a given bdy node value is set only if the btype 
//                           for this node is DIRICHLET.
// RETURNS    :
//************************************************************************************
void NavierStokes::SetVBdyData(GINT  idir, GVecList *bdy_vals)
{

  if ( idir < 1 ) {
    cout << "NavierStokes::SetVBdyData: invalid component direction" << endl;
    exit(1);
  }
  if ( idir > irank_ ) return; // don't set for pressure

  if ( stokes_ == NULL ) return;

  stokes_->SetVBdyData(idir, bdy_vals);

} // end of method SetVBdyData


//************************************************************************************
//************************************************************************************
// METHOD     : GetIntegTime
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE NavierStokes::GetIntegTime()
{
  return time_;
} // end of method GetIntegTime


//************************************************************************************
//************************************************************************************
// METHOD     : GetDivMax
// DESCRIPTION: Computes Max(Div.V)
//
// ARGUMENTS  :
// RETURNS    : max div.v
//************************************************************************************
GDOUBLE NavierStokes::GetDivMax(GINT  iLevel)
{
  if ( stokes_ == NULL ) {
    cout << "NavierStokes::GetDivMax: NULL Stokes solver" << endl;
    exit(1);
  }
 
  return stokes_->GetDivMax(iLevel);

} // end of method GetDivMax


//************************************************************************************
//************************************************************************************
// METHOD     : DivV
// DESCRIPTION: Computes Div.V using Stokes derivatives.
//
// ARGUMENTS  : divv : vector list on the p-grid, whose number of
//                     members = nelems_
// RETURNS    : void
//************************************************************************************
void NavierStokes::DivV(GFieldList *divv)
{

  if ( stokes_ == NULL ) {
    cout << "NavierStokes::DivV: NULL stokes object" << endl;
    exit(1);
  }
  stokes_->DivV(divv, var_list_, irank_);

} // end of method DivV


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : TimeStep
// DESCRIPTION: Computes minimum timestep
//
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GDOUBLE NavierStokes::TimeStep()
{
  GINT     i, j, k, m, n, NN1, NN2, P1, P2;
  GDOUBLE    dx, dy, aux=0.0, auy=0.0, dtimax,
           dtin=-SEHUGE, dtn, dtg;
  Point3D  *vertices;
  GVector *x, *y, *vu1, *vu2;

  u1_list_->start(NULL);
  u2_list_->start(NULL);
  for ( n=0; n<nelems_; n++ )
  {
    P1  = u1_list_->member()->GetElement()->GetOrder(1);
    P2  = u1_list_->member()->GetElement()->GetOrder(2);
    vertices = u1_list_->member()->GetElement()->GetSpVertices();
    x   = u1_list_->member()->GetElement()->GetSpNodes(1);
    y   = u1_list_->member()->GetElement()->GetSpNodes(2);
    vu1 = u1_list_->member()->GetExpCoeffs(0);
    vu2 = u2_list_->member()->GetExpCoeffs(0);
    NN1 = P1 + 1;
    NN2 = P2 + 1;
    for ( j=0,m=0; j<P2; j++ )
    {
      for ( i=0; i<P1; i++,m++ )
      {
        dx       = fabs((*x)(i+1+NN1*j  ) - (*x)(i+NN1*j)) + TINY;
        dy       = fabs((*y)(i+NN1*(j+1)) - (*y)(i+NN1*j)) + TINY;
        aux      = 0.5*((*vu1)(i+1+NN1*j) + (*vu1)(i+NN1*j)) ;
        auy      = 0.5*((*vu2)(i+NN1*(j+1)) + (*vu2)(i+NN1*j)) ;
        dtimax   = MAX( fabs((aux+4.0*nu1_/dx)/dx), fabs((auy+4.0*nu2_/dy)/dy) );
        dtin     = max(dtin,dtimax);
      }
    }
    u1_list_->next();
    u2_list_->next();
  }
  dtn = 1.0 /(dtin + TINY);

  GComm::Allreduce(&dtn, &dtg, 1, GC_GDOUBLE, G_OP_MIN);


  return dtg;

} // end of method TimeStep
#endif



//************************************************************************************
//************************************************************************************
// METHOD     : SetTimestepHistory
// DESCRIPTION: sets timestep history buffer
//
// ARGUMENTS  : GDBuffer *
// RETURNS    : none.
//************************************************************************************
void NavierStokes::SetTimestepHistory(GDBuffer *dthist)
{ 
  advect_->SetTimestepHistory(dthist);
} // end of method SetTimestepHistory


//************************************************************************************
//************************************************************************************
// METHOD     : SetTime
// DESCRIPTION: sets time
//
// ARGUMENTS  : time
// RETURNS    : none.
//************************************************************************************
void NavierStokes::SetTime(GDOUBLE t)
{ 
  time_ = t;
} // end of method SetTime


//************************************************************************************
//************************************************************************************
// METHOD     : SetDoAdvection
// DESCRIPTION: 
// ARGUMENTS  : bAdvect: TRUE or FALSE
// RETURNS    : none.
//************************************************************************************
void NavierStokes::SetDoAdvection(GBOOL bAdvect)
{
  advect_->SetDoAdvection(bAdvect);
} // end of method SetDoAdvection
    

//************************************************************************************
//************************************************************************************
// METHOD     : SetDoDealiasing
// DESCRIPTION: 
// ARGUMENTS  : bDealias: TRUE or FALSE
// RETURNS    : none.
//************************************************************************************
void NavierStokes::SetDoDealiasing(GBOOL bDealias)
{
  advect_->SetDoDealiasing(bDealias);
} // end of method SetDoDealiasing
    


//************************************************************************************
//************************************************************************************
// METHOD     : SetPreconditioner
// DESCRIPTION: 
// ARGUMENTS  :  iwhich: which solver to set for
//               type  : PC type
// RETURNS    : none.
//************************************************************************************
void NavierStokes::SetPreconditioner(GINT iwhich, GPC type)
{
  stokes_->SetPreconditioner(iwhich, type);
} // end of method SetPreconditioner


//************************************************************************************
//************************************************************************************
// METHOD     : GetSolver
// DESCRIPTION: 
// ARGUMENTS  : iwhich: which linear solver to get 
// RETURNS    : specified solver pointer
//************************************************************************************
CG *NavierStokes::GetSolver(GINT iwhich)
{
  return stokes_->GetSolver(iwhich);
} // end of method GetSolver


//************************************************************************************
//************************************************************************************
// METHOD     : ElemListChange
// DESCRIPTION: Checks element list against local pointer copy to
//              determine if element list has been changed.
// ARGUMENTS  : none
// RETURNS    : TRUE if list has changed; else FALSE
//************************************************************************************
GBOOL NavierStokes::ElemListChange()
{
  GVecOpList *vecop;

  if ( uelems_ == NULL || pelems_ == NULL ) {
    cout << "NavierStokes::ElemListChange: NULL element list" << endl;
    exit(1);
  }
  GINT  i=0, esz=uelems_->size();

  if ( esz != pu_[0].size() || esz != nelems_ ) return TRUE;
  
  vecop = advect_->GetVecOp();
  while ( i<esz && (*uelems_)[i] == ((VectorOp*)(*vecop)[i])->GetElem() ) i++;
    
  return ( i < esz );
} // end of method ElemListChange

    
//************************************************************************************
//************************************************************************************
// METHOD     : SetNSubcycles
// DESCRIPTION: sets max number of OIFS subcycles
//
// ARGUMENTS  : GINT nsub
// RETURNS    : none.
//************************************************************************************
void NavierStokes::SetNSubcycles(GINT nsub)
{
  advect_->SetNSubcycles(nsub);
} // end of method SetNSubcycles


//************************************************************************************
//************************************************************************************
// METHOD     : SetFilter
// DESCRIPTION: sets filter
//
// ARGUMENTS  : GLinOpList *
// RETURNS    : none.
//************************************************************************************
void NavierStokes::SetFilter(GLinOpList *filter)
{
  char *serr = "NavierStokes::SetFilter: ";

  if ( stokes_ == NULL ) {
    cout << serr << "NULL Stokes operator" << endl;
    exit(1);
  }
  stokes_->SetFilter(filter);
} // end of method SetFilter


