//************************************************************************************//
// Module       : pcblockjac_helm.hpp
// Date         : 7/17/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a block Jacobi preconditioner object to be used
//                for preconditioning the Helmholtz operator
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#include "pcblockjac_helm.hpp"
#include "helmholtzop.hpp"
#include "diagop.hpp"
#include "mtk.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
PCBlockJac_Helm::PCBlockJac_Helm()
:             LinOp(),
NN_           (0),
bDeletedHere_ (TRUE),
lc1_          (1.0),
lc2_          (1.0),
mc_           (1.0),
elem_         (NULL),
mask_         (NULL),
Hop_          (NULL)
{
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
PCBlockJac_Helm::PCBlockJac_Helm(LinOp *AA)
:             LinOp(AA),
NN_           (0),
bDeletedHere_ (TRUE),
lc1_          (1.0),
lc2_          (1.0),
mc_           (1.0),
elem_         (NULL),
mask_         (NULL),
Hop_          (NULL)
{
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
PCBlockJac_Helm::PCBlockJac_Helm(Elem2D *e, HelmholtzOp *Hin)
:             LinOp(),
NN_           (0),
bDeletedHere_ (TRUE),
lc1_          (1.0),
lc2_          (1.0),
mc_           (1.0),
elem_         (NULL),
mask_         (NULL),
Hop_          (Hin)
{
  SetElem(e);
  if ( Hin != NULL ) bDeletedHere_ = FALSE;
} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Destructor
PCBlockJac_Helm::~PCBlockJac_Helm()
{
  if ( bDeletedHere_ && Hop_  != NULL ) delete Hop_; 
}


//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Helm::SetElem(Elem2D *e)
{

  GINT  j;
  GBOOL bSim;

  if ( e == NULL ) {
    cout << "PCBlockJac_Helm::SetElem: NULL element" << endl;
    exit(1);
  }

  for ( j=0, bSim=TRUE; j<GDIM; j++ ) bSim = bSim && ( N_[j] == (e->GetOrder(j+1)+1) );
  if ( e == elem_ && bSim ) return;

  elem_ = e;
  for ( j=0,NN_=1; j<GDIM; j++ ) {
    N_[j] = elem_->GetOrder(j+1)+1;
    NN_ *= N_[j];
  }
  H_   .Resize(NN_,NN_);
  iH_  .Resize(NN_,NN_);
  icol_.Resize(NN_);
  rcol_.Resize(NN_);
  if ( Hop_ == NULL ) {
    Hop_ = new HelmholtzOp(elem_);
  }
  if ( !ComputeOp() ) {
    cout << "PCBlockJac_Helm::SetElem: ComputeOp failed" << endl;
    exit(1);
  }

} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetMask
// DESCRIPTION: Sets mask
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Helm::SetMask(GVector *m)
{
  mask_ = m;
} // end of method SetMask


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Helm::SetConst(GDOUBLE clc, GDOUBLE cmc)
{


  if ( lc1_ != clc || lc2_ != clc || mc_ != cmc ) {
    if ( elem_ == NULL ) {
      cout << "PCBlockJac_Helm::SetConst: NULL element" << endl;
      exit(1);
    }
    lc1_ = lc2_ = clc;
    mc_ = cmc;
    if ( bDeletedHere_ ) Hop_->SetConst(lc1_, lc2_,mc_);
    if ( !ComputeOp() ) {
      cout << "PCBlockJac_Helm::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
  }
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Helm::SetConst(GDOUBLE clc1, GDOUBLE clc2, GDOUBLE cmc)
{


  if ( lc1_ != clc1 || lc2_ != clc2 || mc_ != cmc ) {
    if ( elem_ == NULL ) {
      cout << "PCBlockJac_Helm::SetConst: NULL element" << endl;
      exit(1);
    }
    lc1_ = clc1;
    lc2_ = clc2;
    mc_ = cmc;
    if ( bDeletedHere_ ) Hop_->SetConst(lc1_, lc2_,mc_);
    if ( !ComputeOp() ) {
      cout << "PCBlockJac_Helm::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
  }
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
GVector  PCBlockJac_Helm::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);

  OpVec_prod(q, newq);

  return newq;

} // end of method \* oeprator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
void PCBlockJac_Helm::OpVec_prod(GVector &q, GVector &ret)
{

  if ( elem_ == NULL ) {
    cout << "PCBlockJac_Helm::operator*: NULL element or operator" << endl;
    exit(1);
  }

  if ( q.dim() != NN_  || ret.dim() != NN_ ) {
    cout << "PCBlockJac_Helm::operator*: incompatible vector(s)" << endl;
    exit(1);
  }

//ret = (*iH) * q;
  MTK::fmatvec_prod(iH_, q, ret);
  if ( mask_ != NULL )  MTK::fvec_point_prod_rep(ret, *mask_);

} // end of method OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeOp
// DESCRIPTION: Computes inverse Helmholtz operator
// ARGUMENTS  : 
//             
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCBlockJac_Helm::ComputeOp()
{

  GBOOL bRet ;

  if ( elem_ == NULL ) return FALSE;

  switch (elem_->ElemType()) {
    case DEFORMED_QUAD:
      bRet = DefmQuadH();
      break;
    case RECT_QUAD:
      bRet = RectQuadH();
      break;
    case TRIANGULAR:
      bRet = TriangleH();
      break;
    default:
      cout << "LaplacianOp::Constructor(3): Invalid element type" << endl;
      exit(1);
  }

#if 0
  // Done with H, so delete data:
  H_.Resize(0,0);
#endif

  return bRet;
} // end of method ComputeOp



//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadH
// DESCRIPTION: Computes Helmholtz operator*vector  for a deformed quad element.
//              Only diagonal^1 elements of operator are considered.
//              Diag elements are:
//                K_ij = Sum_k[D1_ki ^2  G11_pj  + D2_pj^2 G22_ip]
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCBlockJac_Helm::DefmQuadH()
{ 

  GINT      i, j, k, n, m;
  GVector  *G11, *G12, *G21, *G22, *M, *GA[2];
  GMatrix  *D1 , *D2;

  M   = elem_->GetMassMatrix();  // already includes Jacobian
  D1  = elem_->Get1DDerivMatrix(1,FALSE);
  D2  = elem_->Get1DDerivMatrix(2,FALSE);
  G11 = ((DefQuad2D*)elem_)->GetWJMetric(1,1);
  G12 = ((DefQuad2D*)elem_)->GetWJMetric(1,2);
  G21 = G12;
  G22 = ((DefQuad2D*)elem_)->GetWJMetric(2,2);

  if ( D1  == NULL || D2  == NULL ||
       G11 == NULL || G12 == NULL ||
       G21 == NULL || G22 == NULL ) return FALSE;

 
  GMatrix *d1v, *d2v;

  d1v = new GMatrix(NN_,NN_);
  d2v = new GMatrix(NN_,NN_);

  // Do I2 X D1 term:
  for ( k=0; k<N_[1]; k++ ) {
    for ( i=0; i<N_[0]; i++ ) {
      for ( j=0; j<N_[0]; j++ ) {
        n = k * N_[0] + i;
        m = k * N_[0] + j;
        (*d1v)(n,m) = (*D1)(i,j);
      }
    }
  }

  // Do D2 X I1 term:
  for ( i=0; i<N_[1]; i++ ) {
    for ( j=0; j<N_[1]; j++ ) {
      for ( k=0; k<N_[0]; k++ ) {
        n = i * N_[0] + k;
        m = j * N_[0] + k;
        (*d2v)(n,m) = (*D2)(i,j);
      }
    }
  }

  // Incorporate the metric terms:
  GA[0] = G11;
  GA[1] = G12;
  for ( k=0; k<2; k++ )
    for ( i=0; i<NN_; i++ )
      for ( j=0; j<NN_; j++ )
        (*d1v)(i,j) = (*d1v)(i,j) * (*GA[k])(j);

  GA[0] = G21;
  GA[1] = G22;
  for ( k=0; k<2; k++ )
    for ( i=0; i<NN_; i++ )
      for ( j=0; j<NN_; j++ )
        (*d2v)(i,j) = (*d2v)(i,j) * (*GA[k])(j);
  // Multiply by the appropriate transposes, and add up:
  H_   = ( d1v->Transpose() * ( *d1v + *d2v ) )
       + ( d2v->Transpose() * ( *d1v + *d2v ) );

  // Add in mass matrix:
  for ( k=0; k<NN_; k++ ) {
    H_(k,k) += ( mc_*(*M)(k));
  }
  delete d1v;
  delete d2v;

  // Now, invert to form preconditioner:
  return H_.Inverse(iH_);

} // end of method DefmQuadH


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadH
// DESCRIPTION: Computes Helmholtz operator*vector  for a deformed quad element.
//              Only diagonal^1 elements of operator are considered.
//              Diag elements are:
//                K_ij = 
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCBlockJac_Helm::RectQuadH()
{ 
#if 0
  GINT      i, j, k;
  GDOUBLE     L1, L2, R1, R2, RM;
  Point2D  *vert;
  GVector  *M  , *M1, *M2;  
  GMatrix  *K1 , *K2 ;

  M   = elem_->GetMassMatrix();  // already include Jacobian
  M1  = elem_->Get1DWeights(1);
  M2  = elem_->Get1DWeights(2);
  K1   = elem_->Get1DStiffMatrix(1,FALSE);
  K2   = elem_->Get1DStiffMatrix(2,FALSE);
  vert = elem_->GetSpVertices();
  L1   = fabs((vert+1)->x1 - (vert)->x1);
  L2   = fabs((vert+3)->x2 - (vert)->x2);

  if ( L1 == 0 || L2 == 0 ) return FALSE;
  
  R1  = lc1 * L2 / L1;
  R2  = lc2 * L1 / L2;
  RM  = 0.25 * mc_ * L1 * L2;

 
  if ( K1  == NULL || K2  == NULL ||
       M1  == NULL || M2  == NULL ||
       M   == NULL                 ) return FALSE;
  
  H_ = 0.0;
  // Do M2 X K1 term:
  for ( k=0; k<N_[1]; k++ ) {
    for ( i=0; i<N_[0]; i++ ) {
      for ( j=0; j<N_[0]; j++ ) {
        n = k * N_[0] + i;
        m = k * N_[0] + j;
        (*Lap)(n,m) = R1*(*M2)(k) * (*K1)(i,j);
      }
    }
  }

  // Do K2 X M1 term:
  for ( i=0; i<N_[1]; i++ ) {
    for ( j=0; j<N_[1]; j++ ) {
      for ( k=0; k<N_[0]; k++ ) {
        n = i * N_[0] + k;
        m = j * N_[0] + k;
        (*Lap)(n,m) += ( R2*(*M1)(k) * (*K2)(i,j);
      }
    }
  }

  // Add in mass matrix:
  for ( k=0; k<NN_; k++ ) {
    (*Lap)(k,k) += ( mc_*(*MassMatrix)(k));
  }
#endif

  GSHORT   i, j;

  for ( j=0; j<NN_; j++ ) {
    icol_(j) = 1.0 ;
    Hop_->OpVec_prod(icol_, rcol_);
    icol_(j) = 0.0 ;
    for ( i=0; i<NN_; i++ ) H_(i,j) = rcol_(i);
//cout << "PCBlockJac_Helm::RectQuadH: H[" << j << "]=" << H_ << endl;
//cout << "PCBlockJac_Helm::RectQuadH: rcol[" << j << "]=" << rcol_ << endl;
  }

  
  // Now, invert to form preconditioner:
#if 0
  H_.Inverse(iH_);
  GMatrix P(NN_,NN_);
  P = H_ * iH_;
  cout << " P = " << P << endl;
  return TRUE;
#endif

  return H_.Inverse(iH_);

} // end of method RectQuadH


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleH
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCBlockJac_Helm::TriangleH()
{
  return FALSE;
} // end of method TriangleH


