//************************************************************************************//
// Module       : guser_gauss_advect.cpp
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
//************************************************************************************//
#include "gtypes.h"
#include "gaspar_t.h"
#include "guser.h"

//############################################################################################
//############################################################################################
//                                  Problem initialization:
//############################################################################################
//############################################################################################


//#define C(x,y,t,idir)       ( idir==1 ? R0*cos(t) : R0*sin(t) )
#define C(x,y,t,idir)       ( idir==1 ? 1 : 0 )
#define PERGALUMP
//************************************************************************************
//************************************************************************************
// METHOD     : GUserConfig
// DESCRIPTION: User-furnished method to configure the code. This method is
//              called during setup, before GUserInit.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserConfig()
{

  bLinAdvection_ = TRUE;


  // Where appropriate, set the number of quantities to solve for:
  nEvolvedFields_ = 1;

  return TRUE;
} // end of method GUserConfig


//************************************************************************************
//************************************************************************************
// METHOD     : GUserInit
// DESCRIPTION: User-furnished method to initialize lump-advection problems.
//              In this problem, we initialize with a single Gaussian 'sphere', 
//              with Dirichlet or periodic boundary conditions. PERIODIC bcs
//              can be used by defineing PERGALUMP; else undef PERGALUMP.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserInit(int argc, char *argv[])
{
  GINT       i, j, k, m, nlevels, nrefine=0;
  GDOUBLE    x, y, xtheta, ytheta, Rp[2], t0 ;
  GBOOL      bRefining;
  GIBuffer   *bdyindices=NULL;
  GBTBuffer  *bdytype=NULL;
  Elem2D     *e;

#if 1
  for ( i=0; i<nelems_; i++ ) {
    u1a. add(1,uelems.member(i));
    delu.add(1,uelems.member(i));
  }
  if ( GRegisterUserFields(1, &u1a, "V1A") == 0 ) {
   cout << "GUserInit: user field V1A registration failed" << endl;
   exit(1);
 }
  if ( GRegisterUserFields(1, &delu, "delV") == 0 ) {
   cout << "GUserInit: user field delu registration failed" << endl;
   exit(1);
 }
#endif

  bAutoTimeLevelFill_ = FALSE;
  nlevels = ntimelevels_;
  if ( bAutoTimeLevelFill_ ) nlevels = 1;

  if ( !bLinAdvection_ ) {
    cout << "GUserInit: This problem requires a '-linadvect' command line option" << endl;
    exit(1);
  }
  kx         = 2.0*PI / PDISTANCE(gd_[1],gd_[0]);
  ky         = 2.0*PI / PDISTANCE(gd_[2],gd_[1]);
  gL_[0]     = PDISTANCE(gd_[1],gd_[0]);
  gL_[1]     = PDISTANCE(gd_[2],gd_[1]);
//kx         = 1.0;
//ky         = 1.0;
  Rp[0]      = 0.5*(gd_[0].x1 + gd_[1].x1);  // center of coord sys--circle
  Rp[1]      = 0.5*(gd_[0].x2 + gd_[3].x2);
  a1.Resize(1); a2.Resize(1);
  xp.Resize(1); yp.Resize(1);

#if defined(PERGALUMP)
  a1[0]     = 0.05       ; a2[0]     = 0.05       ;  // for periodic Gaussian
#else
  a1[0]     = 0.05       ; a2[0]     = 0.05       ;  // for Dirichlet Gaussian
#endif

  t0 = 0.0;
  if ( bFixedTimeStep_ )
    dt_       /= (pow(2.0,(GINT)aux_));
  else
    Courant_  /= (pow(2.0,(GINT)aux_));


  // xp is the position of the lumps:
//xp[0]     = 0.25*(gd_[0].x1 + gd_[1].x1); yp[0]    = 0.5*(gd_[0].x2 + gd_[3].x2);
  xp[0]     = 0.5 *(gd_[0].x1 + gd_[1].x1); yp[0]    = 0.5*(gd_[0].x2 + gd_[3].x2);
  R0        = sqrt( pow(Rp[0]-xp[0],2) + pow(Rp[1]-yp[0],2) );

  bRefining  = TRUE;
  bDoCoarse_ = FALSE;
  while ( bRefining ) {
cout << "____________________________________________________________________________________" << endl;
cout << "                                  refine_cycle=" << nrefine << " nelems=" << nelems_ << endl;

    time_ = t0;
    time_ -= dt_;
    for ( k=nlevels-1; k>=0 ; k-- ) {
      time_ += dt_;
      for ( i=0; i<nelems_; i++ ) {
        for ( j=0; j<u1[i]->GetExpCoeffs(k)->dim(); j++ ) {
          x           = u1.X(i,1,j);
          y           = u1.X(i,2,j);
          xtheta      = ( x - xp[0] );
          ytheta      = ( y - yp[0] );
          (*cadv_[0])(k,i,j) = C(x,y,time_,1);
          (*cadv_[1])(k,i,j) = C(x,y,time_,2);
#if defined(PERGALUMP)
          u1 (k,i,j) = perGalump(xtheta,ytheta,time_,a1[0], 
                      (*cadv_[0])(k,i,j), (*cadv_[1])(k,i,j));
#else
          u1 (k,i,j) = dirGalump(xtheta,ytheta,time_,a1[0], 
                      (*cadv_[0])(k,i,j), (*cadv_[1])(k,i,j));
#endif
        } // end, node loop
        u1[i]->SetTime    (k,time_);
      } // end, element loop

      if ( k == (nlevels-1) ) GTimeStep(k);
      dthist_[k] = dt_;
    } // end, time level loop

#if 0
    // Smooth initial conditions along child faces:
    for ( j=0; j<nlevels; j++ ) {
      for ( i=0; i<nelems_; i++ ) {
         uptmp[i] = u1[i]->GetExpCoeffs(j);   
      }
      GUtils::H1_Project(uptmp, uelems, (NTreeAdapt*)glop, hDSOp);
    }
    bRefining = FALSE;
#endif

//  Set DIRICHLET BC vector
    for ( i=0; i<nelems_; i++ ) {
      e = uelems[i];
      bdyindices = e->GetBdyIndices();
      bdytype    = e->GetBdyTypes  ();
      u1bdyvals[i]->Resize(bdyindices->dim());  // Remember, ubdyvals points to member data in Field

      for ( j=0; j<bdyindices->dim(); j++ ) {
        m = (*bdyindices)(j);
        if ( (*bdytype)(j) == NOSLIP ) {
          u1bdyvals(i,j) = 0.0;
        }
        if ( (*bdytype)(j) == DIRICHLET ) {
          x           = u1.X(i,1,m);
          y           = u1.X(i,2,m);
          xtheta      = ( x - xp[0] );
          ytheta      = ( y - yp[0] );
          u1bdyvals(i,j) = dirGalump(xtheta,ytheta,time_,a1[0],
                          (*cadv_[0])(0,i,m), (*cadv_[1])(0,i,m));
        }
      }
#if defined(GU_DEBUG_OUTPUT)
     cout << "GUser::u1bdyvals[" << i << "]=" << *u1bdyvals[i] << endl;
#endif
    }

#if 1
    if ( !GAdaptGrid() ) {
      cout << "GUser:: adaption failed" << endl;
      exit(1);
    }
#else
    bElemListChange_ = FALSE;
#endif
    bRefining = bElemListChange_;
    bElemListChange_ = FALSE;
cout << "                                                       refine_cycle=" << nrefine << endl;
cout << "____________________________________________________________________________________" << endl;
    nrefine++;
  } // end of refinement loop

  bElemListChange_ = TRUE;
  GUserTimeDep(time_, dt_);  // must be called to synch cadv_
  
//bFixedTimeStep_ = TRUE;
  bDoCoarse_ = TRUE;

  return TRUE;

} // end of method GUserInit
//############################################################################################
//############################################################################################

//************************************************************************************
//************************************************************************************
// METHOD     : GUserStart
// DESCRIPTION: User-furnished method to provide any information required after
//              a start or restart. For example, the method may set an analytic
//              solution, or set bdy conditions from an analytic solution, etc.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserStart()
{ 
  GINT       i, j, m;    
  GDOUBLE    x, y;
  GIBuffer   *bdyindices=NULL;
  GBTBuffer  *bdytype=NULL;
  Elem2D     *e;

  if ( !bLinAdvection_ ) {
    cout << "GUserInit: This problem requires a '-linadvect' command line option" << endl;
    exit(1);
  }
  
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<u1[i]->GetExpCoeffs(0)->dim(); j++ ) {
      x = u1.X(i,1,j);
      y = u1.X(i,2,j);
      (*cadv_[0])(0,i,j) = C(x,y,time_,1); 
      (*cadv_[1])(0,i,j) = C(x,y,time_,2); 
    }
  }

  for ( i=0; i<nelems_; i++ ) {
    e = uelems[i]; 
    bdyindices = e->GetBdyIndices();
    bdytype    = e->GetBdyTypes  ();
    u1bdyvals[i]->Resize(bdyindices->dim());  // Remember, ubdyvals points to member data in Field

    for ( j=0; j<bdyindices->dim(); j++ ) {
      m = (*bdyindices)(j);
      if ( (*bdytype)(j) == NOSLIP ) {
        u1bdyvals(i,j) = 0.0;
      }
      if ( (*bdytype)(j) == DIRICHLET ) {
        x           = u1.X(i,1,m) - xp[0];
        y           = u1.X(i,2,m) - yp[0];
        u1bdyvals(i,j) = dirGalump(x,y,time_,a1[0],
                        (*cadv_[0])(0,i,m), (*cadv_[1])(0,i,m));
      }
    }
  }

  return TRUE;
} // end of method GUserStart


//************************************************************************************
//************************************************************************************
// METHOD     : GUserTimeDep
// DESCRIPTION: User-furnished method to provide any information required 
//              during time stepping. This method is called at the start 
//              of the time step, before the solver takes a step. 
//              For example, time dependent boundary conditions, 
//              or a non-steady analytic solution might be computed here.
// ARGUMENTS  : ptime: current time
//              pdt  : current time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserTimeDep(GDOUBLE ptime, GDOUBLE pdt)
{ 
  GINT       i, j, k, m;
  GDOUBLE    x, y, tt, *f[2];
  GIBuffer   *bdyindices=NULL;
  GBTBuffer  *bdytype=NULL;
  Elem2D     *e;

  // If there is a grid adaption, reset the advection
  // velocity at all time levels:
  for ( k=0; k<ntimelevels_ && bElemListChange_; k++ ) {
    for ( i=0; i<nelems_; i++ ) {
      tt = u1[i]->GetTime(k);
      f[0] = (*cadv_[0])[i]->GetExpCoeffs(k)->Data();
      f[1] = (*cadv_[1])[i]->GetExpCoeffs(k)->Data();
      for ( j=0; j<u1[i]->GetExpCoeffs(0)->dim(); j++ ) {
        f[0][j] = 1.0;
        f[1][j] = 0.0;
      }
    }
  }

  tt = ptime + pdt;

  // Do analytic solution:
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<u1[i]->GetExpCoeffs(0)->dim(); j++ ) {
      x  = ( u1.X(i,1,j) - xp[0] );
      y  = ( u1.X(i,2,j) - yp[0] );
#if 1
#if defined(PERGALUMP)
          u1a(0,i,j) = perGalump(x,y,tt,a1[0], 
                      (*cadv_[0])(0,i,j), (*cadv_[1])(0,i,j));
#else
          u1a(0,i,j) = dirGalump(x,y,tt,a1[0], 
                      (*cadv_[0])(0,i,j), (*cadv_[1])(0,i,j));
#endif
#endif
    }
  }

  for ( i=0; i<nelems_; i++ ) {
    e = uelems[i];
    bdyindices = e->GetBdyIndices();
    bdytype    = e->GetBdyTypes  ();
    u1bdyvals[i]->Resize(bdyindices->dim());  // Remember, ubdyvals points to member data in Field

    for ( j=0; j<bdyindices->dim(); j++ ) {
      m = (*bdyindices)(j);
      if ( (*bdytype)(j) == NOSLIP ) {
        u1bdyvals(i,j) = 0.0;
      }
      if ( (*bdytype)(j) == DIRICHLET ) {
        x           = u1.X(i,1,m) - xp[0];
        y           = u1.X(i,2,m) - yp[0];
        u1bdyvals(i,j) = dirGalump(x,y,tt,a1[0],
                        (*cadv_[0])(0,i,m), (*cadv_[1])(0,i,m));
      }
    }
  }

  if ( !GSetBdyCond() ) {
    cout << "GUserTimeDep: GSetBdyCond failed" << endl;
    return FALSE;
  }

  return TRUE;
} // end of method GUserTimeDep


//************************************************************************************
//************************************************************************************
// METHOD     : GUserLogConfig
// DESCRIPTION: Provides user with a place to configure static and dynamic components
//              of log file.  Data variables set in the SetStaticParams call must be 
//              globalized, and this may be done by placing them in the 'guser.h' header.

// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserLogConfig()
{
  GBOOL bOk;

  bOk = glogger_.SetDynamicParamDesc("%f %f"
                                   , "#Energy"
                                   , "#L2_ERR"
                                    );
  bOk = bOk &
        glogger_.SetDynamicData(2 , &KE, &errL2
                                 );

  return bOk;
} // end of method GUserLogConfig


//************************************************************************************
//************************************************************************************
// METHOD     : GUserLogUpdate
// DESCRIPTION: Provides user with a place to update variables that are
//              printed in the dynamic log component of the log file.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserLogUpdate()
{
  GINT       i, j, k, NN;
  GDOUBLE    *v, *vke, lE[1], gE[1];
    
  errL2 = ComputeL2Error();
  
  // Compute KE:
  for ( k=0; k<nEvolvedFields_; k++ ) pgfields_[k]->start(NULL);
  utmp.start(NULL);
  for ( i=0; i<uelems.size(); i++ ) {
    NN  = pgfields_[0]->member()->GetExpCoeffs(0)->dim();
    v   = pgfields_[0]->member()->GetExpCoeffs(0)->Data();
    vke = utmp.member()->Data();
    for ( j=0; j<NN; j++ ) vke[j] = v[j]*v[j] ;
    for ( k=1; k<nEvolvedFields_; k++ ) {
      v  = pgfields_[k]->member()->GetExpCoeffs(0)->Data();
      for ( j=0; j<NN; j++ ) vke[j] += 0.5*v[j]*v[j] ;
    }
    for ( k=0; k<nEvolvedFields_; k++ ) pgfields_[k]->next();
    utmp.next();
  }

  GUtils::DoLocal(TRUE);  // turn off internal communication
  lE[0]  = GUtils::ComputeGlobalIntegralV(utmp, uelems);
  GUtils::DoLocal(FALSE);
  GComm::Allreduce(lE, gE, 1, GC_GDOUBLE, G_OP_SUM);

  KE        = gE[0];

  return TRUE;
} // end of method GUserLogUpdate


//************************************************************************************
//************************************************************************************
// METHOD     : GUserTerm
// DESCRIPTION: Provids user with a place to perform some activity at the
//              end of time stepping
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserTerm()
{
  GINT      pe;
  ofstream  os;
  os.open("../gauss.conv",ios::app);

  GUserLogUpdate(); // compute new error

  pe = uelems[0]->GetOrder(1);

  // write error to file:
  os << pe << " " << errL2 << endl; 
  os.close();


  return TRUE;
} // end of method GUserTerm


//************************************************************************************
//************************************************************************************
// METHOD     : dirGalump
// DESCRIPTION: unperiodized analytic galump solution as a function of time.
//                s0^2/s(t)^2 * exp( -((x-x0-cx t)^2 + (y-y0-cy t)^2)/s(t)^2 ) 
//              where
//               s(t)^2 = s0^2 + 2 t nu, is the FWHM as a fcn of t.
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GDOUBLE dirGalump(GDOUBLE x, GDOUBLE y, GDOUBLE t, GDOUBLE sig_0, GDOUBLE cx, GDOUBLE cy)
{ 
  GDOUBLE   galump, sig_t, argr, decayfact;
  GBOOL     bContin=TRUE;

  sig_t     = sqrt(sig_0*sig_0 + 2.0*t*nu_[0]);
  argr      = (x-cx*t)*(x-cx*t) + (y-cy*t)*(y-cy*t);
  decayfact = exp( -argr/(2.0*sig_t*sig_t) );
  
  galump    = pow(sig_0,2)/pow(sig_t,2) * decayfact;

  return galump;
} // end of method dirGalump


//************************************************************************************
//************************************************************************************
// METHOD     : perGalump
// DESCRIPTION: periodized analytic galump solution as a function of time.
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GDOUBLE perGalump(GDOUBLE x, GDOUBLE y, GDOUBLE t, GDOUBLE sig_0, GDOUBLE cx, GDOUBLE cy)
{ 

  GINT    k, n;
  GDOUBLE wsum, prod, argxp, argxm, da, eps=1e-18;
  GDOUBLE xx[2], si[2], sig[2];
  GBOOL   bContin;

  sig[0] = sqrt(sig_0*sig_0 + 2.0*t*nu_[0]);
  sig[1] = sqrt(sig_0*sig_0 + 2.0*t*nu_[1]);
  si [0] = 0.5/(sig[0]*sig[0]);
  si [1] = 0.5/(sig[1]*sig[1]);
  xx [0] = x - cx*t;
  xx [1] = y - cy*t;
    
  prod = 1.0;
  for ( k=0; k<2; k++ ) {
    wsum     = exp(-xx[k]*xx[k]*si[k]);
    n       = 1;
    bContin = TRUE;
    while ( bContin ) {
      argxp = -pow((xx[k]+n*gL_[k]),2.0)*si[k];
      argxm = -pow((xx[k]-n*gL_[k]),2.0)*si[k];
      da    =  exp(argxp) + exp(argxm);
      wsum  += da;
      bContin = da/wsum > eps;
      n++;
    }
    prod *= wsum;
  }

  return  pow(sig_0,2)/pow(sig[0],2)*prod;


}  // end of method perGalump


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeL2Error
// DESCRIPTION: Computes L2 error
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GDOUBLE ComputeL2Error()
{ 

  GINT       i, j, m; 
  GDOUBLE    xtheta, ytheta, errnum, errden;
    
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<u1[i]->GetExpCoeffs(0)->dim(); j++ ) {
      xtheta      = ( u1.X(i,1,j) - xp[0] );
      ytheta      = ( u1.X(i,2,j) - yp[0] );
#if 1
#if defined(PERGALUMP)
      u1a(0,i,j) = perGalump(xtheta,ytheta,time_,a1[0], 
                   (*cadv_[0])(0,i,j), (*cadv_[1])(0,i,j));
#else
      u1a(0,i,j) = dirGalump(xtheta,ytheta,time_,a1[0],
                   (*cadv_[0])(0,i,j), (*cadv_[1])(0,i,j));
#endif
#endif 
      delu(0,i,j) = u1(0,i,j) - u1a(0,i,j);   
    }
  } 
  errden = GUtils::ComputeL2Norm(u1a , 0);
  errnum = GUtils::ComputeL2Norm(delu, 0);

  return (errnum / errden);

}  // end of method ComputeL2Error

