!BOP
!
! !MODULE: stepon -- FV Dynamics specific time-stepping
!
! !INTERFACE:
module stepon

! !USES:
! from cam
   use shr_kind_mod,   only: r8 => shr_kind_r8
   use shr_sys_mod,    only: shr_sys_flush
   use pmgrid,         only: plev, plevp, plat
   use spmd_utils,     only: iam, masterproc
   use constituents,   only: pcnst
   use abortutils,     only: endrun
   use ppgrid,         only: begchunk, endchunk
   use physconst,      only: zvir, cappa
   use physics_types,  only: physics_state, physics_tend
   use dyn_comp,       only: dyn_import_t, dyn_export_t
   use perf_mod
! from homme
   use derivative_mod, only : derivinit, deriv_print, derivative_t
   use quadrature_mod, only : gauss, gausslobatto, quadrature_t
   use edge_mod,       only: EdgeBuffer_t, initEdgeBuffer, FreeEdgeBuffer, &
                             edgeVpack, edgeVunpack
 
   implicit none

   private

!
! !PUBLIC MEMBER FUNCTIONS:
!
  public stepon_init   ! Initialization
  public stepon_run1    ! run method phase 1
  public stepon_run2    ! run method phase 2
  public stepon_run3    ! run method phase 3
  public stepon_final  ! Finalization

!----------------------------------------------------------------------
!
! !DESCRIPTION: Module for dynamics time-stepping.
!
! !REVISION HISTORY:
!
! 2006.05.31  JPE    Created
!
!EOP
!----------------------------------------------------------------------
!BOC
!
! !PRIVATE DATA MEMBERS:
!
  type (derivative_t)   :: deriv           ! derivative struct
  type (quadrature_t)   :: gv,gp           ! quadratures on velocity and pressure grids
  integer :: nets, nete
  type (EdgeBuffer_t) :: edgebuf              ! edge buffer
!-----------------------------------------------------------------------


CONTAINS

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!----------------------------------------------------------------------- 
!BOP
! !ROUTINE:  stepon_init --- Time stepping initialization
!
! !INTERFACE:
subroutine stepon_init( gw, etamid, dyn_in, dyn_out )
! !USES:
  use dimensions_mod, only: nlev
  use hycoef,             only: hyam, hybm


! !OUTPUT PARAMETERS
!
  real(r8), intent(out) :: gw(plat)           ! Gaussian weights
  real(r8), intent(out) :: etamid(plev)       ! vertical coords at midpoints
  type (dyn_import_t), intent(out) :: dyn_in  ! Dynamics import container
  type (dyn_export_t), intent(out) :: dyn_out ! Dynamics export container

! !DESCRIPTION:
!
! Allocate data, initialize values, setup grid locations and other
! work needed to prepare the dynamics to run. Return weights and 
! vertical coords to atmosphere calling program.
!
!EOP
!-----------------------------------------------------------------------
!BOC









  call initEdgeBuffer(edgebuf, (4+pcnst)*nlev+2)

  etamid(:) = hyam(:) + hybm(:)

end subroutine stepon_init

!-----------------------------------------------------------------------

!----------------------------------------------------------------------- 
!BOP
! !ROUTINE:  stepon_run1 -- Phase 1 of dynamics run method.
!
! !INTERFACE:
subroutine stepon_run1( dtime_out, phys_state, phys_tend,               &
                        dyn_in, dyn_out )
  use phys_buffer,    only: pbuf
  use dp_coupling, only: d_p_coupling
  use time_mod,    only: tstep
  implicit none
!
! !OUTPUT PARAMETERS:
!

   real(r8), intent(out) :: dtime_out   ! Time-step
   type(physics_state), intent(inout):: phys_state(begchunk:endchunk)
   type(physics_tend), intent(out):: phys_tend(begchunk:endchunk)
   type (dyn_import_t), intent(out) :: dyn_in  ! Dynamics import container
   type (dyn_export_t), intent(out) :: dyn_out ! Dynamics export container

! !DESCRIPTION:
!
!  Phase 1 run of dynamics. Run the dynamics and couple to physics.
!
!EOP
!-----------------------------------------------------------------------
!BOC
   dtime_out = tstep
   if(tstep <= 0)  stop 'bad tstep'
   !----------------------------------------------------------
   ! Move data into phys_state structure.
   !----------------------------------------------------------
   
   call d_p_coupling (phys_state, phys_tend, pbuf, dyn_out )
   
end subroutine stepon_run1

subroutine stepon_run2(phys_state, phys_tend, dyn_in, dyn_out )
   use bndry_mod,   only: bndry_exchangeV
   use dimensions_mod, only: nlev, nelemd, nv, np
   use dp_coupling,    only: p_d_coupling
   use parallel_mod,   only: par
   use dyn_comp,       only: TimeLevel, PhysTimeLevel
   use phys_buffer,    only: pbuf
   use time_manager,     only: dtime   ! physics timestep

   type(physics_state), intent(inout):: phys_state(begchunk:endchunk)
   type(physics_tend), intent(out):: phys_tend(begchunk:endchunk)
   type (dyn_import_t), intent(out) :: dyn_in  ! Dynamics import container
   type (dyn_export_t), intent(out) :: dyn_out ! Dynamics export container
   integer :: kptr, ie, ic, i, j, k
   real(r8) :: rec2dt


   Call t_startf('dp_coupling_total')
   call t_startf('pd_coupling')
 
   ! copy from phys structures -> dynamics structures
   call p_d_coupling(phys_state, phys_tend, pbuf, dyn_in)

   call t_startf('bndry_exchange')
   ! do boundary exchange
   do ie=1,nelemd
      kptr=0
      call edgeVpack(edgebuf,dyn_in%elem(ie)%state%lnps(:,:,TimeLevel%nm1),1,kptr,dyn_in%elem(ie)%desc)
      kptr=kptr+1
      call edgeVpack(edgebuf,dyn_in%elem(ie)%state%phis, 1, kptr, dyn_in%elem(ie)%desc)
      kptr=kptr+1
      call edgeVpack(edgebuf,dyn_in%elem(ie)%state%FM(:,:,:,:,PhysTimeLevel%nm1),2*nlev,kptr,dyn_in%elem(ie)%desc)
      kptr=kptr+2*nlev
      call edgeVpack(edgebuf,dyn_in%elem(ie)%state%FT(:,:,:,PhysTimeLevel%nm1),nlev,kptr,dyn_in%elem(ie)%desc)
      kptr=kptr+nlev
      call edgeVpack(edgebuf,dyn_in%elem(ie)%state%omega_p,nlev,kptr,dyn_in%elem(ie)%desc)
      kptr=kptr+nlev
#ifdef QFLUX
      call edgeVpack(edgebuf,dyn_in%elem(ie)%state%FQ(:,:,:,:,PhysTimeLevel%nm1),nlev*pcnst,kptr,dyn_in%elem(ie)%desc)
#else
      call edgeVpack(edgebuf,dyn_in%elem(ie)%state%Q(:,:,:,:,TimeLevel%nm1),nlev*pcnst,kptr,dyn_in%elem(ie)%desc)
#endif
   end do

   call bndry_exchangeV(par, edgebuf)
   rec2dt = 1./(2_r8*dtime)

   do ie=1,nelemd
      kptr=0
      call edgeVunpack(edgebuf,dyn_in%elem(ie)%state%lnps(:,:,TimeLevel%nm1),1,kptr,dyn_in%elem(ie)%desc)
      kptr=kptr+1
      call edgeVunpack(edgebuf,dyn_in%elem(ie)%state%phis, 1, kptr, dyn_in%elem(ie)%desc)
      kptr=kptr+1
      call edgeVunpack(edgebuf,dyn_in%elem(ie)%state%FM(:,:,:,:,PhysTimeLevel%nm1),2*nlev,kptr,dyn_in%elem(ie)%desc)
      kptr=kptr+2*nlev
      call edgeVunpack(edgebuf,dyn_in%elem(ie)%state%FT(:,:,:,PhysTimeLevel%nm1),nlev,kptr,dyn_in%elem(ie)%desc)
      kptr=kptr+nlev
      call edgeVunpack(edgebuf,dyn_in%elem(ie)%state%omega_p,nlev,kptr,dyn_in%elem(ie)%desc)
      kptr=kptr+nlev
#ifdef QFLUX
      call edgeVunpack(edgebuf,dyn_in%elem(ie)%state%FQ(:,:,:,:,PhysTimeLevel%nm1),nlev*pcnst,kptr,dyn_in%elem(ie)%desc)
      do ic=1,pcnst
         do k=1,nlev
            do j=1,nv
               do i=1,nv
                  dyn_in%elem(ie)%state%FQ(i,j,k,ic,PhysTimeLevel%nm1)=(dyn_in%elem(ie)%state%FQ(i,j,k,ic,PhysTimeLevel%nm1) - &
                       dyn_in%elem(ie)%state%Q(i,j,k,ic,TimeLevel%nm1))*rec2dt
               end do
            end do
         end do
      end do
#else
      call edgeVunpack(edgebuf,dyn_in%elem(ie)%state%Q(:,:,:,:,TimeLevel%nm1),nlev*pcnst,kptr,dyn_in%elem(ie)%desc)


#endif
   end do
   call t_stopf('bndry_exchange')
	
   call t_stopf('pd_coupling')
   call t_stopf('dp_coupling_total')

end subroutine stepon_run2


subroutine stepon_run3( dtime, etamid, cam_out, phys_state, dyn_in, dyn_out )
   use camsrfexch_types, only: surface_state
   use dyn_comp,    only: dyn_run
   use time_mod,    only: tstep
   real(r8), intent(in) :: dtime   ! Time-step
   real(r8), intent(in)  :: etamid(plev)
   type(surface_state), intent(inout) :: cam_out(:) ! Output from CAM to surface
   type(physics_state), intent(inout):: phys_state(begchunk:endchunk)
   type (dyn_import_t), intent(out) :: dyn_in  ! Dynamics import container
   type (dyn_export_t), intent(out) :: dyn_out ! Dynamics export container
   integer :: rc
  
   call dyn_run(dyn_out,rc)	

end subroutine stepon_run3


!----------------------------------------------------------------------- 
!BOP
! !ROUTINE:  stepon_final --- Dynamics finalization
!
! !INTERFACE:
subroutine stepon_final(dyn_in, dyn_out)

! !PARAMETERS:
  type (dyn_import_t), intent(out) :: dyn_in  ! Dynamics import container
  type (dyn_export_t), intent(out) :: dyn_out ! Dynamics export container
!
! !DESCRIPTION:
!
! Deallocate data needed for dynamics. Finalize any dynamics specific
! files or subroutines.
!
!EOP
!-----------------------------------------------------------------------
!BOC


!EOC
end subroutine stepon_final

!-----------------------------------------------------------------------


end module stepon
