#!/usr/bin/wish -f
#
# This code may (or may not) be part of the COAMPS distribution,
# So it is not protected by the DART copyright agreement.
#
# DART $Id$
#
########################################################################

##############################
# BEGIN PROCEDURES
##############################

# Notebook taken from "Effective Tcl/Tk Programming"

# notebook_create
# ---------------
# Initializes a notebook with the given name and no pages
proc notebook_create {win} {
    global nbInfo

    frame $win -class Notebook
    pack propagate $win 0
    set nbInfo($win-count) 0
    set nbInfo($win-pages) ""
    set nbInfo($win-current) ""
    return $win
}

# notebook_page
# -------------
# Given the name of the notebook and the name of the page, initialize
# that page and add to the notebook
proc notebook_page {win name} {
    global nbInfo

    # names are .nb.page1, .nb.page2, etc.
    set page "$win.page[incr nbInfo($win-count)]"
    lappend nbInfo($win-pages) $page
    set nbInfo($win-page-$name) $page
    frame $page
    if { $nbInfo($win-count) == 1 } {
        after idle [list notebook_display $win $name]
    }
    return $page
}

# notebook_display
# ----------------
# Given the notebook and the page name, displays that page
proc notebook_display {win name} {
    global nbInfo

    set page ""
    if { [info exists nbInfo($win-page-$name)] } {
        set page $nbInfo($win-page-$name)
    } elseif { [winfo exists $win.page$name] } {
        set page $win.page$name
    } elseif { [winfo exists $name ] } {
        set page $name
    }
    if {$page == ""} {
        error "Bad notebook page \"$name\""
    }

    notebook_fix_size $win

    if { $nbInfo($win-current) != "" } {
        pack forget $nbInfo($win-current)
    }
    pack $page -expand yes -fill both
    set nbInfo($win-current) $page
}

# notebook_fix_size
# -----------------
# Given the name of the notebook, scan all the pages and set the
# notebook size to handle the largest page
proc notebook_fix_size {win} {
    global nbInfo

    update idletasks

    set maxWidth  0
    set maxHeight 0

    foreach page $nbInfo($win-pages) {
        set pageWidth [winfo reqwidth $page]
        if { $pageWidth > $maxWidth } {
            set maxWidth $pageWidth
        }

        set pageHeight [winfo reqheight $page]
        if { $pageHeight > $maxHeight } {
            set maxHeight $pageHeight
        }
    }

    set borderWidth [$win cget -borderwidth]
    set maxWidth  [expr $maxWidth + 2 * $borderWidth]
    set maxHeight [expr $maxHeight + 2 * $borderWidth]
    $win configure -width $maxWidth -height $maxHeight
}

# Navigation routines
# -------------------
proc enableNext {} {
    upvar #0 next nextButton
    $nextButton configure -state normal
}
proc enableFinish {} {
    upvar #0 finish finishButton
    $finishButton configure -state normal
}
proc enableBack {} {
    upvar #0 back backButton
    $backButton configure -state normal
}
proc disableNext {} {
    upvar #0 next nextButton
    $nextButton configure -state disabled
}
proc disableFinish {} {
    upvar #0 finish finishButton
    $finishButton configure -state disabled
}
proc disableBack {} {
    upvar #0 back backButton
    $backButton configure -state disabled
}
proc goBack {} {
    global nbInfo
    upvar  wizardNotebook win

    set currentPageName $nbInfo($win-current)
    set currentPageIndex [ lsearch $nbInfo($win-pages) $currentPageName ]
    set firstIndex 0

    if { $currentPageIndex == $firstIndex } { return }

    set backPageIndex [expr $currentPageIndex - 1]

    if { $backPageIndex == $firstIndex } { disableBack }
    enableNext
    disableFinish

    notebook_display $win [lindex $nbInfo($win-pages) $backPageIndex]
}
proc goNext {} {
    global nbInfo
    upvar  wizardNotebook win

    set currentPageName $nbInfo($win-current)
    set currentPageIndex [ lsearch $nbInfo($win-pages) $currentPageName ]
    set lastIndex [expr [llength $nbInfo($win-pages)] - 1]

    if { $currentPageIndex == $lastIndex } { return }

    set nextPageIndex [expr $currentPageIndex + 1]

    if { $nextPageIndex == $lastIndex } { disableNext; enableFinish }
    enableBack

    notebook_display $win [lindex $nbInfo($win-pages) $nextPageIndex]
}
proc goFinish {} { 
    global varData
    global useMeanState
    global numModelLevels

    upvar #0 listUsed varsUsed

    set datFile [open "wizard_generated.restart.dat" w]
    puts $datFile "# This file automatically generated by statevec_wizard GUI\n"

    puts $datFile "### User-selected state variables"
    foreach {varName} $varsUsed {
        puts $datFile "\n# $varName"
        writeRestartData $datFile \
                         $varData($varName-coampsName) \
                         $numModelLevels \
                         $varData($varName-pertType) \
                         $varData($varName-pertMag) \
                         $varData($varName-levelType) \
                         $varData($varName-dartKind) \
                         "UPDATE" \
                         "FALSE" \
                         $varData($varName-canBeNeg)
    }

    if { $useMeanState == 1 } {
        puts $datFile "\n### Model mean state"
        foreach {coampsName levelType dartKind} {
            "exbm" "M" "QTY_EXNER_FUNCTION"
            "exbw" "W" "QTY_EXNER_FUNCTION"
            "thbm" "M" "QTY_POTENTIAL_TEMPERATURE"
        } {
            writeRestartData $datFile $coampsName $numModelLevels 0 0.00 $levelType $dartKind \
                             "FREEZE" "TRUE" 1
}
    }
    close $datFile
    destroy . 
}

proc writeRestartData { dataFile varCoampsName mdlNumLevels varPertType varPertMag \
                        varLevelType varDartKind varUpdate varMean varCanBeNeg }   {
    # Alias for perturbation type
    set pertTypeString(0) "NOPERTS"
    set pertTypeString(1) "UNIFORM"
    set pertTypeString(2) "INDIVID"

    # Alias for allowing negativity
    set posDefString(1) "NOPOSDEF"
    set posDefString(0) "ISPOSDEF"

    if { [string equal "W" $varLevelType] } {
        set varNumLevels [expr $mdlNumLevels + 1]
    } else {
        set varNumLevels $mdlNumLevels 
    }

    puts $dataFile [format "%-5s %-3d %-6s %-6.3f %-1s %-30s %-6s %-5s %-8s" $varCoampsName $varNumLevels  \
                           $pertTypeString($varPertType) $varPertMag $varLevelType $varDartKind $varUpdate \
                           $varMean $posDefString($varCanBeNeg)                                            ]
}

# moveToUsed
# ----------
# Moves the active variable in the "Available" list to the "Used" list
proc moveToUsed {} {
    upvar variablesAvailList availableList
    upvar variablesUsedList  usedList

    set currentAvailable [$availableList get active]
    if { [string length $currentAvailable] > 0 } {
        $usedList      insert end $currentAvailable
        $availableList delete active
    }
}

# moveToAvail
# -----------
# Moves the active variable in the "Used" list to the "Available" list
proc moveToAvail {} {
    upvar variablesAvailList availableList
    upvar variablesUsedList  usedList

    set currentUsed [$usedList get active]
    if { [string length $currentUsed] > 0 } {
        $availableList insert end $currentUsed
        $usedList      delete active
    }
}

# updatePertDisplay
# -----------------
# Updates the display of perturbation type and magnitude of the given variable
proc updatePertDisplay { varName } {
    global pertType
    global pertMag
    global varData

    # The actual entry control
    upvar #0 perturbMagAmount pertMagAmount

    set pertType $varData($varName-pertType)
    set pertMag  $varData($varName-pertMag)

    if { $pertType == 0 } {
        $pertMagAmount configure -state disabled
    } else {
        $pertMagAmount configure -state normal
    }
}

# updatePertData
# --------------
# Changes a variable's perturbation data based on the options selected
proc updatePertData {} {
    global pertType
    global pertMag
    global varData

    upvar #0 variablesToPerturbList varList 

    set curVar [$varList get active]
    if { [string length $curVar] > 0 } {
        set varData($curVar-pertType) $pertType
        set varData($curVar-pertMag)  $pertMag
    }

    updatePertDisplay $curVar
}

proc enablePerts {} {
    global perturbTypeNoPerts
    global perturbTypeUniform
    global perturbTypeIndivid
    global perturbMagAmount

    $perturbTypeNoPerts configure -state normal
    $perturbTypeUniform configure -state normal
    $perturbTypeIndivid configure -state normal
    $perturbMagAmount   configure -state normal
}

proc disablePerts {} {
    global perturbTypeNoPerts
    global perturbTypeUniform
    global perturbTypeIndivid
    global perturbMagAmount

    $perturbTypeNoPerts configure -state disabled
    $perturbTypeUniform configure -state disabled
    $perturbTypeIndivid configure -state disabled
    $perturbMagAmount   configure -state disabled
}

# updateNegData
# -------------
# Changes whether a variable can take on negative values
proc updateNegData {} {
    global canBeNeg
    global varData

    upvar #0 variablesInStateList varList 

    set curVar [$varList get active]
    if { [string length $curVar] > 0 } {
        set varData($curVar-canBeNeg) $canBeNeg
        updateNegDisplay $curVar
    }

}

# updateNegDisplay
# ----------------
# Updates the display of whether a variable is allowed to be < 0
proc updateNegDisplay { varName } {
    global canBeNeg
    global varData

    set canBeNeg $varData($varName-canBeNeg)
}

proc disableNegChoice {} {
    global stateOptionsAllowNegative
    global stateOptionsForbidNegative

    $stateOptionsAllowNegative  configure -state disabled
    $stateOptionsForbidNegative configure -state disabled
}

proc enableNegChoice {} {
    global stateOptionsAllowNegative
    global stateOptionsForbidNegative

    $stateOptionsAllowNegative  configure -state normal
    $stateOptionsForbidNegative configure -state normal
}

proc showTip {helpBar message} {
    $helpBar configure -text $message
}
proc clearTip {helpBar} {
    $helpBar configure -text ""
}

proc showVariableMetadata {varName helpBar} {
    global varData
    showTip $helpBar [format "COAMPS Name: %5s, DART Kind: %s" $varData($varName-coampsName) $varData($varName-dartKind)]
}

##############################
# END PROCEDURES
##############################

##########
# Persistent features
##########
wm title . "DART/COAMPS State Vector Wizard"
option add *Notebook.borderWidth 2      widgetDefault

# Navigation frame
set navigateFrame [ frame  .frNav                                           ]
set back          [ button .frNav.btBack   -text "< Back" -command goBack   ]
set next          [ button .frNav.btNext   -text "Next >" -command goNext   ]
set finish        [ button .frNav.btFinish -text "Finish" -command goFinish ]
pack $back $next $finish -side left -padx 10 -pady 3 -expand y -anchor s

# Help/Status Bar
set statusBar [ label .status -relief sunken -borderwidth 2 -anchor w ]

# Notebook container
set wizardNotebook [ notebook_create .nb]

##########
# Notebook Page 1 - choose state vector elements
##########
set pageVariables  [ notebook_page $wizardNotebook "Variables" ]

# Listboxes containing "available" and "used" state vector elements
set variablesAvail [ labelframe $pageVariables.available -text "Available" ]
set variablesUsed  [ labelframe $pageVariables.used      -text "Used"      ]
set variablesAvailList [listbox $variablesAvail.list -selectmode browse -listvariable listAvail ]
set variablesUsedList  [listbox $variablesUsed.list  -selectmode browse -listvariable listUsed  ]
pack $variablesAvailList -padx 2 -pady 2 -expand y -fill both
pack $variablesUsedList  -padx 2 -pady 2 -expand y -fill both

# Move entries between the lists
set variablesMove     [ frame  $pageVariables.move             ]
set variablesMoveTo   [ button $variablesMove.to   -text "-->" -command moveToUsed  ]
set variablesMoveFrom [ button $variablesMove.from -text "<--" -command moveToAvail ]
pack $variablesMoveTo $variablesMoveFrom -side top

pack $variablesAvail -side left -padx 5 -pady 5 -expand y -fill both
pack $variablesMove  -side left -padx 5 -pady 5 
pack $variablesUsed  -side left -padx 5 -pady 5 -expand y -fill both

##########
# Notebook Page 2 - choose perturbations 
##########
set pagePerturbations [ notebook_page $wizardNotebook "Perturbations" ]

# Listbox containing the variables selected on the previous page
set variablesToPerturb [ labelframe $pagePerturbations.toperturb -text "State Variables" ]
set variablesToPerturbList [ listbox $variablesToPerturb.list -selectmode browse -listvariable listUsed ]
pack $variablesToPerturbList -padx 2 -pady 2 -expand y -fill both
pack $variablesToPerturb     -side left -padx 5 -pady 5 -expand y -fill both

set perturbOptions [ frame $pagePerturbations.pertOptions ]

# Perturbation type
set perturbType        [ labelframe  $perturbOptions.perturbType -text "Perturbation Type"                                         ]
set perturbTypeNoPerts [ radiobutton $perturbType.noPerts        -text "No Perturbations"              -variable pertType -value 0 \
                                                                 -command updatePertData                                           ]
set perturbTypeUniform [ radiobutton $perturbType.uniform        -text "Uniform Field Perturbation"    -variable pertType -value 1 \
                                                                 -command updatePertData                                           ]
set perturbTypeIndivid [ radiobutton $perturbType.individ        -text "Individual Point Perturbation" -variable pertType -value 2 \
                                                                 -command updatePertData                                           ]
pack $perturbTypeNoPerts $perturbTypeUniform $perturbTypeIndivid -side top -padx 2 -pady 2 -anchor w -expand y -fill y

# Perturbation magnitude
set perturbMagnitude  [ frame $perturbOptions.pertMagnitude ]
set perturbMagLabel   [ label $perturbMagnitude.label -text "Perturbation Magnitude" ]
set perturbMagAmount  [ entry $perturbMagnitude.amount -textvariable pertMag ]
pack $perturbMagLabel $perturbMagAmount -side top -padx 2 -pady 2

pack $perturbType $perturbMagnitude -side top -padx 2 -pady 2
pack $perturbOptions -side right -padx 5 -pady 5

##########
# Notebook Page 3 -  allow negativity 
##########
set pageNegativity [ notebook_page $wizardNotebook "AllowNegativity" ]

# Listbox containing the variables in the state vector
set variablesInState [ labelframe $pageNegativity.instate -text "State Variables" ]
set variablesInStateList [ listbox $variablesInState.list -height 10 -width 25 -selectmode browse -listvariable listUsed ]
pack $variablesInStateList -padx 2 -pady 2 -expand y -fill both
pack $variablesInState     -side left -padx 5 -pady 5 -expand y -fill both

set stateOptions [ frame $pageNegativity.stateOptions ]
set stateOptionsAllowNegative  [ radiobutton $stateOptions.allowNeg  -text "Allow all values"       -variable canBeNeg -value 1 ]
set stateOptionsForbidNegative [ radiobutton $stateOptions.forbidNeg -text "Require values >= zero" -variable canBeNeg -value 0 ]
pack $stateOptionsAllowNegative $stateOptionsForbidNegative -side top -padx 2 -pady 2 -anchor w
pack $stateOptions -side top -padx 5 -pady 5 -expand y


##########
# Notebook Page 4 -  miscellaneous parameters
##########
set pageMiscParams [ notebook_page $wizardNotebook "MiscParams" ]

# Include the model mean state?
set meanState       [ labelframe  $pageMiscParams.meanState -text "Mean State" ]
set meanStateUse    [ radiobutton $meanState.use            -text "Use mean state"        -variable useMeanState -value 1 ]
set meanStateNotUse [ radiobutton $meanState.doNotUse       -text "Do not use mean state" -variable useMeanState -value 0 ]
pack $meanStateUse $meanStateNotUse -side top -padx 2 -pady 2 -anchor w

# Divider
set miscDivider [frame $pageMiscParams.divider -width 2 -borderwidth 1 -relief sunken]

# Specify model height
set modelHeightFrame [ frame $pageMiscParams.modelHeight ]
set modelSizeLabel   [ label $modelHeightFrame.sizeLabel -text "Number of model levels" ]
set modelSize        [ entry $modelHeightFrame.size      -textvariable numModelLevels   ]
pack $modelSizeLabel $modelSize -side top -padx 5 -pady 5

pack $meanState        -side left -padx 5 -pady 5 -expand y
pack $miscDivider      -side left -padx 2 -pady 2 -expand y -fill y
pack $modelHeightFrame -side right -padx 5 -pady 5 -expand y

##########
# Variable setup
##########
foreach {varName defaultPertType defaultPertMag defaultNeg levelType coampsName dartKind} {
    "Zonal Wind"                  1 2.00 1 "M" "u2"  "QTY_U_WIND_COMPONENT"
    "Meridional Wind"             1 2.00 1 "M" "v2"  "QTY_V_WIND_COMPONENT"
    "Vertical Velocity"           0 0.00 1 "W" "w2"  "QTY_VERTICAL_VELOCITY"
    "Potential Temperature"       0 0.00 1 "M" "th2" "QTY_POTENTIAL_TEMPERATURE"
    "Specific Humidity"           0 0.00 0 "M" "qv2" "QTY_SPECIFIC_HUMIDITY"
    "Cloud Ice"                   0 0.00 0 "M" "qi2" "QTY_CLOUD_ICE"
    "Cloud Liquid Water"          0 0.00 0 "M" "qc2" "QTY_CLOUD_LIQUID_WATER"
    "Rainwater Mixing Ratio"      0 0.00 0 "M" "qr2" "QTY_RAINWATER_MIXING_RATIO"
    "Snow Mixing Ratio"           0 0.00 0 "M" "qs2" "QTY_SNOW_MIXING_RATIO"
    "Pert. Exner Function"        0 0.00 1 "M" "p2"  "QTY_EXNER_FUNCTION"
} {
    lappend listAvail $varName
    set varData($varName-pertType)   $defaultPertType
    set varData($varName-pertMag)    $defaultPertMag
    set varData($varName-canBeNeg)   $defaultNeg
    set varData($varName-levelType)  $levelType
    set varData($varName-coampsName) $coampsName
    set varData($varName-dartKind)   $dartKind
}

# Default choice is to use the mean state
set useMeanState 1

##########
# Initial display
##########
pack $wizardNotebook -side top -expand yes -fill both -padx 4 -pady 4
pack $navigateFrame -side top -expand no -fill x -anchor s
pack $statusBar -expand no -fill x -anchor s

# Default to first page
notebook_display $wizardNotebook "Variables"
disableBack
disableFinish


##############################
# BEGIN BINDINGS
##############################

# Allow a double-click to move entries between lists
bind $variablesAvailList <Double-ButtonPress-1> {
    set nearestVar [$variablesAvailList get @%x,%y]
    if { [string length $nearestVar] > 0 } {
        $variablesUsedList  insert end $nearestVar
        $variablesAvailList delete @%x,%y
    }
}
bind $variablesUsedList <Double-ButtonPress-1> {
    set nearestVar [$variablesUsedList get @%x,%y]
    if { [string length $nearestVar] > 0 } {
        $variablesAvailList insert end $nearestVar
        $variablesUsedList  delete @%x,%y
    }
}

# Set up default perturbation display
bind $pagePerturbations <Map> {
    set activeVar [$variablesToPerturbList get active]

    if { [string length [$variablesToPerturbList get active]] == 0 } {
        $variablesToPerturbList activate 0
        $variablesToPerturbList selection set 0

        set firstVar [lindex $listUsed 0]
    } else {
        set firstVar $activeVar
    }

    if { [string length $firstVar] > 0 } {
        enablePerts
        enableNext
        updatePertDisplay $firstVar
    } else {
        disablePerts
        disableNext
    }

}

# Set up default perturbation display
bind $pageNegativity <Map> {
    set activeVar [$variablesInStateList get active]

    if { [string length [$variablesToPerturbList get active]] == 0 } {
        $variablesToPerturbList activate 0
        $variablesToPerturbList selection set 0

        set firstVar [lindex $listUsed 0]
    } else {
        set firstVar $activeVar
    }

    if { [string length $firstVar] > 0 } {
        enableNegChoice 
        enableFinish
        updateNegDisplay $firstVar
    } else {
        disableNegChoice
        disableFinish
    }

}

# Number of model levels should be positive
bind $pageMiscParams <Map> {
    if { $numModelLevels <= 0 } {
        disableFinish
    } else {
        enableFinish
    }
}

bind $modelSize <Key> {
    if { $numModelLevels <= 0 } {
        disableFinish
    } else {
        enableFinish
    }
} 

# Update perturbation options when variable changes
bind $variablesToPerturbList <ButtonRelease-1> {
    set activeVar [$variablesToPerturbList get @%x,%y]
    if { [string length $activeVar] > 0 } {
        updatePertDisplay $activeVar
        showVariableMetadata $activeVar $statusBar
    }
}
bind $variablesToPerturbList <<ListboxSelect>> {
    set activeVar [$variablesToPerturbList get active]
    if { [string length $activeVar] > 0 } {
        updatePertDisplay $activeVar
        showVariableMetadata $activeVar $statusBar
    }
}
# Update negativity selection when variable changes
bind $variablesInStateList <ButtonRelease-1> {
    set activeVar [$variablesToPerturbList get @%x,%y]
    if { [string length $activeVar] > 0 } {
        updateNegDisplay $activeVar
        showVariableMetadata $activeVar $statusBar
    }
}
bind $variablesInStateList <<ListboxSelect>> {
    set activeVar [$variablesToPerturbList get active]
    if { [string length $activeVar] > 0 } {
        updateNegDisplay $activeVar
        showVariableMetadata $activeVar $statusBar
    }
}

# Help/Status - navigation
bind $back   <Enter> { showTip  $statusBar "Move back one page" }
bind $back   <Leave> { clearTip $statusBar }
bind $next   <Enter> { showTip  $statusBar "Move ahead one page" }
bind $next   <Leave> { clearTip $statusBar }
bind $finish <Enter> { showTip  $statusBar "Save data file and exit wizard" }
bind $finish <Leave> { clearTip $statusBar }

# Help/Status - Page 1 (Variables)
bind $variablesAvail    <Enter> { showTip  $statusBar "Variables that can be included in the state vector" }
bind $variablesAvail    <Leave> { clearTip $statusBar }
bind $variablesUsed     <Enter> { showTip  $statusBar "Variables that are included in the state vector" }
bind $variablesUsed     <Leave> { clearTip $statusBar }
bind $variablesMoveTo   <Enter> { showTip  $statusBar "Include selected available variable in the state vector" }
bind $variablesMoveTo   <Leave> { clearTip $statusBar }
bind $variablesMoveFrom <Enter> { showTip  $statusBar "Remove selected used variable from the state vector" }
bind $variablesMoveFrom <Leave> { clearTip $statusBar }

# Help/Status - Page 2 (Perturbations)
bind $variablesToPerturb <Enter> { showTip  $statusBar "Variable to perturb" }
bind $variablesToPerturb <Leave> { clearTip $statusBar }
bind $perturbType        <Enter> { showTip  $statusBar "Type of perturbation to apply to this variable" }
bind $perturbType        <Leave> { clearTip $statusBar }
bind $perturbTypeNoPerts <Enter> { showTip  $statusBar "Apply no perturbation" }
bind $perturbTypeNoPerts <Leave> { clearTip $statusBar }
bind $perturbTypeUniform <Enter> { showTip  $statusBar "Apply a single perturbation to an entire level" }
bind $perturbTypeUniform <Leave> { clearTip $statusBar }
bind $perturbTypeIndivid <Enter> { showTip  $statusBar "Apply a different perturbation to each point" }
bind $perturbTypeIndivid <Leave> { clearTip $statusBar }
bind $perturbMagnitude   <Enter> { showTip  $statusBar "Variance of mean-0 Gaussian distribution to draw (additive) perturbations from" }
bind $perturbMagnitude   <Leave> { clearTip $statusBar }

# Help/Status - Page 3 (Negative Values)
bind $variablesInState           <Enter> { showTip  $statusBar "Allow/Reject negative values for this variable" }
bind $variablesInState           <Leave> { clearTip $statusBar }
bind $stateOptions               <Enter> { showTip  $statusBar "Choose range allowed for this variable's value" }
bind $stateOptions               <Leave> { clearTip $statusBar }
bind $stateOptionsAllowNegative  <Enter> { showTip  $statusBar "Allow this variable to take any value (positive or negative)" }
bind $stateOptionsAllowNegative  <Leave> { clearTip $statusBar }
bind $stateOptionsForbidNegative <Enter> { showTip  $statusBar "Allow this variable to take *only* positive (or zero) values" } 
bind $stateOptionsForbidNegative <Leave> { clearTip $statusBar }

# Help/Status - Page 4 (Miscellany)
bind $meanState        <Enter> { showTip  $statusBar "Choose whether to include the model mean state in the state vector" } 
bind $meanState        <Leave> { clearTip $statusBar }
bind $meanStateUse     <Enter> { showTip  $statusBar "Include COAMPS mean state variables in the DART state vector" }
bind $meanStateUse     <Leave> { clearTip $statusBar }
bind $meanStateNotUse  <Enter> { showTip  $statusBar "Do not include COAMPS mean state variables in the DART state vector" }
bind $meanStateNotUse  <Leave> { clearTip $statusBar }
bind $modelHeightFrame <Enter> { showTip  $statusBar "Number of COAMPS model levels ('kka' in namelist)" }
bind $modelHeightFrame <Leave> { clearTip $statusBar }

##############################
# END BINDINGS
##############################

# <next few lines under version control, do not edit>
# $URL$
# $Revision$
# $Date$

