module SDSReader
  !*****************************************************************************
  ! !F90
  !
  ! !Description:
  !  This module contains routines that are used for reading in Scientific Data Sets.
  !  The module is derived and modified from codes by Robert Pincus.
  !  
  ! !Callable routines:
  !    ReadL2DataSet()
  ! 
  ! !Revision History:
  ! $Log: L2Reader.f90,v $
  ! Revision 1.0  1999/10/22  12:43:43  EGMoody
  ! Initial revision.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !  The reads are designed such that no callibration of the data occurs.  This was
  !   done because the SDS are just copied to a new HDF file, in this application.
  !  The generic interface ReadL2DataSet determines the specific subroutine based
  !   upon the type of data set to be read in.
  !  The specific routines read in only one plane of the data per call.  If the SDS
  !   is a multi-banded array, then multiple calls must be made to read in the 
  !   entire SDS.  The variable "plane" determines which band is read in (varies
  !   between zero to the number of bands -1).
  !
  ! !END
  !*****************************************************************************
  
  !Dependencies:
  use typeSizes, only : OneByteIntKind, TwoByteIntKind, FourByteIntKind, &
                        FourByteRealKind, EightByteRealKind
  use hdf,       only : MAX_VAR_DIMS, FAIL,                              &
                        SFrdata

  implicit none

  private

  ! Generic procedure interface:
  interface ReadSDS
     module procedure  ReadOneByteIntData, ReadTwoByteIntData, ReadFourByteIntData, &
                       ReadFourByteRealData, ReadEightByteRealData
  end interface
  !   Generic procedure:
  !   subroutine ReadShortToFloat(SampleDataType, dataSetID, fileName, &
  !                               dataSetName, dimSizes,               &
  !                               SDSDataR1, SDSDataR2, SDSDataR3         )
  !
  !   ******* (kind = ****Byte***Kind),    intent( in) :: SampleDataType
  !   integer (kind = FourByteIntKind),    intent( in) :: dataSetID
  !   character (len = *),                 intent( in) :: fileName, dataSetName
  !   integer (kind = FourByteIntKind),    &
  !            dimension(2),               intent(in)  :: dimSizes
  !   ******* (kind = ***************),    &
  !            dimension(:),     optional, intent(out) :: SDSDataR1
  !   ******* (kind = ***************),    &
  !            dimension(:,:),   optional, intent(out) :: SDSDataR2
  !   ******* (kind = ***************),    &
  !            dimension(:,:,:), optional, intent(out) :: SDSDataR3
  ! !F90
  !
  ! !Description:
  !  ReadL2dataSet reads the specified SDS, given the hdf SDS ID and dimensions of the
  !   array to be read in.
  !  This interface provides for a generic call, meaning that the interface determines
  !   which specific routine to call, given the type(integer or real) of the dataset.
  !
  ! !Input Parameters:
  ! SampleDataType:  Dummy scalar of the data type of the SDS.  Used by the 
  !                       interface to determine which specific routine to call.
  !   dataSetID:     HDF data set ID, as provided by getDataSetID or SFselect
  !   fileName, 
  !    dataSetName:  Character strings containing the name of the dataset
  !                   being read and the file from which it is being read; used
  !                   to make error reports more useful. 
  !   dimSizes:      Dimensions of the SDS, obtained from a call to getDataSetDimSizes;
  !                   used to set the edge values for the SFrdata call.
  !
  ! !Output Parameters:
  !   SDSDataRx:     The SDS will be read into this array.  It can take on the type of
  !                   byte, short, long, float, or double. Rx stands for Rank X.
  !
  ! !Revision History:
  !   See Module revision history at the beginning of the file.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !  The interface was engineered to read in one array from a data set.
  !
  ! !END


  !declare the public routines:
  public :: ReadSDS


  !local variables for these the subroutines:
  integer                                :: status
  !  Used to check for errors in reading the SDS.

  integer (kind = FourByteIntKind  ), &
           dimension(MAX_VAR_DIMS)       :: hdfStart, hdfStride, hdfEdge
  !  hdfstart:  Follows HDF conventions. The array element from which to begin reading the
  !              HDF array. Zero based.
  !  hdfstride: Follows HDF conventions. The frequency with which data should be read in
  !              each dimension. Stride 1 means read every element.
  !  hdfedge:   Follows HDF conventions. The number of elements to read in each dimension. 
  !              If start(:) == 0 and stride(:) == 1, setting edge equal to the shape of 
  !              the underlying array reads all elements. 
  

contains



  ! ----------------------------------------------------------------------------------

  subroutine ReadOneByteIntData( SampleDataType, dataSetID, fileName, &
                                 dataSetName, dimSizes,               &
                                 SDSDataR1, SDSDataR2, SDSDataR3      )
    integer  (kind = OneByteIntKind),     intent( in)  :: SampleDataType
    integer  (kind = FourByteIntKind),    intent( in)  :: dataSetID
    character(len = *),                   intent( in)  :: fileName, dataSetName
    integer  (kind = FourByteIntKind),    &
               dimension(:),              intent( in)  :: dimSizes
    integer  (kind = OneByteIntKind ),    &
               dimension(:),     optional, intent(out) :: SDSdataR1
    integer  (kind = OneByteIntKind ),    &
               dimension(:,:),   optional, intent(out) :: SDSdataR2
    integer  (kind = OneByteIntKind ),    &
               dimension(:,:,:), optional, intent(out) :: SDSdataR3
    ! !F90
    !
    ! !Description:
    !  ReadOneByteIntData is the specific subroutine in the ReadL2DataSet interface,
    !   which reads in the specified Byte SDS, given the hdf SDS ID and 
    !   dimensions of the array to be read in.
    !
    ! !Input Parameters:
    !   dataSetID:    HDF data set ID, as provided by getDataSetID or SFselect
    !   fileName, 
    !    dataSetName: Character strings containing the name of the dataset
    !                  being read and the file from which it is being read; used
    !                  to make error reports more useful. 
    !   dimSizes:     Dimensions of the SDS, obtained from a call to getDataSetDimSizes;
    !                  used to set the edge values for the SFrdata call.
    !   plane:        Integer determining which (single) array section in the third 
    !                  dimension to read.  For a single band, plane = 0.
    !
    ! !Output Parameters:
    !   L2Data:       The specified plane of the byte valued SDS is read into this array.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !  The interface was engineered to read in one array from a data set.
    !
    ! !END

    !Set the variables that define how the SDS is read in:    
    !These are set such that the entire SDS is read in,
    !  regardless of shape or size.
    hdfStart (:)  = 0
    hdfStride(:)  = 1
    hdfEdge  (:) = dimSizes(:)

    !Read in the SDS:
    if ( present(SDSDataR1) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR1)
    else if ( present(SDSDataR2) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR2)
    else if ( present(SDSDataR3) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR3)
    end if

    !Error checking:
    if (status == FAIL) then
       print *, "Error reading data from dataset " //          &
            trim(dataSetname) // " in file " // trim(fileName)
       stop
    end if
    
  end subroutine ReadOneByteIntData

  ! ----------------------------------------------------------------------------------

  subroutine ReadTwoByteIntData( SampleDataType, dataSetID, fileName, &
                                 dataSetName, dimSizes,               &
                                 SDSDataR1, SDSDataR2, SDSDataR3      )
    integer  (kind = TwoByteIntKind),      intent( in) :: SampleDataType
    integer  (kind = FourByteIntKind),     intent( in) :: dataSetID
    character(len = *),                    intent( in) :: fileName, dataSetName
    integer  (kind = FourByteIntKind),     &
               dimension(:),               intent( in) :: dimSizes
    integer  (kind =  TwoByteIntKind  ),   &
               dimension(:),     optional, intent(out) :: SDSdataR1
    integer  (kind =  TwoByteIntKind  ),   &
               dimension(:,:),   optional, intent(out) :: SDSdataR2
    integer  (kind =  TwoByteIntKind  ),   &
               dimension(:,:,:), optional, intent(out) :: SDSdataR3

    ! !F90
    !
    ! !Description:
    !  ReadTwoByteIntData is the specific subroutine in the ReadL2DataSet interface,
    !   which reads in the specified Short SDS, given the hdf SDS ID and 
    !   dimensions of the array to be read in.
    !
    ! !Input Parameters:
    !   dataSetID:    HDF data set ID, as provided by getDataSetID or SFselect
    !   fileName, 
    !    dataSetName: Character strings containing the name of the dataset
    !                  being read and the file from which it is being read; used
    !                  to make error reports more useful. 
    !   dimSizes:     Dimensions of the SDS, obtained from a call to getDataSetDimSizes;
    !                  used to set the edge values for the SFrdata call.
    !   plane:        Integer determining which (single) array section in the third 
    !                  dimension to read.  For a single band, plane = 0.
    !
    ! !Output Parameters:
    !   L2Data:       The specified plane of the short valued SDS is read into this array.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !  The interface was engineered to read in one array from a data set.
    !
    ! !END

    !Set the variables that define how the SDS is read in:    
    !These are set such that the entire SDS is read in,
    !  regardless of shape or size.
    hdfStart (:)  = 0
    hdfStride(:)  = 1
    hdfEdge  (:) = dimSizes(:)

    !Read in the SDS:
    if ( present(SDSDataR1) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR1)
    else if ( present(SDSDataR2) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR2)
    else if ( present(SDSDataR3) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR3)
    end if

    !Error checking:
    if (status == FAIL) then
       print *, "Error reading data from dataset " //          &
            trim(dataSetname) // " in file " // trim(fileName)
       stop
    end if
    
  end subroutine ReadTwoByteIntData

  ! ----------------------------------------------------------------------------------

  subroutine ReadFourByteIntData( SampleDataType, dataSetID, fileName, &
                                  dataSetName, dimSizes,               &
                                  SDSDataR1, SDSDataR2, SDSDataR3      )
    integer  (kind = FourByteIntKind),     intent( in) :: SampleDataType
    integer  (kind = FourByteIntKind),     intent( in) :: dataSetID
    character(len = *),                    intent( in) :: fileName, dataSetName
    integer  (kind = FourByteIntKind),     &
               dimension(:),               intent( in) :: dimSizes
    integer  (kind = FourByteIntKind ),    &
               dimension(:),     optional, intent(out) :: SDSdataR1
    integer  (kind = FourByteIntKind ),    &
               dimension(:,:),   optional, intent(out) :: SDSdataR2
    integer  (kind = FourByteIntKind ),    &
               dimension(:,:,:), optional, intent(out) :: SDSdataR3
    ! !F90
    !
    ! !Description:
    !  ReadFourByteIntData is the specific subroutine in the ReadL2DataSet interface,
    !   which reads in the specified Long SDS, given the hdf SDS ID and 
    !   dimensions of the array to be read in.
    !
    ! !Input Parameters:
    !   dataSetID:    HDF data set ID, as provided by getDataSetID or SFselect
    !   fileName, 
    !    dataSetName: Character strings containing the name of the dataset
    !                  being read and the file from which it is being read; used
    !                  to make error reports more useful. 
    !   dimSizes:     Dimensions of the SDS, obtained from a call to getDataSetDimSizes;
    !                  used to set the edge values for the SFrdata call.
    !   plane:        Integer determining which (single) array section in the third 
    !                  dimension to read.  For a single band, plane = 0.
    !
    ! !Output Parameters:
    !   L2Data:       The specified plane of the long valued SDS is read into this array.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !  The interface was engineered to read in one array from a data set.
    !
    ! !END

    !Set the variables that define how the SDS is read in:    
    !These are set such that the entire SDS is read in,
    !  regardless of shape or size.
    hdfStart (:)  = 0
    hdfStride(:)  = 1
    hdfEdge  (:) = dimSizes(:)

    !Read in the SDS:
    if ( present(SDSDataR1) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR1)
    else if ( present(SDSDataR2) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR2)
    else if ( present(SDSDataR3) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR3)
    end if

    !Error checking:
    if (status == FAIL) then
       print *, "Error reading data from dataset " //          &
            trim(dataSetname) // " in file " // trim(fileName)
       stop
    end if
    
  end subroutine ReadFourByteIntData

  ! ----------------------------------------------------------------------------------

  subroutine ReadFourByteRealData( SampleDataType, dataSetID, fileName, &
                                   dataSetName, dimSizes,               &
                                   SDSDataR1, SDSDataR2, SDSDataR3      )
    real     (kind = FourByteRealKind),   intent( in) :: SampleDataType
    integer  (kind = FourByteIntKind),    intent( in) :: dataSetID
    character(len = *),                   intent( in) :: fileName, dataSetName
    integer  (kind = FourByteIntKind),    &
              dimension(:),               intent( in) :: dimSizes
    real     (kind = FourByteRealKind ),  &
              dimension(:),     optional, intent(out) :: SDSdataR1
    real     (kind = FourByteRealKind ),  &
              dimension(:,:),   optional, intent(out) :: SDSdataR2
    real     (kind = FourByteRealKind ),  &
              dimension(:,:,:), optional, intent(out) :: SDSdataR3
    ! !F90
    !
    ! !Description:
    !  ReadFourByteRealData is the specific subroutine in the ReadL2DataSet interface,
    !   which reads in the specified Float SDS, given the hdf SDS ID and 
    !   dimensions of the array to be read in.
    !
    ! !Input Parameters:
    !   dataSetID:    HDF data set ID, as provided by getDataSetID or SFselect
    !   fileName, 
    !    dataSetName: Character strings containing the name of the dataset
    !                  being read and the file from which it is being read; used
    !                  to make error reports more useful. 
    !   dimSizes:     Dimensions of the SDS, obtained from a call to getDataSetDimSizes;
    !                  used to set the edge values for the SFrdata call.
    !   plane:        Integer determining which (single) array section in the third 
    !                  dimension to read.  For a single band, plane = 0.
    !
    ! !Output Parameters:
    !   L2Data:       The specified plane of the float valued SDS is read into this array.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !  The interface was engineered to read in one array from a data set.
    !
    ! !END

    !Set the variables that define how the SDS is read in:    
    !These are set such that the entire SDS is read in,
    !  regardless of shape or size.
    hdfStart (:)  = 0
    hdfStride(:)  = 1
    hdfEdge  (:) = dimSizes(:)

    !Read in the SDS:
    if ( present(SDSDataR1) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR1)
    else if ( present(SDSDataR2) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR2)
    else if ( present(SDSDataR3) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR3)
    end if

    !Error checking:
    if (status == FAIL) then
       print *, "Error reading data from dataset " //          &
            trim(dataSetname) // " in file " // trim(fileName)
       stop
    end if
    
  end subroutine ReadFourByteRealData

  ! ----------------------------------------------------------------------------------

  subroutine ReadEightByteRealData( SampleDataType, dataSetID, fileName, &
                                    dataSetName, dimSizes,               &
                                    SDSDataR1, SDSDataR2, SDSDataR3      )
    real     (kind = EightByteRealKind),  intent( in) :: SampleDataType
    integer  (kind = FourByteIntKind),    intent( in) :: dataSetID
    character(len = *),                   intent( in) :: fileName, dataSetName
    integer  (kind = FourByteIntKind),    &
              dimension(:),               intent( in) :: dimSizes
    real     (kind = EightByteRealKind ), &
              dimension(:),     optional, intent(out) :: SDSdataR1
    real     (kind = EightByteRealKind ), &
              dimension(:,:),   optional, intent(out) :: SDSdataR2
    real     (kind = EightByteRealKind ), &
              dimension(:,:,:), optional, intent(out) :: SDSdataR3
    ! !F90
    !
    ! !Description:
    !  ReadEightByteRealData is the specific subroutine in the ReadL2DataSet interface,
    !   which reads in the specified Double SDS, given the hdf SDS ID and 
    !   dimensions of the array to be read in.
    !
    ! !Input Parameters:
    !   dataSetID:    HDF data set ID, as provided by getDataSetID or SFselect
    !   fileName, 
    !    dataSetName: Character strings containing the name of the dataset
    !                  being read and the file from which it is being read; used
    !                  to make error reports more useful. 
    !   dimSizes:     Dimensions of the SDS, obtained from a call to getDataSetDimSizes;
    !                  used to set the edge values for the SFrdata call.
    !   plane:        Integer determining which (single) array section in the third 
    !                  dimension to read.  For a single band, plane = 0.
    !
    ! !Output Parameters:
    !   L2Data:       The specified plane of the double valued SDS is read into this array.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !  The interface was engineered to read in one array from a data set.
    !
    ! !END

    !Set the variables that define how the SDS is read in:    
    !These are set such that the entire SDS is read in,
    !  regardless of shape or size.
    hdfStart (:)  = 0
    hdfStride(:)  = 1
    hdfEdge  (:) = dimSizes(:)

    !Read in the SDS:
    if ( present(SDSDataR1) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR1)
    else if ( present(SDSDataR2) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR2)
    else if ( present(SDSDataR3) ) then
       status = SFrdata(dataSetID, hdfStart, hdfStride, hdfEdge, SDSDataR3)
    end if
    
    !Error checking:
    if (status == FAIL) then
       print *, "Error reading data from dataset " //          &
            trim(dataSetname) // " in file " // trim(fileName)
       stop
    end if
    
  end subroutine ReadEightByteRealData

  ! ----------------------------------------------------------------------------------


end module SDSReader
