module Inputs
  !*****************************************************************************
  ! !F90
  !
  ! !Description:
  !  This module contains the input parameters used by the code.  The inputs
  !    are read in from a file created by the PERL script(i.e. change the values
  !    in the PERL script to tailor the program to your needs).  
  !
  ! !Callable Routines:
  !  readInputs()
  !
  ! !Module Revision History: 
  !   $Log:  Inputs.f90, v $
  !
  !   Revision 1.1  1999/12/03  12:12:12  EGMoody
  !    Changes to the reading of the SDS to be subsetted.  Now the names of the
  !      SDS are stored in a seperate input file, which is read in by the
  !      readInputs() routine.
  !    Also added a name to be associated with the Geolocated Point.  This name
  !      will be a prefix to the output HDF file and subsetted granule list file.
  !    No longer needed the L2 input granule path because the input file changed
  !      to include path with the file name.
  !    Added a second print flag to test if we want to print the location being
  !      examined.
  !
  !   Revision 1.0  1999/10/22  09:21:21  EGMoody
  !    Initial version.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !  The path and filename for the input file created by the PERL script is 
  !   hardwired into the code (set to "./inputs/inputs.txt").  If you modify
  !   the path or filename in the PERL script, you must modify the variable
  !   "ProgInputFileName" to reflect these changes.
  !  There is a subroutine, called readInputs, which should be called before 
  !   any processing occurs.  This sets the values of the parameters into 
  !   memory.
  !
  ! !END
  !*****************************************************************************

  !Dependencies:
  use typeSizes,    only : FourByteIntKind, FourByteRealKind


  implicit none

  !make all variables public:
  public

  !****************************************************************************
  !Parameters:
  !****************************************************************************

  !Max lengths:
  !These are set by the PGS toolkit and are not alterable.
  integer  (kind = FourByteIntKind),  parameter :: MaxAttributeLength = 32767
           !value obtained from Rich Hucek.
  integer  (kind = FourByteIntKind),  parameter :: MaxFileNameLength  = 1025
           !value obtained from the PGS include file.

  !****************************************************************************
  !Variables
  !****************************************************************************
  !Desired Lat/Lon:
  real     (kind = FourByteRealKind)            :: desiredLat
  real     (kind = FourByteRealKind)            :: desiredLon
  
  !These let you specify a rectangle around the Lat/Lon point you wish to include:
  real     (kind = FourByteRealKind)            :: EpsilonLat
  real     (kind = FourByteRealKind)            :: EpsilonLon

  !The name that is associated with the geolocated point.  Used as a
  !   prefix to the output HDF file:
  character(len = MaxFileNameLength)            :: LocationName = ""

  !Number of Data Sets that you want:
  integer  (kind = FourByteIntKind)             :: numberOfSDS

  !Array containing the names of the Data Sets:
  character(len = MaxFileNameLength),       &
            dimension(:), allocatable           :: SDS

  !Print Flag: 
  !  Toggles the printing of the name of the granule that is being processed.
  logical                                       :: PrintFlagGranName
  !  Toggles the printing of the name of the location that is being processed.
  logical                                       :: PrintFlagLocName

  !Spatial Subsetting Flag:
  !  An SDS can be spatially subsetted or the entire SDS can be copied
  !    to the new HDF file by setting the flag named 'SpatialSubset".
  !    Setting this flag to true means that you would like to perform
  !    the spatial subsetting, false will dump the entire SDS.
  logical                                       :: SpatialSubsetting

  !Minimum Usable Box Dimension:
  !  Ex. If the smallest dimension of a useable box is 4, then use MinBoxDim of 4.
  integer  (kind = FourByteIntKind)             :: MinBoxDim

  !****************************************************************************
  !These are filenames and directories:
  !****************************************************************************

  !  Program inputs file names:
  character(len = MaxFileNameLength)            :: ProgInputFileName = ""

  !  Output path:
  character(len = MaxFileNameLength)            :: OutputPath = ""
  
  !  This file contains all L2 granule names for MOD04:
  character(len = MaxFileNameLength)            :: all_L2_in_names_file = ""

  !  This contains the file name for the Granule Name File:
  character(len = MaxFileNameLength)            :: GranNameFile = ""

  !  This contains the file name for the SDS Input File:
  character(len = MaxFileNameLength)            :: SDSInputFile = ""

  !****************************************************************************
  !The values to parse out of the CoreMetadata.0:
  !****************************************************************************
  character(len  = 40), dimension(1:4)          :: CoordName = ""
  ! CoordName       : Array containing the names  of the Bounding Coordinates.


contains



  subroutine readInputs
    
    implicit none

    integer(kind = FourByteIntKind)    :: InputLUN, icnt
    character(len = MaxFileNameLength) :: TestSDS
    !*****************************************************************************
    !Read in the inputs from the program inputs files:
    !*****************************************************************************
    !Open the input file:
    ProgInputFileName = "./inputs/F90Inputs.dat"
    inputLUN = 1120
    open(inputLUN, file = trim(ProgInputFileName), form = 'formatted', &
                   status = 'old', action = 'read')
    
    !Read in the desired Lat, Lon, and epsilon:
    read(inputLUN,*) desiredLat
    read(inputLUN,*) desiredLon
    read(inputLUN,*) epsilonLat
    read(inputLUN,*) epsilonLon

    !Read in the Location Name:
    read(inputLUN,*) LocationName

    !Read in the print flag:
    read(inputLUN,*) PrintFlagGranName
    read(inputLUN,*) PrintFlagLocName

    !Read the spatial subsetting flag:
    read(inputLUN,*) SpatialSubsetting

    !Read in the minimum useful box dimension:
    read(inputLUN,*) MinBoxDim
    
    !Read in the File names and Paths:
    !  Output path:
    read(inputLUN,*) OutputPath
    
    !  This file contains all L2 for MOD04:
    read(inputLUN,*) all_L2_in_names_file
    
    !  This contains the file name for the granule name file:
    read(inputLUN,*) GranNameFile

    !  This contains the file name for the SDS input file:
    read(inputLUN,*) SDSInputFile

    close(inputLUN)

    !************************************************************************
    ! Determine the number of SDS names to be read in and allocate space:
    !************************************************************************
    open(inputLUN, file = trim(SDSInputFile), form = 'formatted', &
                   status = 'old', action = 'read')

    ! Read to the end of the file to count the number of SDS to be subsetted:
    numberOfSDS = 0
    TestSDS = "1"
    do while (TestSDS /= "")
       TestSDS = ""
       read(inputLUN,*) TestSDS
       numberofSDS = numberofSDS + 1
    end do

    numberOfSDS = numberOfSDS - 1
    allocate(SDS(numberOfSDS))
    SDS=""

    close(inputLUN)

    !************************************************************************
    ! Read in the SDS names to be subsetted:
    !************************************************************************
    open(inputLUN, file = trim(SDSInputFile), form = 'formatted', &
                   status = 'old', action = 'read')

    do icnt = 1, numberOfSDS

       read(inputLUN,*) SDS(icnt)

    end do

    close(inputLUN)

    !*****************************************************************************
    !Define the Coordinates to Parse from the CoreMetadata.0
    !*****************************************************************************
    CoordName(1) = "EASTBOUNDINGCOORDINATE"
    CoordName(2) = "WESTBOUNDINGCOORDINATE"
    CoordName(3) = "NORTHBOUNDINGCOORDINATE"
    CoordName(4) = "SOUTHBOUNDINGCOORDINATE"
    

  end subroutine readInputs



end module Inputs

