module subsettingUtils
!*****************************************************************************
  ! !F90
  !
  ! !Description:
  !    This module contains the Subsetting utility tools, such as the routines
  !       to find the box corners, create and apply a mask, and copy an SDS
  !       to a box.
  !  
  ! !Callable routines:
  !    findBoxCorners()
  !    findHighResBoxCorners()
  !    ResAndSubsetCheck()
  !    createMask()
  !    applyMask()
  !    copyToBox()
  !    CreateHighResGeo()
  ! 
  ! !Revision History:
  ! $Log: subsettingUtils.f90,v $
  !
  ! Revision 1.1  2000/02/11  11:15:25  EGMoody
  !  Fixed bug in findBoxCorners. The if statement for the south coordinate
  !  was testing the wrong index.
  ! Revision 1.0  1999/12/17  09:21:21  EGMoody
  !  Initial revision.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   The copyToBox and applyMask callable routines are generic interfaces,
  !      designed to handle various data types.  However, they are designed
  !      to handle a single band.  To handle multi-bands, call these multiple
  !      times by looping over the band dimension.
  !   All routines are designed to allow for various sizes of arrays.
  !
  ! !END
  !*****************************************************************************

  !Dependencies:
  use typeSizes,  only : OneByteIntKind, TwoByteIntKind, FourByteIntKind,  &
                         FourByteRealKind, EightByteRealKind
  use Inputs,     only : desiredLat, desiredLon, EpsilonLat, EpsilonLon
  use dataUtils,  only : getFillValue

  implicit none
  private


  interface copyToBox
     module procedure CTBoneByteInt, CTBtwoByteInt, CTBfourByteInt, &
                      CTBfourByteReal, CTBeightByteReal
  end interface
  !subroutine CTB*****Byte****(BoxCorners, CompleteData, SubsettedData)
  !  integer  (kind = FourByteIntKind),           &
  !            dimension(1:4,1:2),     intent(in)  :: BoxCorners
  !  *******  (kind = *****Byte****Kind),         &
  !            dimension(:,:),         intent(in)  :: CompleteData
  !  *******  (kind = *****Byte****Kind),         &
  !            dimension(:,:),         intent(out) :: SubsettedData
  !
  ! !Description:
  !   This routine will copy the data in the original SDS to the box.
  !   The generic iterface will select specific routines based on data type.
  !  
  ! !Input Parameters:
  !    BoxCorners     : Contains the indices of the subsetted box.
  !                      ( dimension (i,j):
  !                         i=1 : Upper Left    j=1: i index of box
  !                         i=2 : Upper Right   j=2: j index of box
  !                         i=3 : Lower Left 
  !                         i=4 : Lower Right                      )           
  !    CompleteData   : The complete SDS, for one waveband.
  !
  ! !Output Parameters:
  !    SubsettedData  : The SDS data that is copied to the box.
  !
  ! !Revision History:
  !   See Module revision history at the beginning of the file.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !    Each call to this routine copies one waveband to a box.
  !    The data type for completData and SubsettedData must be identicle.
  !
  ! !END

  interface applyMask
     module procedure AMaskOBIK, AMaskTBIK, AMaskFBIK, AMaskFBRK, AMaskEBRK
  end interface
  !subroutine AMask*B*K(Data, Mask, origID)
  !  *******  (kind = *****Byte****Kind), &
  !            dimension(:,:)                             :: Data
  !  integer  (kind = FourByteIntKind),   &
  !            dimension(:,:),            intent( in)     :: Mask
  !  integer  (kind = FourByteIntKind),   intent( in)     :: origID
  !
  ! !Description:
  !   This routine will apply the mask, in order to replace unwanted points,
  !      within the box, with fill values.  The routine first obtains the 
  !      fill value for this SDS, then loops over the boxed data, replacing
  !      data with the fill value, when the mask = 0.
  !   The generic iterface will select specific routines based on data type.
  !  
  ! !Input Parameters:
  !    Data   : Dummy scalar of the data type of the SDS.  Used by the 
  !                       interface to determine which specific routine to call.
  !    Mask   : Array containing the mask.  The mask is used to replace unwanted
  !             points in the subsetted box, with fill values.  This is accomplished
  !             by setting the mask to 1 for desired points and 0 for unwanted
  !             points.  Then the mask is applied to the subsetted SDS, in which
  !             when the mask = 0, the fill value replaces the stored value.
  !    origID : 
  !
  ! !Revision History:
  !   See Module revision history at the beginning of the file.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !    Each call to this routine applies the mask to one waveband of the box.
  ! !END



  public :: findBoxCorners, findHighResBoxCorners
  public :: SubsetCheck
  public :: createMask, applyMask
  public :: copyToBox, CreateHighResGeo


  !local variables:
  integer  (kind = FourByteIntKind)             :: i, j, k
  integer  (kind = FourByteIntKind)             :: iprime, jprime



contains



  !*****************************************************************************  
  subroutine findBoxCorners(Lats, Lons, BoxCorners)

    real     (kind = FourByteRealKind),       &
              dimension(:,:),     intent(in )     :: Lats,                   &
                                                     Lons
    integer  (kind = FourByteIntKind),        &
              dimension(1:4,1:2), intent(out)     :: BoxCorners
    ! !Description:
    !   This routine will find the four corners of the box to be subsetted.  This 
    !      is accomplished by finding the points, which deviate(error) the least
    !      from the desired corner points(based on inputted desired lats/lons and
    !      epsilon).  The result will be an odd shaped box(perhaps some sort of 
    !      parallelegram or such), that will have to be redefined to be a rectangle.
    !      The new rectangle will then contain all the points in the desired box
    !      plus some unwanted points included to make this box a rectangle.  These
    !      points will be masked out in calls to applyMask.
    !   The rectangular shape is necessary, as FORTRAN does not allow odd shaped
    !      arrays.
    !
    ! !Input Parameters:
    !    Lats, Lons     : Arrays that store the entire set of geolocation values.
    !
    ! !Output Parameters:
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )           
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    The rectangular shape is necessary, as FORTRAN does not allow odd shaped
    !      arrays.
    !    The indices of the resulting rectangle are store in the BoxCorners array.
    !    The subroutine FindFourPoints is called to determine the closest points
    !      to the desired point.  It is located below this routine.
    !
    ! !END

    !local variables:
    integer  (kind = FourByteIntKind)             :: maxIIndex, maxJIndex
    real     (kind = FourByteRealKind)            :: minDistFromPoint
    integer  (kind = FourByteIntKind), &
              dimension(1:2)                      :: BoxTemp

    !Define the maximum sizes of the granule:
    maxIIndex = size(Lats,1)
    maxJIndex = size(Lats,2)
    
    !Find the four corners of the box based on the desiredLat/Lon:
    call FindFourPoints(  LatVal1 = desiredLat + EpsilonLat, & !UL
                          LonVal1 = desiredLon - EpsilonLon, & !UL
                          LatVal2 = desiredLat + EpsilonLat, & !UR
                          LonVal2 = desiredLon + EpsilonLon, & !UR
                          LatVal3 = desiredLat - EpsilonLat, & !LL
                          LonVal3 = desiredLon - EpsilonLon, & !LL
                          LatVal4 = desiredLat - EpsilonLat, & !LR
                          LonVal4 = desiredLon + EpsilonLon, & !LR
                             Lats = Lats,                    &
                             Lons = Lons,                    &
                           Index1 = BoxCorners(1,1:2),       & !UL
                           Index2 = BoxCorners(2,1:2),       & !UR
                           Index3 = BoxCorners(3,1:2),       & !LL
                           Index4 = BoxCorners(4,1:2)        ) !LR

    !Depending if the satellite is ascending or descending, the
    !  defined box will be correct, otherwise the LR is really
    !  the UL and the LL is the UR (flipped).
    !Check if the points are opposite, and if so, then flip them:
    !UL > LL
    if (BoxCorners(1,2) > BoxCorners(3,2)) then
       BoxTemp(1) = BoxCorners(1,1)
       BoxTemp(2) = BoxCorners(1,2)
       BoxCorners(1,1:2) = BoxCorners(3,1:2)
       BoxCorners(3,1:2) = BoxTemp(1:2)
    end if
    !UR > LR
    if (BoxCorners(2,2) > BoxCorners(4,2)) then
       BoxTemp(1) = BoxCorners(2,1)
       BoxTemp(2) = BoxCorners(2,2)
       BoxCorners(2,1:2) = BoxCorners(4,1:2)
       BoxCorners(4,1:2) = BoxTemp(1:2)
    end if
    !UL > UR
    if (BoxCorners(1,1) > BoxCorners(2,1)) then
       BoxTemp(1) = BoxCorners(1,1)
       BoxTemp(2) = BoxCorners(1,2)
       BoxCorners(1,1:2) = BoxCorners(2,1:2)
       BoxCorners(2,1:2) = BoxTemp(1:2)
    end if
    !LL > LR
    if (BoxCorners(3,1) > BoxCorners(4,1)) then
       BoxTemp(1) = BoxCorners(3,1)
       BoxTemp(2) = BoxCorners(3,2)
       BoxCorners(3,1:2) = BoxCorners(4,1:2)
       BoxCorners(4,1:2) = BoxTemp(1:2)
    end if

    !Lastly, Redefine the box corners:
    !test to see which point will yield the larger box:
    !East Longitude:
    if ( BoxCorners(1,1) < BoxCorners(3,1) ) then
         BoxCorners(3,1) = BoxCorners(1,1)
    else
         BoxCorners(1,1) = BoxCorners(3,1)
    end if
    !West Longitude:
    if ( BoxCorners(2,1) > BoxCorners(4,1) ) then
         BoxCorners(4,1) = BoxCorners(2,1)
    else
         BoxCorners(2,1) = BoxCorners(4,1)
    end if
    !North Latitude:
    if ( BoxCorners(1,2) < BoxCorners(2,2) ) then
         BoxCorners(2,2) = BoxCorners(1,2)
    else
         BoxCorners(1,2) = BoxCorners(2,2)
    end if
    !South Latitude:
    if ( BoxCorners(3,2) > BoxCorners(4,2) ) then
         BoxCorners(4,2) = BoxCorners(3,2)
    else
         BoxCorners(3,2) = BoxCorners(4,2)
    end if

  end subroutine findBoxCorners

  !*****************************************************************************  
  subroutine findHighResBoxCorners( HighResDimSizes,  ResCnvrs,         &
                                    LowResBoxCorners, HighResBoxCorners )
    integer  (kind = FourByteIntKind),    &
              dimension(:),           intent(in )      :: HighResDimSizes
    integer  (kind = FourByteIntKind),    &
                                      intent(in )      :: ResCnvrs
    integer  (kind = FourByteIntKind),    &
              dimension(1:4,1:2),     intent(in )      ::  LowResBoxCorners
    integer  (kind = FourByteIntKind),    &
              dimension(1:4,1:2),     intent(out)      :: HighResBoxCorners
    
    ! !Description:
    !   This routine will determine the four corners of the high resolution box
    !      to be subsetted.  This is accomplished by using the low resolution
    !      box corner points and translating them into high resolution points.
    !   Please see the desing notes for the translation method.
    !   The result will be an odd shaped box(perhaps some sort of 
    !      parallelegram or such), that will have to be redefined to be a rectangle.
    !      The new rectangle will then contain all the points in the desired box
    !      plus some unwanted points included to make this box a rectangle.  These
    !      points will be masked out in calls to applyMask.
    !   The rectangular shape is necessary, as FORTRAN does not allow odd shaped
    !      arrays.
    !
    ! !Input Parameters:
    ! HighResDimSizes   : Used to store the dimensions of the high resolution SDS.
    ! ResCnvrs          : The conversion between high res and low res.
    ! HighResBoxCorners : Contains the indices of the high res subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )   
    !
    ! !Output Parameters:
    ! LowResBoxCorners  : Contains the indices of the low res subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )           
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    ! Here is a background behind the logic of this routine:
    !   The number of points for the high resolution, and the indexing is expressed
    !    by the following example.  Say we have 5km and 1km resolutions and two 5km points.
    !    Then the 1km data would have 6 points, with the two resolutions matching up as
    !    follows:  5km point 1 = 1km point 1, 5km point 2 = 1km point 6.
    !    Thus, for x 5km points, there will be x*5-4 1km points.
    !   There is an additional check between the low and high resolution SDSs arrising
    !    from the number of high resolution points not exactly equaling the number of
    !    low resolution points multiplied by the resolution converter.  Ex. If there
    !    were 200 5km points, there should be 1000 1km points, but this is not always
    !    the case, there are sometimes less and sometime more points.  So, the program
    !    checks this versus a sample high resolution SDS.
    !     For example, if the 1km SDS has an along dimension of 2025, but the 5km SDS 
    !     has an along dimension of 406, then there should be 2030-4 = 2026 1km points
    !     that can be interpolated.  So, the dimension of the 1km box to be subsetted
    !     will be set to have a point at 2021 instead of the max dimension of 2026.
    !     This allows for the correct interpolation.
    !   Note also that if the high res max dimension is larger than the 
    !     low res dim * ResConv + 1, the values beyond low resdim*ResConv+1 will be
    !     excluded from the box, because there won't be a corresponding low res point
    !     to perform the resolution.
    !      For example, if the 1km SDS has an along dimension of 2031, but the 5km SDS
    !      has an along dimension of 406, then there should be 2030-4 = 2026 1km points
    !      that can be interpolated.  So, the dimension will be set to 2026 instead of 
    !      the max along dimension of 2031, to allow for a proper interpolation.
    ! !END

    !local variables:
    integer  (kind = FourByteIntKind)             :: maxI, maxJ

    !Check to make sure the furthermost indicies do not exceed the low res box dims:
    ! (See design notes)
    !  Use the lower right corner point for this check (since it may be the maximum
    !  along and across dimensions).
    MaxI = LowResBoxCorners(4,1) * ResCnvrs - ResCnvrs + 1
    MaxJ = LowResBoxCorners(4,2) * ResCnvrs - ResCnvrs + 1
    !If the MaxI is larger than the maximum high resolution I dimension, then scale
    !  it back until MaxI is equal or less than the max HR I dim.
    do while (MaxI > HighResDimSizes(1))
       MaxI = MaxI - ResCnvrs
    end do
    
    !If the MaxJ is larger than the maximum high resolution J dimension, then scale
    !  it back until MaxJ is equal or less than the max HR J dim.
    do while (MaxJ > HighResDimSizes(2))
       MaxJ = MaxJ - ResCnvrs
    end do
    
    !Define the High Res Box Corners:
    !Upper Left:
    HighResBoxCorners(1,1) = LowResBoxCorners(1,1) * ResCnvrs - ResCnvrs + 1
    HighResBoxCorners(1,2) = LowResBoxCorners(1,2) * ResCnvrs - ResCnvrs + 1
    !Upper Right:
    HighResBoxCorners(2,1) = MaxI
    HighResBoxCorners(2,2) = LowResBoxCorners(2,2) * ResCnvrs - ResCnvrs + 1
    !Lower Left:
    HighResBoxCorners(3,1) = LowResBoxCorners(3,1) * ResCnvrs - ResCnvrs + 1
    HighResBoxCorners(3,2) = MaxJ
    !Lower Right:
    HighResBoxCorners(4,1) = MaxI
    HighResBoxCorners(4,2) = MaxJ

  end subroutine findHighResBoxCorners
  !*****************************************************************************
  !*****************************************************************************
  subroutine FindFourPoints( LatVal1, LonVal1, LatVal2, LonVal2,  &
                             LatVal3, LonVal3, LatVal4, LonVal4,  &
                             Lats,    Lons,                       &
                             Index1,  Index2,  Index3,  Index4   )
    real     (kind = FourByteRealKind),   &
                               intent(in)         :: LatVal1, LonVal1,       &
                                                     LatVal2, LonVal2,       &
                                                     LatVal3, LonVal3,       &
                                                     LatVal4, LonVal4
    real     (kind = FourByteRealKind),   &
              dimension(:,:),  intent(in)         :: Lats,                   &
                                                     Lons
    integer  (kind = FourByteIntKind),    &
              dimension(1:2),  intent(out)        :: Index1, Index2,         &
                                                     Index3, Index4
    ! !Description:
    !   This routine will find the four points in the Lat/Lon data that are closest to 
    !      the Lat/Lon points specified.  This is accomplished by looping through the
    !      the Lat/Lon arrays and computing the distance from the desired Lat/Lon
    !      points (computed as dist = abs(desiredLat-Lat)+abs(desiredLon-Lon)).  The 
    !      Lat/Lon point that is the closest (i.e. has the smallest dist) to the
    !      desired point will be used as the corner point.
    !   This scheme allows for all possible scenarios.  It works for the entire box 
    !      being inside the granule, but more importantly it handles cases where
    !      only one, two, or three points are inside the granule.  In these cases,
    !      one or more desired point(s) lie outside the granule, but the closest point
    !      to these desired points will be found to lie on an edge, and will be the
    !      point with the smallest dist.
    !
    ! !Input Parameters:
    !    LatValX, LonValX : The desired Lat and Lon values.
    !    Lats, Lons       : The Latitude and Longitude arrays.
    !
    ! !Output Parameters:
    !    IndexX           : The indices of the closest points.
    !                         ( j=1: i index of box
    !                           j=2: j index of box )
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !local variables:
    real     (kind = FourByteRealKind)            ::    distFromPoint,       &
                                                     minDistFromPoint1,      &
                                                     minDistFromPoint2,      &
                                                     minDistFromPoint3,      &
                                                     minDistFromPoint4
    !these store the values of the calculated and minimum distance from apoint.

    minDistFromPoint1 = 1000
    minDistFromPoint2 = 1000
    minDistFromPoint3 = 1000
    minDistFromPoint4 = 1000
    do i = 1, size(Lats,1)
       do j = 1, size(Lats,2)

          !check the first point:
          distFromPoint =   abs( LatVal1 - Lats(i,j) ) &
                          + abs( LonVal1 - Lons(i,j) )
          if ( distFromPoint < minDistFromPoint1 ) then
             minDistFromPoint1 = distFromPoint
             Index1(1) = i
             Index1(2) = j
          end if
          
          !check the second point:
          distFromPoint =   abs( LatVal2 - Lats(i,j) ) &
                          + abs( LonVal2 - Lons(i,j) )
          if ( distFromPoint < minDistFromPoint2 ) then
             minDistFromPoint2 = distFromPoint
             Index2(1) = i
             Index2(2) = j
          end if

          !check the third point:
          distFromPoint =   abs( LatVal3 - Lats(i,j) ) &
                          + abs( LonVal3 - Lons(i,j) )
          if ( distFromPoint < minDistFromPoint3 ) then
             minDistFromPoint3 = distFromPoint
             Index3(1) = i
             Index3(2) = j
          end if
          
          !check the fourth point:
          distFromPoint =   abs( LatVal4 - Lats(i,j) ) &
                          + abs( LonVal4 - Lons(i,j) )
          if ( distFromPoint < minDistFromPoint4 ) then
             minDistFromPoint4 = distFromPoint
             Index4(1) = i
             Index4(2) = j
          end if

       end do
    end do

  end subroutine FindFourPoints
  !*****************************************************************************




  !*****************************************************************************
  !SubsetCheck
  !***************************************************************************** 
  subroutine SubsetCheck(dimensionSizes, GeoDimSizes, &
                         SpatialSubset,  CorrectOrder )
    integer  (kind = FourByteIntKind), &
              dimension(:),            intent( in)   :: dimensionSizes,       &
                                                        GeoDimSizes
    logical,                           intent(out)   :: SpatialSubset, &
                                                        CorrectOrder
    ! !Description:
    ! This routine will determine if the SDS is subsettable, and
    !   the order of the dimensions for 2D and 3D arrays.  The order for 2D arrays
    !   should be along, across, for 3D arrays the ordering should be band, along,
    !   across.  The ordering is stored in the MultDimOrder array.
    !
    ! !Input Parameters:
    ! dimensionSizes  : Used to store the dimension of an SDS.
    ! GeoDimSizes     : Stores the dimensions of the Geolocation
    !
    ! !Output Parameters:
    ! SpatialSubset   : Flag for seeing if the SDS is subsettable or not.
    ! CorrectOrder    : Flag to see if the SDS dimensions are in the correct order.
    !                     2D should be along, across
    !                     3D should be band, along, across
    !                     if 2D and flase, then order is across, along
    !                     if 3D and flase, then order is along, across, band
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !Begin testing to determine the rank of the array:

    !Initialize the flags to false, and only set them to true 
    !  if it is determined that the SDS is in the correct order, and subsettable.
    SpatialSubset = .false.
    CorrectOrder = .false. 
    
    if (dimensionSizes(3) > 0) then

       !3D array
       !Test to see if the SDS dimensions are contained 
       !  somewhere within the Geolocation dimensions:
       if (dimensionSizes(1) == GeoDimSizes(1) .or. &
           dimensionSizes(2) == GeoDimSizes(1) .or. &
           dimensionSizes(3) == GeoDimSizes(1)      ) then
          if (dimensionSizes(1) == GeoDimSizes(2) .or. &
              dimensionSizes(2) == GeoDimSizes(2) .or. &
              dimensionSizes(3) == GeoDimSizes(2)      ) then
             !The two Geolocation dims are contained somewhere within the SDS,
             !  now test to see if they are stored either as
             !  band, along, across or along, across, band.
             !  If they are not then this can not be subsetted.
             
             !This could be a high res product, test to see how stored:
             if (dimensionSizes(1) == GeoDimSizes(1) .and. &
                 dimensionSizes(2) == GeoDimSizes(2)       ) then
                !This is stored correctly
                SpatialSubset = .true.
                CorrectOrder  = .true.
             else if (dimensionSizes(2) == GeoDimSizes(1) .and. &
                      dimensionSizes(3) == GeoDimSizes(2)       ) then
                !This is stored incorrectly
                SpatialSubset = .true.

             !Not stored correctly.
             end if

          end if

       !if not true, then it is not subsettable.
       end if


    else if (dimensionSizes(2) > 0) then
       !2D array

       !Test to see if subsettable:
       if (dimensionSizes(1) == GeoDimSizes(1) .or. &
           dimensionSizes(2) == GeoDimSizes(1)      ) then

          !Determine how the array is stored 
          !  and set the dimension order array:
          if (dimensionSizes(1) == GeoDimSizes(1)) then
             !dimension 1 is stored correctly, test second dim:
             if (dimensionSizes(2) == GeoDimSizes(2)) then
                !this array is stored correctly:
                SpatialSubset = .true.
                CorrectOrder  = .true.
             !if not true, then second array not correct, not subsettable.
             end if
          else
             !The dimension order is switched, check the second dim:
             if (dimensionSizes(2) == GeoDimSizes(1)) then
                !The dimensions are correct, but switched:
                SpatialSubset = .true.
                CorrectOrder  = .false.
             !if not true, then second array not correct, not subsettable.
             end if
          end if

       !If not true, then neither dimension matches, so it is not subsettable.
       end if


    !If neither of these are true, than 1D or not subsettable.
    end if



  end subroutine SubsetCheck
  !*****************************************************************************




  !*****************************************************************************
  !copyToBox routines:
  !*****************************************************************************  
  subroutine CreateHighResGeo( ResCnvrs,               &
                               LowResBC, HighResBC,    &
                               LowResData, HighResData )
    integer  (kind = FourByteIntKind),    &
                                      intent(in )      :: ResCnvrs
    integer  (kind = FourByteIntKind),    &
              dimension(1:4,1:2),     intent(in )      :: LowResBC, HighResBC
    real     (kind = FourByteRealKind),   &
              dimension(:,:),         intent(in )      :: LowResData
    real     (kind = FourByteRealKind),   &
              dimension(:,:),         intent(out)      :: HighResData
    ! !Description:
    ! This routine will create the high resolution geolocation by performing
    !   a linear interpolate of the low resolution geolocation.  See the design
    !   notes for explination of the method.
    !
    ! !Input Parameters:
    ! ResCnvrs      : The conversion between high res and low res.
    ! LowResBC,
    !  HighResBC    : Contains the indices of the high res subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )   
    !
    ! LowResData        : Contains the original low resolution data.
    !
    ! !Output Parameters:
    ! HighResData       : Contains the interpolated High resolution data.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !  The interpolation is performed by looping through the low resolution box
    !    and performing an interpolation for four points of the low res box.  The
    !    high res box is filled in this way as the looping continues through the
    !    entire low res box.
    !  The generic routine InterpBox will perform the linear interpolation given
    !    four points of the low resolution grid and the corresponding points of 
    !    the high res grid.  To eliminate interpolation redundancies, the routine
    !    does not perform an interpolation for the first column or row of the 
    !    given high resolution grid.  An implication of this, is that the
    !    entire first row and column of the overall high res box is not 
    !    interpolated by calling this routine.
    !  These special cases are handles using the InterpLine routine which performs
    !    a linear interpolation of a line, not a box.  Given the two low res points
    !    the routine fills in the corresponding high res points.  Succesive calls
    !    over the low res 1st row and column fill in these special cases.  This 
    !    routine is set up such that the very first point in the line is not
    !    interpolated for, thereby eliminating redundancies.  This structure means
    !    that the very first point (1,1) is never computed, instead it is copied
    !    from the lower resolution to the higher resolution.
    !  The routine is structured such that first rows and columns are filled in, 
    !    and then the rest of the box is interpolated.
    !
    ! !END

    !local variables:
    integer  (kind = FourByteIntKind)             :: maxI, maxJ
    integer  (kind = FourByteIntKind)             :: icnt, jcnt
    integer  (kind = FourByteIntKind)             :: ipcnt, jpcnt

    !Find the size of the low resolution data:
    MaxI = Size(LowResData,1)
    MaxJ = Size(LowResData,2)

    !Perform the interpolation for the special cases (first row and column)
    !Copy the first point to the High Res array:
    HighResData(1,1) = LowResData(1,1)

    !Perform the interpolation for the first row:
    !ex. if low res = 5km, and high res = 1km, then for icnt = 1,
    !  the low res points 1 and 2 are inputed, and the high res points 1-6 
    !  are interpolated.
    icnt = 1
    do jcnt = 1, MaxJ - 1
       jpcnt = jcnt * ResCnvrs - ResCnvrs + 1
       call InterpLine(    ResCnvrs = ResCnvrs,                               &
                         LowResLine =  LowResData(icnt, jcnt:jcnt+1),         &
                        HighResLine = HighResData(icnt, jpcnt:jpcnt+ResCnvrs) )
    end do
    
    !Perform the interpolation for the first column:
    !ex. if low res = 5km, and high res = 1km, then for icnt = 1,
    !  the low res points 1 and 2 are inputed, and the high res points 1-6 
    !  are interpolated.
    jcnt = 1
    do icnt = 1, MaxI - 1
       ipcnt = icnt * ResCnvrs - ResCnvrs + 1
       call InterpLine(    ResCnvrs = ResCnvrs,                               &
                         LowResLine =  LowResData(icnt:icnt+1, jcnt),         &
                        HighResLine = HighResData(ipcnt:ipcnt+ResCnvrs, jcnt) )
    end do


    !Interpolate the rest of the box:
    !ex. if low res = 5km, and high res = 1km, then for icnt, jcnt = 1,
    !  the low res points (1,1), (1,2), (2,1), and (2,2) are inputed, 
    !  and the high res box with corner points (1,1),(1,6),(6,1),(6,6) 
    !  are interpolated.
    do icnt = 1, MaxI - 1
       do jcnt = 1, MaxJ - 1
          ipcnt = icnt * ResCnvrs - ResCnvrs + 1
          jpcnt = jcnt * ResCnvrs - ResCnvrs + 1
          call InterpBox(   ResCnvrs = ResCnvrs,                              &
                           LowResBox =  LowResData(icnt:icnt+1, jcnt:jcnt+1), &
                          HighResBox = HighResData(ipcnt:ipcnt+ResCnvrs,      &
                                                   jpcnt:jpcnt+ResCnvrs )     ) 
       end do
    end do


  end subroutine CreateHighResGeo

  !*****************************************************************************  
  subroutine InterpLine( ResCnvrs, LowResLine, HighResLine )
    integer  (kind = FourByteIntKind),    &
                                      intent(in )      :: ResCnvrs
    real     (kind = FourByteRealKind),   &
              dimension(:),           intent(in )      ::  LowResLine
    real     (kind = FourByteRealKind),   &
              dimension(:),           intent(out)      :: HighResLine
    ! !Description:
    ! This routine will interpolate a line of high resolution points,
    !   given the two lower resolution points.  The routine is designed
    !   not to compute the very first point in the line, thereby 
    !   eliminating redundancies.
    !
    ! !Input Parameters:
    ! ResCnvrs      : The conversion between high res and low res.
    ! LowResLine    : Contains the original low resolution data.
    !
    ! !Output Parameters:
    ! HighResLine   : Contains the interpolated High resolution data.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !local variables:
    integer  (kind = FourByteIntKind)             :: cnt
    real     (kind = FourByteRealKind),      &
              dimension(1:2)                      :: length

    !Perform the interpolation for points two through ResCnvrs+1:
    !ie if the high res is 1km and the low res is 5km, then
    ! points 2 through 6 are interpolated from low res points 1 and 2.
    do cnt = 2, ResCnvrs
       length(1) = real(cnt-1)
       length(2) = real(ResCnvrs) - length(1)

       HighResLine(cnt) = LowResLine(1)*length(2) + LowResLine(2)*length(1)
       HighResLine(cnt) = HighResLine(cnt) / real(ResCnvrs)
    end do
    HighResLine(ResCnvrs+1) = LowResLine(2)

  end subroutine InterpLine

  !*****************************************************************************  
  subroutine InterpBox( ResCnvrs, LowResBox, HighResBox )
    integer  (kind = FourByteIntKind),    &
                                      intent(in )      :: ResCnvrs
    real     (kind = FourByteRealKind),   &
              dimension(:,:),         intent(in )      ::  LowResBox
    real     (kind = FourByteRealKind),   &
              dimension(:,:),         intent(out)      :: HighResBox
    ! !Description:
    ! This routine will interpolate a box of high resolution points,
    !   given the four lower resolution points.  The routine is designed
    !   not to compute the very first line or column, thereby 
    !   eliminating redundancies.
    !
    !
    ! !Input Parameters:
    ! ResCnvrs      : The conversion between high res and low res.
    ! LowResBox     : Contains the original low resolution data.
    !
    ! !Output Parameters:
    ! HighResBox    : Contains the interpolated High resolution data.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !local variables:
    integer  (kind = FourByteIntKind)             :: ii,jj,cnt
    real     (kind = FourByteRealKind),      &
              dimension(1:4,1:2)                  :: length
    real     (kind = FourByteRealKind),      &
              dimension(1:4)                      :: area
    real     (kind = FourByteRealKind)            :: TotArea

    !Perform the interpolation by looping through the box starting with
    !  points (2,2), in order to eliminate redundancies:
    do ii = 2, ResCnvrs + 1
       do jj = 2, ResCnvrs + 1

          !compute the lengths
          length(1,1) = ii - 1
          length(1,2) = jj - 1
          length(2,1) = ResCnvrs + 1 - ii
          length(3,2) = ResCnvrs + 1 - jj
          length(2,2) = length(1,2)
          length(3,1) = length(1,1)
          length(4,1) = length(2,1)
          length(4,2) = length(3,2)

          !compute the areas:
          do cnt = 1, 4
             area(cnt) = length(cnt,1) * length(cnt,2)
          end do
          TotArea = ResCnvrs * ResCnvrs

          !Compute the value for the point:
          HighResBox(ii,jj) =   Area(4)*LowResBox(1,1) + Area(3)*LowResBox(2,1) &
                              + Area(2)*LowResBox(1,2) + Area(1)*LowResBox(2,2)
          HighResBox(ii,jj) = HighResBox(ii,jj) / TotArea

       end do
    end do

  end subroutine InterpBox

  !*****************************************************************************
  ! CTB individual routines:
  !*****************************************************************************
  subroutine CTBoneByteInt(BoxCorners, CompleteData, SubsettedData)
    integer  (kind = FourByteIntKind),           &
              dimension(1:4,1:2),     intent(in)  :: BoxCorners
    integer  (kind = OneByteIntKind),            &
              dimension(:,:),         intent(in)  :: CompleteData
    integer  (kind = OneByteIntKind),            &
              dimension(:,:),         intent(out) :: SubsettedData
    ! !Description:
    !   This routine will copy the data in the original SDS to the box.
    !   This specific routine is for one byte integers.
    !  
    ! !Input Parameters:
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )           
    !    CompleteData   : The complete SDS, for one waveband.
    !
    ! !Output Parameters:
    !    SubsettedData  : The SDS data that is copied to the box.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine copies one waveband to a box.
    !    The data type for completData and SubsettedData must be identicle.
    !
    ! !END

    do i = BoxCorners(1,1), BoxCorners(2,1)
       do j = BoxCorners(1,2), BoxCorners(3,2)
                 

          iprime = i - BoxCorners(1,1) + 1
          jprime = j - BoxCorners(1,2) + 1

          !print *,'i,iprime,j,jprime',i,iprime,j,jprime
          SubsettedData(iprime,jprime) = CompleteData(i,j)
          
       end do
    end do
    
  end subroutine CTBoneByteInt
  !*****************************************************************************
  subroutine CTBtwoByteInt(BoxCorners, CompleteData, SubsettedData)
    integer  (kind = FourByteIntKind),           &
              dimension(1:4,1:2),     intent(in)  :: BoxCorners
    integer  (kind = TwoByteIntKind),            &
              dimension(:,:),         intent(in)  :: CompleteData
    integer  (kind = TwoByteIntKind),            &
              dimension(:,:),         intent(out) :: SubsettedData
    ! !Description:
    !   This routine will copy the data in the original SDS to the box.
    !   This specific routine is for two byte integers.
    !  
    ! !Input Parameters:
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )           
    !    CompleteData   : The complete SDS, for one waveband.
    !
    ! !Output Parameters:
    !    SubsettedData  : The SDS data that is copied to the box.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine copies one waveband to a box.
    !    The data type for completData and SubsettedData must be identicle.
    !
    ! !END

    do i = BoxCorners(1,1), BoxCorners(2,1)
       do j = BoxCorners(1,2), BoxCorners(3,2)
                 
          iprime = i - BoxCorners(1,1) + 1
          jprime = j - BoxCorners(1,2) + 1

          SubsettedData(iprime,jprime) = CompleteData(i,j)
          
       end do
    end do
    
  end subroutine CTBtwoByteInt
  !*****************************************************************************
  subroutine CTBfourByteInt(BoxCorners, CompleteData, SubsettedData)
    integer  (kind = FourByteIntKind),           &
              dimension(1:4,1:2),     intent(in)  :: BoxCorners
    integer  (kind = FourByteIntKind),           &
              dimension(:,:),         intent(in)  :: CompleteData
    integer  (kind = FourByteIntKind),           &
              dimension(:,:),         intent(out) :: SubsettedData
    ! !Description:
    !   This routine will copy the data in the original SDS to the box.
    !   This specific routine is for four byte integers.
    !  
    ! !Input Parameters:
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )           
    !    CompleteData   : The complete SDS, for one waveband.
    !
    ! !Output Parameters:
    !    SubsettedData  : The SDS data that is copied to the box.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine copies one waveband to a box.
    !    The data type for completData and SubsettedData must be identicle.
    !
    ! !END

    do i = BoxCorners(1,1), BoxCorners(2,1)
       do j = BoxCorners(1,2), BoxCorners(3,2)
                 
          iprime = i - BoxCorners(1,1) + 1
          jprime = j - BoxCorners(1,2) + 1

          SubsettedData(iprime,jprime) = CompleteData(i,j)
          
       end do
    end do
    
  end subroutine CTBfourByteInt
  !*****************************************************************************
  subroutine CTBfourByteReal(BoxCorners, CompleteData, SubsettedData)
    integer  (kind = FourByteIntKind),           &
              dimension(1:4,1:2),       intent(in)  :: BoxCorners
    real     (kind = FourByteRealKind),          &
              dimension(:,:),           intent(in)  :: CompleteData
    real     (kind = FourByteRealKind),          &
              dimension(:,:),           intent(out) :: SubsettedData
    ! !Description:
    !   This routine will copy the data in the original SDS to the box.
    !   This specific routine is for four byte reals.
    !  
    ! !Input Parameters:
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )           
    !    CompleteData   : The complete SDS, for one waveband.
    !
    ! !Output Parameters:
    !    SubsettedData  : The SDS data that is copied to the box.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine copies one waveband to a box.
    !    The data type for completData and SubsettedData must be identicle.
    !
    ! !END

    do i = BoxCorners(1,1), BoxCorners(2,1)
       do j = BoxCorners(1,2), BoxCorners(3,2)
                 
          iprime = i - BoxCorners(1,1) + 1
          jprime = j - BoxCorners(1,2) + 1
          
          SubsettedData(iprime,jprime) = CompleteData(i,j)
          
       end do
    end do
    
  end subroutine CTBfourByteReal
  !*****************************************************************************
  subroutine CTBeightByteReal(BoxCorners, CompleteData, SubsettedData)
    integer  (kind = FourByteIntKind),           &
              dimension(1:4,1:2),        intent(in)  :: BoxCorners
    real     (kind = EightByteRealKind),         &
              dimension(:,:),            intent(in)  :: CompleteData
    real     (kind = EightByteRealKind),         &
              dimension(:,:),            intent(out) :: SubsettedData
    ! !Description:
    !   This routine will copy the data in the original SDS to the box.
    !   This specific routine is for eight byte reals.
    !  
    ! !Input Parameters:
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                      ( dimension (i,j):
    !                         i=1 : Upper Left    j=1: i index of box
    !                         i=2 : Upper Right   j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                      )           
    !    CompleteData   : The complete SDS, for one waveband.
    !
    ! !Output Parameters:
    !    SubsettedData  : The SDS data that is copied to the box.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine copies one waveband to a box.
    !    The data type for completData and SubsettedData must be identicle.
    !
    ! !END

    do i = BoxCorners(1,1), BoxCorners(2,1)
       do j = BoxCorners(1,2), BoxCorners(3,2)
                 
          iprime = i - BoxCorners(1,1) + 1
          jprime = j - BoxCorners(1,2) + 1

          SubsettedData(iprime,jprime) = CompleteData(i,j)
          
       end do
    end do
    
  end subroutine CTBeightByteReal
  !*****************************************************************************




  !*****************************************************************************
  subroutine createMask(Lats, Lons, Mask)
    real     (kind = FourByteRealKind),  &
              dimension(:,:),            intent(in )     :: Lats,      &
                                                            Lons
    integer  (kind = FourByteIntKind),   &
              dimension(:,:),            intent(out)     :: Mask
    ! !Description:
    !   This routine will create the Mask used to set unwanted points in the box
    !      with fill values.  The routine loops though the box and determines
    !      if the point is within the desired Lat/Lon box.  If so, then the Mask
    !      is set equal to 1, if not, then the mask is set equal to 0.
    !
    ! !Input Parameters:
    !    Lats,
    !     Lons  : Arrays that store the entire set of geolocation values.
    !
    ! !Output Parameters:
    !    Mask   : Array containing the mask.  The mask is used to replace unwanted
    !             points in the subsetted box, with fill values.  This is accomplished
    !             by setting the mask to 1 for desired points and 0 for unwanted
    !             points.  Then the mask is applied to the subsetted SDS, in which
    !             when the mask = 0, the fill value replaces the stored value.BoxCorners
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    The rectangular shape is necessary, as FORTRAN does not allow odd shaped
    !      arrays.
    !    The indices of the resulting rectangle are store in the BoxCorners array.
    !    The subroutine FindFourPoints is called to determine the closest points
    !      to the desired point.  It is located below this routine.
    !
    ! !END


    !local vars:
    integer  (kind = FourByteIntKind)                    :: IDim, JDim
    ! sizes of the Lat/Lon arrays.

    !initialize Mask
    Mask = 1

    !determine sizes of arrays:
    IDim = Size(Lats,1)
    JDim = Size(Lats,2)

    !Loop through array, checking each point for being within the desired box:
    do i = 1, IDim
       do j = 1, JDim

          !Check to see if the point is within the desired Lats:
          if    (      Lats(i,j) <= desiredLat - EpsilonLat &       !South Boundary
                  .or. Lats(i,j) >= desiredLat + EpsilonLat ) then  !North Boundary

                !The point is not within the desired Lats, so set the Mask to 0:
                Mask(i,j) = 0

          else
             !Within the desired Lats, now check the Lons:
             if (      Lons(i,j) <= desiredLon - EpsilonLon &       !West Boundary
                  .or. Lons(i,j) >= desiredLon + EpsilonLon ) then  !East Boundary

                !The point is not within the desired Lons, so set the Mask to 0:
                Mask(i,j) = 0
                
             end if

          end if

       end do
    end do


  end subroutine createMask
  !*****************************************************************************




  !*****************************************************************************
  !applyMask  routines:
  !*****************************************************************************
  subroutine AMaskEBRK(Data, Mask, origID)
    real     (kind = EightByteRealKind), &
              dimension(:,:)                             :: Data
    integer  (kind = FourByteIntKind),   &
              dimension(:,:),            intent( in)     :: Mask
    integer  (kind = FourByteIntKind),   intent( in)     :: origID
    ! !Description:
    !   This routine will apply the mask, in order to replace unwanted points,
    !      within the box, with fill values.  The routine first obtains the 
    !      fill value for this SDS, then loops over the boxed data, replacing
    !      data with the fill value, when the mask = 0.
    !   This specific routine is for eight byte reals.
    !  
    ! !Input Parameters:
    !    Data   : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    Mask   : Array containing the mask.  The mask is used to replace unwanted
    !             points in the subsetted box, with fill values.  This is accomplished
    !             by setting the mask to 1 for desired points and 0 for unwanted
    !             points.  Then the mask is applied to the subsetted SDS, in which
    !             when the mask = 0, the fill value replaces the stored value.
    !    origID : 
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine applies the mask to one waveband of the box.
    ! !END

    !local Vars:
    real     (kind = EightByteRealKind)                  :: fillValue
    integer  (kind = FourByteIntKind)                    :: ISize, JSize

    !get the fill value:
    call getFillValue(    origID = origID, &
                       fillValue = fillValue)

    !obtain the size of the array:
    ISize = Size(Data,1)
    JSize = Size(Data,2)

    !apply the mask:
    do i = 1, ISize
       do j = 1, JSize

          !replace any mask = 0 points with the fill value:
          if (mask(i,j) == 0) then
             Data(i,j) = fillValue
          end if

       end do
    end do


  end subroutine AMaskEBRK
  !*****************************************************************************
  subroutine AMaskFBRK(Data, Mask, origID)
    real     (kind = FourByteRealKind),  &
              dimension(:,:)                             :: Data
    integer  (kind = FourByteIntKind),   &
              dimension(:,:),            intent( in)     :: Mask
    integer  (kind = FourByteIntKind),   intent( in)     :: origID
    ! !Description:
    !   This routine will apply the mask, in order to replace unwanted points,
    !      within the box, with fill values.  The routine first obtains the 
    !      fill value for this SDS, then loops over the boxed data, replacing
    !      data with the fill value, when the mask = 0.
    !   This specific routine is for four byte reals.
    !  
    ! !Input Parameters:
    !    Data   : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    Mask   : Array containing the mask.  The mask is used to replace unwanted
    !             points in the subsetted box, with fill values.  This is accomplished
    !             by setting the mask to 1 for desired points and 0 for unwanted
    !             points.  Then the mask is applied to the subsetted SDS, in which
    !             when the mask = 0, the fill value replaces the stored value.
    !    origID : 
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine applies the mask to one waveband of the box.
    ! !END

    !local Vars:
    real     (kind = FourByteRealKind)                   :: fillValue
    integer  (kind = FourByteIntKind)                    :: ISize, JSize

    !get the fill value:
    call getFillValue(    origID = origID, &
                       fillValue = fillValue)

    !obtain the size of the array:
    ISize = Size(Data,1)
    JSize = Size(Data,2)

    !apply the mask:
    do i = 1, ISize
       do j = 1, JSize

          !replace any mask = 0 points with the fill value:
          if (mask(i,j) == 0) then
             Data(i,j) = fillValue
          end if

       end do
    end do

  end subroutine AMaskFBRK
  !*****************************************************************************
  subroutine AMaskOBIK(Data, Mask, origID)
    integer  (kind = OneByteIntKind),    &
              dimension(:,:)                             :: Data
    integer  (kind = FourByteIntKind),   &
              dimension(:,:),            intent( in)     :: Mask
    integer  (kind = FourByteIntKind),   intent( in)     :: origID
    ! !Description:
    !   This routine will apply the mask, in order to replace unwanted points,
    !      within the box, with fill values.  The routine first obtains the 
    !      fill value for this SDS, then loops over the boxed data, replacing
    !      data with the fill value, when the mask = 0.
    !   This specific routine is for four byte integers.
    !  
    ! !Input Parameters:
    !    Data   : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    Mask   : Array containing the mask.  The mask is used to replace unwanted
    !             points in the subsetted box, with fill values.  This is accomplished
    !             by setting the mask to 1 for desired points and 0 for unwanted
    !             points.  Then the mask is applied to the subsetted SDS, in which
    !             when the mask = 0, the fill value replaces the stored value.
    !    origID : 
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine applies the mask to one waveband of the box.
    ! !END

    !local Vars:
    integer  (kind = OneByteIntKind)                     :: fillValue
    integer  (kind = FourByteIntKind)                    :: ISize, JSize

    !get the fill value:
    call getFillValue(    origID = origID, &
                       fillValue = fillValue)

    !obtain the size of the array:
    ISize = Size(Data,1)
    JSize = Size(Data,2)

    !apply the mask:
    do i = 1, ISize
       do j = 1, JSize

          !replace any mask = 0 points with the fill value:
          if (mask(i,j) == 0) then
             Data(i,j) = fillValue
          end if

       end do
    end do

  end subroutine AMaskOBIK
  !*****************************************************************************
  subroutine AMaskTBIK(Data, Mask, origID)
    integer  (kind = TwoByteIntKind),    &
              dimension(:,:)                             :: Data
    integer  (kind = FourByteIntKind),   &
              dimension(:,:),            intent( in)     :: Mask
    integer  (kind = FourByteIntKind),   intent( in)     :: origID
    ! !Description:
    !   This routine will apply the mask, in order to replace unwanted points,
    !      within the box, with fill values.  The routine first obtains the 
    !      fill value for this SDS, then loops over the boxed data, replacing
    !      data with the fill value, when the mask = 0.
    !   This specific routine is for two byte integers.
    !  
    ! !Input Parameters:
    !    Data   : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    Mask   : Array containing the mask.  The mask is used to replace unwanted
    !             points in the subsetted box, with fill values.  This is accomplished
    !             by setting the mask to 1 for desired points and 0 for unwanted
    !             points.  Then the mask is applied to the subsetted SDS, in which
    !             when the mask = 0, the fill value replaces the stored value.
    !    origID : 
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine applies the mask to one waveband of the box.
    ! !END

    !local Vars:
    integer  (kind = TwoByteIntKind)                     :: fillValue
    integer  (kind = FourByteIntKind)                    :: ISize, JSize

    !get the fill value:
    call getFillValue(    origID = origID, &
                       fillValue = fillValue)

    !obtain the size of the array:
    ISize = Size(Data,1)
    JSize = Size(Data,2)

    !apply the mask:
    do i = 1, ISize
       do j = 1, JSize

          !replace any mask = 0 points with the fill value:
          if (mask(i,j) == 0) then
             Data(i,j) = fillValue
          end if

       end do
    end do

  end subroutine AMaskTBIK
  !*****************************************************************************
  subroutine AMaskFBIK(Data, Mask, origID)
    integer  (kind = FourByteIntKind),   &
              dimension(:,:)                             :: Data
    integer  (kind = FourByteIntKind),   &
              dimension(:,:),            intent( in)     :: Mask
    integer  (kind = FourByteIntKind),   intent( in)     :: origID
    ! !Description:
    !   This routine will apply the mask, in order to replace unwanted points,
    !      within the box, with fill values.  The routine first obtains the 
    !      fill value for this SDS, then loops over the boxed data, replacing
    !      data with the fill value, when the mask = 0.
    !   This specific routine is for one byte integers.
    !  
    ! !Input Parameters:
    !    Data   : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    Mask   : Array containing the mask.  The mask is used to replace unwanted
    !             points in the subsetted box, with fill values.  This is accomplished
    !             by setting the mask to 1 for desired points and 0 for unwanted
    !             points.  Then the mask is applied to the subsetted SDS, in which
    !             when the mask = 0, the fill value replaces the stored value.
    !    origID : 
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Each call to this routine applies the mask to one waveband of the box.
    ! !END

    !local Vars:
    integer  (kind = FourByteIntKind)                    :: fillValue
    integer  (kind = FourByteIntKind)                    :: ISize, JSize

    !get the fill value:
    call getFillValue(    origID = origID, &
                       fillValue = fillValue)

    !obtain the size of the array:
    ISize = Size(Data,1)
    JSize = Size(Data,2)

    !apply the mask:
    do i = 1, ISize
       do j = 1, JSize

          !replace any mask = 0 points with the fill value:
          if (mask(i,j) == 0) then
             Data(i,j) = fillValue
          end if

       end do
    end do


  end subroutine AMaskFBIK
  !*****************************************************************************


end module subsettingUtils
