module dataUtils
  !*****************************************************************************
  ! !F90
  !
  ! !Description:
  !    This module contains data utility functions and subroutines. 
  !  
  ! !Callable routines:
  !    calibrate()
  !    getCoreMetadata() 
  !    getDataSetDimSizes()
  !    getCalibrationInfo()
  !    getDataSetType()
  !    getFillValue()
  ! 
  ! !Revision History:
  ! $Log: dataUtils.f90,v $
  ! Revision 1.0  1999/09/02  12:43:43  EGMoody
  ! Initial revision.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   Many of the hdf routines and module designs were inspired or modified from codes
  !    developed by Robert Pincus (University of Wisconsin and former GSFC programmer).
  !
  ! !END
  !*****************************************************************************

  !Dependencies:
  use typeSizes, only :  OneByteIntKind, TwoByteIntKind, FourByteIntKind,  &
                         FourByteRealKind, EightByteRealKind
  use hdf,       only :  SFfattr, SFgainfo, SFrcatt, SFrnatt, SFginfo, &
                         MAX_NC_NAME, MAX_VAR_DIMS, Fail, &
                         DFNT_CHAR
  implicit none

  private

  public :: getCoreMetadata 
  public :: getDataSetDimSizes
  public :: getCalibrationInfo
  public :: getDataSetType


  !General interfaces:
  interface calibrate
     module procedure calibrateFourByteReal, &
                      calibrateEightByteReal
  end interface
  public :: calibrate
  !function calibrate*****ByteReal(inputArray, offset, scaleFactor) result(outputArray)
  !  real  (kind = *****ByteRealKind),       &
  !         dimension(:, :), intent( in)      :: inputArray
  !  real  (kind = EightByteRealKind),       &
  !                               intent( in) :: offset, scaleFactor
  !  real  (kind = *****ByteRealKind),       &
  !         dimension(size(inputArray, 1),   &
  !                      size(inputArray, 2)) :: outputArray
  ! !F90
  !
  ! !Description:
  !   Applys scale factor and offset correction to a real-valued array (as
  !     would be read from an HDF file) to return a correctly calibrated array.
  !   Generic interface is applied based on the kind of the input and outpu arrays.
  !  
  ! !Input Parameters:
  !    inputArray: real valued rank two array. Typically this contains real values 
  !      as read from an HDF file. 
  !    offset: offset, as would be read from an HDF SDS attribute. 
  !    scaleFactor: linear scaling factor, as would be read from an HDF SDS attribute.  
  !
  ! !Output Parameters:
  !   The function returns an array of rank two of the same size as inputArray, with 
  !     the scale factor and offset applied. 
  !
  ! !Revision History:
  !   See Module revision history at the beginning of the file.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Robert Pincus
  !   Modified by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   Note that this definition of how to apply the calibration information 
  !     is consistent with the HDF convention. 
  !  Both offset and scaleFactor are optional arguments. 
  !
  ! !END



  interface getFillValue
     module procedure getOBIKFV, getTBIKFV, getFBIKFV, getFBRKFV, getEBRKFV
  end interface
  public :: getFillValue
  !subroutine get***KFV(origID, fillValue)
  !  integer   (kind = FourByteIntKind),   intent( in)  :: origID
  !  *******   (kind = *****Byte****Kind), intent( in)  :: fillValue
  !
  ! !F90
  !
  ! !Description:
  !   Reads in the fill value for a specified SDS.
  !   Generic interface is based upon the data type of the fillValue.
  !  
  ! !Input Parameters:
  !   origID: Data handle for accessing a SDS, used by HDF routines and 
  !            passed as argument in subroutines. 
  !
  ! !Output Parameters:
  !   fillValue:  Contains the fill value.
  !
  ! !Revision History:
  !   See Module revision history at the beginning of the file.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !
  ! !END



  !local variables:
  integer (kind = FourByteIntKind)            :: status
  character (len = 2048)                      :: msgbuffer
  ! status      : Used for error checking.
  ! msgbuffer   : Used for error messages.

  contains

    ! ----------------------------------------------------------------------------------
    subroutine getCoreMetadata(fileID, attributeName, attributeValue)
    integer(kind = FourByteIntKind), intent( in) :: fileID
    character(len = *),              intent( in) :: attributeName
    character(len = *),              intent(out) :: attributeValue
    ! !F90
    !
    ! !Description:
    !    Read the value of an attribute from an HDF file. Error trapping
    !      is internal to the subroutine. 
    !  
    ! !Input Parameters:
    !   dataSetID: the HDF ID of the file, array, or dimension from which to read the attribute
    !   attributeName: a character string holding the name of the attribute
    !
    ! !Output Parameters:
    !   attributeValue: the value of the attribute; a character string.
    ! 
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Robert Pincus
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Robert.Pincus@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END
    
    ! Local variables
    character(len = MAX_NC_NAME)       :: name
    integer(kind = FourByteIntKind)    :: attributeType, attributeID, attributeLength
    ! name            : Stores the name of the attribue
    ! attributeType   : Stores the numerical value of the attribute type.
    ! attributeID     : Stores the attribute handle, for HDF calls.
    ! attributeLength : Stores the length of the attribute.

    !obtain the attribute ID:
    attributeID    = SFfattr(fileID, trim(attributeName))
    if (attributeID    == FAIL) then
       print *, "Can't obtain the attribute ID for attribute" // trim(attributeName)
       stop
    end if

    !obtain the attribute info:
    status    = SFgainfo(fileID, attributeID,                 &
                         name, attributeType, attributeLength)

    if (status      == FAIL) then
       print *, "Can't read information about attribute " // trim(attributeName) 
       stop
    end if

    if(attributeLength > len(attributeValue)) then
       print *, "Not enough space supplied to read character attribute " // &
                 trim(attributeName)
       stop
    end if

    if(attributeType /= DFNT_CHAR ) then
       print *, "Attribute " // trim(attributeName) // &
                " is not of type character." 
       stop
    end if

    attributeValue = ""
    status    = SFrcatt (fileID, attributeID, attributeValue)

    if(status == FAIL) then
       print *, "Can't read in the attribute string" // trim(attributeName)
       stop
    end if

  end subroutine getCoreMetadata

  ! ----------------------------------------------------------------------------------

  function getDataSetDimSizes(dataSetID, fileName, dataSetName)
    integer (kind = FourByteIntKind), intent( in) :: dataSetID
    character (len = *), optional,    intent( in) :: fileName, dataSetName
    integer (kind = FourByteIntKind), dimension(MAX_VAR_DIMS) :: getDataSetDimSizes
    ! !F90
    !
    ! !Description:
    !   Gets the dimension sizes (number of elements along each dimension) of an HDF SDS. 
    !  
    ! !Input Parameters:
    !   dataSetID: HDF data set ID, as returned from SFselect of getDataSetID. 
    !   fileName, dataSetName: optional character strings with the name of the file and
    !     the name of the data set. Used in error reporting.
    !
    ! !Output Parameters:
    !    getDataSetDimSizes returns a 4 byte integer array of length MAX_VAR_DIMS. 
    !      The first elements contain the number of elements along the corresponding dimension. 
    ! 
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Robert Pincus
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Robert.Pincus@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Named constants
    !      MAX_VAR_DIMS (netcdf.f90)
    !
    ! !END
    
    ! Local variables
    integer (kind = FourByteIntKind)              :: thisNumDims, thisDataSetType, &
                                                     numAttributes
    character(len = MAX_NC_NAME)                  :: thisDataSetName


    status =  SFginfo(dataSetID, thisDataSetName, thisNumDims, &
                      getDataSetDimSizes, thisDataSetType, numAttributes)
    if (status == FAIL) then
      if(present(fileName) .and. present(dataSetName)) then
        print*, "Can't get information from variable " // &
             trim(dataSetName) // " in file " // trim(fileName) 
     else
        print *, "Failed to get needed information from variable."
      end if
      stop
    end if
    getDataSetDimSizes(thisNumDims + 1 : ) = 0
        
  end function getDataSetDimSizes

  ! --------------------------------------------------------------------------------

  subroutine getCalibrationInfo(dataSetId, scaleFactor, addOffset)
    integer (kind = FourByteIntKind),   intent( in) :: dataSetID
    real    (kind = EightByteRealKind), intent(out) :: scaleFactor, addOffset

    ! !F90
    !
    ! !Description:
    !   Reads the scale_factor and add_offset attributes from a Modis HDF SDS. Both
    !     are assumed to be eight byte reals. Error handling is internal (any mistakes
    !     and execution stops). 
    !
    ! !Input Parameters:
    !   dataSetId: HDF data set ID, as returned by SFstart or getDataSetID
    !
    ! !Output Parameters:
    !   scaleFactor, offset: eight byte real values read from scale_factor and add_offset 
    !     data set attributes. 
    ! 
    ! !Revision History:
    !   $Log: Level2FileInfo.f90,v $
    !   Revision 1.3  1997/12/30  16:02:45  pincus
    !   netCDF constants are now taken from the hdf module; removed reference
    !   to module netcdf. Names of the constants changed.
    !
    !   Revision 1.2  1997/11/10  20:14:47  pincus
    !   Removed tabs to ensure ANSI compliance.
    !
    !   Revision 1.1  1997/10/30  17:44:27  pincus
    !   Initial revision
    !
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Robert Pincus
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Robert.Pincus@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   Named constants:
    !     FAIL: (hdf.f90)
    !
    ! !END
  

    ! Read scale factor and offset, which are real valued attributes for all variables.
    status = SFrnatt(dataSetID, SFfattr(dataSetID, "scale_factor"), scaleFactor)

    if (status == FAIL) then
      msgbuffer = "Error reading scale factor."
      stop
    end if

    status = SFrnatt(dataSetID, SFfattr(dataSetID, "add_offset"), addOffset)

    if (status == FAIL) then
      msgbuffer = "Error reading offset" 
      stop
    end if

  end subroutine getCalibrationInfo

  ! --------------------------------------------------------------------------------

  function calibrateFourByteReal(inputArray, offset, scaleFactor) result(outputArray)
    real  (kind = FourByteRealKind),     &
           dimension(:, :), intent( in)      :: inputArray
    real  (kind = EightByteRealKind),    &
                           intent( in)       :: offset, scaleFactor
    real  (kind = FourByteRealKind),     &
          dimension(size(inputArray, 1), &
                        size(inputArray, 2)) :: outputArray
    ! !F90
    !
    ! !Description:
    !   Applys scale factor and offset correction to a real-valued array (as
    !   would be read from an HDF file) to return a correctly calibrated array.
    !  
    ! !Input Parameters:
    !    inputArray: real valued rank two array. Typically this contains real values 
    !      as read from an HDF file. 
    !    offset: offset, as would be read from an HDF SDS attribute. 
    !    scaleFactor: linear scaling factor, as would be read from an HDF SDS attribute.  
    !
    ! !Output Parameters:
    !   The function returns an array of rank two of the same size as inputArray, with 
    !     the scale factor and offset applied. 
    !
    ! !Revision History:
    !   $Log: Level2Read.f90,v $
    !   Revision 1.4  1997/12/30  16:02:36  pincus
    !   netCDF constants are now taken from the hdf module; removed reference
    !   to module netcdf. Names of the constants changed.
    !
    !   Revision 1.3  1997/11/13  15:15:51  pincus
    !   Added support for reading bit field values from a two dimensional SDS.
    !   (Previously, all bit field SDSs were supposed to have three dimensions.)
    !
    !   Revision 1.2  1997/11/10  20:33:22  pincus
    !   Removed tabs for ANSI compliance.
    !
    !   Revision 1.1  1997/10/30  17:46:07  pincus
    !   Initial revision
    !
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Robert Pincus
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Robert.Pincus@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   Note that this definition of how to apply the calibration information 
    !     is consistent with the HDF convention. 
    !  Both offset and scaleFactor are optional arguments. 
    !   
    !
    ! !END
    outputArray(:, :) = inputArray(:, :)
    if (abs(offset) > tiny(offset)) &
      outputArray = outputArray - offset
    if (abs(scaleFactor - 1.0) > tiny(scaleFactor)) &
      outputArray = outputArray * scaleFactor
  end function calibrateFourByteReal

  ! --------------------------------------------------------------------------------

  function calibrateEightByteReal(inputArray, offset, scaleFactor) result(outputArray)
    real  (kind = EightByteRealKind),       &
           dimension(:, :), intent( in)      :: inputArray
    real  (kind = EightByteRealKind),       &
                                 intent( in) :: offset, scaleFactor
    real  (kind = EightByteRealKind),       &
           dimension(size(inputArray, 1),   &
                        size(inputArray, 2)) :: outputArray
    ! !F90
    !
    ! !Description:
    !   Applys scale factor and offset correction to a real-valued array (as
    !   would be read from an HDF file) to return a correctly calibrated array.
    !  
    ! !Input Parameters:
    !    inputArray: real valued rank two array. Typically this contains real values 
    !      as read from an HDF file. 
    !    offset: offset, as would be read from an HDF SDS attribute. 
    !    scaleFactor: linear scaling factor, as would be read from an HDF SDS attribute.  
    !
    ! !Output Parameters:
    !   The function returns an array of rank two of the same size as inputArray, with 
    !     the scale factor and offset applied. 
    !
    ! !Revision History:
    !   $Log: Level2Read.f90,v $
    !   Revision 1.4  1997/12/30  16:02:36  pincus
    !   netCDF constants are now taken from the hdf module; removed reference
    !   to module netcdf. Names of the constants changed.
    !
    !   Revision 1.3  1997/11/13  15:15:51  pincus
    !   Added support for reading bit field values from a two dimensional SDS.
    !   (Previously, all bit field SDSs were supposed to have three dimensions.)
    !
    !   Revision 1.2  1997/11/10  20:33:22  pincus
    !   Removed tabs for ANSI compliance.
    !
    !   Revision 1.1  1997/10/30  17:46:07  pincus
    !   Initial revision
    !
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Robert Pincus
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Robert.Pincus@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   Note that this definition of how to apply the calibration information 
    !     is consistent with the HDF convention. 
    !  Both offset and scaleFactor are optional arguments. 
    !   
    !
    ! !END
    outputArray(:, :) = inputArray(:, :)
    if (abs(offset) > tiny(offset)) &
      outputArray = outputArray - offset
    if (abs(scaleFactor - 1.0) > tiny(scaleFactor)) &
      outputArray = outputArray * scaleFactor
  end function calibrateEightByteReal

  ! ----------------------------------------------------------------------------------

  function getDataSetType(dataSetID, fileName, dataSetName)
    integer (kind = FourByteIntKind), intent( in) :: dataSetID
    character (len = *), optional,    intent( in) :: fileName, dataSetName
    integer (kind = FourByteIntKind)              :: getDataSetType
    ! !F90
    !
    ! !Description:
    !   Returns the HDF data set type (i.e. DFNT_FLOAT32) of a given HDF data set. This function
    !     is an error checking wrapper for SFginfo. 
    !  
    ! !Input Parameters:
    !   dataSetID: HDF data set ID, as returned from SFselect of getDataSetID. 
    !   fileName, dataSetName: optional character strings with the name of the file and
    !     the name of the data set. Used in error reporting.
    !
    ! !Output Parameters:
    !    getDataSetType returns a scalar 4 byte integer with the data set type. The definitions 
    !      of the types are in the hdf module. 
    ! 
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Robert Pincus
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Robert.Pincus@gsfc.nasa.gov
    !
    ! !Design Notes:
    !    Named constants
    !      MAX_NC_NAME, MAX_VAR_DIMS (hdf.f90)
    !
    ! !END
    
    ! Local variables
    integer (kind = FourByteIntKind)              :: thisNumDims, numAttributes
    character(len = MAX_NC_NAME)                  :: thisDataSetName
    integer (kind = FourByteIntKind), &
                              dimension(MAX_VAR_DIMS) :: thisDimensionSizes

    status =  SFginfo(dataSetID, thisDataSetName, thisNumDims, &
                      thisDimensionSizes, getDataSetType, numAttributes)
    if (status == FAIL) then
       if(present(fileName) .and. present(dataSetName)) then
          msgbuffer = "Failed to get needed information from variable " // &
               trim(dataSetName) // " in file " // trim(fileName) 
       else
          msgbuffer = "Failed to get needed information from variable."
       end if
       stop
    end if
        
  end function getDataSetType

  ! ----------------------------------------------------------------------------------

  subroutine getOBIKFV(origID, fillValue)

    integer   (kind = FourByteIntKind),   intent( in)  :: origID
    integer   (kind =  OneByteIntKind),   intent( in)  :: fillValue
    !
    ! !F90
    !
    ! !Description:
    !   Reads in the fill value for a specified SDS.
    !   Generic interface is based upon the data type of the fillValue.
    !  
    ! !Input Parameters:
    !   origID: Data handle for accessing a SDS, used by HDF routines and 
    !            passed as argument in subroutines. 
    !
    ! !Output Parameters:
    !   fillValue:  Contains the fill value.
    !
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END
    integer   (kind = FourByteIntKind)                 :: attrID
    character (len = 100)                              :: attrName

    attrName = "_FillValue"

    attrID = SFfattr(origID, trim(attrName))
    if (attrID == FAIL) then
      msgbuffer = "Error obtaining attribute ID for one byte integer fill value"
      stop
    end if

    !byte
    status = sfrnatt(origID, attrID, fillValue)
    if (status == FAIL) then
      msgbuffer = "Error obtaining one byte integer fill value"
      stop
    end if

  end subroutine getOBIKFV

  ! ----------------------------------------------------------------------------------

  subroutine getTBIKFV(origID, fillValue)

    integer   (kind = FourByteIntKind),   intent( in)  :: origID
    integer   (kind =  TwoByteIntKind),   intent( in)  :: fillValue
    !
    ! !F90
    !
    ! !Description:
    !   Reads in the fill value for a specified SDS.
    !   Generic interface is based upon the data type of the fillValue.
    !  
    ! !Input Parameters:
    !   origID: Data handle for accessing a SDS, used by HDF routines and 
    !            passed as argument in subroutines. 
    !
    ! !Output Parameters:
    !   fillValue:  Contains the fill value.
    !
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END
    integer   (kind = FourByteIntKind)                 :: attrID
    character (len = 100)                              :: attrName

    attrName = "_FillValue"

    attrID = SFfattr(origID, trim(attrName))
    if (attrID == FAIL) then
      msgbuffer = "Error obtaining attribute ID for two byte integer fill value"
      stop
    end if

    !short
    status = sfrnatt(origID, attrID, FillValue)
    if (status == FAIL) then
      msgbuffer = "Error obtaining two byte integer fill value"
      stop
    end if

  end subroutine getTBIKFV

  ! ----------------------------------------------------------------------------------

  subroutine getFBIKFV(origID, fillValue)

    integer   (kind = FourByteIntKind),   intent( in)  :: origID
    integer   (kind = FourByteIntKind),   intent( in)  :: fillValue
    !
    ! !F90
    !
    ! !Description:
    !   Reads in the fill value for a specified SDS.
    !   Generic interface is based upon the data type of the fillValue.
    !  
    ! !Input Parameters:
    !   origID: Data handle for accessing a SDS, used by HDF routines and 
    !            passed as argument in subroutines. 
    !
    ! !Output Parameters:
    !   fillValue:  Contains the fill value.
    !
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END
    integer   (kind = FourByteIntKind)                 :: attrID
    character (len = 100)                              :: attrName

    attrName = "_FillValue"

    attrID = SFfattr(origID, trim(attrName))
    if (attrID == FAIL) then
      msgbuffer = "Error obtaining attribute ID for four byte integer fill value"
      stop
    end if

    !long
    status = sfrnatt(origID, attrID, FillValue)
    if (status == FAIL) then
      msgbuffer = "Error obtaining four byte integer fill value"
      stop
    end if

  end subroutine getFBIKFV

  ! ----------------------------------------------------------------------------------

  subroutine getFBRKFV(origID, fillValue)

    integer   (kind = FourByteIntKind),   intent( in)  :: origID
    real      (kind = FourByteRealKind),  intent( in)  :: fillValue
    !
    ! !F90
    !
    ! !Description:
    !   Reads in the fill value for a specified SDS.
    !   Generic interface is based upon the data type of the fillValue.
    !  
    ! !Input Parameters:
    !   origID: Data handle for accessing a SDS, used by HDF routines and 
    !            passed as argument in subroutines. 
    !
    ! !Output Parameters:
    !   fillValue:  Contains the fill value.
    !
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    integer   (kind = FourByteIntKind)                 :: attrID
    character (len = 100)                              :: attrName

    attrName = "_FillValue"

    attrID = SFfattr(origID, trim(attrName))
    if (attrID == FAIL) then
      msgbuffer = "Error obtaining attribute ID for four byte real fill value"
      stop
    end if

    !float
    status = sfrnatt(origID, attrID, fillValue)
    if (status == FAIL) then
      msgbuffer = "Error obtaining four byte real fill value"
      stop
    end if

  end subroutine getFBRKFV

  ! ----------------------------------------------------------------------------------

  subroutine getEBRKFV(origID, fillValue)

    integer   (kind = FourByteIntKind),   intent( in)  :: origID
    real      (kind = EightByteRealKind), intent( in)  :: fillValue
    !
    ! !F90
    !
    ! !Description:
    !   Reads in the fill value for a specified SDS.
    !   Generic interface is based upon the data type of the fillValue.
    !  
    ! !Input Parameters:
    !   origID: Data handle for accessing a SDS, used by HDF routines and 
    !            passed as argument in subroutines. 
    !
    ! !Output Parameters:
    !   fillValue:  Contains the fill value.
    !
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    integer   (kind = FourByteIntKind)                 :: attrID
    character (len = 100)                              :: attrName

    attrName = "_FillValue"

    attrID = SFfattr(origID, trim(attrName))
    if (attrID == FAIL) then
      msgbuffer = "Error obtaining attribute ID for eight byte real fill value"
      stop
    end if

    !double
    status = sfrnatt(origID, attrID, fillValue)
    if (status == FAIL) then
      msgbuffer = "Error obtaining eight byte real fill value"
      stop
    end if

  end subroutine getEBRKFV

  ! ----------------------------------------------------------------------------------

end module dataUtils
