#!/usr/local/bin/perl -w

#References and Credits:
#     Written by
#      Eric Moody
#      Climate and Radiation Branch, Code 913
#      NASA/GSFC
#      Greenbelt MD 20771
#      Eric.Moody@gsfc.nasa.gov

#Version 1.0

#PERL Script for the Single Resolution Subsetting program.
#  This script is used to define input parameters, paths, and filenames.
#    It also reads in parameters and data from 2 .txt files, in order to
#    define multiple points to be subsetted, and multiple directories that
#    contain the L2 input granules.

#  The script provides for the following capabilities:
#    Multiple directories for L2 granules can be specified.
#    The same input granules can be subsetted for 
#      multiple locations(lat/lon/epsilon point).
#    Location name included in the name of the output subsetted granule.
#    Include only the specified MOD in the list of input granules to
#      be considered.
#    An SDS can be spatially subsetted or the entire SDS can be copied
#      to the new HDF file by setting the flag named 'SpatialSubset".
#      Setting this flag to true means that you would like to perform
#      the spatial subsetting, false will dump the entire SDS.

#  The script is set up as follows:
#    The first section allows you to define some run-time paramers (such
#     as pring flag, minimum subsetted box dimension, and the MOD being
#     examined).
#    The second section allows you to specify the code home directory, 
#     output and input paths.
#    The third section allows you to define the base name for the
#     subsetted output file name.
#    The fourth section allows you to define input file names used in
#     this PERL script.
#    The fifth section allows you to define input file names used as 
#     input to the F90 code.

#  Note that the first FIVE parts of this script can be altered to define run
#   parameters.  Sections SIX and beyond contain code to create the list of
#   L2 granules to be processed, creation of the F90Inputs.dat file,
#   and runs the F90 code for the multiple locations. 
#   It is highly recommended that you do not make changes in 
#   these (SIX and beyond) sections.

#Design Notes:
#   The PERL script generates a file that contains all of the L2 granules to
#     be considered for subsetting.  Multiple locations can be specified with
#     an input text file that contains path names for the various directories.
#     The script loops through the paths and adds the names of the specified
#     granule type (MODXY) to a text file.  This text file is read in by the
#     F90 code and the granules listed will be examined for subsetting.
#   Only one list of L2 input granules is produced per running of the PERL
#     script.  This means that all specified locations will use the same list
#      of input L2 granule files.
#   The PERL script generates a file that contains a list of input parameters
#     and file/path names use by the F90 code.  This list is generated from 
#     variables specified below and from the Locations text file.
#   Multiple locations for subsetting are handled by reading in the locations
#     from the Locations text file.   This file can be altered to specify a 
#     location name, Lat, Lon, and Epsilon.  The script first reads in the
#     number of locations, then performs a set of commands for each location.
#     For each iteration of the loop, the script reads in the lat/lon/epsilon 
#     and location name.  Then the script writes out the parameters and 
#     filenames/paths to the F90Inputs.dat file.  The script then runs the
#     F90 code.

#***********************************************************************
#1) First set up some of the F90 Program Inputs:
#***********************************************************************

#Print Flag, Toggles printing the name of the location being processed:
$PrintFlagLocName = ".true.";

#Print Flag, Toggles printing the name of the granule being processed:
$PrintFlagGranName = ".true.";

#Spatial subsetting flag.  Setting this to true will perform the spatial
#   subsetting, false will dump the entire SDS to the new HDF file.
$SpatialSubsetting = ".true.";

#MinBoxDim, Minimum useable box dimension.  Ex. If the smallest dimension
#   of a useable box is 4, then set MinBoxDim to 4.
$MinBoxDim = 1;

#The MOD that is being examined.
#  The form must be MODXY, ex. MOD05 or MOD35
$MODForm = "MOD";


#***********************************************************************
#2) Now specify the code home directory, output and input paths:
#***********************************************************************

#code home directory:
#  Note that this path DOES NOT end with a /
#$homeDir = "/atmos/moody/Coordinate/SnglResSubsetter";
#This will issue the pwd command to obtain the current directory:
$homeDir = `pwd`;
#remove the trailing whitespace:
$homeDir =~ s/^(\S*)\s.*/$1/;

#code output directories:
#  Note that this path DOES NOT end with a /
$outputDir = "$homeDir/output";

#Path of the input files:
#  Note that this path DOES NOT end with a /
#  Note:  It is highly recommended that this name and path are not
#         altered, as they are hard-wired into the F90 code. 
#         If you do change this name or path, you must alter the
#         file "Inputs.f90" to reflect this change.
$inputDir = "$homeDir/inputs";


#***********************************************************************
#3) Define base name for the subsetted output file names:
#***********************************************************************
#The base for the output file that contains
#   the list of subsetted L2 granules.  The
#   output path and Location prefix will 
#   be placed in front of this base.
#   ex. for Alexandria Egypt:
#       ALEXEGTSubGrans.txt
$BaseSubGranList = "SubGrans.txt";


#***********************************************************************
#4) Define input file names to this PERL Script:
#***********************************************************************
#Note:  These are the files that you alter for runs.

#Path and name of the file that contains the list of 
#   granule directories containing the granules to be analyzed:
$granPathList = "$inputDir/GranPathList.txt";

#Path and name of the file that contains the list of
#   locations to be analyzed:
$LocationList = "$inputDir/LocationList.txt";


#***********************************************************************
#5) Define input file names to the F90 code:
#***********************************************************************
#Note:  These are the files that are created by the PERL script
#       for input into the F90 code.

#Path and name of the file that contains
#   the list of L2 granules to be processed:
$granList = "$inputDir/GranuleFlnms.dat";

#Path and name of the file that contains
#   the input list of SDS names to be subsetted:
$SDSList = "$inputDir/SDSList.txt";

#Path and name of the file that contains the list of
#  inputs to the F90 program:
#  Note:  It is highly recommended that this name and path are not
#         altered, as they are hard-wired into the F90 code. 
#         If you do change this name or path, you must alter the
#         file "Inputs.f90" to reflect this change.
$progInputsFile = "$inputDir/F90Inputs.dat";





#***********************************************************************
#End of the Definition section.  It is highly recommended that no
#  changes be made to the remaining part of the script.
#***********************************************************************
print "\nThis is version 1.0 of the Single Resolution Subsetter.\n\n";

#***********************************************************************
#6) Create the list of L2 Granules to process:
#***********************************************************************
#Open the input granule path list:
open(INPUT, "<$granPathList");

#Read the number of paths:
$numGranPaths = <INPUT>;

#Open the granule list:
open(OUTPUT,">$granList");

#Loop over the number of granule paths:
for ( $icnt = 0; $icnt < $numGranPaths; $icnt++) {
  #read in the inputpath:
  $inputpath = <INPUT>;
  #remove the trailing whitespace:
  $inputpath =~ s/^(\S*)\s.*/$1/;
  &OutputFiles($MODForm,$inputpath);
}

#Add a line with '' to specify the end of the list
print OUTPUT "''\n";

#Close the granule list:
close OUTPUT;

#Close the input granule path list:
close INPUT;

#***********************************************************************
#***********************************************************************
# 7) Loop over the number of Locations, creating the program inputs file
#      and running the F90 program for each location:
#***********************************************************************
#***********************************************************************
#Open the location file:
open(INPUT, "<$LocationList");

#Read in the number of locations:
$numLocations = <INPUT>;

#Loop over the number of locations:
for ( $icnt = 0; $icnt < $numLocations; $icnt++) {
  
  #***********************************************************************
  #Create the program inputs file and write the data:
  #***********************************************************************
  
  #Name to be associated with this current Lat/Lon/Epsilon point:
  #    This name will be used in the output HDF name:
  $LocationName = <INPUT>;
  $LocationName =~ s/^(\S*)\s.*/$1/;

  #Desired Latitude/Longitude, i.e. the geolocated point of interest:
  $desiredLat = <INPUT>;
  $desiredLat =~ s/^(\S*)\s.*/$1/;
  $desiredLon = <INPUT>;
  $desiredLon =~ s/^(\S*)\s.*/$1/;
  #Epsilon, the range around the geolocation point, in degrees:
  $epsilonLat = <INPUT>;
  $epsilonLat =~ s/^(\S*)\s.*/$1/;
  $epsilonLon = <INPUT>;
  $epsilonLon =~ s/^(\S*)\s.*/$1/;

  #Path and name of the file that contains
  #   the list of subsetted L2 granules:
  $subsettedGranList = "$outputDir/$LocationName$BaseSubGranList";


  #Write the input parameters to file:
  open(OUTPUT,">$progInputsFile");
  print OUTPUT "$desiredLat\n";
  print OUTPUT "$desiredLon\n";
  print OUTPUT "$epsilonLat\n";
  print OUTPUT "$epsilonLon\n";
  print OUTPUT "$LocationName\n";
  print OUTPUT "$PrintFlagGranName\n";
  print OUTPUT "$PrintFlagLocName\n";
  print OUTPUT "$SpatialSubsetting\n";
  print OUTPUT "$MinBoxDim\n";
  print OUTPUT "'$outputDir'\n";
  print OUTPUT "'$granList'\n";
  print OUTPUT "'$subsettedGranList'\n";
  print OUTPUT "'$SDSList'\n";
  close OUTPUT;
  
  
  #***********************************************************************
  #Run the code:
  #***********************************************************************
  chdir($homeDir);
  system "*.exe";
  
}

close(INPUT);


#***********************************************************************
#The OutputFiles Subroutine:
#***********************************************************************
sub OutputFiles
  {
    local($inform,$inpath) = @_;
    chdir($inpath);
    $currentDir = `pwd`;
    $currentDir =~ s/^(\S*)\s.*/$1/;
    $currentDir = "$currentDir/";
    #Only perform if the directory exists(changed to)
    if ($currentDir eq $inpath ) {
      foreach my $filename (<$inform*hdf>) {
	print OUTPUT "'$inpath$filename'\n";
      }  
    }
    return;
  }
