program SnglResSubsetter
  !*****************************************************************************
  ! !F90
  !
  ! !Description:
  !    This program will loop through a list of granules, and determine, by using the
  !     bounding coordinates in the CoreMetadata.0 and the geolocation arrays in the
  !     granules, which granules are located within a given area around a specified 
  !     latitude and longitude.  For these useful granules, the program reads in 
  !     specified Scientific Data Sets (SDS), subsets the SDS for the desired area,
  !     and then outputs them to a new HDF file.  The Geolocation is also copied to
  !     the new HDF file (at the 10km resolution) and the new HDF file contains copies
  !     of all the global attributes, SDS attributes, and SDS dimensions and names.
  !    The program has the capacity to perform a spatial subsetting defined by the
  !     latitude/longitude point and surrounding box, or a flag can be set, in which
  !     a SDS's entire data will be copied to the new HDF.  The flag SpatialSubsetting
  !     can be set in the PERL script. Setting the flag to true performs the spatial 
  !     subsetting, setting to flase will dump the entire SDS to the new HDF file.
  !    The program allows for all of the SDS or a select list of SDSs to be subsetted.
  !     If the first line of the Location List file is "All", then all of the SDSs will
  !     be subsetted.  Otherwise, the user can create a list of SDSs to subset in this
  !     Location List file.
  !    The subsetting is done in two parts.  First, the bounding box, and a mask are
  !     defined using geolocation.  Then, for an SDS, the data within this bounding box
  !     is copied into a temporary array and the mask is applied to this subsetted data.
  !    The bounding box and mask are determined from the geolocation data.  First, the
  !     four corner points are found that define the box.  These four points are
  !     determined to be the closest to the desired box corners.  Then, a mask is
  !     generated to filter out points that are within this box, but are not within the 
  !     desired geolocated box. 
  !  
  ! !Inputs:
  !    Various input parameters are defined in the PERL script that runs this program.
  !     These parameters are written to a text file(by the PERL script), and read into
  !     this program in the Inputs.f90 module(via a call to readInputs).  The filename
  !     and path of this input file is hardwired into the code as "./inputs/inputs.txt".
  !     If this path or name is altered in the PERL script, then these changes must be
  !     mirrored in the Inputs.f90 module by altering the variable "ProgInputFileName".
  !     Some of the input parameters include:
  !      Desired Geolocation point and box (in degrees) around that point.
  !      Minimum dimension of a useful box.
  !      Number and names of SDS to be read in/ouputted.
  !      Print flag for printing out the name of the granule being examined.
  !      File names and paths for input granules, granule list, output paths, etc.
  !    The granules to be examined are defined by a input file.  The PERL script is 
  !     directed to read the names of all the granules in a specified directory(i.e.
  !     for a certain day) and then write those granule names to a text file.  This
  !     text file is read in by this program in order to test all of the granules in
  !     the directory.
  !
  ! !Outputs:
  !    The useful granules have the specified SDS and geolocation outputted to an HDF.
  !    The names of the subsetted granules is stored in a text file, for referencing
  !        as a packing list.
  ! 
  ! !Revision History:
  ! $Log: SnglResSubsetter.f90,v $
  !
  ! Revision 1.0  2000/04/21  12:12:12  EGMoody
  ! Initial Public Release of code.
  !
  ! Revision 0.3  2000/12/17  12:12:12  EGMoody
  ! Corrected how the granule name was seperated from the full path.
  !
  ! Revision 0.2  1999/12/03  12:12:12  EGMoody
  ! Modified the granule names input file to include path name.  This was
  !   included so that a user can specify multiple directories for
  !   input granules.  Also had to adjust for this when reading in the 
  !   granule name and seperating off the path name. This made the 
  !   variable MOD_L2_in_path obsolite.
  ! The names of the SDSs to be outputted is now specified in a seperate file
  !   instead of being included in the PERL script.  This file is directly
  !   read in by this F90 program instead of from the PERL input file.
  ! The input granule list file was changed such that it no longer includes
  !   the number of granules to be examined.  The counting is done in this code
  !   by reading in all of the granule names until there is a line with "".
  !   At this point the file is rewound and the looping over these files begins.
  ! Included a second print flag for a location name associated with
  !   the inputted Lat/Lon point.  
  ! This location name is also added to the front of the output HDF 
  !   file and the granule list file.
  !
  ! Revision 0.1  1999/10/22  12:43:43  EGMoody
  ! Initial revision.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   This specific code had been tailored specifically for subsetting granules that
  !    have SDS at the same resolution (L1B, MOD05,MOD06, MOD35).  If you desire to
  !    examine other MODs, please contact Eric Moody for a different version.
  !   This program can handle SDS of rank 1, 2, and 3.  For higher ranked arrays,
  !    alterations must be made.
  !   This program handles SDS that are not related to the geolocation by copying them
  !    directly to the new HDF file.
  !   This code can only handle two forms of Multibanded arrays of rank 3.  There is
  !    logic to handle arrays arranged as (Along, Across, Band), or (Band, Along, Across).
  !    If the SDS is (..., Band, ...), then subsetting will not be performed.
  !   If, upon determination that a granule is useful, it is found
  !    that the subsetted box is too small, then the granule is not used.  The smallest
  !    useful subsetted box dimension is defined in the PERL script.
  !   A file is generated that contains a listing of the new HDF files that were created.
  !   Many of the hdf routines and module designs were inspired or modified from codes
  !    developed by Robert Pincus (University of Wisconsin and former GSFC programmer).
  !
  ! !END
  !*****************************************************************************

  !Modules directly copied and altered from codes developed by Robert Pincus:
  use hdf,             only  : MAX_VAR_DIMS, SFfinfo, SFselect, SFGinfo, FAIL
  use hdfUtils,        only  : getFileID, getDataSetID, closeFile, endDataSet
  use typeSizes,       only  : FourByteIntKind,                                 &
                               FourByteRealKind, EightByteRealKind,             &
                               byteSizesOK

  !Modules created/altered for this code:
  use Inputs,          only  : MaxFileNameLength, MaxAttributeLength,           &
                               desiredLon, desiredLat, EpsilonLat, EpsilonLon,  &
                               LocationName,                                    &
                               numberOfSDS, SDS,                                &
                               PrintFlagGranName, PrintFlagLocName,             &
                               SpatialSubsetting,                               &
                               MinBoxDim,                                       &
                               all_L2_in_names_file,                            &
                               CoordName,                                       &
                               readInputs
  use dataUtils,       only  : getDatasetDimSizes,                              &
                               getCoreMetadata,                                 &
                               getDataSetType
  use SDSReader,       only  : ReadSDS
  use parse,           only  : parser
  use output,          only  : initializeGranuleFile, outputGranuleName,        &
                               initializeNewHDF, createSDS, writeSDS
  use subsettingUtils, only  : findBoxCorners, SubsetCheck,                     &
                               copyToBox, createMask, applyMask
  use subsetting,      only  : subsetDecisionTree

  implicit none


  !*****************************************************************************
  !Variables and parameters:
  !*****************************************************************************

  !Generic counters
  integer  (kind = FourByteIntKind)             :: i, j, k, l


  !Variables used for filehandling
  integer  (kind = FourByteIntKind)             :: inputFileID
  integer  (kind = FourByteIntKind)             :: newHDFID 
  integer  (kind = FourByteIntKind)             :: status
  character(len  = MaxFileNameLength)           :: fileName
  character (len = maxattributelength)          :: newHDFfilename
  character(len  = MaxFileNameLength)           :: fileNameNoPath
  integer  (kind = fourByteIntKind)             :: lengthOfString
  integer  (kind = fourByteIntKind)             :: StringPosition
  character(len  = MaxFileNameLength)           :: Gen_File
  integer  (kind = FourByteIntKind)             :: FileLUN
  integer  (kind = FourByteIntKind)             :: ierr
  integer  (kind = FourByteIntKind)             :: FileCounter
  integer  (kind = FourByteIntKind)             :: NumGranFiles
  ! inputFileID     : File handle for an input  HDF file, used by HDF routines.
  ! newHDFID        : File handle for an output HDF file, used by HDF routines.
  ! status          : Used for error checking.
  ! fileNameNoPath,
  !  fileName,
  !  newHDFfilename,
  !  Gen_File       : Used for storing File and Path names.
  ! lengthOfString  : Used to store the length of a character string.
  ! StringPosition  : Used to store a position in a string.
  ! FileLUN         : The file handle used in OPEN, READ, CLOSE F90 statements.
  ! ierr            : Contains value for IOSTAT. 0=Success, otherwise=Fail
  ! FileCounter     : Integer used as a counter to loop around the granules.
  ! NumGranFiles    : The number of granules for the specified PGE.


  !Variables used for parsing the CoreMetadata:
  character(len  = MaxAttributeLength)          :: CoreString
  real     (kind = EightByteRealKind),  &
            dimension(1:4)                      :: CoordVal
  ! CoreString      : String where the CoreMetadata.0 is stored.
  ! CoordVal        : Array containing the values of the Bounding Coordinates.


  !Variables used for Subsetting the SDS and Latitude and Longitude arrays:
  real     (kind = FourByteRealKind),   &
            allocatable, dimension(:,:)         :: Lats, LatsBox,     &
                                                   Lons, LonsBox
  integer  (kind = FourByteIntKind)             :: latID, lonID
  logical                                       :: potentialGranule
  logical                                       :: useableGranule
  integer  (kind = FourByteIntKind),    &
            dimension(1:4,1:2)                  :: BoxCorners
  integer  (kind = FourByteIntKind),    &
            dimension(MAX_VAR_DIMS)             :: boxDimensionSizes, GeoDimSizes
  integer  (kind = FourByteIntKind),    &
            allocatable, dimension(:,:)         :: Mask
  real     (kind =  FourByteRealKind)           :: dummyFloat  = 0.0
  ! Lats, Lons        : Arrays that store the entire set of geolocation values.
  ! LatsBox, LonsBox  : Arrays that store the Subsetted Box of geolocation values.
  ! latID, lonID      : SDS data handles used by HDF routines, for the Geolocation.
  ! potentialGranule  : Stores result of a granule's bounding coordinates test.
  ! useableGranule    : Stores result of a granule's Lat/Lon test.
  ! BoxCorners        : Contains the indices of the subsetted box.
  !     dimension (i,j):
  !     i=1 : Upper Left            j=1: i index of box
  !     i=2 : Upper Right           j=2: j index of box
  !     i=3 : Lower Left 
  !     i=4 : Lower Right
  ! boxDimensionSizes : The dimensions of the subsetted box.
  ! GeoDimSizes       : The dimensions of the geolocation arrays.
  ! Mask              : Array containing the mask.  The mask is used to replace unwanted 
  !                     points in the subsetted box, with fill values.  This is accomplished
  !                     by setting the mask to 1 for desired points and 0 for unwanted
  !                     points.  Then the mask is applied to the subsetted SDS, in which
  !                     when the mask = 0, the fill value replaces the stored value.
  ! dummyFloat        : Dummy scalars used as sample data types in calls to subsetSDS.

  !Variables used for reading in and outputting desired SDS:
  character(len  = MaxFileNameLength)           :: SDSName  
  integer  (kind = FourByteIntKind)             :: SDSID
  integer  (kind = FourByteIntKind)             :: TotNumOfSDS
  logical                                       :: SubsetAllSDS, SpatialSubset,  &
                                                   CorrectOrder, OutputSDS
  integer  (kind = FourByteIntKind)             :: BegSDSLoop, EndSDSLoop
  integer  (kind = FourByteIntKind),    &
            dimension(MAX_VAR_DIMS)             :: dimensionSizes
  integer  (kind = FourByteIntKind)             :: DataType
  integer  (kind = FourByteIntKind)             :: rank
  integer  (kind = FourByteIntKind)             :: nattrs
  ! SDSName         : Stores the name of the SDS being used, used by HDF routines 
  !                    and passed as argument in subroutines.
  ! SDSID           : Data handle for accessing a SDS, used by HDF routines and 
  !                    passed as argument in subroutines.
  ! TotNumOfSDS     : Total number of SDSs in the HDF file.
  ! SubsetAllSDS    : Flag for seeing if the user wishes all SDSs to be subsetted.
  ! SpatialSubset   : Flag for seeing if the SDS is subsettable or not.
  ! CorrectOrder    : Flag to see if the SDS dimensions are in the correct order.
  !                     2D should be along, across
  !                     3D should be band, along, across
  !                     if 2D and flase, then order is across, along
  !                     if 3D and flase, then order is along, across, band
  ! Beg, EndSDSLoop : Beginning and ending counters for the SDS loop.
  ! dimensionSizes  : Used to store the dimension of an SDS.
  ! DataType        : Stores the integer value of the data type.
  !                    ( hdf include file lists the types as:  
  !                      Double = DFNT_FLOAT64 =  6
  !                       Float = DFNT_FLOAT32 =  5
  !                       Short = DFNT_INT16   = 22
  !                        Byte = DFNT_INT8    = 20 )
  ! rank            : The rank of the SDS in the HDF granule.
  ! nattrs          : Number of attributes.


  !*****************************************************************************
  !*****************************************************************************
  ! Begin Processing:
  !*****************************************************************************
  !*****************************************************************************


  !*****************************************************************************
  !Check to ensure type sizes are OK for this system
  !*****************************************************************************
  if ( .not. byteSizesOK() ) then
     print *; print *, "The type sizes are not correct"; print *
     stop
  end if


  !*****************************************************************************
  !Define the desired SDS and file paths:
  !*****************************************************************************
  call readInputs

  !*****************************************************************************
  !Initialize the useful granule name output file:
  !*****************************************************************************
  call initializeGranuleFile
  
  !*****************************************************************************
  !Open the list of L2 granules:
  !*****************************************************************************
  !define the F90 file handle:
  FileLUN = 1020

  !open the file containing the L2 granule names:
  open(FileLUN, file=trim(all_L2_in_names_file), form='formatted',              &
       status='old', action='read', iostat = ierr)

  !Check for an error opening the file:
  if (ierr /= 0) then
     Print *,"Could not open", trim(all_L2_in_names_file)
     stop
  end if

  !Determine the number of granules to be examined by reading all of them in:
  NumGranFiles = 0
  fileName = "1"
  do while (fileName /= "")
     fileName = ""
     read(FileLUN,*) fileName
     NumGranFiles = NumGranFiles + 1
  end do
  NumGranFiles = NumGranFiles - 1
  
  !rewind the file:
  rewind(FileLUN)

  !*****************************************************************************
  !Loop through the granules:
  !*****************************************************************************
  !Print the name of the granule being processed, if desired:
  if (PrintFlagLocName) then
     print *,trim(LocationName)
  end if

  fileloop : do FileCounter = 1, NumGranFiles

     !*****************************************************************************
     !Open the granule and obtain the "CoreMetadata.0" string
     !*****************************************************************************

     !Read in the next granule filename with path:
     fileName = ""
     read(FileLUN,*) fileName

     !Save just the name for output purposes:
     fileNameNoPath = ""
     lengthOfString = len(trim(fileName))
     !Determine where the name of the file begins:
     stringPosition = 1
     do i = 1, lengthOfString - 1
        if (fileName(i:i) == '/') then
           stringPosition = i
        end if
     end do
     !Store just the name:
     fileNameNoPath = fileName(stringPosition+1:lengthOfString)
     
     !Print the name of the granule being processed, if desired:
     if (PrintFlagGranName) then
        print *,trim(fileName)
     end if

     !Open the granule for reading:
     inputFileID = getFileID(trim(fileName))

     !Read in the CoreMetadata.0 character string:
     CoreString = ""
     call getCoreMetadata(inputFileID, "CoreMetadata.0", CoreString)


     !*****************************************************************************
     !Begin Parsing CoreMetadata.0 for bounding coordinates and 
     !Check bounding coordinates to see if this is a potential granule
     !*****************************************************************************

     !reset error and test flags, respectively:
     CoordVal     = -50000.0
     potentialGranule = .false.

     !Read and test east bounding coordinate:
     call parser( string = CoreString,        &
                  name   = CoordName(1),      &
                  value  = CoordVal (1) )   
     east: if ( CoordVal(1) >= desiredLon - EpsilonLon ) then

        !Read and test west bouding coordinate:
        call parser( string = CoreString,        &
                     name   = CoordName(2),      &
                     value  = CoordVal (2) )       
        west: if ( CoordVal(2) <= desiredLon + EpsilonLat ) then

           !Read and test north bounding coordinate:
           call parser( string = CoreString,        &
                        name   = CoordName(3),      &
                        value  = CoordVal (3) )
           north: if ( CoordVal(3) >= desiredLat - EpsilonLat ) then

              !Read and test south bounding coordinate:
              call parser( string = CoreString,        &
                           name   = CoordName(4),      &
                           value  = CoordVal (4) )
              south: if ( CoordVal(4) <= desiredLat + EpsilonLat ) then
                 !This is a potential granule:
                 potentialGranule = .true.
              end if south

           end if north

        end if west

     end if east
  

     !*****************************************************************************
     !If this is a potential granule, read in the Latitude and Longitude data sets:
     !*****************************************************************************
     potentGranule: if ( potentialGranule ) then

        !Obtain the data set ID for Latitude and Longitude:
        latID = getDataSetID(fileID = inputfileID, dataSetName = "Latitude")
        lonID = getDataSetID(fileId = inputfileID, dataSetName = "Longitude")
        
        !Obtain the dimension of the SDS and allocate the Lat/Lon arrays:
        dimensionSizes = getDataSetDimSizes(dataSetId = latID)
        GeoDimSizes = dimensionSizes
        allocate(Lats (dimensionSizes(1), dimensionSizes(2)), stat = status)
        allocate(Lons (dimensionSizes(1), dimensionSizes(2)), stat = status)
        
        !Read in the Lats and Lons arrays:
        call ReadSDS( SampleDataType = dummyFloat,     &  
                           dataSetID = latID,          &
                            fileName = trim(fileName), &
                         dataSetName = "Latitude",     &
                            dimSizes = dimensionSizes, &
                           SDSDataR2 = Lats            )
        call ReadSDS( SampleDataType = DummyFloat,     &
                           dataSetID = lonID,          &
                            fileName = trim(fileName), &
                         dataSetName = "Longitude",    &
                            dimSizes = dimensionSizes, &
                           SDSDataR2 = Lons            )
        
        !!End access to the Lat and Lon data sets:
        call endDataSet(latID)
        call endDataSet(lonID)
        

        !*****************************************************************************
        !Determine if the desired point is contained in this granule Lat/Lon data:
        !*****************************************************************************
        
        !Reset the test flag:
        useableGranule = .false.
        
        !Proceed with the test:
        do i = 1, dimensionSizes(1)
           do j = 1, dimensionSizes(2)
              
              !First check to see if this point is within an epsilon of the desired Lon:
              LatCheck: if ( Lons(i,j) >= (desiredLon-epsilonLon)      &
                       .and. Lons(i,j) <= (desiredLon+epsilonLon) )    then
                 
                 !Check to see if this point is within an epsilon of the desired Lat:
                 LonCheck: if ( Lats(i,j) >= (desiredLat-epsilonLat)   &
                          .and. Lats(i,j) <= (desiredLat+epsilonLat) ) then
                    !Granule is useable:
                    useableGranule = .true.
                    exit
                 end if LonCheck
                 
              end if LatCheck
              
           end do
        end do


        !*****************************************************************************
        !If useable granule, determine the data points to output:
        !*****************************************************************************
        if ( useableGranule ) then

           !Test if spatial subsetting is to be performed:
           if (SpatialSubsetting) then
              !Perform the spatial subsetting:
              !Obtain the corner points of the box that contains the desired data:
              BoxCorners = 1
              call findBoxCorners(        Lats = Lats,           &
                                          Lons = Lons,           &
                                    BoxCorners = BoxCorners     )
           else
              !Copy the entire array to the new HDF file:
              !Upper Left:
              BoxCorners(1,1) = 1
              BoxCorners(1,2) = 1
              !Upper Right:
              BoxCorners(2,1) = dimensionSizes(1)
              BoxCorners(2,2) = 1
              !Lower Left:
              BoxCorners(3,1) = 1
              BoxCorners(3,2) = dimensionSizes(2)
              !Lower Right:
              BoxCorners(4,1) = dimensionSizes(1)
              BoxCorners(4,2) = dimensionSizes(2)

           end if

           !Check to make sure the box is large enough to be useful:
           BoxDimensionSizes = 0
           BoxDimensionSizes(1) = abs( BoxCorners(1,1) - BoxCorners(2,1) ) + 1
           BoxDimensionSizes(2) = abs( BoxCorners(1,2) - BoxCorners(3,2) ) + 1
           if (      BoxDimensionSizes(1) < MinBoxDim          &
                .or. BoxDimensionSizes(2) < MinBoxDim )  then
              !too small to be useful
              useableGranule = .false.
           end if
           
        end if


        !*****************************************************************************
        !Testing for useable granule complete, now work with the Lat/Lon arrays:
        !*****************************************************************************
        LatLonOutput: if (useableGranule) then

           !*****************************************************************************
           ! Subset the Lats/Lons to the desired box:
           !*****************************************************************************
           !allocate the necessary space for the Lats/Lons and the Mask:
           allocate( LatsBox( 1:boxDimensionSizes(1), 1:boxDimensionSizes(2) ) )
           allocate( LonsBox( 1:boxDimensionSizes(1), 1:boxDimensionSizes(2) ) )
           allocate( Mask   ( 1:boxDimensionSizes(1), 1:boxDimensionSizes(2) ) )
           
           !copy the array to this box:
           call copyToBox(     BoxCorners = BoxCorners,  &
                             CompleteData = Lats,        &
                            SubsettedData = LatsBox    )
           call copyToBox(     BoxCorners = BoxCorners,  &
                             CompleteData = Lons,        &
                            SubsettedData = LonsBox    )

           !Create the mask to replace unwanted points, in the box, with fill values:
           !determine if the data is to be spatially subsetted:
           if (SpatialSubsetting) then
              !Create the mask based on Lat/Lon points;
              call createMask(  Lats = LatsBox,  &
                                Lons = LonsBox,  &
                                Mask = Mask      )
           else
              !Set the mask equal to 1, so that all points are desired:
              Mask = 1
           end if

           !Apply the mask to the Lats/Lons arrays:
           call applyMask(   Data = LatsBox,  &
                             Mask = Mask,     &
                           origID = latID     )
           call applyMask(   Data = LonsBox,  &
                             Mask = Mask,     &
                           origID = lonID     )


           !*****************************************************************************
           !Initialize the new HDF file, and write it's name t the Granule list:
           !*****************************************************************************
           !Write the name of the new HDF to the granule name file:
           call outputGranuleName(granuleName = trim(fileNameNoPath))

           !Create the new HDF file and copy over global attributes.
           call initializeNewHDF(  newHDFfilename = trim(fileNameNoPath),    &
                                      inputFileID = inputFileID,             &
                                         newHDFID = newHDFID                 )

           !*****************************************************************************
           !Output the lat and lons:
           !*****************************************************************************
           !Output the Longitude array:
           SDSName = "Longitude"
           rank = 2
           !Create the SDS:
           call createSDS( inputFileID = inputFileID,          &
                              newHDFID = newHDFID,             &
                               SDSName = trim(SDSName),        &
                              dimSizes = BoxDimensionSizes,    &
                                  rank = rank                  )
           
           !Copy the data to the new HDF:
           call writeSDS( SampleDataType = dummyFloat,         &
                                newHDFID = newHDFID,           &
                                 SDSName = trim(SDSName),      &
                                dimSizes = BoxDimensionSizes,  &
                                    rank = rank,               &
                               SDSdataR2 = LonsBox             )

           !Output the Latitude array:
           SDSName = "Latitude"
           rank = 2
           !Create the SDS:
           call createSDS( inputFileID = inputFileID,         &
                              newHDFID = newHDFID,            &
                               SDSName = trim(SDSName),       &
                              dimSizes = BoxDimensionSizes,   &
                                  rank = rank                 )
           
           !Copy the data to the new HDF:
           call writeSDS( SampleDataType = dummyFloat,        &
                                newHDFID = newHDFID,          &
                                 SDSName = trim(SDSName),     &
                                dimSizes = BoxDimensionSizes, &
                                    rank = rank,              &
                               SDSdataR2 = LatsBox            )
           
           !deallocate the Lons/Lats boxes:
           deallocate(LonsBox)
           deallocate(LatsBox)
           

        end if LatLonOutput

        !Deallocate the Lat/Lon arrays:
        deallocate(Lons)
        deallocate(Lats)
        
        
        !*****************************************************************************
        !If this is a useable granule, read in and output the data sets for this granule:
        !*****************************************************************************

        useGranule: if ( useableGranule ) then
           
           !*****************************************************************************
           !Loop throught the specified SDS' to read in, subset, and output:
           !*****************************************************************************

           !Determine if all of the SDS, or only a selected list are to be subsetted:
           if (trim(SDS(1)) == "All") then
              !determine the number of SDSs:
              status = SFFinfo(inputFileID,TotNumOfSDS,nattrs)
              BegSDSLoop = 0
              EndSDSLoop = TotNumOfSDS - 1
              SubsetAllSDS = .true.
           else
              BegSDSLoop = 1
              EndSDSLoop = numberOfSDS
              SubsetAllSDS = .false.
                 
           end if
           ProductLoop: do i = BegSDSLoop, EndSDSLoop

              !Determine which SDS to read in and determine type and sizes:
              if (SubsetAllSDS) then
                 !read in the next SDS in the HDF:
                 SDSID = SFselect(inputFileID,i)
                 !Obtain the SDS name, dimensionSizes and datatype:
                 status = SFginfo(SDSID, SDSName, rank, dimensionSizes, datatype, nattrs)
                 if (status == FAIL) print *,'Could not obain SDS info for SDS number ',i
              else
                 !Read in the next from the SDS list:
                 SDSName = trim(SDS(i))
                 !Obtain the data set ID:
                 SDSID = getDataSetID(fileID = inputfileID, dataSetName = trim(SDSName))
              end if
              
              !Obtain the dimensions of this SDS
              dimensionSizes = getDataSetDimSizes(dataSetId = SDSID)

              !Obtain the data type:
              dataType = getDataSetType(SDSID, trim(fileName), trim(SDSName))

              !Determine the resolution, dimension arrangement, and if subsettable:
              !Check to see if the SDS is 3D, 2D, or 1D:
              call SubsetCheck(dimensionSizes, GeoDimSizes,  &
                               SpatialSubset, CorrectOrder     )
              !If there is to be no spatial subsetting set the flag:
              if (.not. SpatialSubsetting) then
                 SpatialSubset = .false.
              end if

              !Check to see if these are the Lat or Lon arrays:
              OutputSDS = .true.
              if (trim(SDSName) == "Latitude"  .or. &
                  trim(SDSName) == "Longitude" ) then
                 OutputSDS = .false.
              end if
              
              if (OutputSDS) then
                               
                 !Call the Subsetting Decision Tree to Subset this SDS:
                 call subsetDecisionTree(     dataType = dataType,          &
                                              filename = trim(fileName),    &
                                           inputFileID = inputFileID,       &
                                              newHDFID = newHDFID,          &
                                               SDSName = trim(SDSName),     &
                                                 SDSID = SDSID,             &
                                         SpatialSubset = SpatialSubset,     &
                                          CorrectOrder = CorrectOrder,      &
                                        dimensionSizes = dimensionSizes,    &
                                     boxDimensionSizes = boxDimensionSizes, &
                                            boxCorners = boxCorners,        &
                                                  Mask = Mask               )

              !END check for subsetting:
              end if

              !End the access to the SDS
              call endDataSet(SDSID)
              
           end do ProductLoop
           
           !end access to new HDF:
           call closeFile(newHDFID)

           !deallocate the mask:
           deallocate(Mask)

        end if useGranule
        
     end if potentGranule

     !Close the granule hdf file:
     call closeFile(inputFileID) 
     
  end do fileloop
  !Close the list of L2 granules file:
  close(FileLUN)
  !deallocate the SDS name array:
  deallocate(SDS)


end program SnglResSubsetter
  

