module subsetting
  !*****************************************************************************
  ! !F90
  !
  ! !Description:
  !    This module contains the Subsetting decision tree and the routines which
  !      read in the data, subsets the data, and output the subsetted data.
  !  
  ! !Callable routines:
  !    subsetDecisionTree()
  ! 
  ! !Revision History:
  ! $Log: subsetting.f90,v $
  ! Revision 1.0  1999/10/22  12:43:43  EGMoody
  ! Initial revision.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   The decision tree determines the rank of the SDS, and then calls the 
  !     appropriate subsetting routine based on rank and data type.
  !   There is a generic subsetting interface (subsetSDS) which is called by
  !     the decision tree.  The interface then calls the appropriate subsetting
  !     subroutine based on:
  !        1)  The data type of a sample data type
  !        2)  If the variable numberWavebands is present (indicating a
  !               multiband SDS)
  !   This is a list of the following subsetting subroutines:
  !        note:  SB = Single Band, MB = Multi Band
  !               IK = Integer Kind, RK = Real Kind
  !               OB = One Byte, TB = Two Byte, FB = Four Byte, EB = Eight Byte
  !
  !               SBsubsetOBIK, SBsubsetTBIK, SBsubsetFBIK,
  !               MBsubsetOBIK, MBsubsetTBIK, MBsubsetFBIK
  !               SBsubsetFBRK, SBsubsetEBRK,             
  !               MBsubsetFBRK, MBsubsetEBRK
  !
  ! !END
  !*****************************************************************************

  !Dependencies:
  use typeSizes,       only: OneByteIntKind, TwoByteIntKind, FourByteIntKind,  &
                             FourByteRealKind, EightByteRealKind
  use hdf,             only: DFNT_FLOAT64, DFNT_FLOAT32,                       &
                             DFNT_INT32, DFNT_UINT32,                          &
                             DFNT_INT16, DFNT_UINT16,                          &
                             DFNT_INT8,  DFNT_UINT8
  use SDSReader,       only: ReadSDS  
  use subsettingUtils, only: applyMask, copyToBox
  use Output,          only: createSDS, writeSDS

  implicit none

  private

  public :: subsetDecisionTree
  
  !Generic interface:
  interface subsetSDS
     module procedure subsetOBIK, subsetTBIK, subsetFBIK, &
                      subsetFBRK, subsetEBRK
  end interface
  ! subroutine subset*B*K( SampleDataType, numWavebands,                 & 
  !                        filename, inputFileID,                        &
  !                        newHDFID, SDSName, SDSID,                     &
  !                        dimensionSizes, boxDimensionSizes, boxCorners,&
  !                        Mask                                          )
  !
  !  *******  (kind = *****Byte****Kind),     intent(in)   :: SampleDataType
  !  integer  (kind = FourByteIntKind),       intent(in)   :: numWavebands
  !  character(len  = *),                     intent(in)   :: fileName
  !  integer  (kind = FourByteIntKind),       intent(in)   :: inputFileID
  !  integer  (kind = FourByteIntKind),       intent(in)   :: newHDFID 
  !  character(len  = *),                     intent(in)   :: SDSName
  !  integer  (kind = FourByteIntKind),       intent(in)   :: SDSID
  !  integer  (kind = FourByteIntKind),    &
  !            dimension(:),                  intent(in)   :: dimensionSizes, &
  !                                                           boxDimensionSizes
  !  integer  (kind = FourByteIntKind),    &
  !           dimension(:,:),                 intent(in)   :: BoxCorners
  !  integer  (kind = FourByteIntKind),    &
  !            dimension(:,:),                intent(in)   :: Mask
  !
  ! !Description:
  !   This routine will read in the entire SDS, copy the designated box to an
  !     array the size of the box, mask the boxed data, create the SDS in the 
  !     new HDF file(and copy the SDS attributes and dimension names), then
  !     write out the masked data to the new SDS.
  !  
  ! !Input Parameters:
  !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
  !                       interface to determine which specific routine to call.
  !    numWavebands   : May or maynot be in the subroutine.  If present, then the
  !                       SDS is multibanded, if not present, then single-banded.
  !                       Used by interface to determine the specific routine to call.
  !    fileName       : Used for storing input HDF file and Path name.
  !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
  !    newHDFID       : File handle for an output HDF file, used by HDF routines.
  !    SDSName        : Stores the name of the SDS being subsetted
  !    SDSID          : Data handle for accessing the SDS being subsetted
  !    dimensionSizes : Used to store the dimension of an SDS.
  ! boxDimensionSizes : Used to store the dimensions of the subsetted box.
  !    BoxCorners     : Contains the indices of the subsetted box.
  !                       dimension (i,j):
  !                        i=1 : Upper Left            j=1: i index of box
  !                        i=2 : Upper Right           j=2: j index of box
  !                        i=3 : Lower Left 
  !                        i=4 : Lower Right
  !    Mask           : Array containing the mask.  The mask is used to replace unwanted
  !                     points in the subsetted box, with fill values.  This is accomplished
  !                     by setting the mask to 1 for desired points and 0 for unwanted
  !                     points.  Then the mask is applied to the subsetted SDS, in which
  !                     when the mask = 0, the fill value replaces the stored value.
  !
  ! !Revision History:
  !   See Module revision history at the beginning of the file.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   The subroutines ReadSDS, copyToBox, applyMask, createSDS, and writeSDS
  !      are generic interfaces, accepting all kinds of data, etc.
  !   The arrays that store the SDS data and subsetted data are allocatable, thereby
  !      being flexible enough to handle any size for the SDS.
  !
  ! !END

  !local variables:
  integer  (kind = FourByteIntKind)             :: waveCounter
  ! waveCounter : Counter to loop over multibanded SDS.



contains



  !*****************************************************************************
  !*****************************************************************************
  ! Subsetting decision tree:
  !*****************************************************************************
  !*****************************************************************************

  !*****************************************************************************
  subroutine SubsetDecisionTree( dataType,                                      & 
                                 filename, inputFileID,                         &
                                 newHDFID, SDSName, SDSID,                      &
                                 SpatialSubset, CorrectOrder,                   &
                                 dimensionSizes, boxDimensionSizes, boxCorners, &
                                 Mask                                           )

    integer  (kind = FourByteIntKind)                     :: dataType
    character(len  = *),                     intent(in)   :: fileName
    integer  (kind = FourByteIntKind),       intent(in)   :: inputFileID
    integer  (kind = FourByteIntKind),       intent(in)   :: newHDFID 
    character(len  = *),                     intent(in)   :: SDSName
    integer  (kind = FourByteIntKind),       intent(in)   :: SDSID
    logical,                                 intent(in)   :: SpatialSubset, CorrectOrder
    integer  (kind = FourByteIntKind),    &
              dimension(:),                  intent(in)   :: dimensionSizes
    integer  (kind = FourByteIntKind),    &
              dimension(:),                  intent(in)   :: boxDimensionSizes
    integer  (kind = FourByteIntKind),    &
              dimension(:,:),                intent(in)   :: BoxCorners
    integer  (kind = FourByteIntKind),    &
              dimension(:,:),                intent(in)   :: Mask
    ! !Description:
    !   The decision tree determines the data type and number of wavebands of the 
    !      SDS to be subsetted.  Then, through a case structure, it calls subsetSDS
    !      with the appropriate dummy data type and includes the variable 
    !      numberWavebands if the SDS is multibanded.
    !   It is a generic interface which can handle any resolution product.
    !  
    ! !Input Parameters:
    !    dataType       : Stores the integer value of the data type.
    !                      ( hdf include file lists the types as:  
    !                        Double = DFNT_FLOAT64 =  6
    !                         Float = DFNT_FLOAT32 =  5
    !                         Short = DFNT_INT16   = 22
    !                          Byte = DFNT_INT8    = 20 )
    !    numWavebands   : Contains the number of wave bands for this SDS.
    !    fileName       : Used for storing input HDF file and Path name.
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    SDSID          : Data handle for accessing the SDS being subsetted
    !    SpatialSubset  : Flag for seeing if the SDS is subsettable or not.
    !     CorrectOrder  : Flag to see if the SDS dimensions are in the correct order.
    !                     2D should be along, across
    !                     3D should be band, along, across
    !                     if 2D and flase, then order is across, along
    !                     if 3D and flase, then order is along, across, band
    !    dimensionSizes : Used to store the dimension of an SDS.
    ! boxDimensionSizes : Used to store the dimensions of the subsetted box.
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                       ( dimension (i,j):
    !                         i=1 : Upper Left            j=1: i index of box
    !                         i=2 : Upper Right           j=2: j index of box
    !                         i=3 : Lower Left 
    !                         i=4 : Lower Right                               )
    !    Mask           : Array containing the mask.  The mask is used to replace unwanted
    !                     points in the subsetted box, with fill values.  This is accomplished
    !                     by setting the mask to 1 for desired points and 0 for unwanted
    !                     points.  Then the mask is applied to the subsetted SDS, in which
    !                     when the mask = 0, the fill value replaces the stored value.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The subroutine subsetSDS is a generic interface based upon differences in data
    !     type and number of bands.
    !   Dummy data is used as dummy input data types for subsetSDS.
    !
    ! !END


    !local vars:
    integer  (kind =   OneByteIntKind )           :: dummyByte   = 0
    integer  (kind =   TwoByteIntKind )           :: dummyShort  = 0
    integer  (kind =  FourByteIntKind )           :: dummyLong   = 0
    real     (kind =  FourByteRealKind)           :: dummyFloat  = 0.0
    real     (kind = EightByteRealKind)           :: dummyDouble = 0.0
    !dummy scalars used as sample data types in calls to subsetSDS.
    integer  (kind = FourByteIntKind)             :: rank
    !rank          : Rank of the array; 1=single dimension, 2=onebanded, 3= multibanded.
    
    !Define the rank of the SDS to be stored in the new HDF file:
    if (dimensionSizes(3) > 0) then
       rank = 3
    else if (dimensionSizes(2) > 0) then
       rank = 2
    else if (dimensionSizes(1) > 0) then
       rank = 1
    end if

    !Check for the multiple integer dataType numbers:
    if (dataType == DFNT_UINT32) then
        dataType =  DFNT_INT32
    end if
    if (dataType == DFNT_UINT16) then
        dataType =  DFNT_INT16
    end if
    if (dataType == DFNT_UINT8) then
        dataType =  DFNT_INT8
    end if

    !Select the appropriate data type of the SDS:
    select case(dataType)
       
       
    case(DFNT_FLOAT64)
       !Double case
       call subsetSDS(  SampleDataType = dummyDouble,       &
                              filename = trim(fileName),    &
                           inputFileID = inputFileID,       &
                              newHDFID = newHDFID,          &
                               SDSName = trim(SDSName),     &
                                 SDSID = SDSID,             &
                        dimensionSizes = dimensionSizes,    &
                     boxDimensionSizes = boxDimensionSizes, &
                            boxCorners = boxCorners,        &
                                  rank = rank,              &
                         SpatialSubset = SpatialSubset,     &
                          CorrectOrder = CorrectOrder,      &
                                  Mask = Mask               )
       
    case(DFNT_FLOAT32)
       !Float case
       call subsetSDS(  SampleDataType = dummyFloat,        &
                              filename = trim(fileName),    &
                           inputFileID = inputFileID,       &
                              newHDFID = newHDFID,          &
                               SDSName = trim(SDSName),     &
                                 SDSID = SDSID,             &
                        dimensionSizes = dimensionSizes,    &
                     boxDimensionSizes = boxDimensionSizes, &
                            boxCorners = boxCorners,        &
                                  rank = rank,              &
                         SpatialSubset = SpatialSubset,     &
                          CorrectOrder = CorrectOrder,      &
                                  Mask = Mask               )
       
    case(DFNT_INT32)
       !Long case
       call subsetSDS(  SampleDataType = dummyLong,         &
                              filename = trim(fileName),    &
                           inputFileID = inputFileID,       &
                              newHDFID = newHDFID,          &
                               SDSName = trim(SDSName),     &
                                 SDSID = SDSID,             &
                        dimensionSizes = dimensionSizes,    &
                     boxDimensionSizes = boxDimensionSizes, &
                            boxCorners = boxCorners,        &
                                  rank = rank,              &
                         SpatialSubset = SpatialSubset,     &
                          CorrectOrder = CorrectOrder,      &
                                  Mask = Mask               )
       
    case(DFNT_INT16)
       !Short case
       call subsetSDS(  SampleDataType = dummyShort,        &
                              filename = trim(fileName),    &
                           inputFileID = inputFileID,       &
                              newHDFID = newHDFID,          &
                               SDSName = trim(SDSName),     &
                                 SDSID = SDSID,             &
                        dimensionSizes = dimensionSizes,    &
                     boxDimensionSizes = boxDimensionSizes, &
                            boxCorners = boxCorners,        &
                                  rank = rank,              &
                         SpatialSubset = SpatialSubset,     &
                          CorrectOrder = CorrectOrder,      &
                                  Mask = Mask               )
       
    case(DFNT_INT8)
       !Byte case
       call subsetSDS(  SampleDataType = dummyByte,         &
                              filename = trim(fileName),    &
                           inputFileID = inputFileID,       &
                              newHDFID = newHDFID,          &
                               SDSName = trim(SDSName),     &
                                 SDSID = SDSID,             &
                        dimensionSizes = dimensionSizes,    &
                     boxDimensionSizes = boxDimensionSizes, &
                            boxCorners = boxCorners,        &
                                  rank = rank,              &
                         SpatialSubset = SpatialSubset,     &
                          CorrectOrder = CorrectOrder,      &
                                  Mask = Mask               )
       
    end select


  end subroutine SubsetDecisionTree
  !*****************************************************************************

  !*****************************************************************************
  !*****************************************************************************
  ! Subsetting Routines:
  !*****************************************************************************
  !*****************************************************************************
  
  !*****************************************************************************
  ! Double Case:
  !*****************************************************************************

  !*****************************************************************************
  subroutine subsetEBRK( SampleDataType,                               & 
                         filename, inputFileID,                        &
                         newHDFID, SDSName, SDSID,                     &
                         dimensionSizes, boxDimensionSizes, boxCorners,&
                         rank, SpatialSubset, CorrectOrder, Mask       )

    real     (kind = EightByteRealKind),     intent(in)   :: SampleDataType
    character(len  = *),                     intent(in)   :: fileName
    integer  (kind = FourByteIntKind),       intent(in)   :: inputFileID
    integer  (kind = FourByteIntKind),       intent(in)   :: newHDFID 
    character(len  = *),                     intent(in)   :: SDSName
    integer  (kind = FourByteIntKind),       intent(in)   :: SDSID
    integer  (kind = FourByteIntKind),    &
              dimension(:),                  intent(in)   :: dimensionSizes, &
                                                             boxDimensionSizes
    integer  (kind = FourByteIntKind),    &
             dimension(:,:),                 intent(in)   :: BoxCorners
    integer  (kind = FourByteIntKind),       intent(in)   :: rank
    logical,                                 intent(in)   :: SpatialSubset, CorrectOrder
    integer  (kind = FourByteIntKind),    &
              dimension(:,:),                intent(in)   :: Mask
    ! !Description:
    !   Specific routine to read in an eight byte real SDS.
    !   This routine will read in the entire SDS, copy the designated box to an
    !     array the size of the box, mask the boxed data, create the SDS in the 
    !     new HDF file(and copy the SDS attributes and dimension names), then
    !     write out the masked data to the new SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    fileName       : Used for storing input HDF file and Path name.
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    SDSID          : Data handle for accessing the SDS being subsetted
    !    dimensionSizes : Used to store the dimension of an SDS.
    ! boxDimensionSizes : Used to store the dimensions of the subsetted box.
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                       dimension (i,j):
    !                        i=1 : Upper Left            j=1: i index of box
    !                        i=2 : Upper Right           j=2: j index of box
    !                        i=3 : Lower Left 
    !                        i=4 : Lower Right
    !    rank           : Rank of the array; 1=single dimension, 2=onebanded, 3= multibanded.
    !    SpatialSubset  : Flag for seeing if the SDS is subsettable or not.
    !     CorrectOrder  : Flag to see if the SDS dimensions are in the correct order.
    !                     2D should be along, across
    !                     3D should be band, along, across
    !                     if 2D and flase, then order is across, along
    !                     if 3D and flase, then order is along, across, band
    !    Mask           : Array containing the mask.  The mask is used to replace unwanted
    !                     points in the subsetted box, with fill values.  This is accomplished
    !                     by setting the mask to 1 for desired points and 0 for unwanted
    !                     points.  Then the mask is applied to the subsetted SDS, in which
    !                     when the mask = 0, the fill value replaces the stored value.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The subroutines ReadSDS, copyToBox, applyMask, createSDS, and writeSDS
    !      are generic interfaces, accepting all kinds of data, etc.
    !   The arrays that store the SDS data and subsetted data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !local vars:
    integer  (kind = FourByteIntKind)                     :: numWavebands
    !numWavebands  : Contains the number of wave bands for this SDS.
    integer  (kind = FourByteIntKind),    &
              dimension(:),       allocatable             :: TempBoxDimSizes
    !TempBoxDimSizes : Allows the adjustment of the box dimensions.
    real     (kind = EightByteRealKind),  &
              dimension(:,:,:),   allocatable             ::    eightByteRealDataR3
    real     (kind = EightByteRealKind),  &
              dimension(:,:),     allocatable             ::    eightByteRealDataR2
    real     (kind = EightByteRealKind),  &
              dimension(:),       allocatable             ::    eightByteRealDataR1
    real     (kind = EightByteRealKind),  &
              dimension(:, :, :), allocatable             :: boxEightByteRealDataR3
    real     (kind = EightByteRealKind),  &
              dimension(:, :),    allocatable             :: boxEightByteRealDataR2
    ! The SDS will be read into these arrays, and then outputted.  They were 
    !  engineered to be allocatable, to allow for varying granule and SDS sizes.
    !  Rx stands for Rank X.
    ! The Box array will contain the subsetted data. Rx stands for Rank X.

    !Allocate the temporary box dimensions array:
    allocate(TempBoxDimSizes(size(dimensionSizes)))
    TempBoxDimSizes = 0

    !Split the cases into ranks:
    select case(rank)
       
    case(3)
       !Rank = 3 (i.e. Multi-banded 3 dimensional):
       !Allocate the data array:
       allocate(   EightByteRealDataR3(    dimensionSizes(1),  &
                                           dimensionSizes(2),  &
                                           dimensionSizes(3))  )

       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR3 = EightByteRealDataR3 )
       
       !Subset the data, if acceptable format:
       if (SpatialSubset) then
          !Subset the SDS:
          !Allocate the box array:
          if (CorrectOrder) then
             !Band, Along, Across
             TempBoxDimSizes(1) = BoxDimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(2)
             TempBoxDimSizes(3) = dimensionSizes(3)
             numWavebands = TempBoxDimSizes(3)
          else
             !along, across, band
             TempBoxDimSizes(1) = dimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(1)
             TempBoxDimSizes(3) = BoxDimensionSizes(2)
             numWavebands = dimensionSizes(1)
          end if
          allocate(boxEightByteRealDataR3( TempBoxDimSizes(1),  &
                                           TempBoxDimSizes(2),  &
                                           TempBoxDimSizes(3))  )  

          !Perform the subsetting one band at a time:
          do waveCounter = 0, numWavebands - 1
             if (CorrectOrder) then
                !Band, Along, Across
                call copyToBox(    BoxCorners = BoxCorners,                 &
                                 CompleteData = EightByteRealDataR3(:,:,waveCounter+1),  &
                                SubsettedData = boxEightByteRealDataR3(:,:,waveCounter+1))
                call applyMask(   Data = boxEightByteRealDataR3(:,:,waveCounter+1),    &
                                  Mask = Mask,                              &
                                origID = SDSID                              )
             else
                !Along, Across, Band
                call copyToBox(    BoxCorners = BoxCorners,                 &
                                 CompleteData = EightByteRealDataR3(waveCounter+1,:,:),  &
                                SubsettedData = boxEightByteRealDataR3(waveCounter+1,:,:))
                call applyMask(   Data = boxEightByteRealDataR3(waveCounter+1,:,:),    &
                                  Mask = Mask,                              &
                                origID = SDSID                              )
             end if
          end do
          
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = TempBoxDimSizes,         & 
                                 rank = rank                     )
          
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = TempBoxDimSizes,       &
                                   rank = rank,                  &
                              SDSdataR3 = boxEightByteRealDataR3 )
          
          !Deallocate the box array:
          deallocate(boxEightByteRealDataR3)

       else       
          !Do not perform the subsetting.
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
          
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR3 = eightByteRealDataR3    ) 

       end if

       !Deallocate the data array:
       deallocate(EightByteRealDataR3)

    case(2)
       !Rank = 2 (i.e. Two dimensional SDS):
       !Allocate the data array:
       allocate(   EightByteRealDataR2(    dimensionSizes(1),  &
                                           dimensionSizes(2))  )
       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR2 = EightByteRealDataR2 )

       !Subset the data, if acceptable format:
       if (SpatialSubset) then
          !Subset the SDS:
          !Allocate the box array:
          allocate(boxEightByteRealDataR2( BoxDimensionSizes(1),  &
                                           BoxDimensionSizes(2))  )
       
          !Subset:
          call copyToBox(    BoxCorners = BoxCorners,            &
                           CompleteData = EightByteRealDataR2,   &
                          SubsettedData = boxEightByteRealDataR2 )
          call applyMask(   Data = boxEightByteRealDataR2,       &
                            Mask = Mask,                         &
                          origID = SDSID                         )
          
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = BoxDimensionSizes,       & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = BoxDimensionSizes,     &
                                   rank = rank,                  &
                              SDSdataR2 = boxEightByteRealDataR2 )

          !Deallocate the box array:
          deallocate(boxEightByteRealDataR2)

       else

          !Do not subset:
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR2 = eightByteRealDataR2    )


       end if

       !Deallocate the data arrays:
       deallocate(EightByteRealDataR2)

    case(1)
       !Rank = 1 (Single Dimensional, no subsetting):
       !Allocate the arrays:
       allocate(  EightByteRealDataR1( dimensionSizes(1) )  )

       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR1 = EightByteRealDataR1 )
              
       !Create the SDS in the new HDF:
       call createSDS( inputFileID = inputFileID,             &
                          newHDFID = newHDFID,                &
                           SDSName = trim(SDSName),           &
                          dimSizes = dimensionSizes,          & 
                              rank = rank                     )
    
       !Output the SDS to the new HDF:
       call writeSDS( SampleDataType = SampleDataType,        &  
                            newHDFID = newHDFID,              &
                             SDSName = trim(SDSName),         &
                            dimSizes = dimensionSizes,        &
                                rank = rank,                  &
                           SDSdataR1 = EightByteRealDataR1    )

       !Deallocate the data arrays:
       deallocate(EightByteRealDataR1)

    !end the case:
    end select

    !deallocate temporary box dimension array:
    deallocate(TempBoxDimSizes)
 
  end subroutine subsetEBRK

  !*****************************************************************************

  !*****************************************************************************
  ! Float Case:
  !*****************************************************************************

  !*****************************************************************************
  
  subroutine subsetFBRK( SampleDataType,                               & 
                         filename, inputFileID,                        &
                         newHDFID, SDSName, SDSID,                     &
                         dimensionSizes, boxDimensionSizes, boxCorners,&
                         rank, SpatialSubset, CorrectOrder, Mask       )

    real     (kind = FourByteRealKind),      intent(in)   :: SampleDataType
    character(len  = *),                     intent(in)   :: fileName
    integer  (kind = FourByteIntKind),       intent(in)   :: inputFileID
    integer  (kind = FourByteIntKind),       intent(in)   :: newHDFID 
    character(len  = *),                     intent(in)   :: SDSName
    integer  (kind = FourByteIntKind),       intent(in)   :: SDSID
    integer  (kind = FourByteIntKind),    &
              dimension(:),                  intent(in)   :: dimensionSizes, &
                                                             boxDimensionSizes
    integer  (kind = FourByteIntKind),    &
             dimension(:,:),                 intent(in)   :: BoxCorners
    integer  (kind = FourByteIntKind),       intent(in)   :: rank
    logical,                                 intent(in)   :: SpatialSubset, CorrectOrder
    integer  (kind = FourByteIntKind),    &
              dimension(:,:),                intent(in)   :: Mask
    ! !Description:
    !   Specific routine to read in a four byte real SDS.
    !   This routine will read in the entire SDS, copy the designated box to an
    !     array the size of the box, mask the boxed data, create the SDS in the 
    !     new HDF file(and copy the SDS attributes and dimension names), then
    !     write out the masked data to the new SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    fileName       : Used for storing input HDF file and Path name.
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    SDSID          : Data handle for accessing the SDS being subsetted
    !    dimensionSizes : Used to store the dimension of an SDS.
    ! boxDimensionSizes : Used to store the dimensions of the subsetted box.
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                       dimension (i,j):
    !                        i=1 : Upper Left            j=1: i index of box
    !                        i=2 : Upper Right           j=2: j index of box
    !                        i=3 : Lower Left 
    !                        i=4 : Lower Right
    !    rank           : Rank of the array; 1=single dimension, 2=onebanded, 3= multibanded.
    !    SpatialSubset  : Flag for seeing if the SDS is subsettable or not.
    !     CorrectOrder  : Flag to see if the SDS dimensions are in the correct order.
    !                     2D should be along, across
    !                     3D should be band, along, across
    !                     if 2D and flase, then order is across, along
    !                     if 3D and flase, then order is along, across, band
    !    Mask           : Array containing the mask.  The mask is used to replace unwanted
    !                     points in the subsetted box, with fill values.  This is accomplished
    !                     by setting the mask to 1 for desired points and 0 for unwanted
    !                     points.  Then the mask is applied to the subsetted SDS, in which
    !                     when the mask = 0, the fill value replaces the stored value.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The subroutines ReadSDS, copyToBox, applyMask, createSDS, and writeSDS
    !      are generic interfaces, accepting all kinds of data, etc.
    !   The arrays that store the SDS data and subsetted data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !local vars:
    integer  (kind = FourByteIntKind)                     :: numWavebands
    !numWavebands  : Contains the number of wave bands for this SDS.
    integer  (kind = FourByteIntKind),    &
              dimension(:),       allocatable             :: TempBoxDimSizes
    !TempBoxDimSizes : Allows the adjustment of the box dimensions.
    real     (kind = FourByteRealKind ),  &
              dimension(:,:,:),   allocatable             ::     fourByteRealDataR3
    real     (kind = FourByteRealKind ),  &
              dimension(:,:),     allocatable             ::     fourByteRealDataR2
    real     (kind = FourByteRealKind ),  &
              dimension(:),       allocatable             ::     fourByteRealDataR1
    real     (kind = FourByteRealKind ),  &
              dimension(:, :, :), allocatable             ::  boxFourByteRealDataR3
    real     (kind = FourByteRealKind ),  &
              dimension(:, :),    allocatable             ::  boxFourByteRealDataR2
    ! The SDS will be read into these arrays, and then outputted.  They were 
    !  engineered to be allocatable, to allow for varying granule and SDS sizes.
    ! The Box array will contain the subsetted data. Rx stands for Rank X.

    !Allocate the temporary box dimensions array:
    allocate(TempBoxDimSizes(size(dimensionSizes)))
    TempBoxDimSizes = 0

    !Split the cases into ranks:
    select case(rank)
       
    case(3)
       !Rank = 3 (i.e. Multi-banded 3 dimensional):
       !Allocate the data array:
       allocate(   FourByteRealDataR3(    dimensionSizes(1),  &
                                          dimensionSizes(2),  &
                                          dimensionSizes(3))  )
       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR3 = FourByteRealDataR3  )
       
       !Subset the data, if acceptable format:
       if (SpatialSubset) then
          !Subset the data:
          !Allocate the box array:
          if (CorrectOrder) then
             !band, along, across
             TempBoxDimSizes(1) = BoxDimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(2)
             TempBoxDimSizes(3) = dimensionSizes(3)
             numWavebands = TempBoxDimSizes(3)
          else
             !along, across, band
             TempBoxDimSizes(1) = dimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(1)
             TempBoxDimSizes(3) = BoxDimensionSizes(2)
             numWavebands = dimensionSizes(1)
          end if
          allocate(boxFourByteRealDataR3( TempBoxDimSizes(1),  &
                                          TempBoxDimSizes(2),  &
                                          TempBoxDimSizes(3))  )

          !Perform this one band at a time:
          do waveCounter = 0, numWavebands - 1
             if (CorrectOrder) then
                !Band, Along, Across
                call copyToBox(    BoxCorners = BoxCorners,                 &
                                 CompleteData = FourByteRealDataR3(:,:,waveCounter+1),  &
                                SubsettedData = boxFourByteRealDataR3(:,:,waveCounter+1))
                call applyMask(   Data = boxFourByteRealDataR3(:,:,waveCounter+1),    &
                                  Mask = Mask,                             &
                                origID = SDSID                             )
             else
                !Along, Across, Band
                call copyToBox(    BoxCorners = BoxCorners,                 &
                                 CompleteData = FourByteRealDataR3(waveCounter+1,:,:),  &
                                SubsettedData = boxFourByteRealDataR3(waveCounter+1,:,:))
                call applyMask(   Data = boxFourByteRealDataR3(waveCounter+1,:,:),    &
                                  Mask = Mask,                             &
                                origID = SDSID                             )
             end if
          end do
          
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = TempBoxDimSizes,         & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = TempBoxDimSizes,       &
                                   rank = rank,                  &
                              SDSdataR3 = boxFourByteRealDataR3  )

          !deallocate the box array:
          deallocate(boxFourByteRealDataR3)

       else
          !Do not perform the subsetting.
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR3 = FourByteRealDataR3     )
       end if
       
       !Deallocate the data arrays:
       deallocate(FourByteRealDataR3)

    case(2)
       !Rank = 2 (i.e. Two dimensional SDS):
       !Allocate the data arrays:
       allocate(   FourByteRealDataR2(    dimensionSizes(1),  &
                                          dimensionSizes(2))  )

       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR2 = FourByteRealDataR2  )
       
       !Subset the data, if acceptable format:
       if (SpatialSubset) then
          !Subset the data:
          !Allocate the box array:
          allocate(boxFourByteRealDataR2( BoxDimensionSizes(1),  &
                                          BoxDimensionSizes(2))  )

          !Subset
          call copyToBox(    BoxCorners = BoxCorners,           &
                           CompleteData = FourByteRealDataR2,   &
                          SubsettedData = boxFourByteRealDataR2 )
          call applyMask(   Data = boxFourByteRealDataR2,       &
                            Mask = Mask,                        &
                          origID = SDSID                        )

          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = BoxDimensionSizes,       & 
                                 rank = rank                     )
          
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = BoxDimensionSizes,     &
                                   rank = rank,                  &
                              SDSdataR2 = boxFourByteRealDataR2  )

          !Deallocate the box array:
          deallocate(boxFourByteRealDataR2)

       else
          !Do not subset:
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
          
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR2 = FourByteRealDataR2     )
       end if

       !Deallocate the data arrays:
       deallocate(FourByteRealDataR2)

    case(1)
       !Rank = 1 (Single Dimensional, no subsetting):
       !Allocate the arrays:
       allocate(  FourByteRealDataR1( dimensionSizes(1) )  )

       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR1 = FourByteRealDataR1  )
              
       !Create the SDS in the new HDF:
       call createSDS( inputFileID = inputFileID,             &
                          newHDFID = newHDFID,                &
                           SDSName = trim(SDSName),           &
                          dimSizes = dimensionSizes,          & 
                              rank = rank                     )
    
       !Output the SDS to the new HDF:
       call writeSDS( SampleDataType = SampleDataType,        &  
                            newHDFID = newHDFID,              &
                             SDSName = trim(SDSName),         &
                            dimSizes = dimensionSizes,        &
                                rank = rank,                  &
                           SDSdataR1 = FourByteRealDataR1     )

       !Deallocate the data arrays:
       deallocate(   FourByteRealDataR1)


    !end the case:
    end select

    !deallocate temporary box dimension array:
    deallocate(TempBoxDimSizes)
 
  end subroutine subsetFBRK

  !*****************************************************************************

  !*****************************************************************************
  ! Long Case:
  !*****************************************************************************

  !*****************************************************************************
  
  subroutine subsetFBIK( SampleDataType,                               & 
                         filename, inputFileID,                        &
                         newHDFID, SDSName, SDSID,                     &
                         dimensionSizes, boxDimensionSizes, boxCorners,&
                         rank, SpatialSubset, CorrectOrder, Mask       )

    integer  (kind = FourByteIntKind),       intent(in)   :: SampleDataType
    character(len  = *),                     intent(in)   :: fileName
    integer  (kind = FourByteIntKind),       intent(in)   :: inputFileID
    integer  (kind = FourByteIntKind),       intent(in)   :: newHDFID 
    character(len  = *),                     intent(in)   :: SDSName
    integer  (kind = FourByteIntKind),       intent(in)   :: SDSID
    integer  (kind = FourByteIntKind),    &
              dimension(:),                  intent(in)   :: dimensionSizes, &
                                                             boxDimensionSizes
    integer  (kind = FourByteIntKind),    &
             dimension(:,:),                 intent(in)   :: BoxCorners
    integer  (kind = FourByteIntKind),       intent(in)   :: rank
    logical,                                 intent(in)   :: SpatialSubset, CorrectOrder
    integer  (kind = FourByteIntKind),    &
              dimension(:,:),                intent(in)   :: Mask
    ! !Description:
    !   Specific routine to read in a four byte integer SDS.
    !   This routine will read in the entire SDS, copy the designated box to an
    !     array the size of the box, mask the boxed data, create the SDS in the 
    !     new HDF file(and copy the SDS attributes and dimension names), then
    !     write out the masked data to the new SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    fileName       : Used for storing input HDF file and Path name.
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    SDSID          : Data handle for accessing the SDS being subsetted
    !    dimensionSizes : Used to store the dimension of an SDS.
    ! boxDimensionSizes : Used to store the dimensions of the subsetted box.
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                       dimension (i,j):
    !                        i=1 : Upper Left            j=1: i index of box
    !                        i=2 : Upper Right           j=2: j index of box
    !                        i=3 : Lower Left 
    !                        i=4 : Lower Right
    !    rank           : Rank of the array; 1=single dimension, 2=onebanded, 3= multibanded.
    !    SpatialSubset  : Flag for seeing if the SDS is subsettable or not.
    !     CorrectOrder  : Flag to see if the SDS dimensions are in the correct order.
    !                     2D should be along, across
    !                     3D should be band, along, across
    !                     if 2D and flase, then order is across, along
    !                     if 3D and flase, then order is along, across, band
    !    Mask           : Array containing the mask.  The mask is used to replace unwanted
    !                     points in the subsetted box, with fill values.  This is accomplished
    !                     by setting the mask to 1 for desired points and 0 for unwanted
    !                     points.  Then the mask is applied to the subsetted SDS, in which
    !                     when the mask = 0, the fill value replaces the stored value.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The subroutines ReadSDS, copyToBox, applyMask, createSDS, and writeSDS
    !      are generic interfaces, accepting all kinds of data, etc.
    !   The arrays that store the SDS data and subsetted data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !local vars:
    integer  (kind = FourByteIntKind)                     :: numWavebands
    !numWavebands  : Contains the number of wave bands for this SDS.
    integer  (kind = FourByteIntKind),    &
              dimension(:),       allocatable             :: TempBoxDimSizes
    !TempBoxDimSizes : Allows the adjustment of the box dimensions.
    integer  (kind = FourByteIntKind  ),  &
              dimension(:,:,:),   allocatable             ::     fourByteIntDataR3
    integer  (kind = FourByteIntKind  ),  &
              dimension(:,:),     allocatable             ::     fourByteIntDataR2
    integer  (kind = FourByteIntKind  ),  &
              dimension(:),       allocatable             ::     fourByteIntDataR1
    integer  (kind = FourByteIntKind  ),  &
              dimension(:, :, :), allocatable             ::  boxFourByteIntDataR3
    integer  (kind = FourByteIntKind  ),  &
              dimension(:, :),    allocatable             ::  boxFourByteIntDataR2
    ! The SDS will be read into these arrays, and then outputted.  They were 
    !  engineered to be allocatable, to allow for varying granule and SDS sizes.
    ! The Box array will contain the subsetted data. Rx stands for Rank X.

    !Allocate the temporary box dimensions array:
    allocate(TempBoxDimSizes(size(dimensionSizes)))
    TempBoxDimSizes = 0

    !Split the cases into ranks:
    select case(rank)
       
    case(3)
       !Rank = 3 (i.e. Multi-banded 3 dimensional):
       !Allocate the data array:
       allocate(   FourByteIntDataR3(    dimensionSizes(1),  &
                                         dimensionSizes(2),  &
                                         dimensionSizes(3))  )
       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR3 = FourByteIntDataR3   )

       !Subset the data, if acceptable format:
       if (SpatialSubset) then
       
          !Subset the data:
          !Allocate the box array:
          if (CorrectOrder) then
             !Band, Along, Across
             TempBoxDimSizes(1) = BoxDimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(2)
             TempBoxDimSizes(3) = dimensionSizes(3)
             numWavebands = TempBoxDimSizes(3)
          else
             !along, across, band
             TempBoxDimSizes(1) = dimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(1)
             TempBoxDimSizes(3) = BoxDimensionSizes(2)
             numWavebands = dimensionSizes(1)
          end if
          allocate(boxFourByteIntDataR3( TempBoxDimSizes(1),  &
                                         TempBoxDimSizes(2),  &
                                         TempBoxDimSizes(3))  )

          !Perform this one band at a time:
          do waveCounter = 0, numWavebands - 1
             if (CorrectOrder) then
                !Band, Along, Across
                call copyToBox(    BoxCorners = BoxCorners,               &
                                 CompleteData = FourByteIntDataR3(:,:,waveCounter+1),  &
                                SubsettedData = boxFourByteIntDataR3(:,:,waveCounter+1))
                call applyMask(   Data = boxFourByteIntDataR3(:,:,waveCounter+1),    &
                                  Mask = Mask,                            &
                                origID = SDSID                            )
             else
                !along, across, band
                call copyToBox(    BoxCorners = BoxCorners,               &
                                 CompleteData = FourByteIntDataR3(waveCounter+1,:,:),  &
                                SubsettedData = boxFourByteIntDataR3(waveCounter+1,:,:))
                call applyMask(   Data = boxFourByteIntDataR3(waveCounter+1,:,:),    &
                                  Mask = Mask,                            &
                                origID = SDSID                            )
             end if
          end do
          
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = TempBoxDimSizes,         & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = TempBoxDimSizes,       &
                                   rank = rank,                  &
                              SDSdataR3 = boxFourByteIntDataR3   )
          
          !Deallocate the box array:
          deallocate(boxFourByteIntDataR3)

       else
          !Do not perform the subsetting.
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR3 = FourByteIntDataR3      )
       end if

       !Deallocate the data arrays:
       deallocate(FourByteIntDataR3)

    case(2)
       !Rank = 2 (i.e. Two dimensional SDS):
       !Allocate the data arrays:
       allocate(   FourByteIntDataR2(    dimensionSizes(1),  &
                                         dimensionSizes(2))  )

       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR2 = FourByteIntDataR2   )
       
       !Subset the data, if acceptable format:
       if (SpatialSubset) then

          !allocate the box arrays:
          allocate(boxFourByteIntDataR2( BoxDimensionSizes(1),  &
                                         BoxDimensionSizes(2))  )
          !Subset the data:
          call copyToBox(    BoxCorners = BoxCorners,          &
                           CompleteData = FourByteIntDataR2,   &
                          SubsettedData = boxFourByteIntDataR2 )
          call applyMask(   Data = boxFourByteIntDataR2,       &
                            Mask = Mask,                       &
                          origID = SDSID                       )
          
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = BoxDimensionSizes,       & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = BoxDimensionSizes,     &
                                   rank = rank,                  &
                              SDSdataR2 = boxFourByteIntDataR2   )

          !Deallocate the box arrays:
          deallocate(boxFourByteIntDataR2)

       else
          !Do not subset:
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR2 = FourByteIntDataR2      )

       end if

       !Deallocate the data arrays:
       deallocate(FourByteIntDataR2)

    case(1)
       !Rank = 1 (Single Dimensional, no subsetting):
       !Allocate the arrays:
       allocate(  FourByteIntDataR1( dimensionSizes(1) )  )

       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR1 = FourByteIntDataR1   )
              
       !Create the SDS in the new HDF:
       call createSDS( inputFileID = inputFileID,             &
                          newHDFID = newHDFID,                &
                           SDSName = trim(SDSName),           &
                          dimSizes = dimensionSizes,          & 
                              rank = rank                     )
    
       !Output the SDS to the new HDF:
       call writeSDS( SampleDataType = SampleDataType,        &  
                            newHDFID = newHDFID,              &
                             SDSName = trim(SDSName),         &
                            dimSizes = dimensionSizes,        &
                                rank = rank,                  &
                           SDSdataR1 = FourByteIntDataR1      )

       !Deallocate the data arrays:
       deallocate(   FourByteIntDataR1)


    !end the case:
    end select

    !deallocate temporary box dimension array:
    deallocate(TempBoxDimSizes)
 
  end subroutine subsetFBIK

  !*****************************************************************************

  !*****************************************************************************
  ! Short Case:
  !*****************************************************************************

  !*****************************************************************************
  
  subroutine subsetTBIK( SampleDataType,                               & 
                         filename, inputFileID,                        &
                         newHDFID, SDSName, SDSID,                     &
                         dimensionSizes, boxDimensionSizes, boxCorners,&
                         rank, SpatialSubset, CorrectOrder, Mask       )

    integer  (kind = TwoByteIntKind),        intent(in)   :: SampleDataType
    character(len  = *),                     intent(in)   :: fileName
    integer  (kind = FourByteIntKind),       intent(in)   :: inputFileID
    integer  (kind = FourByteIntKind),       intent(in)   :: newHDFID 
    character(len  = *),                     intent(in)   :: SDSName
    integer  (kind = FourByteIntKind),       intent(in)   :: SDSID
    integer  (kind = FourByteIntKind),    &
              dimension(:),                  intent(in)   :: dimensionSizes, &
                                                             boxDimensionSizes
    integer  (kind = FourByteIntKind),    &
             dimension(:,:),                 intent(in)   :: BoxCorners
    integer  (kind = FourByteIntKind),       intent(in)   :: rank
    logical,                                 intent(in)   :: SpatialSubset, CorrectOrder
    integer  (kind = FourByteIntKind),    &
              dimension(:,:),                intent(in)   :: Mask
    ! !Description:
    !   Specific routine to read in a two byte integer SDS.
    !   This routine will read in the entire SDS, copy the designated box to an
    !     array the size of the box, mask the boxed data, create the SDS in the 
    !     new HDF file(and copy the SDS attributes and dimension names), then
    !     write out the masked data to the new SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    fileName       : Used for storing input HDF file and Path name.
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    SDSID          : Data handle for accessing the SDS being subsetted
    !    dimensionSizes : Used to store the dimension of an SDS.
    ! boxDimensionSizes : Used to store the dimensions of the subsetted box.
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                       dimension (i,j):
    !                        i=1 : Upper Left            j=1: i index of box
    !                        i=2 : Upper Right           j=2: j index of box
    !                        i=3 : Lower Left 
    !                        i=4 : Lower Right
    !    rank           : Rank of the array; 1=single dimension, 2=onebanded, 3= multibanded.
    !    SpatialSubset  : Flag for seeing if the SDS is subsettable or not.
    !     CorrectOrder  : Flag to see if the SDS dimensions are in the correct order.
    !                     2D should be along, across
    !                     3D should be band, along, across
    !                     if 2D and flase, then order is across, along
    !                     if 3D and flase, then order is along, across, band
    !    Mask           : Array containing the mask.  The mask is used to replace unwanted
    !                     points in the subsetted box, with fill values.  This is accomplished
    !                     by setting the mask to 1 for desired points and 0 for unwanted
    !                     points.  Then the mask is applied to the subsetted SDS, in which
    !                     when the mask = 0, the fill value replaces the stored value.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The subroutines ReadSDS, copyToBox, applyMask, createSDS, and writeSDS
    !      are generic interfaces, accepting all kinds of data, etc.
    !   The arrays that store the SDS data and subsetted data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !local vars:
    integer  (kind = FourByteIntKind)                     :: numWavebands
    !numWavebands  : Contains the number of wave bands for this SDS.
    integer  (kind = FourByteIntKind),    &
              dimension(:),       allocatable             :: TempBoxDimSizes
    !TempBoxDimSizes : Allows the adjustment of the box dimensions.
    integer  (kind =  TwoByteIntKind  ),  &
              dimension(:,:,:),   allocatable             ::      twoByteIntDataR3
    integer  (kind =  TwoByteIntKind  ),  &
              dimension(:,:),     allocatable             ::      twoByteIntDataR2
    integer  (kind =  TwoByteIntKind  ),  &
              dimension(:),       allocatable             ::      twoByteIntDataR1
    integer  (kind =  TwoByteIntKind  ),  &
              dimension(:, :, :), allocatable             ::   boxTwoByteIntDataR3
    integer  (kind =  TwoByteIntKind  ),  &
              dimension(:, :),    allocatable             ::   boxTwoByteIntDataR2
    ! The SDS will be read into these arrays, and then outputted.  They were 
    !  engineered to be allocatable, to allow for varying granule and SDS sizes.
    ! The Box array will contain the subsetted data. Rx stands for Rank X.

    !Allocate the temporary box dimensions array:
    allocate(TempBoxDimSizes(size(dimensionSizes)))
    TempBoxDimSizes = 0

    !Split the cases into ranks:
    select case(rank)
       
    case(3)
       !Rank = 3 (i.e. Multi-banded 3 dimensional):
       !Allocate the arrays:
       allocate(   TwoByteIntDataR3(    dimensionSizes(1),  &
                                        dimensionSizes(2),  &
                                        dimensionSizes(3))  )
       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR3 = TwoByteIntDataR3    )

       !Subset the data, if acceptable format:
       if (SpatialSubset) then       

          !Subset the data:
          !Allocate the box array:
          if (CorrectOrder) then
             !Band, Along, Across
             TempBoxDimSizes(1) = BoxDimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(2)
             TempBoxDimSizes(3) = dimensionSizes(3)
             numWavebands = TempBoxDimSizes(3)
          else
             !along, across, band
             TempBoxDimSizes(1) = dimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(1)
             TempBoxDimSizes(3) = BoxDimensionSizes(2)
             numWavebands = dimensionSizes(1)
          end if
          allocate(boxTwoByteIntDataR3( TempBoxDimSizes(1),  &
                                        TempBoxDimSizes(2),  &
                                        TempBoxDimSizes(3))  )


          !Perform this one band at a time:
          do waveCounter = 0, numWavebands - 1
             if (CorrectOrder) then
                !Band, Along, Across
                call copyToBox(    BoxCorners = BoxCorners,            &
                                 CompleteData = TwoByteIntDataR3(:,:,waveCounter+1),  &
                                SubsettedData = boxTwoByteIntDataR3(:,:,waveCounter+1))
                call applyMask(   Data = boxTwoByteIntDataR3(:,:,waveCounter+1),    &
                                  Mask = Mask,                          &
                                origID = SDSID                          )
             else
                !along, across, band 
                call copyToBox(    BoxCorners = BoxCorners,            &
                                 CompleteData = TwoByteIntDataR3(waveCounter+1,:,:),  &
                                SubsettedData = boxTwoByteIntDataR3(waveCounter+1,:,:))
                call applyMask(   Data = boxTwoByteIntDataR3(waveCounter+1,:,:),    &
                                  Mask = Mask,                          &
                                origID = SDSID                          )
             end if
          end do
          
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = TempBoxDimSizes,         & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = TempBoxDimSizes,       &
                                   rank = rank,                  &
                              SDSdataR3 = boxTwoByteIntDataR3    )

          !Deallocate the box arrays:
          deallocate(boxTwoByteIntDataR3)

       else
          !Do not perform the subsetting.
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR3 = TwoByteIntDataR3       )
       end if

       !Deallocate the data arrays:
       deallocate(   TwoByteIntDataR3)

    case(2)
       !Rank = 2 (i.e. Two dimensional SDS):
       !Allocate the data arrays:
       allocate(   TwoByteIntDataR2(    dimensionSizes(1),  &
                                        dimensionSizes(2))  )
       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR2 = TwoByteIntDataR2    )

       !Subset the data, if acceptable format:
       if (SpatialSubset) then
          !allocate the box array:
          allocate(boxTwoByteIntDataR2( BoxDimensionSizes(1),  &
                                        BoxDimensionSizes(2))  )

          !Subset the data:
          call copyToBox(    BoxCorners = BoxCorners,         &
                           CompleteData = TwoByteIntDataR2,   &
                          SubsettedData = boxTwoByteIntDataR2 )
          call applyMask(   Data = boxTwoByteIntDataR2,       &
                            Mask = Mask,                      &
                           origID = SDSID                     )

          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = BoxDimensionSizes,       & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = BoxDimensionSizes,     &
                                   rank = rank,                  &
                              SDSdataR2 = boxTwoByteIntDataR2    )

          !Deallocate the box array:
          deallocate(boxTwoByteIntDataR2)

       else
          !Do not subset:
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR2 = TwoByteIntDataR2       )
       end if

       !Deallocate the data arrays:
       deallocate(   TwoByteIntDataR2)

    case(1)
       !Rank = 1 (Single Dimensional, no subsetting):
       !Allocate the arrays:
       allocate(  TwoByteIntDataR1( dimensionSizes(1) )  )

       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR1 = TwoByteIntDataR1    )
              
       !Create the SDS in the new HDF:
       call createSDS( inputFileID = inputFileID,             &
                          newHDFID = newHDFID,                &
                           SDSName = trim(SDSName),           &
                          dimSizes = dimensionSizes,          & 
                              rank = rank                     )
    
       !Output the SDS to the new HDF:
       call writeSDS( SampleDataType = SampleDataType,        &  
                            newHDFID = newHDFID,              &
                             SDSName = trim(SDSName),         &
                            dimSizes = dimensionSizes,        &
                                rank = rank,                  &
                           SDSdataR1 = TwoByteIntDataR1       )

       !Deallocate the data arrays:
       deallocate(   TwoByteIntDataR1)

    !end the case:
    end select

    !deallocate temporary box dimension array:
    deallocate(TempBoxDimSizes)
    
  end subroutine subsetTBIK

  !*****************************************************************************

  !*****************************************************************************
  ! Byte Case:
  !*****************************************************************************

  !*****************************************************************************
  
  subroutine subsetOBIK( SampleDataType,                               & 
                         filename, inputFileID,                        &
                         newHDFID, SDSName, SDSID,                     &
                         dimensionSizes, boxDimensionSizes, boxCorners,&
                         rank, SpatialSubset, CorrectOrder, Mask       )

    integer  (kind = OneByteIntKind),        intent(in)   :: SampleDataType
    character(len  = *),                     intent(in)   :: fileName
    integer  (kind = FourByteIntKind),       intent(in)   :: inputFileID
    integer  (kind = FourByteIntKind),       intent(in)   :: newHDFID 
    character(len  = *),                     intent(in)   :: SDSName
    integer  (kind = FourByteIntKind),       intent(in)   :: SDSID
    integer  (kind = FourByteIntKind),    &
              dimension(:),                  intent(in)   :: dimensionSizes, &
                                                             boxDimensionSizes
    integer  (kind = FourByteIntKind),    &
             dimension(:,:),                 intent(in)   :: BoxCorners
    integer  (kind = FourByteIntKind),       intent(in)   :: rank
    logical,                                 intent(in)   :: SpatialSubset, CorrectOrder
    integer  (kind = FourByteIntKind),    &
              dimension(:,:),                intent(in)   :: Mask
    ! !Description:
    !   Specific routine to read in a one byte integer SDS.
    !   This routine will read in the entire SDS, copy the designated box to an
    !     array the size of the box, mask the boxed data, create the SDS in the 
    !     new HDF file(and copy the SDS attributes and dimension names), then
    !     write out the masked data to the new SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    fileName       : Used for storing input HDF file and Path name.
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    SDSID          : Data handle for accessing the SDS being subsetted
    !    dimensionSizes : Used to store the dimension of an SDS.
    ! boxDimensionSizes : Used to store the dimensions of the subsetted box.
    !    BoxCorners     : Contains the indices of the subsetted box.
    !                       dimension (i,j):
    !                        i=1 : Upper Left            j=1: i index of box
    !                        i=2 : Upper Right           j=2: j index of box
    !                        i=3 : Lower Left 
    !                        i=4 : Lower Right
    !    rank           : Rank of the array; 1=single dimension, 2=onebanded, 3= multibanded.
    !    SpatialSubset  : Flag for seeing if the SDS is subsettable or not.
    !     CorrectOrder  : Flag to see if the SDS dimensions are in the correct order.
    !                     2D should be along, across
    !                     3D should be band, along, across
    !                     if 2D and flase, then order is across, along
    !                     if 3D and flase, then order is along, across, band
    !    Mask           : Array containing the mask.  The mask is used to replace unwanted
    !                     points in the subsetted box, with fill values.  This is accomplished
    !                     by setting the mask to 1 for desired points and 0 for unwanted
    !                     points.  Then the mask is applied to the subsetted SDS, in which
    !                     when the mask = 0, the fill value replaces the stored value.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The subroutines ReadSDS, copyToBox, applyMask, createSDS, and writeSDS
    !      are generic interfaces, accepting all kinds of data, etc.
    !   The arrays that store the SDS data and subsetted data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !local vars:
    integer  (kind = FourByteIntKind)                     :: numWavebands
    !numWavebands  : Contains the number of wave bands for this SDS.
    integer  (kind = FourByteIntKind),    &
              dimension(:),       allocatable             :: TempBoxDimSizes
    !TempBoxDimSizes : Allows the adjustment of the box dimensions.
    integer  (kind = OneByteIntKind ),    &
              dimension(:,:,:),   allocatable             ::      oneByteIntDataR3
    integer  (kind = OneByteIntKind ),    &
              dimension(:,:),     allocatable             ::      oneByteIntDataR2
    integer  (kind = OneByteIntKind ),    &
              dimension(:),       allocatable             ::      oneByteIntDataR1
    integer  (kind = OneByteIntKind ),    &
              dimension(:, :, :), allocatable             ::   boxOneByteIntDataR3
    integer  (kind = OneByteIntKind ),    &
              dimension(:, :),    allocatable             ::   boxOneByteIntDataR2
    ! The SDS will be read into these arrays, and then outputted.  They were 
    !  engineered to be allocatable, to allow for varying granule and SDS sizes.
    ! The Box array will contain the subsetted data. Rx stands for Rank X.

    !Allocate the temporary box dimensions array:
    allocate(TempBoxDimSizes(size(dimensionSizes)))
    TempBoxDimSizes = 0

    !Split the cases into ranks:
    select case(rank)
       
    case(3)
       !Rank = 3 (i.e. Multi-banded 3 dimensional):
       !Allocate the arrays:
       allocate(   OneByteIntDataR3(    dimensionSizes(1),  &
                                        dimensionSizes(2),  &
                                        dimensionSizes(3))  )
       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR3 = OneByteIntDataR3    )

       !Subset the data, if acceptable format:
       if (SpatialSubset) then       
          
          !Subset the data:
          !Allocate the box array:
          if (CorrectOrder) then
             !Band, Along, Across
             TempBoxDimSizes(1) = BoxDimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(2)
             TempBoxDimSizes(3) = dimensionSizes(3)
             numWavebands = TempBoxDimSizes(3)
          else
             !along, across, band
             TempBoxDimSizes(1) = dimensionSizes(1)
             TempBoxDimSizes(2) = BoxDimensionSizes(1)
             TempBoxDimSizes(3) = BoxDimensionSizes(2)
             numWavebands = dimensionSizes(1)
          end if
          allocate(boxOneByteIntDataR3( TempBoxDimSizes(1),  &
                                        TempBoxDimSizes(2),  &
                                        TempBoxDimSizes(3))  )

          !Subset the data:
          !Perform this one band at a time:
          do waveCounter = 0, numWavebands - 1
             if (CorrectOrder) then
                !Band, Along, Across
                call copyToBox(    BoxCorners = BoxCorners,              &
                                 CompleteData = OneByteIntDataR3(:,:,waveCounter+1),  &
                                SubsettedData = boxOneByteIntDataR3(:,:,waveCounter+1))
                call applyMask(   Data = boxOneByteIntDataR3(:,:,waveCounter+1),    &
                                  Mask = Mask,                           &
                                origID = SDSID                           )
             else
                !along, across, band 
                call copyToBox(    BoxCorners = BoxCorners,              &
                                 CompleteData = OneByteIntDataR3(waveCounter+1,:,:),  &
                                SubsettedData = boxOneByteIntDataR3(waveCounter+1,:,:))
                call applyMask(   Data = boxOneByteIntDataR3(waveCounter+1,:,:),    &
                                  Mask = Mask,                           &
                                origID = SDSID                           )
             end if
          end do

          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = TempBoxDimSizes,         & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = TempBoxDimSizes,       &
                                   rank = rank,                  &
                              SDSdataR3 = boxOneByteIntDataR3    )
       
          !Deallocate the box arrays:
          deallocate(boxOneByteIntDataR3)
          
       else
          !Do not perform the subsetting.
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,             &
                             newHDFID = newHDFID,                &
                              SDSName = trim(SDSName),           &
                             dimSizes = dimensionSizes,          & 
                                 rank = rank                     )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = dimensionSizes,        &
                                   rank = rank,                  &
                              SDSdataR3 = OneByteIntDataR3       )
       
       end if

       !Deallocate the data arrays:
       deallocate(   OneByteIntDataR3)

    case(2)
       !Rank = 2 (i.e. Two dimensional SDS):
       !Allocate the data arrays:
       allocate(   OneByteIntDataR2(    dimensionSizes(1),  &
                                        dimensionSizes(2))  )
       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR2 = OneByteIntDataR2    )

       !Subset the data, if acceptable format:
       if (SpatialSubset) then
          !allocate the box array:
          allocate(boxOneByteIntDataR2( BoxDimensionSizes(1),  &
                                        BoxDimensionSizes(2))  )

          !Subset the data:
          call copyToBox(    BoxCorners = BoxCorners,          &
                           CompleteData = OneByteIntDataR2,    &
                          SubsettedData = boxOneByteIntDataR2  )
          call applyMask(   Data = boxOneByteIntDataR2,        &
                            Mask = Mask,                       &
                           origID = SDSID                      )

          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,          &
                             newHDFID = newHDFID,             &
                              SDSName = trim(SDSName),        &
                             dimSizes = BoxDimensionSizes,    & 
                                 rank = rank                  )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,        &  
                               newHDFID = newHDFID,              &
                                SDSName = trim(SDSName),         &
                               dimSizes = BoxDimensionSizes,     &
                                   rank = rank,                  &
                              SDSdataR2 = boxOneByteIntDataR2    )
          !Deallocate the box array:
          deallocate(boxOneByteIntDataR2)

       else
          !Do not subset:
          !Create the SDS in the new HDF:
          call createSDS( inputFileID = inputFileID,          &
                             newHDFID = newHDFID,             &
                              SDSName = trim(SDSName),        &
                             dimSizes = dimensionSizes,       & 
                                 rank = rank                  )
    
          !Output the SDS to the new HDF:
          call writeSDS( SampleDataType = SampleDataType,     &  
                               newHDFID = newHDFID,           &
                                SDSName = trim(SDSName),      &
                               dimSizes = dimensionSizes,     &
                                   rank = rank,               &
                              SDSdataR2 = OneByteIntDataR2    )

       end if

       !Deallocate the data arrays:
       deallocate(   OneByteIntDataR2)

    case(1)
       !Rank = 1 (Single Dimensional, no subsetting):
       !Allocate the arrays:
       allocate(  OneByteIntDataR1( dimensionSizes(1) )  )

       !Read in the Data:
       call ReadSDS( SampleDataType = SampleDataType,     &
                          dataSetID = SDSID,              &
                           fileName = trim(fileName),     &
                        dataSetName = trim(SDSName),      &
                           dimSizes = dimensionSizes,     &
                          SDSDataR1 = OneByteIntDataR1    )
              
       !Create the SDS in the new HDF:
       call createSDS( inputFileID = inputFileID,             &
                          newHDFID = newHDFID,                &
                           SDSName = trim(SDSName),           &
                          dimSizes = dimensionSizes,          & 
                              rank = rank                     )
    
       !Output the SDS to the new HDF:
       call writeSDS( SampleDataType = SampleDataType,        &  
                            newHDFID = newHDFID,              &
                             SDSName = trim(SDSName),         &
                            dimSizes = dimensionSizes,        &
                                rank = rank,                  &
                           SDSdataR1 = OneByteIntDataR1       )

       !Deallocate the data arrays:
       deallocate(   OneByteIntDataR1)


    !end the case:
    end select

    !deallocate temporary box dimension array:
    deallocate(TempBoxDimSizes)
     
  end subroutine SubsetOBIK

  !*****************************************************************************
  !*****************************************************************************




end module subsetting



