module Output
  !*****************************************************************************
  ! !F90
  !
  ! !Description:
  !    This module contains the subroutines used to initialize and output data to
  !      the various types of output files (text, HDF).
  !    Included in the module are routines to initialize a new HDF file, create a
  !      SDS and write a single or multibanded SDS to the HDF file.
  !    There are also routines to intialize the Useful granule text file, and to
  !      write the name of a useful granule to this text file.
  !  
  ! !Callable routines:
  !    initializeGranuleFile()
  !    outputGranuleName()
  !    initializeNewHDF()
  !    createSDS()
  !    createHRGeo()
  !    writeSDS()
  ! 
  ! !Revision History:
  ! $Log: Output.f90,v $
  !
  ! Revision 1.0  1999/12/17  12:43:43  EGMoody
  ! Initial revision.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   There is a generic SDS writing interface (writeSDS) which is called to 
  !     write out an SDS.  The interface then calls the appropriate subsetting
  !     subroutine based on:
  !        1)  The data type of a sample data type
  !        2)  The number of dimensions of the variable SDSDate.  If two dimensions
  !              are present, then it is singlebanded, if three, multibanded.
  !   There are two routines to create SDS in the new HDF file.  createSDS will 
  !     create an SDS for any resolution.  createHRGeo is a specialized form of
  !     createSDS, which will create the high resolution Geolocation SDS.
  !   This is a list of the following SDS writing subroutines:
  !        note:  IK = Integer Kind, RK = Real Kind
  !               OB = One Byte, TB = Two Byte, FB = Four Byte, EB = Eight Byte
  !
  !               writeOBIK,   writeTBIK,   writeFBIK
  !               writeFBRK,   writeEBRK,
  !
  ! !END
  !*****************************************************************************

  !Dependencies:
  use typeSizes, only  :  OneByteIntKind, TwoByteIntKind, FourByteIntKind, &
                          FourByteRealKind, EightByteRealKind
  use Inputs,    only  :  maxFileNameLength, MaxAttributeLength,           &
                          LocationName, OutputPath, GranNameFile
  use HDF,       only  :  Max_Var_Dims,                                    &
                          SFstart, SFend,                                  &
                          SFcreate,  SFn2index, SFselect, SFwdata,         &
                          SFdimid, SFsdmname, SFendacc,                    &
                          SFfinfo, SFginfo, SFgdinfo, SFgainfo,            & 
                          SFfattr, SFrcatt, SFscatt, SFrnatt, SFsnatt,     &
                          DFACC_CREATE, DFACC_WRITE, FAIL,                 &
                          DFNT_CHAR, DFNT_FLOAT, DFNT_DOUBLE,              &
                          DFNT_INT8, DFNT_UINT8,                           &
                          DFNT_INT16, DFNT_UINT16,                         &
                          DFNT_INT32, DFNT_UINT32


  implicit none

  private

  !Generic Interface:
  interface writeSDS
     module procedure writeOBIK, writeTBIK, writeFBIK, writeFBRK, writeEBRK
  end interface
  !subroutine write*B*K( SampleDataType, newHDFID, SDSName, dimSizes, rank, &
  !                      SDSdataR1,SDSdataR2,SDSdataR3      )
  !
  !  *******   (kind = ******Byte****Kind), intent( in)    :: SampleDataType
  !  integer   (kind = FourByteIntKind),    intent( in)    :: newHDFID 
  !  character (len = *),                   intent( in)    :: SDSName
  !  integer   (kind = FourByteIntKind),    &
  !             dimension(:),               intent( in)    :: dimSizes
  !  integer   (kind = FourByteIntKind),    intent( in)    :: rank
  !  *******   (kind = ******Byte****Kind ),&
  !             dimension(:),     optional  intent( in)    :: SDSdataR1
  !  *******   (kind = ******Byte****Kind ),&
  !             dimension(:,:),   optional, intent( in)    :: SDSdataR2
  !  *******   (kind = ******Byte****Kind ),&
  !             dimension(:,:,:), optional, intent( in)    :: SDSdataR3
  !
  ! !Description:
  !   This routine will output the entire SDS to the new HDF file.  The generic
  !     interface determines the appropriate subsetting subroutine based on:
  !        1)  The data type of a sample data type
  !  
  ! !Input Parameters:
  !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
  !                       interface to determine which specific routine to call.
  !    newHDFID       : File handle for an output HDF file, used by HDF routines.
  !    SDSName        : Stores the name of the SDS being subsetted
  !    dimSizes       : Used to store the dimension of an SDS.
  !    rank           : The rank of the SDS in the HDF granule.
  !    SDSdataRx      : The subsetted data to be outputted to the HDF file.
  !                        Rx stands for Rank X.
  !
  ! !Revision History:
  !   See Module revision history at the beginning of the file.
  !
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   The arrays that store the subsetted SDS data are allocatable, thereby
  !      being flexible enough to handle any size for the SDS.
  !
  ! !END


  public :: initializeGranuleFile, outputGranuleName
  public :: initializeNewHDF
  public :: createSDS, createHRGeo, writeSDS



  !local variables:
  !counters:
  integer  (kind = fourByteIntKind)             :: i,j,k

  !HDF variables:
  integer  (kind = FourByteIntKind)             :: status
  integer  (kind = FourByteIntKind  ),  &
             dimension(MAX_VAR_DIMS)            :: hdfStart, hdfStride, hdfEdge
  integer  (kind = FourByteIntKind)             :: sds_id, sds_index
  !  status   : Used for error checking.
  !  hdfstart : Follows HDF conventions. The array element from which to begin reading the
  !              HDF array. Zero based.
  !  hdfstride: Follows HDF conventions. The frequency with which data should be read in
  !              each dimension. Stride 1 means read every element.
  !  hdfedge  : Follows HDF conventions. The number of elements to read in each dimension. 
  !              If start(:) == 0 and stride(:) == 1, setting edge equal to the shape of 
  !              the underlying array reads all elements. 

contains



  ! ----------------------------------------------------------------------------------
  ! Granule name file routines:
  ! ----------------------------------------------------------------------------------
  subroutine initializeGranuleFile
    ! !Description:
    !   This routine initializes the useful granule text file.  The file is meant as
    !    a packing list for the new HDF files that were created.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !local variables:
    integer  (kind = FourByteIntKind)             :: GenLUN
    integer  (kind = FourByteIntKind)             :: ierr
    ! GenLUN         : The file handle used in OPEN, READ, CLOSE F90 statements.
    ! ierr            : Contains value for IOSTAT. 0=Success, otherwise=Fail

    !open the formatted granule name file:
    GenLUN = 1005
    open(GenLUN, file = trim(GranNameFile), form = "formatted", status = "replace", &
               iostat = ierr)

    !Check for an error opening this file:
    if (ierr /= 0) then
       Print *, "Could not open", trim(GranNameFile)
       Stop
    end if

    !close the formatted granule name file:
    close(GenLUN)
    
  end subroutine initializeGranuleFile

  ! ----------------------------------------------------------------------------------

  subroutine outputGranuleName(granuleName)
    character (len = *), intent( in)                  :: granuleName
    ! !Description:
    !   This routine writes the name of a useful granule to the 
    !    useful granule text file.  The file is meant as
    !    a packing list for the new HDF files that were created.
    !
    ! !Inputs:
    !  GranuleName:  The name of the useful granule file to be written
    !                  to the file.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !local variables:
    integer  (kind = FourByteIntKind)             :: GenLUN
    character(len  = MaxFileNameLength)           :: Gen_File
    integer  (kind = FourByteIntKind)             :: ierr
    integer  (kind = fourByteIntKind)             :: lengthOfString
    ! GenLUN         : The file handle used in OPEN, READ, CLOSE F90 statements.
    ! Gen_File       : Used for storing File and Path names.
    ! ierr           : Contains value for IOSTAT. 0=Success, otherwise=Fail
    ! lengthOfString : Used to store the length of a character string.

    !open the formatted granule name file:
    GenLUN = 1005
    open(GenLUN, file = trim(GranNameFile), form = "formatted", status = "old", &
         position = "append", iostat = ierr)

    !Check for an error opening this file:
    if (ierr /= 0) then
       Print *, "Could not open", trim(GranNameFile)
       Stop
    end if

    !write the name of the granule to the granule name file:
    lengthOfString = len(trim(granuleName))
    write(Gen_File,'(A,A)') trim(LocationName),trim(granuleName)
    write(GenLUN,*) trim(Gen_File)

    !close the formatted granule name file:
    close(GenLUN)

  end subroutine outputGranuleName
  ! ----------------------------------------------------------------------------------
  ! ----------------------------------------------------------------------------------


  ! ----------------------------------------------------------------------------------
  ! new HDF output routines:
  ! ----------------------------------------------------------------------------------

  subroutine initializeNewHDF(newHDFfilename, inputFileID, newHDFID)
    character (len = *),                intent( in)  :: newHDFfilename
    integer   (kind = FourByteIntKind), intent( in)  :: inputFileID
    integer   (kind = FourByteIntKind), intent(out)  :: newHDFID
    ! !Description:
    !   This routine will initialize the new HDF file and copy over the global
    !    attributes (such as the metadata) from the original granule.
    !  
    ! !Input Parameters:
    !    newHDFfilename : The name, only, of the new HDF file.
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !
    ! !Output Parameters:
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The number of global attributes is determined and then the 
    !    subroutine "copyAttr" is called to copy each attribute to the
    !    new HDF file.
    !
    ! !END

    !local variables:
    character(len  = MaxFileNameLength)              :: Gen_File
    integer (kind = FourByteIntKind)                 :: ndatasets, nglobal_attr
    character(len  = MaxFileNameLength)              :: newHDFwithPath
    ! Gen_File       : Used for storing File and Path names.
    ! ndatasets      : return from sfginfo, contains the number of datasets.
    ! nglobal_attr   : return from sfginfo, contains the number of global attributes.
    ! newHDFwithPath : Contains the full path and file name for the new HDF file.

    !define the new HDF filename with path:
    write(Gen_File,'(A,A,A,A)') trim(OutputPath),"/",trim(LocationName),trim(newHDFfilename)
    newHDFwithPath = trim(Gen_File)

    !create the new HDF file:
    newHDFID = SFstart(trim(newHDFwithPath),DFACC_CREATE)
    if (newHDFID == Fail) then
       Print *, "Can't Create the NEW HDF File", trim(newHDFwithPath)
       stop
    end if

    !Close the new HDF file:
    status = SFend(newHDFID)
    if (status == fail) then
       Print *, "Can't End access to the NEW HDF File", trim(newHDFwithPath)
       stop
    end if

    !reopen the HDF file:
    newHDFID = SFstart(trim(newHDFwithPath),DFACC_WRITE)
    if (newHDFID == Fail) then
       Print *, "Can't Reopen the NEW HDF File", trim(newHDFwithPath)
       stop
    end if

    !determine the number of global attributes:
    status = SFFinfo(inputFileID,ndatasets,nglobal_attr)
    if (status == fail) then
       Print *, "Can't obtain the number of glabal attributes for the orig. HDF"
       stop
    end if

    !loop through the attributes, copying each one to the new HDF file:
    do i = 0, nglobal_attr-1

       call copyAttr(    origID = inputFileID,   &
                          newID = newHDFID,      &
                      attrIndex = i              )
       
    end do
    
    
  end subroutine initializeNewHDF

  ! ----------------------------------------------------------------------------------

  subroutine createSDS(inputFileID, newHDFID, SDSName, dimSizes, rank)
    integer   (kind = FourByteIntKind), intent( in)  :: inputFileID
    integer   (kind = FourByteIntKind), intent( in)  :: newHDFID 
    character (len = *),                intent( in)  :: SDSName
    integer   (kind = FourByteIntKind),    &
               dimension(*),            intent( in)  :: dimSizes
    integer   (kind = FourByteIntKind), intent( in)  :: rank
    ! !Description:
    !   This routine will create the desired SDS in the new HDF file.  Upon
    !    successfully creating the SDS, the local SDS dimension names and 
    !    attributes are copied to the new HDF file.
    !  
    ! !Input Parameters:
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    dimSizes       : Used to store the dimension of an SDS.
    !    rank           : The rank of the SDS in the HDF granule.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !local variables:
    integer (kind = FourByteIntKind)                 :: newsds_id = 0
    integer (kind = FourByteIntKind)                 :: dim_id = 0, newdim_id = 0
    character (len = maxattributelength)             :: dim_name = "", dummySDSName = ""
    integer (kind = FourByteIntKind)                 :: dim_number=0, dim_size = 0

    integer (kind = FourByteIntKind)                 :: dummyrank = 0,        &
                                                        numberType = 0,       &
                                                        nattrs = 0
    integer (kind = FourByteIntKind),    &
              dimension(MAX_VAR_DIMS)                :: dummyDimSz = 0
    ! newsds_id   : Data handle for accessing the SDS being created.
    ! dim_id,      
    !  newdim_id  : Data handle for accessing the dimension.
    ! dim_name    : Name of the dimension.
    ! dim_number  : The number of the dimension (starts from 0).
    ! dim_size    : The size of the dimension.
    ! numberType  : Stores the integer value of the data type.
    !                 ( hdf include file lists the types as:  
    !                    Double = DFNT_FLOAT64 =  6
    !                     Float = DFNT_FLOAT32 =  5
    !                     Short = DFNT_INT16   = 22
    !                      Byte = DFNT_INT8    = 20 )
    ! nattrs      : Number of attributes.
    ! dummy vars  : These are dummy variables used to store extra 
    !                 information obtained from calls to sfginfo.

    !
    !obtain information about the original SDS:
    !

    !determine the sds_index for the SDS:
    sds_index = SFn2index( inputFileID, trim(SDSName) )
    if (sds_index == fail) then
       Print *, "Can't obtain the SDS ID for SDS", trim(SDSName)
       stop
    end if

    !get access to this SDS:
    sds_id = SFselect(inputFileID,sds_index)
    if (sds_id == fail) then
       Print *, "Can't obtain access (ID) to the SDS", trim(SDSName)
       stop
    end if

    !get general info:
    status = SFginfo(sds_id,dummySDSName,dummyrank,DummyDimSz,numberType,nattrs)
    if (status == fail) then
       Print *, "Can't obtain the general info. for SDS", trim(SDSName)
       stop
    end if


    !
    !create a new SDS in the new HDF:
    !
    !create the new SDS
    newsds_id = SFcreate(newHDFID, trim(SDSName), numberType, rank, dimsizes(1:rank))
    if (newsds_id == fail) then
       Print *, "Can't create the new SDS", trim(SDSName)
       stop
    end if
    
    !set the names of the dimensions of the new SDS:
    do i = 0, rank-1

       !obtain the name of the original dimension:
       dim_id = sfdimid(sds_id,i)
       if (dim_id == fail) then
          Print *, "Can't obtain the dimension ID for SDS", trim(SDSName)
          stop
       end if
       status = sfgdinfo(dim_id,dim_name,dim_size,numberType,dim_number)
       if (status == fail) then
          Print *, "Can't obtain the dimension info. for SDS", trim(SDSName)
          stop
       end if

       !rename the dimension of the new SDS:
       newdim_id = sfdimid(newsds_id,i)
       if (newdim_id == fail) then
          Print *, "Can't obtain the new dimension ID for SDS", trim(SDSName)
          stop
       end if

       status = sfsdmname(newdim_id, trim(dim_name))
       if (status == fail) then
          Print *, "Can't name the new dimension for dimension", trim(dim_name)
          stop
       end if
    
    end do
    
    !Copy SDS attributes:
    do i = 0, nattrs-1

       call copyAttr(    origID = sds_ID,        &
                          newID = newsds_id,     &
                      attrIndex = i              )

    end do


    !end access to this SDS:
    status = sfendacc(newsds_id)
    if (status == fail) then
       Print *, "Can't end access to the new SDS", trim(SDSName)
       stop
    end if


  end subroutine createSDS
  
  ! ----------------------------------------------------------------------------------

  subroutine createHRGeo(inputFileID, newHDFID, SDSName, SDSCopyName, HRSDSID, &
                         dimSizes, rank)
    integer   (kind = FourByteIntKind), intent( in)  :: inputFileID
    integer   (kind = FourByteIntKind), intent( in)  :: newHDFID 
    character (len = *),                intent( in)  :: SDSName, SDSCopyName
    integer   (kind = FourByteIntKind), intent( in)  :: HRSDSID
    integer   (kind = FourByteIntKind),    &
               dimension(*),            intent( in)  :: dimSizes
    integer   (kind = FourByteIntKind), intent( in)  :: rank
    ! !Description:
    !   This routine will create the desired high resolutions Geolocation in 
    !    the new HDF file.  It copies the local attributes from the low res
    !    Geolocation, and it copies the dimension names from a high resolution
    !    sample SDS.
    !  
    ! !Input Parameters:
    !    inputFileID    : File handle for an  input HDF file, used by HDF routines.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    SDSCopyName    : Stores the name of the SDS to copy the attributes from.
    !    HRSDSID        : Stores the SDS ID of the high resolution SDS to
    !                        copy the names of the dimesions from.
    !    dimSizes       : Used to store the dimension of an SDS.
    !    rank           : The rank of the SDS in the HDF granule.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !local variables:
    integer (kind = FourByteIntKind)                 :: newsds_id
    integer (kind = FourByteIntKind)                 :: dim_id, newdim_id
    character (len = maxattributelength)             :: dim_name, dummySDSName = ""
    integer (kind = FourByteIntKind)                 :: dim_number, dim_size

    integer (kind = FourByteIntKind)                 :: dummyrank = 0,        &
                                                        numberType = 0,       &
                                                        nattrs = 0
    integer (kind = FourByteIntKind),    &
              dimension(MAX_VAR_DIMS)                :: dummyDimSz = 0
    ! newsds_id   : Data handle for accessing the SDS being created.
    ! dim_id,      
    !  newdim_id  : Data handle for accessing the dimension.
    ! dim_name    : Name of the dimension.
    ! dim_number  : The number of the dimension (starts from 0).
    ! dim_size    : The size of the dimension.
    ! numberType  : Stores the integer value of the data type.
    !                 ( hdf include file lists the types as:  
    !                    Double = DFNT_FLOAT64 =  6
    !                     Float = DFNT_FLOAT32 =  5
    !                     Short = DFNT_INT16   = 22
    !                      Byte = DFNT_INT8    = 20 )
    ! nattrs      : Number of attributes.
    ! dummy vars  : These are dummy variables used to store extra 
    !                 information obtained from calls to sfginfo.

    !
    !obtain information about the SDS to copy local attributes from:
    !

    !determine the sds_index for the SDS to copy from:
    sds_index = SFn2index( inputFileID, trim(SDSCopyName) )
    if (sds_index == fail) then
       Print *, "Can't obtain the SDS ID for SDS", trim(SDSCopyName)
       stop
    end if

    !get access to this SDS:
    sds_id = SFselect(inputFileID,sds_index)
    if (sds_id == fail) then
       Print *, "Can't obtain access (ID) to the SDS", trim(SDSCopyName)
       stop
    end if

    !get general info:
    status = SFginfo(sds_id,dummySDSName,dummyrank,DummyDimSz,numberType,nattrs)
    if (status == fail) then
       Print *, "Can't obtain the general info. for SDS", trim(SDSCopyName)
       stop
    end if

    !
    !create a new SDS in the new HDF:
    !
    !create the new SDS
    newsds_id = SFcreate(newHDFID, trim(SDSName), numberType, rank, dimsizes(1:rank))
    if (newsds_id == fail) then
       Print *, "Can't create the new SDS", trim(SDSName)
       stop
    end if

    !set the names of the dimensions of the new SDS (by copying them from
    !   the sample high resolution SDS):
    do i = 0, rank-1

       !obtain the name of the original dimension from the sample:
       dim_id = sfdimid(HRSDSID,i)
       if (dim_id == fail) then
          Print *, "Can't obtain the dimension ID for the high res sample"
          stop
       end if
       status = sfgdinfo(dim_id,dim_name,dim_size,numberType,dim_number)
       if (status == fail) then
          Print *, "Can't obtain the dimension info. for the high res sample"
          stop
       end if

       !rename the dimension of the new SDS:
       newdim_id = sfdimid(newsds_id,i)
       if (newdim_id == fail) then
          Print *, "Can't obtain the new dimension ID for SDS", trim(SDSName)
          stop
       end if
       status = sfsdmname(newdim_id, trim(dim_name))
       if (status == fail) then
          Print *, "Can't name the new dimension for dimension", trim(dim_name)
          stop
       end if
    
    end do
    
    !Copy SDS attributes:
    do i = 0, nattrs-1

       call copyAttr(    origID = sds_ID,        &
                          newID = newsds_id,     &
                      attrIndex = i              )

    end do


    !end access to this SDS:
    status = sfendacc(newsds_id)
    if (status == fail) then
       Print *, "Can't end access to the new SDS", trim(SDSName)
       stop
    end if


  end subroutine createHRGeo

  ! ----------------------------------------------------------------------------------
  ! writeSDS routines:
  ! ----------------------------------------------------------------------------------

  ! ----------------------------------------------------------------------------------
  subroutine writeOBIK( SampleDataType, newHDFID, SDSName, dimSizes, rank, &
                        SDSdataR1, SDSdataR2, SDSdataR3 )

    integer   (kind = OneByteIntKind),     intent( in)    :: SampleDataType
    integer   (kind = FourByteIntKind),    intent( in)    :: newHDFID 
    character (len = *),                   intent( in)    :: SDSName
    integer   (kind = FourByteIntKind),    &
               dimension(:),               intent( in)    :: dimSizes
    integer   (kind = FourByteIntKind),    intent( in)    :: rank
    integer   (kind = OneByteIntKind ),    &
               dimension(:),     optional, intent( in)    :: SDSdataR1
    integer   (kind = OneByteIntKind ),    &
               dimension(:,:),   optional, intent( in)    :: SDSdataR2
    integer   (kind = OneByteIntKind ),    &
               dimension(:,:,:), optional, intent( in)    :: SDSdataR3
    ! !Description:
    !   This routine will output the entire SDS to the new HDF file.  
    !   This specific routine is for a one byte integer SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    dimSizes       : Used to store the dimension of an SDS.
    !    rank           : The rank of the SDS in the HDF granule.
    !    SDSdataRx      : The subsetted data to be outputted to the HDF file.
    !                         The Rx stands for Rank X.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The arrays that store the subsetted SDS data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !initialize the hdf variables:
    hdfStart  = 0
    hdfStride = 1
    hdfEdge   = 1

    !define the hdf variables used:
    hdfEdge (1:rank) = DimSizes(1:rank)

    !determine the sds_index for the SDS:
    sds_index = SFn2index( newHDFID, trim(SDSName) )

    !get access to this SDS:
    sds_id = SFselect(newHDFID,sds_index)

    !write the data to the new HDF file:
    if (rank==1) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR1)
    else if (rank==2) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR2)
    else
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR3)
    end if
    
    !error checking:
    if (sds_index == fail .or. &
        sds_id    == fail .or. &
        status    == fail)     then
       print *,'Could not write data for SDS', trim(SDSName)
       stop
    end if
    !end access to this SDS:
    status = sfendacc(sds_id)
    if (status  == fail) then
       print *,'Could not end access to SDS', trim(SDSName)
       stop
    end if

  end subroutine writeOBIK

  ! ----------------------------------------------------------------------------------
  
  subroutine writeTBIK( SampleDataType, newHDFID, SDSName, dimSizes, rank, &
                        SDSdataR1, SDSdataR2, SDSdataR3 )

    integer   (kind = TwoByteIntKind),     intent( in)    :: SampleDataType
    integer   (kind = FourByteIntKind),    intent( in)    :: newHDFID 
    character (len = *),                   intent( in)    :: SDSName
    integer   (kind = FourByteIntKind),    &
               dimension(:),               intent( in)    :: dimSizes
    integer   (kind = FourByteIntKind),    intent( in)    :: rank
    integer  (kind =  TwoByteIntKind  ),   &
               dimension(:),     optional, intent( in)    :: SDSdataR1
    integer  (kind =  TwoByteIntKind  ),   &
               dimension(:,:),   optional, intent( in)    :: SDSdataR2
    integer  (kind =  TwoByteIntKind  ),   &
               dimension(:,:,:), optional, intent( in)    :: SDSdataR3
    ! !Description:
    !   This routine will output the entire SDS to the new HDF file.  
    !   This specific routine is for a two byte integer SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    dimSizes       : Used to store the dimension of an SDS.
    !    rank           : The rank of the SDS in the HDF granule.
    !    SDSdataRx      : The subsetted data to be outputted to the HDF file.
    !                         The Rx stands for Rank X.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The arrays that store the subsetted SDS data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !initialize the hdf variables:
    hdfStart  = 0
    hdfStride = 1
    hdfEdge   = 1

    !define the hdf variables used:
    hdfEdge (1:rank) = DimSizes(1:rank)

    !determine the sds_index for the SDS:
    sds_index = SFn2index( newHDFID, trim(SDSName) )

    !get access to this SDS:
    sds_id = SFselect(newHDFID,sds_index)

    !write the data to the new HDF file:
    if (rank==1) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR1)
    else if (rank==2) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR2)
    else
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR3)
    end if

    !error checking:
    if (sds_index == fail .or. &
        sds_id    == fail .or. &
        status    == fail)     then
       print *,'Problems writing data for SDS', trim(SDSName)
       stop
    end if

    !end access to this SDS:
    status = sfendacc(sds_id)
    if (status  == fail) then
       print *,'Could not end access to SDS', trim(SDSName)
       stop
    end if

  end subroutine writeTBIK

  ! ----------------------------------------------------------------------------------
  
  subroutine writeFBIK( SampleDataType, newHDFID, SDSName, dimSizes, rank, &
                        SDSdataR1, SDSdataR2, SDSdataR3 )

    integer   (kind = FourByteIntKind),    intent( in)    :: SampleDataType
    integer   (kind = FourByteIntKind),    intent( in)    :: newHDFID 
    character (len = *),                   intent( in)    :: SDSName
    integer   (kind = FourByteIntKind),    &
               dimension(:),               intent( in)    :: dimSizes
    integer   (kind = FourByteIntKind),    intent( in)    :: rank
    integer   (kind = FourByteIntKind ),   &
               dimension(:),     optional, intent( in)    :: SDSdataR1
    integer   (kind = FourByteIntKind ),   &
               dimension(:,:),   optional, intent( in)    :: SDSdataR2
    integer   (kind = FourByteIntKind ),   &
               dimension(:,:,:), optional, intent( in)    :: SDSdataR3
    ! !Description:
    !   This routine will output the entire SDS to the new HDF file.  
    !   This specific routine is for a four byte integer SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    dimSizes       : Used to store the dimension of an SDS.
    !    rank           : The rank of the SDS in the HDF granule.
    !    SDSdataRx      : The subsetted data to be outputted to the HDF file.
    !                         The Rx stands for Rank X.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The arrays that store the subsetted SDS data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !initialize the hdf variables:
    hdfStart  = 0
    hdfStride = 1
    hdfEdge   = 1

    !define the hdf variables used:
    hdfEdge (1:rank) = DimSizes(1:rank)

    !determine the sds_index for the SDS:
    sds_index = SFn2index( newHDFID, trim(SDSName) )

    !get access to this SDS:
    sds_id = SFselect(newHDFID,sds_index)

    !write the data to the new HDF file:
    if (rank==1) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR1)
    else if (rank==2) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR2)
    else
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR3)
    end if
    
    !error checking:
    if (sds_index == fail .or. &
        sds_id    == fail .or. &
        status    == fail)     then
       print *,'Problems writing data for SDS', trim(SDSName)
       stop
    end if

    !end access to this SDS:
    status = sfendacc(sds_id)
    if (status  == fail) then
       print *,'Could not end access to SDS', trim(SDSName)
       stop
    end if

  end subroutine writeFBIK
  
  ! ----------------------------------------------------------------------------------
  
  subroutine writeFBRK( SampleDataType, newHDFID, SDSName, dimSizes, rank, &
                        SDSdataR1, SDSdataR2, SDSdataR3 )

    real      (kind = FourByteRealKind),   intent( in)    :: SampleDataType
    integer   (kind = FourByteIntKind),    intent( in)    :: newHDFID 
    character (len = *),                   intent( in)    :: SDSName
    integer   (kind = FourByteIntKind),    &
               dimension(:),               intent( in)    :: dimSizes
    integer   (kind = FourByteIntKind),    intent( in)    :: rank
    real      (kind = FourByteRealKind ),  &
               dimension(:),     optional, intent( in)    :: SDSdataR1
    real      (kind = FourByteRealKind ),  &
               dimension(:,:),   optional, intent( in)    :: SDSdataR2
    real      (kind = FourByteRealKind ),  &
               dimension(:,:,:), optional, intent( in)    :: SDSdataR3
    ! !Description:
    !   This routine will output the entire SDS to the new HDF file.  
    !   This specific routine is for a four byte real SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    dimSizes       : Used to store the dimension of an SDS.
    !    rank           : The rank of the SDS in the HDF granule.
    !    SDSdataRx      : The subsetted data to be outputted to the HDF file.
    !                         The Rx stands for Rank X.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The arrays that store the subsetted SDS data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !initialize the hdf variables:
    hdfStart  = 0
    hdfStride = 1
    hdfEdge   = 1

    !define the hdf variables used:
    hdfEdge (1:rank) = DimSizes(1:rank)

    !determine the sds_index for the SDS:
    sds_index = SFn2index( newHDFID, trim(SDSName) )

    !get access to this SDS:
    sds_id = SFselect(newHDFID,sds_index)

    !write the data to the new HDF file:
    if (rank==1) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR1)
    else if (rank==2) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR2)
    else
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR3)
    end if
    
    !error checking:
    if (sds_index == fail .or. &
        sds_id    == fail .or. &
        status    == fail)     then
       print *,'Problems writing data for SDS', trim(SDSName)
       stop
    end if

    !end access to this SDS:
    status = sfendacc(sds_id)
    if (status  == fail) then
       print *,'Could not end access to SDS', trim(SDSName)
       stop
    end if

  end subroutine writeFBRK
  
  ! ----------------------------------------------------------------------------------
  
  subroutine writeEBRK( SampleDataType, newHDFID, SDSName, dimSizes, rank, &
                        SDSdataR1, SDSdataR2, SDSdataR3 )

    real      (kind = EightByteRealKind),  intent( in)    :: SampleDataType
    integer   (kind = FourByteIntKind),    intent( in)    :: newHDFID 
    character (len = *),                   intent( in)    :: SDSName
    integer   (kind = FourByteIntKind),    &
               dimension(*),               intent( in)    :: dimSizes
    integer   (kind = FourByteIntKind),    intent( in)    :: rank
    real      (kind = EightByteRealKind ), &
               dimension(:),     optional, intent( in)    :: SDSdataR1
    real      (kind = EightByteRealKind ), &
               dimension(:,:),   optional, intent( in)    :: SDSdataR2
    real      (kind = EightByteRealKind ), &
               dimension(:,:,:), optional, intent( in)    :: SDSdataR3
    ! !Description:
    !   This routine will output the entire SDS to the new HDF file.  
    !   This specific routine is for an eight byte real SDS.
    !  
    ! !Input Parameters:
    !    SampleDataType : Dummy scalar of the data type of the SDS.  Used by the 
    !                       interface to determine which specific routine to call.
    !    newHDFID       : File handle for an output HDF file, used by HDF routines.
    !    SDSName        : Stores the name of the SDS being subsetted
    !    dimSizes       : Used to store the dimension of an SDS.
    !    rank           : The rank of the SDS in the HDF granule.
    !    SDSdataRx      : The subsetted data to be outputted to the HDF file.
    !                         The Rx stands for Rank X.
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   The arrays that store the subsetted SDS data are allocatable, thereby
    !      being flexible enough to handle any size for the SDS.
    !
    ! !END

    !initialize the hdf variables:
    hdfStart  = 0
    hdfStride = 1
    hdfEdge   = 1

    !define the hdf variables used:
    hdfEdge (1:rank) = DimSizes(1:rank)

    !determine the sds_index for the SDS:
    sds_index = SFn2index( newHDFID, trim(SDSName) )

    !get access to this SDS:
    sds_id = SFselect(newHDFID,sds_index)

    !write the data to the new HDF file:
    if (rank==1) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR1)
    else if (rank==2) then
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR2)
    else
       status = SFwdata(sds_id, hdfStart,hdfStride,hdfEdge,SDSdataR3)
    end if
    
    !error checking:
    if (sds_index == fail .or. &
        sds_id    == fail .or. &
        status    == fail)     then
       print *,'Problems writing data for SDS', trim(SDSName)
       stop
    end if

    !end access to this SDS:
    status = sfendacc(sds_id)
    if (status  == fail) then
       print *,'Could not end access to SDS', trim(SDSName)
       stop
    end if
  end subroutine writeEBRK
  
  ! ----------------------------------------------------------------------------------

  ! ----------------------------------------------------------------------------------
  ! ----------------------------------------------------------------------------------

  subroutine copyAttr( origID, newID, attrIndex)
    
    integer   (kind = FourByteIntKind), intent( in)  :: origID
    integer   (kind = FourByteIntKind), intent( in)  :: newID 
    integer   (kind = FourByteIntKind), intent( in)  :: attrIndex
    ! !Description:
    !   This routine will copy a attribute (local SDS or global) from the 
    !    original HDF file, to the new HDF file.
    !   The routine was designed to copy only a single attribute.  Multiple
    !    calls of this routine are required to copy multiple attributes.
    !  
    ! !Input Parameters:
    !    origID       : File handle for original HDF file, used by HDF routines.
    !    newID        : File handle for new HDF file, used by HDF routines.
    !    attrIndex    : The index of the attribute (start from zero).
    !
    ! !Revision History:
    !   See Module revision history at the beginning of the file.
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END

    !local variables:
    integer (kind = FourByteIntKind)                 :: attrID
    character (len = maxattributelength)             :: attrName

    character (len = maxattributelength)             ::   charAttr
    integer (kind =  OneByteIntKind),    &
              dimension(:), allocatable              ::   byteAttr
    integer (kind =  TwoByteIntKind),    &
              dimension(:), allocatable              ::   shortAttr
    integer (kind = FourByteIntKind),    &
              dimension(:), allocatable              ::   longAttr
    real    (kind = FourByteRealKind),   &
              dimension(:), allocatable              ::  floatAttr
    real    (kind = EightByteRealKind),  &
              dimension(:), allocatable              :: doubleAttr
    integer (kind = FourByteIntKind)                 :: numberType = 0
    integer (kind = FourByteIntKind)                 :: count

       
    !obtain information about the original attribute:
    !  get the name, type, and count
    status = SFgainfo(origID,attrIndex,attrName,numberType,count)
    if (status == fail) then
       print *,'Could not get info about the attribute', trim(attrName)
       stop
    end if

    !obtain the attribute ID:
    attrID = SFfattr(origID, trim(attrName))
    if (attrID == fail) then
       print *,'Could not obtain the attribute ID for', trim(attrName)
       stop
    end if

    !Make sure that integer number types are not UINT:
    if (numberType == DFNT_UINT8) then
        numberType =  DFNT_INT8
    end if
    if (numberType == DFNT_UINT16) then
        numberType =  DFNT_INT16
    end if
    if (numberType == DFNT_UINT32) then
        numberType =  DFNT_INT32
    end if
    
    if (numberType == DFNT_CHAR) then
       !character
       charAttr = ""
       !read in the original attribute value:
       status = sfrcatt(origID, attrID, charAttr)
       if (status == fail) then
          print *,'Could not read the original attribute', trim(attrName)
          stop
       end if

       !write the attribute value to the new HDF:
       status = sfscatt(newID, trim(attrName), numberType, count, trim(charAttr))
       if (status == fail) then
          print *,'Could not write the attribute', trim(attrName)
          stop
       end if
    else
       
       if (numberType == DFNT_FLOAT) then
          !float
          !allocate the dimension:
          allocate (floatAttr(1:count))
          !read in the original attribute value:
          status = sfrnatt(origID, attrID, floatAttr)
          if (status == fail) then
             print *,'Could not read in the original attribute', trim(attrName)
             stop
          end if
          !write the attribute value to the new HDF:
          status = sfsnatt(newID, trim(attrName), numberType, count, floatAttr)
          if (status == fail) then
             print *,'Could not write the attribute', trim(attrName)
             stop
          end if
          !deallocate the dimension:
          deallocate (floatAttr)
          
       else if (numberType == DFNT_DOUBLE) then
          !double
          !allocate the dimension:
          allocate (doubleAttr(1:count))
          !read in the original attribute value:
          status = sfrnatt(origID, attrID, doubleAttr)
          if (status == fail) then
             print *,'Could not read in the original attribute', trim(attrName)
             stop
          end if
          !write the attribute value to the new HDF:
          status = sfsnatt(newID, trim(attrName), numberType, count, doubleAttr)
          if (status == fail) then
             print *,'Could not write the attribute', trim(attrName)
             stop
          end if
          !deallocate the dimension:
          deallocate (doubleAttr)
          
       else if (numberType == DFNT_INT8) then
          !byte
          !allocate the dimension:
          allocate (byteAttr(1:count))
          !read in the original attribute value:
          status = sfrnatt(origID, attrID, byteAttr)
          if (status == fail) then
             print *,'Could not read in the original attribute', trim(attrName)
             stop
          end if
          !write the attribute value to the new HDF:
          status = sfsnatt(newID, trim(attrName), numberType, count, byteAttr)
          if (status == fail) then
             print *,'Could not write the attribute', trim(attrName)
             stop
          end if
          !deallocate the dimension:
          deallocate (byteAttr)
          
       else if (numberType == DFNT_INT16) then
          !short
          !allocate the dimension:
          allocate (shortAttr(1:count))
          !read in the original attribute value:
          status = sfrnatt(origID, attrID, shortAttr)
          if (status == fail) then
             print *,'Could not read in the original attribute', trim(attrName)
             stop
          end if
          !write the attribute value to the new HDF:
          status = sfsnatt(newID, trim(attrName), numberType, count, shortAttr)
          if (status == fail) then
             print *,'Could not write the attribute', trim(attrName)
             stop
          end if
          !deallocate the dimension:
          deallocate (shortAttr)
          
       else if (numberType == DFNT_INT32) then
          !long
          !allocate the dimension:
          allocate (longAttr(1:count))
          !read in the original attribute value:
          status = sfrnatt(origID, attrID, longAttr)
          if (status == fail) then
             print *,'Could not read in the original attribute', trim(attrName)
             stop
          end if
          !write the attribute value to the new HDF:
          status = sfsnatt(newID, trim(attrName), numberType, count, longAttr)
          if (status == fail) then
             print *,'Could not write the attribute', trim(attrName)
             stop
          end if
          !deallocate the dimension:
          deallocate (longAttr)
          
       end if
       
    end if
    
  end subroutine copyAttr

  ! ----------------------------------------------------------------------------------

end module Output
