module hdfUtils
  !*****************************************************************************
  ! !F90
  !
  ! !Description:
  !    This module contains the hdf utility routines.
  !  
  ! !Callable routines:
  !   getFileID()
  !   closeFile()
  !   getDataSetID()
  !   endDataSet()
  ! 
  ! Module Revision History
  !   $Log: hdfUtils.f90,v $
  !   Revision 1.8  1999/11/15  12:35:13  EGMoody
  !   Added the routines GetFileID and CloseFile.
  !   Removed GetCoreMetadata and placed it in the module dataUtils.
  !
  !   Revision 1.7  1998/02/13  16:40:08  pincus
  !   Separated functionality of subroutine getDataSetList into function
  !   getNumDataSets (which gets the number of data sets) and subroutine
  !   getDataSetList (which returns the names of the datasets).
  !   Added prologs to hidden specific procedures underlying generic procedures
  !   to conform with ECS requirements.
  !
  !   Revision 1.6  1997/12/30  15:36:00  pincus
  !   netCDF constants are now taken from the hdf module; removed reference
  !   to module netcdf. Names of the constants changed.
  !
  !   Revision 1.5  1997/11/20  21:53:48  pincus
  !   Removed explicit conversion of integer types to placate FORCHECK;
  !   removed unneeded variables.
  !
  !   Revision 1.4  1997/11/01  20:29:42  pincus
  !   Specified the entities used from module hdf.
  !
  !   Revision 1.3  1997/10/23  21:37:04  pincus
  !   Corrected check of argument sizes in vector versions of readAttribute.
  !
  !   Revision 1.2  1997/10/23  19:04:43  pincus
  !   Added function getAttributeLength, generic procedure readAttribute;
  !   changed endDataSet to subroutine.
  !
  !   Revision 1.1  1997/07/15  17:22:13  pincus
  !   Initial revision
  ! !Team-Unique Header:
  !   Cloud Retrieval Group, NASA Goddard Space Flight Center
  !
  ! !References and Credits:
  !   Written by
  !    Eric Moody
  !    Climate and Radiation Branch, Code 913
  !    NASA/GSFC
  !    Greenbelt MD 20771
  !    Eric.Moody@gsfc.nasa.gov
  !
  ! !Design Notes:
  !   These routines were modified from codes written by Robert Pincus.
  !
  ! !END
  !*****************************************************************************
  
  !Dependencies:
  use typeSizes,      only : FourByteIntKind
  use hdf,            only : SFstart, SFend, SFselect, SFn2index,   &
                             SFgainfo, SFrcatt, SFfattr, SFendacc,  &
                             MAX_NC_NAME, FAIL,DFNT_CHAR, DFACC_READ
  use CharacterUtils, only : IntToChar

  implicit none
  private
  
  public :: getFileID, closeFile
  public :: getDataSetID, endDataSet

  
contains


  ! ----------------------------------------------------------------------------------
  Function Getfileid(Filename)

    Character (Len = *),              Intent( In) :: Filename
    Integer (Kind = Fourbyteintkind)              :: Getfileid

    ! !F90
    !
    ! !Description:
    !   Get Hdf File Id Given File Name. Serves Mostly 
    !     As An Error-Checking Wrapper To Hdf Call Sfstart. 
    !  
    ! !Input Parameters:
    !   Filename : Name Of The Hdf File To Be Opened.
    !
    ! !Output Parameters:
    !    Getfileid Returns The Hdf Id Of The File. If The Id Can'T Be Found 
    !      The Function Stops Execution. 
    ! 
    ! !Revision History:
    !    See Module Revision History In Data Section Of Module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, Nasa Goddard Space Flight Center
    !
    ! !References And Credits:
    !   Written By
    !    Eric Moody
    !    Climate And Radiation Branch, Code 913
    !    Nasa/Gsfc
    !    Greenbelt Md 20771
    !    Eric.Moody@Gsfc.Nasa.Gov
    !
    ! !Design Notes:
    !   Named Constants:
    !     Fail       (Hdf.F90)
    !
    ! !End
    Getfileid = Sfstart(Trim(Filename), Dfacc_Read)

    If (GetfileID == Fail) Then
       Print *, "Can'T Obtain The Input File Id For File" // Trim(Filename)
       Stop
    End If

  End Function Getfileid

  ! ----------------------------------------------------------------------------------

  subroutine closeFile(fileID)

    integer (kind = FourByteIntKind)              :: fileID

    ! !F90
    !
    ! !Description:
    !   Get HDF file ID given file name. Serves mostly 
    !     as an error-checking wrapper to HDF call SFstart. 
    !  
    ! !Input Parameters:
    !   fileName : Name of the HDF file to be opened.
    !
    ! !Output Parameters:
    !    getFileID returns the HDF ID of the file. If the ID can't be found 
    !      the function stops execution. 
    ! 
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Eric Moody
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Eric.Moody@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   Named constants:
    !     FAIL       (hdf.f90)
    !
    ! !END

    integer                            :: status

    status = SFend(fileID)

    if (status == FAIL) then
       print *, "Can't end access to input file"
       STOP
    end if

  end subroutine closeFile

  ! ----------------------------------------------------------------------------------

  function getDataSetID(fileID, dataSetName)

    integer (kind = FourByteIntKind), intent( in) :: fileID
    character (len = *),              intent( in) :: dataSetName
    integer (kind = FourByteIntKind)              :: getDataSetID
    character (len = 2048)                        :: msgbuffer


    ! !F90
    !
    ! !Description:
    !   Get HDF dataset ID given HDF file ID and name of data set. Serves mostly 
    !     as an error-checking wrapper to HDF call SFselect. 
    !  
    ! !Input Parameters:
    !   fileID: HDF file ID, from a call to SFstart.
    !   dataSetName: character string containing name of data set to get ID. 
    !
    ! !Output Parameters:
    !    getDataSetID returns the HDF dataset ID of the dataset. If the ID can't be found 
    !      the function stops execution. 
    ! 
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Robert Pincus
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Robert.Pincus@gsfc.nasa.gov
    !
    ! !Design Notes:
    !   Named constants:
    !     FAIL       (hdf.f90)
    !
    ! !END
    getDataSetID = SFselect(fileID, SFn2index(fileID, trim(dataSetName)))

    if(getDataSetID == FAIL) then
       print *, "Can't get dataSetID for dataSet " // &
                   trim(dataSetName) // " in file ID" // &
                   IntToChar(fileID)
       stop
    end if

  end function getDataSetID

 ! ----------------------------------------------------------------------------------

  subroutine endDataSet(dataSetID)
    integer (kind = FourByteIntKind), intent( in) :: dataSetID
    ! !F90
    !
    ! !Description:
    !   End access to an HDF SDS. This function is an error-checking wrapper to 
    !     HDF call SFselect. 
    !  
    ! !Input Parameters:
    !    dataSetID: HDF data set ID.
    !
    ! !Output Parameters:
    !    None. 
    ! 
    ! !Revision History:
    !    See module revision history in data section of module. 
    !
    ! !Team-Unique Header:
    !   Cloud Retrieval Group, NASA Goddard Space Flight Center
    !
    ! !References and Credits:
    !   Written by
    !    Robert Pincus
    !    Climate and Radiation Branch, Code 913
    !    NASA/GSFC
    !    Greenbelt MD 20771
    !    Robert.Pincus@gsfc.nasa.gov
    !
    ! !Design Notes:
    !
    ! !END
    
    ! Local variable
    integer                       :: status
    character (len = 2048)        :: msgbuffer

    status = SFendacc(dataSetID)

    if(status == FAIL) then
      print *,"Error ending access to data set." 
      stop
    end if

  end subroutine endDataSet


end module hdfUtils
