! DART software - Copyright UCAR. This open source software is provided
! by UCAR, "as is", without charge, subject to all terms of use at
! http://www.image.ucar.edu/DAReS/DART/DART_download
!
! $Id$

!----------------------------------------------------------------------
! WARNING!!  The file obs_def_mod.f90 is AUTOGENERATED by the
! 'preprocess' program.  Any changes made to this file will be
! overwritten when it is regenerated.  To make permanent changes
! in the code, edit DEFAULT_obs_def_mod.F90, or edit the
! observation specific obs_def_xxx_mod.f90 files.
!----------------------------------------------------------------------


!----------------------------------------------------------------------
!----------------------------------------------------------------------
! Any observation-specific modules which contain executable code to
! handle the interpolation and/or reading and writing of the obs will
! have the module code inserted here.  Then models only have to use
! the obs_def_mod module, and only include obs_def_mod.f90 in their
! path_names files.  To change the observation types/kinds, edit the
! model-specific 'input.nml' file and add or remove lines from the
! &preprocess_nml section, the 'input_files' list.
!
! Start of any user-defined executable module code
!----------------------------------------------------------------------

!---------------------------------------------------------------------------  
! Start of code inserted from ../../../obs_def/obs_def_gps_mod.f90
!---------------------------------------------------------------------------  
                                                                              
module obs_def_gps_mod

use        types_mod, only : r8, missing_r8, RAD2DEG, DEG2RAD, PI
use    utilities_mod, only : register_module, error_handler, E_ERR, E_MSG, &
                             file_exist, open_file, close_file, nmlfileunit, &
                             check_namelist_read, find_namelist_in_file, &
                             do_output, do_nml_file, do_nml_term, &
                             ascii_file_format
use     location_mod, only : location_type, set_location, get_location, &
                             write_location, read_location, vert_is_height, &
                             VERTISHEIGHT
use time_manager_mod, only : time_type, read_time, write_time, &
                             set_time, set_time_missing, interactive_time
!use  assim_model_mod, only : interpolate

use     obs_kind_mod, only : QTY_U_WIND_COMPONENT, &
                             QTY_V_WIND_COMPONENT, QTY_SURFACE_PRESSURE, &
                             QTY_TEMPERATURE, QTY_SPECIFIC_HUMIDITY, &
                             QTY_PRESSURE, QTY_GPSRO

implicit none
private

public :: set_gpsro_ref, get_gpsro_ref, write_gpsro_ref, read_gpsro_ref, &
          get_expected_gpsro_ref, interactive_gpsro_ref

! version controlled file description for error handling, do not edit
character(len=256), parameter :: source   = &
   "$URL$"
character(len=32 ), parameter :: revision = "$Revision$"
character(len=128), parameter :: revdate  = "$Date$"

logical, save :: module_initialized = .false.

! Storage for the special information required for GPS RO observations
!

! Because we are currently only generating one observation type
! (GPSRO_REFRACTIVITY), there must be enough of these to cover all gps
! obs in all obs_seq files that are read in (e.g. for obs_diag if you
! cover multiple days or weeks, you must have enough room for all of them.)
! the local operator needs none of this additional info; the best approach
! would be to keep a single QTY_GPSRO, but make 2 observation types.
! the local has no additional metadata; the nonlocal needs one of these
! allocated and filled in.
integer :: max_gpsro_obs = 100000

type gps_nonlocal_type
   private
   character(len=6) :: gpsro_ref_form
   real(r8)         :: ray_direction(3)
   real(r8)         :: rfict
   real(r8)         :: step_size
   real(r8)         :: ray_top
end type gps_nonlocal_type

type(gps_nonlocal_type), allocatable :: gps_data(:)

namelist /obs_def_gps_nml/ max_gpsro_obs

character(len=129) :: string1, string2
integer  :: ii
integer  :: keycount

contains

!------------------------------------------------------------------------------


  subroutine initialize_module
!------------------------------------------------------------------------------
!
! initialize global gps private key number and allocate space for obs data
integer :: rc, iunit


call register_module(source, revision, revdate)
module_initialized = .true.

! global count of all gps observations from any input file
keycount = 0

! Read the namelist entry
call find_namelist_in_file("input.nml", "obs_def_gps_nml", iunit)
read(iunit, nml = obs_def_gps_nml, iostat = rc)
call check_namelist_read(iunit, rc, "obs_def_gps_nml")

! Record the namelist values used for the run ...
if (do_nml_file()) write(nmlfileunit, nml=obs_def_gps_nml)
if (do_nml_term()) write(     *     , nml=obs_def_gps_nml)

! find max number of gps obs which can be stored, and initialize type
allocate(gps_data(max_gpsro_obs), stat = rc)
if (rc /= 0) then
   write(string1, *) 'initial allocation failed for gps observation data,', &
                       'itemcount = ', max_gpsro_obs
   call error_handler(E_ERR,'initialize_module', string1, &
                      source, revision, revdate)
endif

end subroutine initialize_module



 subroutine set_gpsro_ref(gpskey, nx, ny, nz, rfict0, ds, htop, subset0)
!------------------------------------------------------------------------------
!
! increment key and set all private data for this observation

integer,          intent(out) :: gpskey
real(r8),         intent(in)  :: nx, ny, nz, rfict0, ds, htop
character(len=6), intent(in)  :: subset0

if ( .not. module_initialized ) call initialize_module

keycount = keycount + 1
gpskey = keycount

if(gpskey > max_gpsro_obs) then
   write(string1, *) 'key (',gpskey,') exceeds max_gpsro_obs (',max_gpsro_obs,')'
   string2 = 'Increase max_gpsro_obs in input.nml &obs_def_gps_nml namelist.'
   call error_handler(E_ERR,'read_gpsro_ref', string1, &
                      source, revision, revdate, text2=string2)
endif

gps_data(gpskey)%ray_direction(1) = nx
gps_data(gpskey)%ray_direction(2) = ny
gps_data(gpskey)%ray_direction(3) = nz
gps_data(gpskey)%gpsro_ref_form   = subset0

gps_data(gpskey)%rfict     = rfict0
gps_data(gpskey)%step_size = ds
gps_data(gpskey)%ray_top   = htop

end subroutine set_gpsro_ref


 subroutine get_gpsro_ref(gpskey, nx, ny, nz, rfict0, ds, htop, subset0)
!------------------------------------------------------------------------------
!
! return all private data for this observation

integer,          intent(in)  :: gpskey
real(r8),         intent(out) :: nx, ny, nz, rfict0, ds, htop
character(len=6), intent(out) :: subset0

if ( .not. module_initialized ) call initialize_module

if (gpskey < 1 .or. gpskey > keycount) then
   write(string1, *) 'key (',gpskey,') out of valid range (1<=key<=',keycount,')'
   call error_handler(E_ERR,'get_gpsro_ref', string1, &
                      source, revision, revdate)
endif

nx = gps_data(gpskey)%ray_direction(1)
ny = gps_data(gpskey)%ray_direction(2)
nz = gps_data(gpskey)%ray_direction(3)
subset0 = gps_data(gpskey)%gpsro_ref_form

rfict0 = gps_data(gpskey)%rfict
ds     = gps_data(gpskey)%step_size
htop   = gps_data(gpskey)%ray_top

end subroutine get_gpsro_ref



 subroutine write_gpsro_ref(gpskey, ifile, fform)
!------------------------------------------------------------------------------
!

integer,          intent(in)           :: gpskey, ifile
character(len=*), intent(in), optional :: fform


if ( .not. module_initialized ) call initialize_module

! Write the 5 character identifier for verbose formatted output
! Write out the obs_def key for this observation
if (ascii_file_format(fform)) then
   write(ifile,11) gpskey
   write(ifile, *) gps_data(gpskey)%rfict, gps_data(gpskey)%step_size, &
                   gps_data(gpskey)%ray_top, &
                  (gps_data(gpskey)%ray_direction(ii), ii=1, 3), &
                   gps_data(gpskey)%gpsro_ref_form
11  format('gpsroref', i8)
else
   write(ifile) gpskey
   write(ifile) gps_data(gpskey)%rfict, gps_data(gpskey)%step_size, &
                gps_data(gpskey)%ray_top, &
               (gps_data(gpskey)%ray_direction(ii), ii=1, 3), &
                gps_data(gpskey)%gpsro_ref_form
endif

end subroutine write_gpsro_ref



 subroutine read_gpsro_ref(gpskey, ifile, fform)
!------------------------------------------------------------------------------
!
! Every GPS observation has its own (metadata) gpskey.
! When you read multiple gps observation sequence files, it is necessary
! to track the total number of metadata gpskeys read, not just the number
! in the current file.
!

integer,          intent(out)          :: gpskey
integer,          intent(in)           :: ifile
character(len=*), intent(in), optional :: fform

integer :: keyin    ! the metadata key in the current obs sequence

real(r8) :: nx, ny, nz, rfict0, ds, htop
character(len=6) :: subset0
character(len=8) :: header

if ( .not. module_initialized ) call initialize_module

if (ascii_file_format(fform)) then
   read(ifile, FMT='(a8, i8)') header, keyin    ! throw away keyin
   if(header /= 'gpsroref') then
       call error_handler(E_ERR,'read_gpsro_ref', &
       'Expected header "gpsroref" in input file', source, revision, revdate)
   endif
   read(ifile, *) rfict0, ds, htop, nx, ny, nz, subset0
else
   read(ifile) keyin          ! read and throw away
   read(ifile) rfict0, ds, htop, nx, ny, nz, subset0
endif


! increment key and set all private data for this observation
call set_gpsro_ref(gpskey, nx, ny, nz, rfict0, ds, htop, subset0)

end subroutine read_gpsro_ref


subroutine interactive_gpsro_ref(gpskey)
!----------------------------------------------------------------------
!
! Interactively prompt for the info needed to create a gps refractivity
! observation.  Increments the key number and returns it.

integer, intent(out) :: gpskey

real(r8) :: nx, ny, nz, rfict0, ds, htop
character(len=6) :: subset0
integer :: gpstype


if ( .not. module_initialized ) call initialize_module

!Now interactively obtain reflectivity type information
! valid choices are local or non-local

write(*, *)
write(*, *) 'Beginning to inquire information on reflectivity type.'
write(*, *)

100 continue
write(*, *) 'Enter 1 for local refractivity (GPSREF)'
write(*, *) 'Enter 2 for non-local refractivity/excess phase delay (GPSEXC)'
write(*, *)

read(*,*) gpstype

select case (gpstype)
   case (1)
      subset0 = 'GPSREF'
   case (2)
      subset0 = 'GPSEXC'
   case default
      write(*,*) 'Bad value, must enter 1 or 2'
      goto 100
end select

if (gpstype == 2) then
    ! FIXME:  i have no idea what valid values are for any
   !  of the following items, so i cannot add any error checking or
   !  guidance for the user.

   write(*, *)
   write(*, *) 'Enter X, Y, Z value for ray direction'
   write(*, *)
   read(*,*) nx, ny, nz

   write(*, *)
   write(*, *) 'Enter local curvature radius'
   write(*, *)
   read(*,*) rfict0

   write(*, *)
   write(*, *) 'Enter step size'
   write(*, *)
   read(*,*) ds

   write(*, *)
   write(*, *) 'Enter ray top'
   write(*, *)
   read(*,*) htop
else
   nx = 0.0
   ny = 0.0
   nz = 0.0
   rfict0 = 0.0
   ds = 0.0
   htop = 0.0
endif

! increment key and set all private data for this observation
call set_gpsro_ref(gpskey, nx, ny, nz, rfict0, ds, htop, subset0)

write(*, *)
write(*, *) 'End of specialized section for gps observation data.'
write(*, *) 'You will now have to enter the regular obs information.'
write(*, *)

end subroutine interactive_gpsro_ref

 subroutine get_expected_gpsro_ref(state_vector, location, gpskey, ro_ref, istatus)
!------------------------------------------------------------------------------
!
! Purpose: Calculate GPS RO local refractivity or non_local (integrated)
!          refractivity (excess phase, Sergey Sokolovskiy et al., 2005)
!------------------------------------------------------------------------------
!
! inputs:
!    state_vector:    DART state vector
!
! output parameters:
!    ro_ref: modeled local refractivity (N-1)*1.0e6 or non_local
!            refractivity (excess phase, m)
!            (according to the input data parameter subset)
!    istatus:  =0 normal; =1 outside of domain.
!------------------------------------------------------------------------------
!  Author: Hui Liu
!  Version 1.1: June 15, 2004: Initial version CAM
!
!  Version 1.2: July 29, 2005: revised for new obs_def and WRF
!------------------------------------------------------------------------------
implicit none

real(r8),            intent(in)  :: state_vector(:)
type(location_type), intent(in)  :: location
integer,             intent(in)  :: gpskey
real(r8),            intent(out) :: ro_ref
integer,             intent(out) :: istatus

! local variables

real(r8) :: nx, ny, nz       ! unit tangent direction of ray at perigee
real(r8) :: xo, yo, zo       ! perigee location in Cartesian coordinate

real(r8) :: ref_perigee, ref00, ref1, ref2, dist_to_perigee
real(r8) :: phase
real(r8) :: xx, yy, zz, height1, lat1, lon1, delta_phase1, delta_phase2

integer  :: iter, istatus0
real(r8) :: lon, lat, height, obsloc(3)

if ( .not. module_initialized ) call initialize_module

if ( .not. vert_is_height(location)) then
   write(string1, *) 'vertical location must be height; gps obs key ', gpskey
   call error_handler(E_ERR,'get_expected_gpsro_ref', string1, &
                      source, revision, revdate)
endif

obsloc   = get_location(location)


lon      = obsloc(1)                       ! degree: 0 to 360
lat      = obsloc(2)                       ! degree: -90 to 90
height   = obsloc(3)                       ! (m)

! calculate refractivity at perigee

call ref_local(state_vector, location, height, lat, lon, ref_perigee, istatus0)
! if istatus > 0, the interpolation failed and we should return failure now.
if(istatus0 > 0) then
   istatus = istatus0
   ro_ref = missing_r8
   return
endif

choose: if(gps_data(gpskey)%gpsro_ref_form == 'GPSREF') then
    ! use local refractivity

    ro_ref = ref_perigee * 1.0e6      ! in (N-1)*1.0e6, same with obs

else  ! gps_data(gpskey)%gpsro_ref_form == 'GPSEXC'

    ! otherwise, use non_local refractivity(excess phase delay)

    ! Initialization
    phase = 0.0_r8
    dist_to_perigee =  0.0_r8   ! distance to perigee from a point of the ray

    nx = gps_data(gpskey)%ray_direction(1)
    ny = gps_data(gpskey)%ray_direction(2)
    nz = gps_data(gpskey)%ray_direction(3)

    ! convert location of the perigee from geodetic to Cartesian coordinate

    call geo2carte (height, lat, lon, xo, yo, zo, gps_data(gpskey)%rfict )

    ! currently, use a straight line passing the perigee point as ray model.
    ! later, more sophisticated ray models can be used.
    !
    ! Start the horizontal integrate of the model refractivity along a
    ! straight line path in cartesian coordinate
    !
    ! (x-xo)/a = (y-yo)/b = (z-zo)/c,  (a,b,c) is the line direction

    ref1 = ref_perigee
    ref2 = ref_perigee

    iter = 0
    do

       iter = iter + 1
       dist_to_perigee = dist_to_perigee + gps_data(gpskey)%step_size

       !  integrate to one direction of the ray for one step
       xx = xo + dist_to_perigee * nx
       yy = yo + dist_to_perigee * ny
       zz = zo + dist_to_perigee * nz

       ! convert the location of the point to geodetic coordinates
       ! height(m), lat, lon(deg)

       call carte2geo(xx, yy, zz, height1, lat1, lon1, gps_data(gpskey)%rfict )
       if (height1 >= gps_data(gpskey)%ray_top) exit

       ! get the refractivity at this ray point(ref00)
       call ref_local(state_vector, location, height1, lat1, lon1, ref00, istatus0)
       ! when any point of the ray is problematic, return failure
       if(istatus0 > 0) then
         istatus = istatus0
         ro_ref = missing_r8
         return
       endif

       ! get the excess phase due to this ray interval
       delta_phase1 = (ref1 + ref00) * gps_data(gpskey)%step_size * 0.5_r8

       ! save the refractivity for integration of next ray interval
       ref1 = ref00

       ! integrate to the other direction of the ray
       xx = xo - dist_to_perigee * nx
       yy = yo - dist_to_perigee * ny
       zz = zo - dist_to_perigee * nz

       call carte2geo (xx, yy, zz, height1, lat1, lon1, gps_data(gpskey)%rfict )

       ! get the refractivity at this ray point(ref00)
       call ref_local(state_vector, location, height1, lat1, lon1, ref00, istatus0)
       ! when any point of the ray is problematic, return failure
       if(istatus0 > 0) then
         istatus = istatus0
         ro_ref = missing_r8
         return
       endif

       ! get the excess phase due to this ray interval
       delta_phase2 = (ref2 + ref00) * gps_data(gpskey)%step_size * 0.5_r8

       ! save the refractivity for integration of next ray interval
       ref2 = ref00

       phase = phase + delta_phase1 + delta_phase2
       ! print*, 'phase= ',  phase, delta_phase1, delta_phase2

    end do

    ! finish the integration of the excess phase along the ray

    ro_ref = phase    ! in m

    ! print*, 'xx = ', lon, lat, height, ro_ref

endif choose

! if the original height was too high, for example.  do not return a
! negative or 0 excess phase or refractivity.
if (ro_ref == missing_r8 .or. ro_ref <= 0.0_r8) then
   istatus = 5
   ro_ref = missing_r8
   return
endif

! ended ok, return local refractivity or non-local excess phase accumulated value
istatus = 0

end subroutine get_expected_gpsro_ref



 subroutine ref_local(state_vector, location, height, lat, lon, ref00, istatus0)
!------------------------------------------------------------------------------
!
! Calculate local refractivity at any GPS ray point (height, lat, lon)
!
! inputs:
!    height, lat, lon:  GPS observation location (units: m, degree)
!
! output:
!    ref00: modeled local refractivity at ray point(unit: N-1, ~1.0e-4 to e-6)
!
!------------------------------------------------------------------------------
implicit none

real(r8), intent(in) :: state_vector(:)
real(r8), intent(in) :: lon, lat, height

real(r8), intent(out) :: ref00
integer,  intent(out) :: istatus0

real(r8), parameter::  rd = 287.05_r8, rv = 461.51_r8, c1 = 77.6d-6 , &
                       c2 = 3.73d-1,  rdorv = rd/rv
real(r8) :: lon2, t, q, p, tv, ew
type(location_type) :: location, location2
integer :: which_vert

if ( .not. module_initialized ) call initialize_module

! for integration of GPS ray path beyond the wraparound point
lon2 = lon
if(lon > 360.0_r8 ) lon2 = lon - 360.0_r8
if(lon <   0.0_r8 ) lon2 = lon + 360.0_r8

which_vert = VERTISHEIGHT
location2 = set_location(lon2, lat, height,  which_vert)

! set return values assuming failure, so we can simply return if any
! of the interpolation calls below fail.
istatus0 = 3
ref00 = missing_r8

call error_handler(E_ERR, 'this is not distributed', 'yet')
!HKcall interpolate(state_vector, location2,  QTY_TEMPERATURE,       t, istatus0)
!HKif (istatus0 > 0) return
!HKcall interpolate(state_vector, location2,  QTY_SPECIFIC_HUMIDITY, q, istatus0)
!HKif (istatus0 > 0) return
!HKcall interpolate(state_vector, location2,  QTY_PRESSURE,          p, istatus0)
!HKif (istatus0 > 0) return

!  required variable units for calculation of GPS refractivity
!   t :  Kelvin, from top to bottom
!   q :  kg/kg, from top to bottom
!   p :  mb

p     = p * 0.01_r8      ! to mb

tv    = t * (1.0_r8+(rv/rd - 1.0_r8)*q)         ! virtual temperature
ew    = q * p/(rdorv + (1.0_r8-rdorv)*q )
ref00 = c1*p/t + c2*ew/(t**2)              ! (N-1)

! now we have succeeded, set istatus to good
istatus0 = 0

end subroutine ref_local


 subroutine geo2carte (s1, s2, s3, x1, x2, x3, rfict0)
!------------------------------------------------------------------------------
!
!  Converts geodetical coordinates to cartesian with a reference sphere
!------------------------------------------------------------------------------
!  input parameters:
!   s - geodetical coordinates
!        (height (m), latitude (degree), longitude (degree))
!                     -90 to 90           0 to 360
!  output parameters:
!   x - cartesian coordinates (m) connected with the earth(x, y, z-coordinate)
!------------------------------------------------------------------------------
implicit none
real(r8), intent(in)  :: s1, s2, s3, rfict0    ! units: m
real(r8), intent(out) ::   x1, x2 ,x3
real(r8) :: g3, g4

if ( .not. module_initialized ) call initialize_module

g3 = s1 + rfict0
g4 = g3 * cos(s2*DEG2RAD)
x1 = g4 * cos(s3*DEG2RAD)
x2 = g4 * sin(s3*DEG2RAD)
x3 = g3 * sin(s2*DEG2RAD)

end subroutine geo2carte


 subroutine carte2geo (x1, x2, x3, s1, s2, s3, rfict0)
!------------------------------------------------------------------------------
!
!  Converts cartesian coordinates to geodetical.
!
!   input parameters:
!        x - cartesian coordinates (x, y, z-coordinate, unit: m)
!
!   output parameters:
!        s - geodetical coordinates
!            (height (m), latitude (deg), longitude (deg))
!                          -90 to 90         0 to 360
!------------------------------------------------------------------------------
implicit none
real(r8), intent(in)  :: x1, x2, x3, rfict0
real(r8), intent(out) :: s1, s2, s3

real(r8), parameter :: crcl  = 2.0_r8 * PI, &
                       crcl2 = 4.0_r8 * PI

real(r8) :: rho, sphi, azmth

if ( .not. module_initialized ) call initialize_module

rho   = sqrt (x1**2 + x2**2 + x3**2 )
sphi  = x3/rho
s1    = rho - rfict0
s2    = asin (sphi)
azmth = atan2 (x2, x1)
s3    = mod((azmth + crcl2), crcl)

s2    = s2 * RAD2DEG
s3    = s3 * RAD2DEG

end  subroutine carte2geo

end module obs_def_gps_mod

                                                                              
!---------------------------------------------------------------------------  
! End of code inserted from ../../../obs_def/obs_def_gps_mod.f90
!---------------------------------------------------------------------------  
!---------------------------------------------------------------------------  
! Start of code inserted from ../../../obs_def/obs_def_altimeter_mod.f90
!---------------------------------------------------------------------------  
                                                                              
module obs_def_altimeter_mod

use        types_mod, only : r8, missing_r8, t_kelvin
use    utilities_mod, only : register_module, error_handler, E_ERR, E_MSG
use     location_mod, only : location_type, set_location, get_location , write_location, &
                             read_location
!use  assim_model_mod, only : interpolate
use     obs_kind_mod, only : QTY_SURFACE_PRESSURE, QTY_SURFACE_ELEVATION

implicit none
private

public :: get_expected_altimeter, compute_altimeter

! version controlled file description for error handling, do not edit
character(len=256), parameter :: source   = &
   "$URL$"
character(len=32 ), parameter :: revision = "$Revision$"
character(len=128), parameter :: revdate  = "$Date$"

logical, save :: module_initialized = .false.

contains

!----------------------------------------------------------------------

subroutine initialize_module

call register_module(source, revision, revdate)
module_initialized = .true.

end subroutine initialize_module


subroutine get_expected_altimeter(state_vector, location, altimeter_setting, istatus)

real(r8),            intent(in)  :: state_vector(:)
type(location_type), intent(in)  :: location
real(r8),            intent(out) :: altimeter_setting     ! altimeter (hPa)
integer,             intent(out) :: istatus

real(r8) :: psfc                ! surface pressure value   (Pa)
real(r8) :: hsfc                ! surface elevation level  (m above SL)

if ( .not. module_initialized ) call initialize_module

!  interpolate the surface pressure to the desired location
!HK call interpolate(state_vector, location, QTY_SURFACE_PRESSURE, psfc, istatus)
if (istatus /= 0) then
   altimeter_setting = missing_r8
   return
endif

!  interpolate the surface elevation to the desired location
!HK call interpolate(state_vector, location, QTY_SURFACE_ELEVATION, hsfc, istatus)
if (istatus /= 0) then
   altimeter_setting = missing_r8
   return
endif

!  Compute the altimeter setting given surface pressure and height, altimeter is hPa
altimeter_setting = compute_altimeter(psfc * 0.01_r8, hsfc)

if (altimeter_setting < 880.0_r8 .or. altimeter_setting >= 1100.0_r8) then
   altimeter_setting = missing_r8
   if (istatus == 0) istatus = 1
   return
endif

return
end subroutine get_expected_altimeter


function compute_altimeter(psfc, hsfc)

real(r8), parameter :: k1 = 0.190284_r8
real(r8), parameter :: k2 = 8.4228807E-5_r8

real(r8), intent(in) :: psfc  !  (hPa)
real(r8), intent(in) :: hsfc  !  (m above MSL)

real(r8) :: compute_altimeter !  (hPa)

compute_altimeter = ((psfc - 0.3_r8) ** k1 + k2 * hsfc) ** (1.0_r8 / k1)

return
end function compute_altimeter

!----------------------------------------------------------------------------

end module obs_def_altimeter_mod

                                                                              
!---------------------------------------------------------------------------  
! End of code inserted from ../../../obs_def/obs_def_altimeter_mod.f90
!---------------------------------------------------------------------------  
!---------------------------------------------------------------------------  
!No module code needed for ../../../obs_def/obs_def_reanalysis_bufr_mod.f90
!---------------------------------------------------------------------------  

!----------------------------------------------------------------------
! End of any user-defined executable module code
!----------------------------------------------------------------------
!----------------------------------------------------------------------

!----------------------------------------------------------------------
!----------------------------------------------------------------------
! Start of main obs_def_mod module code
!----------------------------------------------------------------------

module obs_def_mod

! Contains the basic parts of a module for defining and evaluating observation
! definitions. Can evaluate identity observations as is. The DART preprocess
! program is used to add in extra observation kinds at the indicated spots in
! the code.

use        types_mod, only : r8, missing_i, missing_r8, RAD2DEG
use    utilities_mod, only : register_module, error_handler, E_ERR, E_MSG, &
                             ascii_file_format
use     location_mod, only : location_type, read_location, write_location, &
                             interactive_location, set_location_missing
use time_manager_mod, only : time_type, read_time, write_time, set_time, &
                             set_time_missing, interactive_time
use  assim_model_mod, only : get_state_meta_data_distrib, interpolate_distrib !HK
use     obs_kind_mod, only : assimilate_this_type_of_obs, evaluate_this_type_of_obs, &
                             max_defined_types_of_obs, get_name_for_type_of_obs, map_type_of_obs_table, &
                             get_type_of_obs_from_menu
use ensemble_manager_mod, only : ensemble_type

!HK temporary
use mpi_utilities_mod, only    : my_task_id



!----------------------------------------------------------------------
! This list is autogenerated by the 'preprocess' program.  To add types
! or kinds, edit the obs_def_xxx_mod.f90 files, and then add/remove them
! from the 'input_files' variable of the &preprocess_nml namelist
! in the model-specific work/input.nml file.
!
! Start of obs_def_xxx_mod specific types and kinds
!----------------------------------------------------------------------

!---------------------------------------------------------------------------  
                                                                              
use obs_kind_mod, only : TEMPERATURE
use obs_kind_mod, only : SPECIFIC_HUMIDITY
use obs_kind_mod, only : PRESSURE
use obs_kind_mod, only : GPSRO_REFRACTIVITY
use obs_kind_mod, only : RADIOSONDE_SURFACE_ALTIMETER
use obs_kind_mod, only : DROPSONDE_SURFACE_ALTIMETER
use obs_kind_mod, only : MARINE_SFC_ALTIMETER
use obs_kind_mod, only : LAND_SFC_ALTIMETER
use obs_kind_mod, only : METAR_ALTIMETER
use obs_kind_mod, only : RADIOSONDE_U_WIND_COMPONENT
use obs_kind_mod, only : RADIOSONDE_V_WIND_COMPONENT
use obs_kind_mod, only : RADIOSONDE_GEOPOTENTIAL_HGT
use obs_kind_mod, only : RADIOSONDE_SURFACE_PRESSURE
use obs_kind_mod, only : RADIOSONDE_TEMPERATURE
use obs_kind_mod, only : RADIOSONDE_SPECIFIC_HUMIDITY
use obs_kind_mod, only : DROPSONDE_U_WIND_COMPONENT
use obs_kind_mod, only : DROPSONDE_V_WIND_COMPONENT
use obs_kind_mod, only : DROPSONDE_SURFACE_PRESSURE
use obs_kind_mod, only : DROPSONDE_TEMPERATURE
use obs_kind_mod, only : DROPSONDE_SPECIFIC_HUMIDITY
use obs_kind_mod, only : AIRCRAFT_U_WIND_COMPONENT
use obs_kind_mod, only : AIRCRAFT_V_WIND_COMPONENT
use obs_kind_mod, only : AIRCRAFT_TEMPERATURE
use obs_kind_mod, only : AIRCRAFT_SPECIFIC_HUMIDITY
use obs_kind_mod, only : ACARS_U_WIND_COMPONENT
use obs_kind_mod, only : ACARS_V_WIND_COMPONENT
use obs_kind_mod, only : ACARS_TEMPERATURE
use obs_kind_mod, only : ACARS_SPECIFIC_HUMIDITY
use obs_kind_mod, only : MARINE_SFC_U_WIND_COMPONENT
use obs_kind_mod, only : MARINE_SFC_V_WIND_COMPONENT
use obs_kind_mod, only : MARINE_SFC_TEMPERATURE
use obs_kind_mod, only : MARINE_SFC_SPECIFIC_HUMIDITY
use obs_kind_mod, only : MARINE_SFC_PRESSURE
use obs_kind_mod, only : LAND_SFC_U_WIND_COMPONENT
use obs_kind_mod, only : LAND_SFC_V_WIND_COMPONENT
use obs_kind_mod, only : LAND_SFC_TEMPERATURE
use obs_kind_mod, only : LAND_SFC_SPECIFIC_HUMIDITY
use obs_kind_mod, only : LAND_SFC_PRESSURE
use obs_kind_mod, only : SAT_U_WIND_COMPONENT
use obs_kind_mod, only : SAT_V_WIND_COMPONENT
use obs_kind_mod, only : ATOV_TEMPERATURE
use obs_kind_mod, only : AIRS_TEMPERATURE
use obs_kind_mod, only : AIRS_SPECIFIC_HUMIDITY
                                                                              
use obs_kind_mod, only : QTY_TEMPERATURE
use obs_kind_mod, only : QTY_SPECIFIC_HUMIDITY
use obs_kind_mod, only : QTY_PRESSURE
use obs_kind_mod, only : QTY_GPSRO
use obs_kind_mod, only : QTY_SURFACE_PRESSURE
use obs_kind_mod, only : QTY_U_WIND_COMPONENT
use obs_kind_mod, only : QTY_V_WIND_COMPONENT
use obs_kind_mod, only : QTY_GEOPOTENTIAL_HEIGHT
                                                                              
!---------------------------------------------------------------------------  
                                                                              

!----------------------------------------------------------------------
! End of obs_def_xxx_mod specific types and kinds
!----------------------------------------------------------------------

!----------------------------------------------------------------------
! This section is autogenerated by the 'preprocess' program.
!
! Start of any obs_def_xxx_mod specific use statements
!----------------------------------------------------------------------

! If any observation types required specialized code, the module code
! will have been added above, and now a use statement will be generated
! here so the generic obs_def_mod has access to the code.

  use obs_def_gps_mod, only : get_expected_gpsro_ref, interactive_gpsro_ref, &
                              read_gpsro_ref, write_gpsro_ref
   use obs_def_altimeter_mod, only : get_expected_altimeter, compute_altimeter

!----------------------------------------------------------------------
! End of any obs_def_xxx_mod specific use statements
!----------------------------------------------------------------------


implicit none
private

interface assignment(=)
   module procedure copy_obs_def
end interface

public :: init_obs_def, get_obs_def_key, get_obs_def_location, get_obs_def_type_of_obs, &
   get_obs_def_time, get_obs_def_error_variance, set_obs_def_location, &
   set_obs_def_type_of_obs, set_obs_def_time, set_obs_def_error_variance, &
   set_obs_def_key, interactive_obs_def, write_obs_def, read_obs_def, &
   obs_def_type, destroy_obs_def, copy_obs_def, &
   assignment(=), get_name_for_type_of_obs, get_expected_obs_from_def_distrib_state !HK

! version controlled file description for error handling, do not edit
character(len=256), parameter :: source   = &
   "$URL$"
character(len=32 ), parameter :: revision = "$Revision$"
character(len=128), parameter :: revdate  = "$Date$"

type obs_def_type
! In revision, obs_kind module is responsible for taking care of
! identity obs kinds, too
   private
   type(location_type)   :: location   ! center of mass, so to speak
   integer               :: kind
   type(time_type)       :: time
   real(r8)              :: error_variance
   integer               :: key        ! Used by specialized observation types
end type obs_def_type

logical, save :: module_initialized = .false.

contains

!----------------------------------------------------------------------------

subroutine initialize_module

call register_module(source, revision, revdate)
module_initialized = .true.

! Note that there is no namelist for this module now that
! obs_kind has been revised

end subroutine initialize_module


!----------------------------------------------------------------------------

subroutine init_obs_def(obs_def, location, kind, time, error_variance)
! Need to add additional component arguments as optionals as needed

! Constructor for an obs_def

type(obs_def_type), intent(out) :: obs_def
type(location_type), intent(in) :: location
integer,             intent(in) :: kind
type(time_type),     intent(in) :: time
real(r8),            intent(in) :: error_variance

if ( .not. module_initialized ) call initialize_module

obs_def%location = location
obs_def%kind = kind
obs_def%time = time
obs_def%error_variance = error_variance
! No key assigned for standard observation defs
obs_def%key = -1

end subroutine init_obs_def

!---------------------------------------------------------------------

subroutine copy_obs_def(obs_def1, obs_def2)

! Copy function to be overloaded with '='

type(obs_def_type), intent(out) :: obs_def1
type(obs_def_type), intent(in) :: obs_def2

if ( .not. module_initialized ) call initialize_module

obs_def1%location       = obs_def2%location
obs_def1%kind           = obs_def2%kind
obs_def1%time           = obs_def2%time
obs_def1%error_variance = obs_def2%error_variance
obs_def1%key            = obs_def2%key
!deallocate(obs_def1%platform_qc)
!allocate(obs_def1%platform_qc(size(obs_def2%platform_qc))
! Should this be pointer assignment or regular
!obs_def1%platform_qc >= or == obs_def2%platform_qc
!obs_def1%aperture = obs_def2%aperture

end subroutine copy_obs_def

!----------------------------------------------------------------------------

function get_obs_def_key(obs_def)

type(obs_def_type), intent(in) :: obs_def
integer                        :: get_obs_def_key

if ( .not. module_initialized ) call initialize_module

get_obs_def_key = obs_def%key

end function get_obs_def_key

!----------------------------------------------------------------------------

function get_obs_def_error_variance(obs_def)

type(obs_def_type), intent(in) :: obs_def
real(r8)                       :: get_obs_def_error_variance

if ( .not. module_initialized ) call initialize_module

get_obs_def_error_variance = obs_def%error_variance

end function get_obs_def_error_variance

!----------------------------------------------------------------------------

function get_obs_def_location(obs_def)

! Returns observation location.

type(location_type)            :: get_obs_def_location
type(obs_def_type), intent(in) :: obs_def

if ( .not. module_initialized ) call initialize_module

get_obs_def_location = obs_def%location

end function get_obs_def_location

!----------------------------------------------------------------------------

function get_obs_def_type_of_obs(obs_def)

! Returns observation kind

integer                        :: get_obs_def_type_of_obs
type(obs_def_type), intent(in) :: obs_def

if ( .not. module_initialized ) call initialize_module

get_obs_def_type_of_obs = obs_def%kind

end function get_obs_def_type_of_obs

!----------------------------------------------------------------------------

function get_obs_def_time(obs_def)

! Returns observation time

type(time_type)                :: get_obs_def_time
type(obs_def_type), intent(in) :: obs_def

if ( .not. module_initialized ) call initialize_module

get_obs_def_time = obs_def%time

end function get_obs_def_time

!----------------------------------------------------------------------------

function get_name_for_type_of_obs(obs_kind_ind)

! Returns observation name

integer, intent(in) :: obs_kind_ind
character(len = 32) :: get_name_for_type_of_obs

if ( .not. module_initialized ) call initialize_module

get_name_for_type_of_obs = get_name_for_type_of_obs(obs_kind_ind)

end function get_name_for_type_of_obs

!----------------------------------------------------------------------------

subroutine set_obs_def_location(obs_def, location)

! Sets the location of an obs_def

type(obs_def_type), intent(inout) :: obs_def
type(location_type),   intent(in) :: location

if ( .not. module_initialized ) call initialize_module

obs_def%location = location

end subroutine set_obs_def_location

!----------------------------------------------------------------------------

subroutine set_obs_def_error_variance(obs_def, error_variance)

! Sets the error variance of an obs_def

type(obs_def_type), intent(inout) :: obs_def
real(r8), intent(in) :: error_variance

if ( .not. module_initialized ) call initialize_module

obs_def%error_variance = error_variance

end subroutine set_obs_def_error_variance

!----------------------------------------------------------------------------

subroutine set_obs_def_key(obs_def, key)

! Sets the key of an obs_def

type(obs_def_type), intent(inout) :: obs_def
integer,            intent(in)    :: key

if ( .not. module_initialized ) call initialize_module

obs_def%key = key

end subroutine set_obs_def_key

!----------------------------------------------------------------------------

subroutine set_obs_def_type_of_obs(obs_def, kind)

! Sets the kind of an obs_def

type(obs_def_type), intent(inout) :: obs_def
integer,               intent(in) :: kind

if ( .not. module_initialized ) call initialize_module

obs_def%kind = kind

end subroutine set_obs_def_type_of_obs

!----------------------------------------------------------------------------

subroutine set_obs_def_time(obs_def, time)

! Sets the time of an obs_def

type(obs_def_type), intent(inout) :: obs_def
type(time_type), intent(in) :: time

if ( .not. module_initialized ) call initialize_module

obs_def%time = time

end subroutine set_obs_def_time

!----------------------------------------------------------------------------

subroutine get_expected_obs_from_def_distrib_state(key, obs_def, obs_kind_ind, ens_index, &
   state_time, isprior, istatus, assimilate_this_ob, evaluate_this_ob, expected_obs, state_ens_handle, win)

! Compute forward operator for a particular obs_def
integer,            intent(in)  :: key
type(obs_def_type), intent(in)  :: obs_def
integer,            intent(in)  :: obs_kind_ind, ens_index
type(time_type),    intent(in)  :: state_time
logical,            intent(in)  :: isprior
integer,            intent(out) :: istatus(:)
logical,            intent(out) :: assimilate_this_ob, evaluate_this_ob

type(location_type) :: location
type(time_type)     :: obs_time
integer             :: obs_key
real(r8)            :: error_var

!HK
real(r8), intent(out) :: expected_obs(:)
integer, intent(in) :: win !> window for one sided communication
type(ensemble_type) state_ens_handle

! Load up the assimilate and evaluate status for this observation kind
assimilate_this_ob = assimilate_this_type_of_obs(obs_kind_ind)
evaluate_this_ob = evaluate_this_type_of_obs(obs_kind_ind)

! If not being assimilated or evaluated return with missing_r8 and istatus 0
if(assimilate_this_ob .or. evaluate_this_ob) then
   ! for speed, access directly instead of using accessor functions
   location  = obs_def%location
   obs_time  = obs_def%time
   obs_key   = obs_def%key
   error_var = obs_def%error_variance

   ! Compute the forward operator.  In spite of the variable name,
   ! obs_kind_ind is in fact a 'type' index number.  use the function
   ! get_quantity_for_type_of_obs from the obs_kind_mod if you want to map
   ! from a specific type to a generic kind.  the third argument of
   ! a call to the 'interpolate()' function must be a kind index and
   ! not a type.  normally the preprocess program does this for you.
   select case(obs_kind_ind)

      ! arguments available to an obs_def forward operator code are:
      !   state        -- the entire model state vector
      !   state_time   -- the time of the state vector data
      !   ens_index    -- the ensemble number
      !   location     -- observation location
      !   obs_kind_ind -- the index of the observation specific type
      !   obs_time     -- the time of the observation
      !   error_var    -- the observation error variance
      !   isprior      -- true for prior eval; false for posterior
      !
      ! the routine must return values for:
      !   obs_val -- the computed forward operator value
      !   istatus -- return code: 0=ok, >0 is error, <0 reserved for system use
      !
      ! to call interpolate() directly, the arg list MUST BE:
      !  interpolate(state, location, QTY_xxx, obs_val, istatus)
      !
      ! the preprocess program generates lines like this automatically,
      ! and this matches the interfaces in each model_mod.f90 file.
      !
      ! CASE statements and algorithms for specific observation kinds are
      ! inserted here by the DART preprocess program.

      ! DART PREPROCESS GET_EXPECTED_OBS_FROM_DEF INSERTED HERE
      case(RADIOSONDE_TEMPERATURE)
         call interpolate_distrib(location, QTY_TEMPERATURE, istatus, expected_obs, state_ens_handle, win)
      case(ACARS_TEMPERATURE)
         call interpolate_distrib(location, QTY_TEMPERATURE, istatus, expected_obs, state_ens_handle, win)
      case(AIRCRAFT_TEMPERATURE)
         call interpolate_distrib(location, QTY_TEMPERATURE, istatus, expected_obs, state_ens_handle, win)
      case(RADIOSONDE_U_WIND_COMPONENT)
         call interpolate_distrib(location, QTY_U_WIND_COMPONENT, istatus, expected_obs, state_ens_handle, win)
      case(RADIOSONDE_V_WIND_COMPONENT)
         call interpolate_distrib(location, QTY_V_WIND_COMPONENT, istatus, expected_obs, state_ens_handle, win)
      case(AIRCRAFT_U_WIND_COMPONENT)
         call interpolate_distrib(location, QTY_U_WIND_COMPONENT, istatus, expected_obs, state_ens_handle, win)
      case(AIRCRAFT_V_WIND_COMPONENT)
         call interpolate_distrib(location, QTY_V_WIND_COMPONENT, istatus, expected_obs, state_ens_handle, win)
      case(ACARS_U_WIND_COMPONENT)
         call interpolate_distrib(location, QTY_U_WIND_COMPONENT, istatus, expected_obs, state_ens_handle, win)
      case(ACARS_V_WIND_COMPONENT)
         call interpolate_distrib(location, QTY_V_WIND_COMPONENT, istatus, expected_obs, state_ens_handle, win)
      case(SAT_U_WIND_COMPONENT)
         call interpolate_distrib(location, QTY_U_WIND_COMPONENT, istatus, expected_obs, state_ens_handle, win)
      case(SAT_V_WIND_COMPONENT)
         call interpolate_distrib(location, QTY_V_WIND_COMPONENT, istatus, expected_obs, state_ens_handle, win)



      ! If the observation kind is not available, it is an error. The DART
      ! preprocess program should provide code for all available kinds.
      case DEFAULT
         call error_handler(E_ERR, 'get_expected_obs_from_def_distrib_state', &
            'Attempt to evaluate or assimilate undefined obs_kind type.', &
             source, revision, revdate)
   end select
else
   ! Not computing forward operator for this kind
   expected_obs(:) = missing_r8
   istatus = 0
endif

end subroutine get_expected_obs_from_def_distrib_state

!----------------------------------------------------------------------------


  subroutine read_obs_def(ifile, obs_def, key, obs_val, fform)
!----------------------------------------------------------------------------
! subroutine read_obs_def(ifile, obs_def, key, obs_val, fform)
!
! ifile
! obs_def
! key
! obs_val    needed if you want to perform operations based on value
! fform
!
! Reads an obs_def from file which is just an integer unit number in the
! current preliminary implementation.

integer,                    intent(in)    :: ifile
type(obs_def_type),         intent(inout) :: obs_def
integer,                    intent(in)    :: key
real(r8),                   intent(inout) :: obs_val
character(len=*), optional, intent(in)    :: fform

character(len=5)  :: header
integer           :: o_index
logical           :: is_ascii
character(len=32) :: fileformat   ! here for backwards compatibility only

if ( .not. module_initialized ) call initialize_module

is_ascii = ascii_file_format(fform)

! here for backwards compatibility only; after the next release,
! remove this and force people in their own obs_def_xxx_mod.f90 code
! to set the last arg to read/write to be fform instead of fileformat
if (is_ascii) then
   fileformat = 'formatted'
else
   fileformat = 'unformatted'
endif

! Begin by reading five character ascii header, then location, kind, error variance, index

! Need to add additional error checks on read
if (is_ascii) then
   read(ifile, '(a5)') header
   if(header /= 'obdef') then
      call error_handler(E_ERR,'read_obs_def', &
         'Expected header "obdef" in input file', source, revision, revdate)
   endif
endif

! Read the location, kind, time and error variance
obs_def%location = read_location(ifile, fform)
if (is_ascii) then
   read(ifile, '(a5)' ) header
   if(header /= 'kind ') then
      call error_handler(E_ERR,'read_kind', &
         'Expected kind header "kind " in input file', &
          source, revision, revdate)
   endif
   read(ifile, *) o_index
else
   read(ifile)    o_index
endif

! Negative value is identity obs, doesn't need mapped
! Positive value must use mapping to get to proper index in obs_kind
if(o_index < 0) then
   obs_def%kind = o_index
else
   obs_def%kind = map_type_of_obs_table(o_index)
endif

! This kind may have its own module that needs to read more
select case(obs_def%kind)
   ! More complicated kinds may require reading additional information from
   ! an observation sequence file. Case code to do this is inserted here by
   ! the DART preprocess program.

         case(GPSRO_REFRACTIVITY)
            call read_gpsro_ref(obs_def%key, ifile, fform)
         case(RADIOSONDE_SURFACE_ALTIMETER, DROPSONDE_SURFACE_ALTIMETER, MARINE_SFC_ALTIMETER, &
              LAND_SFC_ALTIMETER, METAR_ALTIMETER)
            continue
   case(TEMPERATURE)
      continue
   case(SPECIFIC_HUMIDITY)
      continue
   case(PRESSURE)
      continue
   case(RADIOSONDE_U_WIND_COMPONENT)
      continue
   case(RADIOSONDE_V_WIND_COMPONENT)
      continue
   case(RADIOSONDE_GEOPOTENTIAL_HGT)
      continue
   case(RADIOSONDE_SURFACE_PRESSURE)
      continue
   case(RADIOSONDE_TEMPERATURE)
      continue
   case(RADIOSONDE_SPECIFIC_HUMIDITY)
      continue
   case(DROPSONDE_U_WIND_COMPONENT)
      continue
   case(DROPSONDE_V_WIND_COMPONENT)
      continue
   case(DROPSONDE_SURFACE_PRESSURE)
      continue
   case(DROPSONDE_TEMPERATURE)
      continue
   case(DROPSONDE_SPECIFIC_HUMIDITY)
      continue
   case(AIRCRAFT_U_WIND_COMPONENT)
      continue
   case(AIRCRAFT_V_WIND_COMPONENT)
      continue
   case(AIRCRAFT_TEMPERATURE)
      continue
   case(AIRCRAFT_SPECIFIC_HUMIDITY)
      continue
   case(ACARS_U_WIND_COMPONENT)
      continue
   case(ACARS_V_WIND_COMPONENT)
      continue
   case(ACARS_TEMPERATURE)
      continue
   case(ACARS_SPECIFIC_HUMIDITY)
      continue
   case(MARINE_SFC_U_WIND_COMPONENT)
      continue
   case(MARINE_SFC_V_WIND_COMPONENT)
      continue
   case(MARINE_SFC_TEMPERATURE)
      continue
   case(MARINE_SFC_SPECIFIC_HUMIDITY)
      continue
   case(MARINE_SFC_PRESSURE)
      continue
   case(LAND_SFC_U_WIND_COMPONENT)
      continue
   case(LAND_SFC_V_WIND_COMPONENT)
      continue
   case(LAND_SFC_TEMPERATURE)
      continue
   case(LAND_SFC_SPECIFIC_HUMIDITY)
      continue
   case(LAND_SFC_PRESSURE)
      continue
   case(SAT_U_WIND_COMPONENT)
      continue
   case(SAT_V_WIND_COMPONENT)
      continue
   case(ATOV_TEMPERATURE)
      continue
   case(AIRS_TEMPERATURE)
      continue
   case(AIRS_SPECIFIC_HUMIDITY)
      continue

! A negative value means identity observations, just move along
   case (:-1)
      continue

   case DEFAULT
      call error_handler(E_ERR, 'read_obs_def', &
         'Attempt to read for undefined obs_kind type.', &
         source, revision, revdate)
end select

! Read the time for the observation
obs_def%time = read_time(ifile, fform)

if (is_ascii) then
   read(ifile, *) obs_def%error_variance
else
   read(ifile)    obs_def%error_variance
endif

end subroutine read_obs_def

!----------------------------------------------------------------------------

subroutine write_obs_def(ifile, obs_def, key, fform)

! Writes an obs_def to file.

integer,                    intent(in) :: ifile
type(obs_def_type),         intent(in) :: obs_def
integer,                    intent(in) :: key
character(len=*), intent(in), optional :: fform

logical           :: is_ascii
character(len=32) :: fileformat   ! here for backwards compatibility only

if ( .not. module_initialized ) call initialize_module

is_ascii = ascii_file_format(fform)

! here for backwards compatibility only; after the next release,
! remove this and force people in their own obs_def_xxx_mod.f90 code
! to set the last arg to read/write to be fform instead of fileformat
if (is_ascii) then
   fileformat = 'formatted'
else
   fileformat = 'unformatted'
endif

! Write the 5 character identifier for verbose formatted output
if (is_ascii) write(ifile, '("obdef")')

! Write out the location, kind and error variance
call write_location(ifile, obs_def%location, fform)
if (is_ascii) then
   write(ifile, '("kind")' )
   write(ifile, *) obs_def%kind
else
   write(ifile)    obs_def%kind
endif

! This kind may have its own module that needs to write more
select case(obs_def%kind)
   ! More complicated kinds may require writing additional information from
   ! an observation sequence file. Case code to do this is inserted here by
   ! the DART preprocess program.

         case(GPSRO_REFRACTIVITY)
            call write_gpsro_ref(obs_def%key, ifile, fform)
         case(RADIOSONDE_SURFACE_ALTIMETER, DROPSONDE_SURFACE_ALTIMETER, MARINE_SFC_ALTIMETER, &
              LAND_SFC_ALTIMETER, METAR_ALTIMETER)
            continue
   case(TEMPERATURE)
      continue
   case(SPECIFIC_HUMIDITY)
      continue
   case(PRESSURE)
      continue
   case(RADIOSONDE_U_WIND_COMPONENT)
      continue
   case(RADIOSONDE_V_WIND_COMPONENT)
      continue
   case(RADIOSONDE_GEOPOTENTIAL_HGT)
      continue
   case(RADIOSONDE_SURFACE_PRESSURE)
      continue
   case(RADIOSONDE_TEMPERATURE)
      continue
   case(RADIOSONDE_SPECIFIC_HUMIDITY)
      continue
   case(DROPSONDE_U_WIND_COMPONENT)
      continue
   case(DROPSONDE_V_WIND_COMPONENT)
      continue
   case(DROPSONDE_SURFACE_PRESSURE)
      continue
   case(DROPSONDE_TEMPERATURE)
      continue
   case(DROPSONDE_SPECIFIC_HUMIDITY)
      continue
   case(AIRCRAFT_U_WIND_COMPONENT)
      continue
   case(AIRCRAFT_V_WIND_COMPONENT)
      continue
   case(AIRCRAFT_TEMPERATURE)
      continue
   case(AIRCRAFT_SPECIFIC_HUMIDITY)
      continue
   case(ACARS_U_WIND_COMPONENT)
      continue
   case(ACARS_V_WIND_COMPONENT)
      continue
   case(ACARS_TEMPERATURE)
      continue
   case(ACARS_SPECIFIC_HUMIDITY)
      continue
   case(MARINE_SFC_U_WIND_COMPONENT)
      continue
   case(MARINE_SFC_V_WIND_COMPONENT)
      continue
   case(MARINE_SFC_TEMPERATURE)
      continue
   case(MARINE_SFC_SPECIFIC_HUMIDITY)
      continue
   case(MARINE_SFC_PRESSURE)
      continue
   case(LAND_SFC_U_WIND_COMPONENT)
      continue
   case(LAND_SFC_V_WIND_COMPONENT)
      continue
   case(LAND_SFC_TEMPERATURE)
      continue
   case(LAND_SFC_SPECIFIC_HUMIDITY)
      continue
   case(LAND_SFC_PRESSURE)
      continue
   case(SAT_U_WIND_COMPONENT)
      continue
   case(SAT_V_WIND_COMPONENT)
      continue
   case(ATOV_TEMPERATURE)
      continue
   case(AIRS_TEMPERATURE)
      continue
   case(AIRS_SPECIFIC_HUMIDITY)
      continue

   ! A negative value means identity observations, just move along
   case (:-1)
      continue

   case DEFAULT
      call error_handler(E_ERR, 'write_obs_def', &
         'Attempt to write for undefined obs_kind type.', &
         source, revision, revdate)
end select

call write_time(ifile, obs_def%time, fform)

if (is_ascii) then
   write(ifile, *) obs_def%error_variance
else
   write(ifile)    obs_def%error_variance
endif

end subroutine write_obs_def


subroutine interactive_obs_def(obs_def, key)
!---------------------------------------------------------------------------
!
! Allows interactive creation of an observation

type(obs_def_type), intent(inout) :: obs_def
integer,               intent(in) :: key

if ( .not. module_initialized ) call initialize_module

! Get the observation kind WANT A STRING OPTION, TOO?
obs_def%kind = get_type_of_obs_from_menu()

! Input any special stuff for this kind
select case(obs_def%kind)
   ! More complicated kinds may require inputting additional information to
   ! define an observation. Case code to do this is inserted here by the
   ! DART preprocess program.

         case(GPSRO_REFRACTIVITY)
            call interactive_gpsro_ref(obs_def%key)
         case(RADIOSONDE_SURFACE_ALTIMETER, DROPSONDE_SURFACE_ALTIMETER, MARINE_SFC_ALTIMETER, &
              LAND_SFC_ALTIMETER, METAR_ALTIMETER)
            continue
   case(TEMPERATURE)
      continue
   case(SPECIFIC_HUMIDITY)
      continue
   case(PRESSURE)
      continue
   case(RADIOSONDE_U_WIND_COMPONENT)
      continue
   case(RADIOSONDE_V_WIND_COMPONENT)
      continue
   case(RADIOSONDE_GEOPOTENTIAL_HGT)
      continue
   case(RADIOSONDE_SURFACE_PRESSURE)
      continue
   case(RADIOSONDE_TEMPERATURE)
      continue
   case(RADIOSONDE_SPECIFIC_HUMIDITY)
      continue
   case(DROPSONDE_U_WIND_COMPONENT)
      continue
   case(DROPSONDE_V_WIND_COMPONENT)
      continue
   case(DROPSONDE_SURFACE_PRESSURE)
      continue
   case(DROPSONDE_TEMPERATURE)
      continue
   case(DROPSONDE_SPECIFIC_HUMIDITY)
      continue
   case(AIRCRAFT_U_WIND_COMPONENT)
      continue
   case(AIRCRAFT_V_WIND_COMPONENT)
      continue
   case(AIRCRAFT_TEMPERATURE)
      continue
   case(AIRCRAFT_SPECIFIC_HUMIDITY)
      continue
   case(ACARS_U_WIND_COMPONENT)
      continue
   case(ACARS_V_WIND_COMPONENT)
      continue
   case(ACARS_TEMPERATURE)
      continue
   case(ACARS_SPECIFIC_HUMIDITY)
      continue
   case(MARINE_SFC_U_WIND_COMPONENT)
      continue
   case(MARINE_SFC_V_WIND_COMPONENT)
      continue
   case(MARINE_SFC_TEMPERATURE)
      continue
   case(MARINE_SFC_SPECIFIC_HUMIDITY)
      continue
   case(MARINE_SFC_PRESSURE)
      continue
   case(LAND_SFC_U_WIND_COMPONENT)
      continue
   case(LAND_SFC_V_WIND_COMPONENT)
      continue
   case(LAND_SFC_TEMPERATURE)
      continue
   case(LAND_SFC_SPECIFIC_HUMIDITY)
      continue
   case(LAND_SFC_PRESSURE)
      continue
   case(SAT_U_WIND_COMPONENT)
      continue
   case(SAT_V_WIND_COMPONENT)
      continue
   case(ATOV_TEMPERATURE)
      continue
   case(AIRS_TEMPERATURE)
      continue
   case(AIRS_SPECIFIC_HUMIDITY)
      continue

   ! A negative value means identity observations, just move along
   case (:-1)
      continue
   case DEFAULT
      call error_handler(E_ERR, 'interactive_obs_def', &
         'Attempt to interactively create undefined obs_kind type.', &
         source, revision, revdate)
end select

! If the kind is an identity observation, don't need to call location
! Get location from state meta_data
if(obs_def%kind < 0) then
! Get the location of this from model
   call get_state_meta_data(-1 * obs_def%kind, obs_def%location)
else! Get the location
   call interactive_location(obs_def%location)
endif

! Get the time
call interactive_time(obs_def%time)

write(*, *) 'Input error variance for this observation definition '
read(*, *) obs_def%error_variance

! TJH -- might want to do some sort of error checking (i.e. for positive values)

end subroutine interactive_obs_def

!----------------------------------------------------------------

subroutine destroy_obs_def(obs_def)
! TECHNICALLY NEED TO CALL DESTRUCTORS FOR ALL SUBCOMPONENTS,
! NO ALLOCATED STORAGE YET

type(obs_def_type), intent(inout) :: obs_def

if ( .not. module_initialized ) call initialize_module

call set_obs_def_location(obs_def, set_location_missing() )
obs_def%kind = missing_i
call set_obs_def_time(obs_def, set_time_missing() )
call set_obs_def_error_variance( obs_def, missing_r8)

end subroutine destroy_obs_def

!---------------------------------------------------------------------------
!---------------------------------------------------------------------------

end module obs_def_mod

! <next few lines under version control, do not edit>
! $URL$
! $Id$
! $Revision$
! $Date$
