! Data Assimilation Research Testbed -- DART
! Copyright 2004, Data Assimilation Initiative, University Corporation for Atmospheric Research
! Licensed under the GPL -- www.gpl.org/licenses/gpl.html

MODULE module_netcdf_interface

! <next five lines automatically updated by CVS, do not edit>
! $Source: /home/thoar/CVS.REPOS/DART/models/wrf/WRF_BC/module_netcdf_interface.f90,v $
! $Revision: 1.10 $
! $Date: 2004/07/01 16:05:11 $
! $Author: caya $
! $Id: module_netcdf_interface.f90,v 1.10 2004/07/01 16:05:11 caya Exp $

  use        types_mod, only : r8
  use    utilities_mod, only : error_handler, E_ERR

  use netcdf

  implicit none
  private

public  get_dims_cdf,        &
        get_gl_att_real_cdf, &
        put_gl_att_real_cdf, &
        get_var_3d_real_cdf, &
        get_var_2d_real_cdf, &
        put_var_3d_real_cdf, &
        put_var_2d_real_cdf, &
        get_times_cdf,       &
        put_time_cdf,        &
        netcdf_read_write_var, &
        netcdf_read_write_char

!-----------------------------------------------------------------------
! CVS Generated file description for error handling, do not edit
character(len=128) :: &
source   = "$Source: /home/thoar/CVS.REPOS/DART/models/wrf/WRF_BC/module_netcdf_interface.f90,v $", &
revision = "$Revision: 1.10 $", &
revdate  = "$Date: 2004/07/01 16:05:11 $"


CONTAINS

!--------------------------------------------------------------------

subroutine get_dims_cdf( file, var, idims, ndims, debug )

  implicit none

  character (len=80),     intent(in) :: file
  character (len=*),      intent(in) :: var
  logical,                intent(in) :: debug
  integer, intent(out), dimension(4) :: idims
  integer,               intent(out) :: ndims

  integer  :: cdfid, id_data, i
  integer  :: dimids(4)

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_NOWRITE, cdfid) )

  call check( nf90_inq_varid(cdfid, var, id_data) )

  call check( nf90_Inquire_Variable(cdfid, id_data, ndims=ndims, dimids=dimids) )

  if(debug) write(6,*) ' number of dims for ',var,' ',ndims

  do i=1,ndims
     call check( nf90_inquire_dimension(cdfid, dimids(i), len=idims(i)) )
     if(debug) write(6,*) ' dimension ',i,idims(i)
  enddo

  call check( nf90_close(cdfid) )

contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'get_dims_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check

end subroutine get_dims_cdf

!-------------------------------------------------------------------------------

subroutine get_gl_att_real_cdf( file, att_name, value, debug )

  implicit none

  character (len=80), intent(in) :: file
  character (len=*),  intent(in) :: att_name
  logical,            intent(in) :: debug
  real(r8),          intent(out) :: value

  integer :: cdfid

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_NOWRITE, cdfid) )

  call check( nf90_get_att(cdfid, nf90_global, att_name, value) )

  if(debug) write(6,*) ' global attribute ',att_name,' is ',value

  call check( nf90_close(cdfid) )

contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'get_gl_att_real_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check

end subroutine get_gl_att_real_cdf

!-------------------------------------------------------------------------------

subroutine put_gl_att_real_cdf( file, att_name, value, debug )

  implicit none

  character (len=80), intent(in) :: file
  character (len=*),  intent(in) :: att_name
  logical,            intent(in) :: debug
  real(r8),           intent(in) :: value

  integer :: cdfid

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_WRITE, cdfid) )

  call check( nf90_redef(cdfid) )
  call check( nf90_put_att(cdfid, nf90_global, att_name, value) )

  if(debug) write(6,*) ' global attribute ',att_name,' is ',value

  call check( nf90_close(cdfid) )

contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'get_gl_att_real_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check


end subroutine put_gl_att_real_cdf


!--------------------------------------------------------------------

subroutine get_var_3d_real_cdf( file, var, data, &
     i1, i2, i3, time, debug )

  implicit none

  integer,                    intent(in) :: i1, i2, i3, time
  character (len=80),         intent(in) :: file
  logical,                    intent(in) :: debug
  character (len=*),          intent(in) :: var
  real(r8), dimension(i1,i2,i3), intent(out) :: data

  integer            :: cdfid, id_data
  character (len=80) :: varnam
  integer            :: ndims, idims(4), dimids(4)
  integer            :: i, ivtype

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_NOWRITE, cdfid) )

  call check( nf90_inq_varid(cdfid, var, id_data) )

  call check( nf90_Inquire_Variable(cdfid, id_data, name=varnam, xtype=ivtype, ndims=ndims, dimids=dimids) )

  if(debug) then
     write(6,*) ' number of dims for ',var,' ',ndims
     write(unit=*, fmt='(a,i6)') ' ivtype=', ivtype
     write(unit=*, fmt='(a, a)') ' varnam=', trim(varnam)
     write(unit=*, fmt='(a,i6)') ' kind(data)=', kind(data)
  endif

  do i=1,ndims
     call check( nf90_inquire_dimension(cdfid, dimids(i), len=idims(i)) )
     if(debug) write(6,*) ' dimension ',i,idims(i)
  enddo

!  check the dimensions

  if(  (i1 /= idims(1)) .or.  &
       (i2 /= idims(2)) .or.  &
       (i3 /= idims(3)) .or.  &
       (time > idims(4))     )  then

     write(6,*) ' error in get_var_3d_real read, dimension problem '
     write(6,*) i1, idims(1)
     write(6,*) i2, idims(2)
     write(6,*) i3, idims(3)
     write(6,*) time, idims(4)
     stop

  end if

!  get the data
  
  call check( nf90_get_var(cdfid, id_data, data, start = (/ 1, 1, 1, time /)) )

  if(debug) write(unit=*, fmt='(a,e24.12)') ' Sample data=', data(1,1,1)

  call check( nf90_close(cdfid) )

contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'get_var_3d_real_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check

end subroutine get_var_3d_real_cdf

!--------------------------------------------------------------------

subroutine get_var_2d_real_cdf( file, var, data, &
     i1, i2, time, debug )

  implicit none

  integer,                 intent(in) :: i1, i2, time
  character (len=80),      intent(in) :: file
  logical,                 intent(in) :: debug
  character (len=*),       intent(in) :: var
  real(r8), dimension(i1,i2), intent(out) :: data

  integer            :: cdfid, id_data
  character (len=80) :: varnam
  integer            :: ndims, idims(4), dimids(4)
  integer            :: i, ivtype

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_NOWRITE, cdfid) )

  call check( nf90_inq_varid(cdfid, var, id_data) )

  call check( nf90_Inquire_Variable(cdfid, id_data, name=varnam, xtype=ivtype, ndims=ndims, dimids=dimids) )

  if(debug) then
     write(6,*) ' number of dims for ',var,' ',ndims
     write(unit=*, fmt='(a,i6)') ' ivtype=', ivtype
     write(unit=*, fmt='(a, a)') ' varnam=', trim(varnam)
  endif

  do i=1,ndims
     call check( nf90_inquire_dimension(cdfid, dimids(i), len=idims(i)) )
     if(debug) write(6,*) ' dimension ',i,idims(i)
  enddo

!  check the dimensions

  if(  (i1 /= idims(1)) .or.  &
       (i2 /= idims(2)) .or.  &
       (time > idims(3))     )  then

     write(6,*) ' error in get_var_2d_real read, dimension problem '
     write(6,*) i1, idims(1)
     write(6,*) i2, idims(2)
     write(6,*) time, idims(4)
     stop

  end if

!  get the data

  call check( nf90_get_var(cdfid, id_data, data, start = (/ 1, 1, 1 /)) )

  if(debug) write(unit=*, fmt='(a,e24.12)') ' Sample data=', data(1,1)

  call check( nf90_close(cdfid) )

contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'get_var_2d_real_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check

end subroutine get_var_2d_real_cdf

!--------------------------------------------------------------------

subroutine put_var_3d_real_cdf( file, var, data, &
     i1, i2, i3, time, debug )

  implicit none

  integer,                       intent(in) :: i1, i2, i3, time
  character (len=80),            intent(in) :: file
  logical,                       intent(in) :: debug
  character (len=*),             intent(in) :: var
  real(r8), dimension(i1,i2,i3), intent(in) :: data

  integer            :: cdfid, id_data
  character (len=80) :: varnam
  integer            :: ndims, idims(4), dimids(4)
  integer            :: i, ivtype

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_WRITE, cdfid) )

  call check( nf90_inq_varid(cdfid, var, id_data) )

  call check( nf90_Inquire_Variable(cdfid, id_data, name=varnam, xtype=ivtype, ndims=ndims, dimids=dimids) )

  if(debug) write(6,*) ' number of dims for ',var,' ',ndims

  do i=1,ndims
     call check( nf90_inquire_dimension(cdfid, dimids(i), len=idims(i)) )
     if(debug) write(6,*) ' dimension ',i,idims(i)
  enddo

!  check the dimensions

  if(  (i1 /= idims(1)) .or.  &
       (i2 /= idims(2)) .or.  &
       (i3 /= idims(3)) .or.  &
       (time > idims(4))     )  then

     write(6,*) ' error in put_var_3d_real read, dimension problem '
     write(6,*) i1, idims(1)
     write(6,*) i2, idims(2)
     write(6,*) i3, idims(3)
     write(6,*) time, idims(4)
     stop

  end if

!  write the data
  
  call check( nf90_put_var(cdfid, id_data, data, start = (/ 1, 1, 1, time /)) )

  call check( nf90_close(cdfid) )

contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'put_var_3d_real_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check

end subroutine put_var_3d_real_cdf

!--------------------------------------------------------------------

subroutine put_var_2d_real_cdf( file, var, data, &
     i1, i2, time, debug )

  implicit none

  integer,                    intent(in) :: i1, i2, time
  character (len=80),         intent(in) :: file
  logical,                    intent(in) :: debug
  character (len=*),          intent(in) :: var
  real(r8), dimension(i1,i2), intent(in) :: data

  integer            :: cdfid, id_data
  character (len=80) :: varnam
  integer            :: ndims, idims(3), dimids(3)
  integer            :: i, ivtype

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_WRITE, cdfid) )

  call check( nf90_inq_varid(cdfid, var, id_data) )

  call check( nf90_Inquire_Variable(cdfid, id_data, name=varnam, xtype=ivtype, ndims=ndims, dimids=dimids) )

  if(debug) write(6,*) ' number of dims for ',var,' ',ndims

  do i=1,ndims
     call check( nf90_inquire_dimension(cdfid, dimids(i), len=idims(i)) )
     if(debug) write(6,*) ' dimension ',i,idims(i)
  enddo

!  check the dimensions

  if(  (i1 /= idims(1)) .or.  &
       (i2 /= idims(2)) .or.  &
       (time > idims(3))     )  then

     write(6,*) ' error in put_var_2d_real read, dimension problem '
     write(6,*) i1, idims(1)
     write(6,*) i2, idims(2)
     write(6,*) time, idims(3)
     stop

  end if

!  write the data
  
  call check( nf90_put_var(cdfid, id_data, data, start = (/ 1, 1, time /)) )

  call check( nf90_close(cdfid) )

contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'put_var_2d_real_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check

end subroutine put_var_2d_real_cdf

!--------------------------------------------------------------------

subroutine get_times_cdf( file, time_name, times, n_times, max_times, debug )

  implicit none

  integer,            intent(in)  :: max_times
  integer,            intent(out) :: n_times
  character (len=80), intent(in)  :: file, time_name
  character (len=19), intent(out) :: times(max_times)
  logical,            intent(in)  :: debug

  integer            :: cdfid, id_time
  character (len=80) :: varnam, time1
  integer            :: ndims, idims(max_times)
  integer            :: istart(max_times),iend(max_times), dimids(max_times)
  integer            :: i, ivtype

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_NOWRITE, cdfid) )

  call check( nf90_inq_varid(cdfid, time_name, id_time) )

  call check( nf90_Inquire_Variable(cdfid, id_time, name=varnam, xtype=ivtype, ndims=ndims, dimids=dimids) )

  do i=1,ndims
     call check( nf90_inquire_dimension(cdfid, dimids(i), len=idims(i)) )
     if(debug) write(6,*) ' dimension ',i,idims(i)
  enddo

!  get the times

  n_times = idims(2)
  do i=1,idims(2)
    istart(1) = 1
    iend(1) = idims(1)
    istart(2) = i
    iend(2) = 1

    call check( nf90_get_var(cdfid, id_time, times(i), start = (/ 1, i /), &
                             count = (/idims(1), 1/) ) )
!    rcode = NF_GET_VARA_TEXT  ( cdfid, id_time,  &
!                                istart, iend,    &
!                                times(i)          )
    time1 = times(i)

    if(debug) write(6,*) trim(file), time1(1:19)
  enddo

  call check( nf90_close(cdfid) )
contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'get_times_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check

end subroutine get_times_cdf

!--------------------------------------------------------------------

subroutine put_time_cdf( file, time_name, char, itime, debug )

  implicit none

  integer,            intent(in)  :: itime
  character (len=80), intent(in)  :: file, time_name
  character (len=19), intent(out) :: char
  logical,            intent(in)  :: debug

  integer            :: cdfid, id_time

  if(debug) write(6,*) ' open netcdf file ', trim(file)

  call check( nf90_open(file, NF90_WRITE, cdfid) )

  call check( nf90_inq_varid(cdfid, time_name, id_time) )

  call check( nf90_put_var(cdfid, id_time, char, start = (/ 1, itime /)) )

  call check( nf90_close(cdfid) )

contains

  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent (in) :: istatus

    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'put_time_cdf', &
       trim(nf90_strerror(istatus)), source, revision, revdate)

  end subroutine check

end subroutine put_time_cdf

!**********************************************************************

subroutine netcdf_read_write_var( variable, ncid, var_id, var,          &
                                  start, kount, stride, map, in_or_out, debug, ndims )

! Rewritten to use some F90 interface and totally replace the dart_to_wrf 
! module routine. There used to be TWO routines that did the same thing.
! 
! As such 'kount', 'stride','map' all become meaningless because
! the whole process is slaved to simply replace an existing netcdf
! variable with a conformable variable -- no possibility for 
! us to write a subset of the domain.

integer                    :: ncid, var_id, ndims
real(r8), dimension(ndims) :: var
character (len=6)          :: in_or_out
integer, dimension(ndims)  :: start, kount, stride, map
character (len=*)          :: variable
logical                    :: debug
character (len=129)        :: error_string

if(debug) write(6,*) ' var for io is ',variable
call check( nf90_inq_varid(ncid, variable, var_id) )
if(debug) write(6,*) variable, ' id = ',var_id

if( in_or_out(1:5) == "INPUT" ) then

   call check( nf90_get_var(ncid, var_id, var, start=start, count=kount, stride=stride) )

else if( in_or_out(1:6) == "OUTPUT" ) then

   call check( nf90_put_var(ncid, var_id, var, start, kount, stride, map) )

else

  write(error_string,*)' unknown IO function for var_id ',var_id, in_or_out
  call error_handler(E_ERR,'netcdf_read_write_var', &
       error_string, source, revision,revdate)

end if

contains
  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent ( in) :: istatus
    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'netcdf_read_write_var', &
       trim(nf90_strerror(istatus)), source, revision, revdate)
  end subroutine check

end subroutine netcdf_read_write_var

!**********************************************************************

subroutine netcdf_read_write_char( variable, ncid, var_id, var,          &
                                  start, kount, stride, map, in_or_out, debug, ndims )

! Rewritten to use some F90 interface and totally replace the dart_to_wrf 
! module routine. There used to be TWO routines that did the same thing.
! 
! As such 'kount', 'stride','map' all become meaningless because
! the whole process is slaved to simply replace an existing netcdf
! variable with a conformable variable -- no possibility for 
! us to write a subset of the domain.

integer                            :: ncid, var_id, ndims
character(len=1), dimension(ndims) :: var
character (len=6)                  :: in_or_out
integer, dimension(ndims)          :: start, kount, stride, map
character (len=*)                  :: variable
logical                            :: debug
character (len=129) :: error_string

if(debug) write(6,*) ' var for io is ',variable
call check( nf90_inq_varid(ncid, variable, var_id) )
if(debug) write(6,*) variable, ' id = ',var_id

if( in_or_out(1:5) == "INPUT" ) then

  call check( nf90_get_var(ncid, var_id, var(1:kount(1)), start=start, count=kount, stride=stride) )

else if( in_or_out(1:6) == "OUTPUT" ) then

  call check( nf90_put_var(ncid, var_id, var(1:kount(1)), start, kount, stride, map) )

else

  write(error_string,*)' unknown IO function for var_id ',var_id, in_or_out
  call error_handler(E_ERR,'netcdf_read_write_char', &
       error_string, source, revision,revdate)

end if

contains
  ! Internal subroutine - checks error status after each netcdf, prints 
  !                       text message each time an error code is returned. 
  subroutine check(istatus)
    integer, intent ( in) :: istatus
    if(istatus /= nf90_noerr) call error_handler(E_ERR, 'netcdf_read_write_char', &
       trim(nf90_strerror(istatus)), source, revision, revdate)
  end subroutine check
end subroutine netcdf_read_write_char

END MODULE module_netcdf_interface
