load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_code.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_csm.ncl"
load "./src/TrackColors.ncl"
load "./src/MapWindow.ncl"
load "./src/PlotHeader.ncl"
load "./src/symbols.ncl"
load "./src/GetTrackData.ncl"
load "./src/GetLineNums.ncl"
load "./src/PlotTrack.ncl"
load "./src/PlotLegends.ncl"
load "./src/Cleanup.ncl"

begin

  ;----------------------------------------------------------------------
  ; Get DataSource and file info from command line
  ;----------------------------------------------------------------------
      tkres = True

      ; Make sure we have a datafile to work with
      if (.not. isvar("inputFILE") ) then
        print(" ")
        print(" ### MUST SUPPLY a inputFILE ### ")
        print(" ")
        print("     Something like: ")
        print("     ncl CreateTracks.ncl inputFILE=track_2005082800_12k stormNAME=Katrina DataSource=RIP4 wksTYPE=X11 ")
        print("     REMEMBER TO ADD QUOTES" )
        exit
      end if
      tkres@inputFILE = inputFILE

    ; DataSource = AFWA (data from WRF model)
    ; DataSource = RIP4 (data from RIP)
      if (.not. isvar("DataSource") ) then
        DataSource = "AFWA"
      end if
      if (DataSource .ne. "RIP4" .and. DataSource .ne. "AFWA") then
        print("'" + DataSource + "'" + " is not a valid DataSource")
        print("  Current options are :")
        print("  RIP4 for data generated by rip4")
        print("  AFWA for data generated by the WRF model")
        exit
      end if
      tkres@DataSource = DataSource
      print("Input is " + tkres@DataSource + " type data")

    ; Do we have any special data that comes in at a different interval from default
    ; As we only care about this for RIP4 data (AFWA has all the info in the file)
    ; We are going to make the default 3h, for 12km RIP4 data
      if (.not. isvar("DataInt") .or. DataInt .eq. 0.0) then
         DataInt = 3.0
      end if
      tkres@SetInterval = DataInt

  ;----------------------------------------------------------------------
  ; Are we dealing with SUMMARIES
  ;----------------------------------------------------------------------
      if (.not. isvar("plotType") ) then
         plotType = 0
      end if
      if (plotType .eq. 0) then
         filename = inputFILE
         tkres@plotType = 0
         print("Doing a single time plot")
      else
         filename   = (systemfunc("cut -d' ' -f1 " + inputFILE)) 
         TRHeader = new(dimsizes(filename),string)
         tkres@plotType = plotType
         print("Doing summary plot type " + tkres@plotType)
      end if

  ;----------------------------------------------------------------------
  ; Basic Setup                  
  ;----------------------------------------------------------------------

  ; open the workstation
    if (.not. isvar("wksTYPE") ) then
      wksTYPE = "pdf"
    end if
    tkres@wksTYPE = wksTYPE
    if (.not. isvar("outputFILE") ) then
      outputFILE = "hur_track"
    end if
    tkres@outputFILE = outputFILE
    wks = gsn_open_wks(wksTYPE,outputFILE)  

  ; set up header
    if (.not. isvar("stormNAME") ) then
      header = "ARW Forecast"
      stormNAME = "UnKnown"
    else
      header = "ARW Forecast: " + stormNAME
    end if
    tkres@header = header
    tkres@stormNAME = stormNAME

    ; are we doing the best track
    if (.not. isvar("bTrack") ) then
      if (tkres@plotType .eq. 0) then
        bTrack = False
      else
        bTrack = True
      end if
    end if
    tkres@bTrack = bTrack
    if (.not. isvar("btFile") ) then
      btFile = "BestTrack"
    end if
    tkres@btFile = btFile

    ; less common settings
    ; rotate the text angle on the track line
    if (.not. isvar("textAngle") ) then
      textAngle=0.0
    end if
    tkres@textAngle = textAngle
    ; rotate the output .pdf file
    if (.not. isvar("rot") ) then
      rot=-90
    end if
    tkres@rotPDF = rot

    ; change the start number for summary 4 plots
    if (.not. isvar("StartLineColor") ) then
      StartLineColor=5 
    end if
    tkres@StartLineColor = StartLineColor

    ; Change the default directory used for web pages
    ; This is only valid for RIP4 input data
    if (.not. isvar("tarDIR") ) then
      tarDIR="hur12" 
    end if
    tkres@tarDIR = tarDIR
    

  ; Set return character
    cr = "~C~"


  ;----------------------------------------------------------------------
  ; Read storm track and info
  ;----------------------------------------------------------------------

    do iNumFil=0,dimsizes(filename)-1

      tkres@iNumFil  = iNumFil
      tkres@filename = filename(iNumFil)

      if ( DataSource .eq. "RIP4" ) then   ; need to clean up the datafile
       ; First figure out some line numbers
         line_breaks = GetLineNums(tkres)
         system ( "csh -c 'rm -f TimeInfo' ")
         system ( "csh -c 'rm -f TrackInfo' ")
         system ( "csh -c 'rm -f TrackInput' ")
         system ( "csh -c 'CreateTracks.csh "+tkres@filename+" "+tkres@SetInterval+" "+line_breaks(0)+" "+line_breaks(1)+" "+line_breaks(2)+" ' ")
      end if
  
      ll_lev_wnd     = GetTrackData(tkres)
                       lat = ll_lev_wnd(0,:)
                       lon = ll_lev_wnd(1,:)
                       lev = ll_lev_wnd(2,:)
                       wnd = ll_lev_wnd(3,:)
      interval       = GetInterval(tkres)
                       tkres@interval = interval
      dd_hh          = Get_dd_hh(tkres)
      time_start_end = GetTimeInfo(tkres)
      cat            = GetCatInfo(tkres)
                       tkres@DataType = cat(0)

  ; set up the data the way we want to plot it
    slev   = sprintf("  %4.0f",lev)
    swnd   = sprintf("  %5.1f",wnd)
    wndlev = slev + "    " + cr + swnd + "    "  

    ; Create top right header
    if (tkres@plotType .eq. 0) then
      TRHeader = time_start_end
    else
      TRHeader(iNumFil) = time_start_end(0)
    end if

  ;----------------------------------------------------------------------
  ; Set up the window
  ;----------------------------------------------------------------------

    ; Set user over ride of plot domain to true. If user does not use
    ;     the override commands, this will be set to False again
    tkres@latMINset = True
    tkres@latMAXset = True
    tkres@lonMINset = True
    tkres@lonMAXset = True
    if (iNumFil .eq. 0) then
      ; defaults if not overwritten on the command line
      if (.not. isvar("latMIN") ) then
        latMIN = 25.
        tkres@latMINset = False
      end if
      tkres@latMIN = latMIN
      if (.not. isvar("latMAX") ) then
        latMAX = 40.
        tkres@latMAXset = False
      end if
      tkres@latMAX = latMAX
      if (.not. isvar("lonMIN") ) then
        lonMIN = -90.
        tkres@lonMINset = False
      end if
      tkres@lonMIN = lonMIN
      if (.not. isvar("lonMAX") ) then
        lonMAX = -75.
        tkres@lonMAXset = False
      end if
      tkres@lonMAX = lonMAX
      if (.not. isvar("disEDGE") ) then
        disEDGE = 2.
      end if
      tkres@disEDGE= disEDGE

      ; Make sure the final window is bigger than the data spread
        MapWindow(lat,lon,wnd,tkres)
  
      print("The Track Plot Window is set to")
      print("   Latitute  from " + tkres@min_lat + " to " + tkres@max_lat)
      print("   Longitute from " + tkres@min_lon + " to " + tkres@max_lon)
      print("The buffer between the data and the plot edge is " + tkres@disEDGE + " degrees")

    ;----------------------------------------------------------------------
    ; Set up resources and create map
    ;----------------------------------------------------------------------

    ; Get colors to use
      track_colors = TrackColors()
      gsn_define_colormap(wks,track_colors)
      ;gsn_draw_colormap(wks)

    ; Set some map resources and zoom in           
      res = False
      MapResources(res,tkres)

    ; Create the map. 
      map = gsn_csm_map(wks,res)
    ; Add Header top left 
      PlotHeaderL(wks,map,tkres)

      draw(map)

    end if

  ;----------------------------------------------------------------------
  ; The plotting section.
  ;----------------------------------------------------------------------

  ; Draw the track line
    PlotTrackLine(wks,map,lat,lon,wnd,cat,tkres)

  ; Add Symbols over line
    PlotTrackSymbols(wks,map,lat,lon,wnd,cat,tkres)
 
  ; Add pressure/wind information on track line
  ; Add date information on track line
  ; Only when needed - determine by plotType
    PlotTrackInfo(wks,map,lat,lon,wnd,wndlev,dd_hh,tkres)

    delete(ll_lev_wnd)
    delete(lat)
    delete(lon)
    delete(lev)
    delete(wnd)
    delete(slev)
    delete(swnd)
    delete(wndlev)

  end do


  ; Add time information top right
    PlotHeaderR(wks,map,TRHeader,tkres)

  ; Add hurricane legend in bottom left
    HurricaneLegend(wks,map,tkres)
 
  ; Add pressure/wind and dd/hh legend to bottom right
    SymbolLegend(wks,map,tkres)


  ; Finally - are we doing a best track
    if (tkres@bTrack) then
      PlotBestTrack(wks,map,tkres)
    end if


  ; We are done
  frame(wks)
  delete(wks)


  ; Cleanup if needed
  makePNG(time_start_end,tkres)

end
