MODULE module_nrutils

  IMPLICIT NONE  ! ALWAYS, ALWAYS, ALWAYS!!!!!

  PRIVATE   ! make all variable declarations private unless
            ! explicity made public

  ! Make the subroutines we will actually call public
  PUBLIC :: locate, four1, fourrow, bilinear, sort2, indexx
  PUBLIC :: tridag, mprove, lubksb, ludcmp

  ! NRTYPE.F

  INTEGER, PARAMETER :: I4B = SELECTED_INT_KIND(9)
  INTEGER, PARAMETER :: I2B = SELECTED_INT_KIND(4)
  INTEGER, PARAMETER :: I1B = SELECTED_INT_KIND(2)
  INTEGER, PARAMETER :: SP = KIND(1.0)
  INTEGER, PARAMETER :: DP = KIND(1.0D0)
  INTEGER, PARAMETER :: SPC = KIND((1.0,1.0))
  INTEGER, PARAMETER :: DPC = KIND((1.0D0,1.0D0))
  INTEGER, PARAMETER :: LGT = KIND(.true.)
  REAL(SP), PARAMETER :: PI=3.141592653589793238462643383279502884197_sp
  REAL(SP), PARAMETER :: PIO2=1.57079632679489661923132169163975144209858_sp
  REAL(SP), PARAMETER :: TWOPI=6.283185307179586476925286766559005768394_sp
  REAL(SP), PARAMETER :: SQRT2=1.41421356237309504880168872420969807856967_sp
  REAL(SP), PARAMETER :: EULER=0.5772156649015328606065120900824024310422_sp
  REAL(DP), PARAMETER :: PI_D=3.141592653589793238462643383279502884197_dp
  REAL(DP), PARAMETER :: PIO2_D=1.57079632679489661923132169163975144209858_dp
  REAL(DP), PARAMETER :: TWOPI_D=6.283185307179586476925286766559005768394_dp
  TYPE sprs2_sp
     INTEGER(I4B) :: n,len
     REAL(SP), DIMENSION(:), POINTER :: val
     INTEGER(I4B), DIMENSION(:), POINTER :: irow
     INTEGER(I4B), DIMENSION(:), POINTER :: jcol
  END TYPE sprs2_sp
  TYPE sprs2_dp
     INTEGER(I4B) :: n,len
     REAL(DP), DIMENSION(:), POINTER :: val
     INTEGER(I4B), DIMENSION(:), POINTER :: irow
     INTEGER(I4B), DIMENSION(:), POINTER :: jcol
  END TYPE sprs2_dp

  ! NRUTIL.F variable declarations

  INTEGER(I4B), PARAMETER :: NPAR_ARTH=16,NPAR2_ARTH=8
  INTEGER(I4B), PARAMETER :: NPAR_GEOP=4,NPAR2_GEOP=2
  INTEGER(I4B), PARAMETER :: NPAR_CUMSUM=16
  INTEGER(I4B), PARAMETER :: NPAR_CUMPROD=8
  INTEGER(I4B), PARAMETER :: NPAR_POLY=8
  INTEGER(I4B), PARAMETER :: NPAR_POLYTERM=8

  ! NR.F interfaces

!  INTERFACE four1
!     SUBROUTINE four1_dp(data,isign)
!       COMPLEX(DPC), DIMENSION(:), INTENT(INOUT) :: data
!       INTEGER(I4B), INTENT(IN) :: isign
!     END SUBROUTINE four1_dp
!
!     SUBROUTINE four1_sp(data,isign)
!       COMPLEX(SPC), DIMENSION(:), INTENT(INOUT) :: data
!       INTEGER(I4B), INTENT(IN) :: isign
!     END SUBROUTINE four1_sp
!  END INTERFACE

!  INTERFACE fourrow
!     SUBROUTINE fourrow_dp(data,isign)
!       COMPLEX(DPC), DIMENSION(:,:), INTENT(INOUT) :: data
!       INTEGER(I4B), INTENT(IN) :: isign
!     END SUBROUTINE fourrow_dp
!
!     SUBROUTINE fourrow_sp(data,isign)
!       COMPLEX(SPC), DIMENSION(:,:), INTENT(INOUT) :: data
!       INTEGER(I4B), INTENT(IN) :: isign
!     END SUBROUTINE fourrow_sp
!  END INTERFACE

  INTERFACE four1
     MODULE PROCEDURE four1_dp, four1_sp
  END INTERFACE

  INTERFACE fourrow
     MODULE PROCEDURE fourrow_dp, fourrow_sp
  END INTERFACE

  ! NRUTIL.F interfaces

  INTERFACE arth
     MODULE PROCEDURE arth_r, arth_d, arth_i
  END INTERFACE

  INTERFACE assert
     MODULE PROCEDURE assert1,assert2,assert3,assert4,assert_v
  END INTERFACE

  INTERFACE assert_eq
     MODULE PROCEDURE assert_eq2,assert_eq3,assert_eq4,assert_eqn
  END INTERFACE

  INTERFACE icomp_xchg
     MODULE PROCEDURE icomp_xchg_sp, icomp_xchg_i4b
  END INTERFACE

  INTERFACE imaxloc
     MODULE PROCEDURE imaxloc_r,imaxloc_i
  END INTERFACE

  INTERFACE indexx
     MODULE PROCEDURE indexx_sp, indexx_i4b
  END INTERFACE

  INTERFACE outerprod
     MODULE PROCEDURE outerprod_r,outerprod_d
  END INTERFACE

  INTERFACE swap
     MODULE PROCEDURE swap_i,swap_r,swap_rv,swap_c, &
          swap_cv,swap_cm,swap_z,swap_zv,swap_zm, &
          masked_swap_rs,masked_swap_rv,masked_swap_rm
  END INTERFACE

CONTAINS

!-----------------------------------------------------------------------------
!
! NRUTILS.F
!
!-----------------------------------------------------------------------------
! assert
!-----------------------------------------------------------------------------

  SUBROUTINE assert1(n1,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    LOGICAL, INTENT(IN) :: n1
    if (.not. n1) then
       write (*,*) 'nrerror: an assertion failed with this tag:',string
       STOP 'program terminated by assert1'
    end if
  END SUBROUTINE assert1

  SUBROUTINE assert2(n1,n2,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    LOGICAL, INTENT(IN) :: n1,n2
    if (.not. (n1 .and. n2)) then
       write (*,*) 'nrerror: an assertion failed with this tag:',string
       STOP 'program terminated by assert2'
    end if
  END SUBROUTINE assert2

  SUBROUTINE assert3(n1,n2,n3,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    LOGICAL, INTENT(IN) :: n1,n2,n3
    if (.not. (n1 .and. n2 .and. n3)) then
       write (*,*) 'nrerror: an assertion failed with this tag:',string
       STOP 'program terminated by assert3'
    end if
  END SUBROUTINE assert3

  SUBROUTINE assert4(n1,n2,n3,n4,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    LOGICAL, INTENT(IN) :: n1,n2,n3,n4
    if (.not. (n1 .and. n2 .and. n3 .and. n4)) then
       write (*,*) 'nrerror: an assertion failed with this tag:',string
       STOP 'program terminated by assert4'
    end if
  END SUBROUTINE assert4

  SUBROUTINE assert_v(n,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    LOGICAL, DIMENSION(:), INTENT(IN) :: n
    if (.not. all(n)) then
       write (*,*) 'nrerror: an assertion failed with this tag:',string
       STOP 'program terminated by assert_v'
    end if
  END SUBROUTINE assert_v

!-----------------------------------------------------------------------------
! assert_eq
!-----------------------------------------------------------------------------

  FUNCTION assert_eq2(n1,n2,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    INTEGER, INTENT(IN) :: n1,n2
    INTEGER :: assert_eq2
    if (n1 == n2) then
       assert_eq2=n1
    else
       write (*,*) 'nrerror: an assert_eq failed with this tag:', &
            string
       STOP 'program terminated by assert_eq2'
    end if
  END FUNCTION assert_eq2

  FUNCTION assert_eq3(n1,n2,n3,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    INTEGER, INTENT(IN) :: n1,n2,n3
    INTEGER :: assert_eq3
    if (n1 == n2 .and. n2 == n3) then
       assert_eq3=n1
    else
       write (*,*) 'nrerror: an assert_eq failed with this tag:', &
            string
       STOP 'program terminated by assert_eq3'
    end if
  END FUNCTION assert_eq3

  FUNCTION assert_eq4(n1,n2,n3,n4,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    INTEGER, INTENT(IN) :: n1,n2,n3,n4
    INTEGER :: assert_eq4
    if (n1 == n2 .and. n2 == n3 .and. n3 == n4) then
       assert_eq4=n1
    else
       write (*,*) 'nrerror: an assert_eq failed with this tag:', &
            string
       STOP 'program terminated by assert_eq4'
    end if
  END FUNCTION assert_eq4

  FUNCTION assert_eqn(nn,string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    INTEGER, DIMENSION(:), INTENT(IN) :: nn
    INTEGER :: assert_eqn
    if (all(nn(2:) == nn(1))) then
       assert_eqn=nn(1)
    else
       write (*,*) 'nrerror: an assert_eq failed with this tag:', &
            string
       STOP 'program terminated by assert_eqn'
    end if
  END FUNCTION assert_eqn

!-----------------------------------------------------------------------------
! arth
!-----------------------------------------------------------------------------

  FUNCTION arth_r(first,increment,n)
    REAL(SP), INTENT(IN) :: first,increment
    INTEGER(I4B), INTENT(IN) :: n
    REAL(SP), DIMENSION(n) :: arth_r
    INTEGER(I4B) :: k,k2
    REAL(SP) :: temp
    if (n > 0) arth_r(1)=first
    if (n <= NPAR_ARTH) then
       do k=2,n
          arth_r(k)=arth_r(k-1)+increment
       end do
    else
       do k=2,NPAR2_ARTH
          arth_r(k)=arth_r(k-1)+increment
       end do
       temp=increment*NPAR2_ARTH
       k=NPAR2_ARTH
       do
          if (k >= n) exit
          k2=k+k
          arth_r(k+1:min(k2,n))=temp+arth_r(1:min(k,n-k))
          temp=temp+temp
          k=k2
       end do
    end if
  END FUNCTION arth_r

  FUNCTION arth_d(first,increment,n)
    REAL(DP), INTENT(IN) :: first,increment
    INTEGER(I4B), INTENT(IN) :: n
    REAL(DP), DIMENSION(n) :: arth_d
    INTEGER(I4B) :: k,k2
    REAL(DP) :: temp
    if (n > 0) arth_d(1)=first
    if (n <= NPAR_ARTH) then
       do k=2,n
          arth_d(k)=arth_d(k-1)+increment
       end do
    else
       do k=2,NPAR2_ARTH
          arth_d(k)=arth_d(k-1)+increment
       end do
       temp=increment*NPAR2_ARTH
       k=NPAR2_ARTH
       do
          if (k >= n) exit
          k2=k+k
          arth_d(k+1:min(k2,n))=temp+arth_d(1:min(k,n-k))
          temp=temp+temp
          k=k2
       end do
    end if
  END FUNCTION arth_d

  FUNCTION arth_i(first,increment,n)
    INTEGER(I4B), INTENT(IN) :: first,increment,n
    INTEGER(I4B), DIMENSION(n) :: arth_i
    INTEGER(I4B) :: k,k2,temp
    if (n > 0) arth_i(1)=first
    if (n <= NPAR_ARTH) then
       do k=2,n
          arth_i(k)=arth_i(k-1)+increment
       end do
    else
       do k=2,NPAR2_ARTH
          arth_i(k)=arth_i(k-1)+increment
       end do
       temp=increment*NPAR2_ARTH
       k=NPAR2_ARTH
       do
          if (k >= n) exit
          k2=k+k
          arth_i(k+1:min(k2,n))=temp+arth_i(1:min(k,n-k))
          temp=temp+temp
          k=k2
       end do
    end if
  END FUNCTION arth_i

!-----------------------------------------------------------------------------
! swap
!-----------------------------------------------------------------------------

  SUBROUTINE swap_i(a,b)
    INTEGER(I4B), INTENT(INOUT) :: a,b
    INTEGER(I4B) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_i

  SUBROUTINE swap_r(a,b)
    REAL(SP), INTENT(INOUT) :: a,b
    REAL(SP) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_r

  SUBROUTINE swap_rv(a,b)
    REAL(SP), DIMENSION(:), INTENT(INOUT) :: a,b
    REAL(SP), DIMENSION(SIZE(a)) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_rv

  SUBROUTINE swap_c(a,b)
    COMPLEX(SPC), INTENT(INOUT) :: a,b
    COMPLEX(SPC) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_c

  SUBROUTINE swap_cv(a,b)
    COMPLEX(SPC), DIMENSION(:), INTENT(INOUT) :: a,b
    COMPLEX(SPC), DIMENSION(SIZE(a)) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_cv

  SUBROUTINE swap_cm(a,b)
    COMPLEX(SPC), DIMENSION(:,:), INTENT(INOUT) :: a,b
    COMPLEX(SPC), DIMENSION(size(a,1),size(a,2)) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_cm

  SUBROUTINE swap_z(a,b)
    COMPLEX(DPC), INTENT(INOUT) :: a,b
    COMPLEX(DPC) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_z

  SUBROUTINE swap_zv(a,b)
    COMPLEX(DPC), DIMENSION(:), INTENT(INOUT) :: a,b
    COMPLEX(DPC), DIMENSION(SIZE(a)) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_zv

  SUBROUTINE swap_zm(a,b)
    COMPLEX(DPC), DIMENSION(:,:), INTENT(INOUT) :: a,b
    COMPLEX(DPC), DIMENSION(size(a,1),size(a,2)) :: dum
    dum=a
    a=b
    b=dum
  END SUBROUTINE swap_zm

  SUBROUTINE masked_swap_rs(a,b,mask)
    REAL(SP), INTENT(INOUT) :: a,b
    LOGICAL(LGT), INTENT(IN) :: mask
    REAL(SP) :: swp
    if (mask) then
       swp=a
       a=b
       b=swp
    end if
  END SUBROUTINE masked_swap_rs

  SUBROUTINE masked_swap_rv(a,b,mask)
    REAL(SP), DIMENSION(:), INTENT(INOUT) :: a,b
    LOGICAL(LGT), DIMENSION(:), INTENT(IN) :: mask
    REAL(SP), DIMENSION(size(a)) :: swp
    where (mask)
       swp=a
       a=b
       b=swp
    end where
  END SUBROUTINE masked_swap_rv

  SUBROUTINE masked_swap_rm(a,b,mask)
    REAL(SP), DIMENSION(:,:), INTENT(INOUT) :: a,b
    LOGICAL(LGT), DIMENSION(:,:), INTENT(IN) :: mask
    REAL(SP), DIMENSION(size(a,1),size(a,2)) :: swp
    where (mask)
       swp=a
       a=b
       b=swp
    end where
  END SUBROUTINE masked_swap_rm

!-----------------------------------------------------------------------------
! nrerror
!-----------------------------------------------------------------------------

  SUBROUTINE nrerror(string)
    CHARACTER(LEN=*), INTENT(IN) :: string
    write (*,*) 'nrerror: ',string
    STOP 'program terminated by nrerror'
  END SUBROUTINE nrerror

!-----------------------------------------------------------------------------
! outerprod
!-----------------------------------------------------------------------------

  FUNCTION outerprod_r(a,b)
    REAL(SP), DIMENSION(:), INTENT(IN) :: a,b
    REAL(SP), DIMENSION(size(a),size(b)) :: outerprod_r
    outerprod_r = SPREAD(a,dim=2,ncopies=size(b)) * &
                  SPREAD(b,dim=1,ncopies=size(a))
  END FUNCTION outerprod_r

  FUNCTION outerprod_d(a,b)
    REAL(DP), DIMENSION(:), INTENT(IN) :: a,b
    REAL(DP), DIMENSION(size(a),size(b)) :: outerprod_d
    outerprod_d = SPREAD(a,dim=2,ncopies=size(b)) * &
                  SPREAD(b,dim=1,ncopies=size(a))
  END FUNCTION outerprod_d

!-----------------------------------------------------------------------------
! imaxloc
!-----------------------------------------------------------------------------

  FUNCTION imaxloc_r(arr)
    REAL(SP), DIMENSION(:), INTENT(IN) :: arr
    INTEGER(I4B) :: imaxloc_r
    INTEGER(I4B), DIMENSION(1) :: imax
    imax=MAXLOC(arr(:))
    imaxloc_r=imax(1)
  END FUNCTION imaxloc_r

  FUNCTION imaxloc_i(iarr)
    INTEGER(I4B), DIMENSION(:), INTENT(IN) :: iarr
    INTEGER(I4B), DIMENSION(1) :: imax
    INTEGER(I4B) :: imaxloc_i
    imax=MAXLOC(iarr(:))
    imaxloc_i=imax(1)
  END FUNCTION imaxloc_i

!-----------------------------------------------------------------------------
!
! BILINEAR.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE BILINEAR(X1A,X2A,YA,M,N,X1,X2,Y)
    ! Performs bilinear interpolation (Num. Rec., Section 3.6)
    ! Given an array of YA(j,k), where j varies from 1 to M, and k
    ! varies from 1 to N, and where X1A(j) describes the values of the
    ! first index, and X2A(k) describes the values of the second index
    ! (both must be monotonically increasing), this subroutine returns
    ! the value of Y as the value of YA evaluated at the location (X1,X2)
    IMPLICIT NONE
    INTEGER M,N               ! Array sizes
    REAL X1,X2,Y,X1A(M),X2A(N),YA(M,N)
    REAL T,U,Y1,Y2,Y3,Y4      ! Temporary values
    INTEGER J,K               ! Indices of locations near X1,X2

    ! Find j and k such that x1a(j) <= x1 <= x1a(j+1) and
    !                        x2a(k) <= x2 <= x2a(k+1)
    CALL LOCATE(X1A,M,X1,J)
    CALL LOCATE(X2A,N,X2,K)

    ! Here we are saying if our value lies outside our box range,
    ! just use the value at the edge of the range
    IF (j < 1) j=1
    IF (j > M-1) j=M-1
    IF (k < 1) k=1
    IF (k > N-1) k=N-1

    ! Define the values of the four points around X1,X2
    Y1=YA(J  ,K  )
    Y2=YA(J+1,K  )
    Y3=YA(J+1,K+1)
    Y4=YA(J  ,K+1)

    ! Set up variables for convenience
    T =(X1-X1A(J))/(X1A(J+1)-X1A(J))
    U =(X2-X2A(K))/(X2A(K+1)-X2A(K))

    ! Calculate the final value
    Y = (1.-T)*(1.-U)*Y1 + &
         T *(1.-U)*Y2 + &
         T *    U *Y3 + &
         (1.-T)*    U *Y4

  END SUBROUTINE BILINEAR

!-----------------------------------------------------------------------------
!
! LOCATE.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE LOCATE(xx,n,x,j)
    ! given array xx of length n, and given a value x, returns a value j
    ! such that x is between xx(j) and xx(j+1).  xx must be monotonic, either
    ! increasing or decreasing.  j=0 or j=n is returned to indicate that
    ! x is out of range
    IMPLICIT NONE
    INTEGER, INTENT(IN) :: n
    REAL, DIMENSION(N), INTENT(IN) :: xx
    REAL, INTENT(IN) :: x
    INTEGER, INTENT(OUT) :: j

    INTEGER :: jl,ju,jm
    jl=0
    ju=n+1
    DO WHILE (ju-jl > 1)
       jm=(ju+jl)/2
       IF ((xx(n) > xx(1)).EQV.(x > xx(jm))) THEN
          jl=jm
       ELSE
          ju=jm
       ENDIF
    END DO
    j=jl
  END SUBROUTINE LOCATE

!-----------------------------------------------------------------------------
!
! FOUR1.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE four1_sp(data,isign)
    IMPLICIT NONE
    COMPLEX(SPC), DIMENSION(:), INTENT(INOUT) :: data
    INTEGER(I4B), INTENT(IN) :: isign
    COMPLEX(SPC), DIMENSION(:,:), ALLOCATABLE :: dat,temp
    COMPLEX(DPC), DIMENSION(:), ALLOCATABLE :: w,wp
    REAL(DP), DIMENSION(:), ALLOCATABLE :: theta
    INTEGER(I4B) :: n,m1,m2,j
    n=size(data)
    call assert(iand(n,n-1)==0, 'n must be a power of 2 in four1_sp')
    m1=2**ceiling(0.5_sp*log(real(n,sp))/0.693147_sp)
    m2=n/m1
    allocate(dat(m1,m2),theta(m1),w(m1),wp(m1),temp(m2,m1))
    dat=reshape(data,shape(dat))
    call fourrow(dat,isign)
    theta=arth(0,isign,m1)*TWOPI_D/n
    wp=cmplx(-2.0_dp*sin(0.5_dp*theta)**2,sin(theta),kind=dpc)
    w=cmplx(1.0_dp,0.0_dp,kind=dpc)
    do j=2,m2
       w=w*wp+w
       dat(:,j)=dat(:,j)*w
    end do
    temp=transpose(dat)
    call fourrow(temp,isign)
    data=reshape(temp,shape(data))
    deallocate(dat,w,wp,theta,temp)
  END SUBROUTINE four1_sp

  SUBROUTINE four1_dp(data,isign)
    IMPLICIT NONE
    COMPLEX(DPC), DIMENSION(:), INTENT(INOUT) :: data
    INTEGER(I4B), INTENT(IN) :: isign
    COMPLEX(DPC), DIMENSION(:,:), ALLOCATABLE :: dat,temp
    COMPLEX(DPC), DIMENSION(:), ALLOCATABLE :: w,wp
    REAL(DP), DIMENSION(:), ALLOCATABLE :: theta
    INTEGER(I4B) :: n,m1,m2,j
    n=size(data)
    call assert(iand(n,n-1)==0, 'n must be a power of 2 in four1_dp')
    m1=2**ceiling(0.5_sp*log(real(n,sp))/0.693147_sp)
    m2=n/m1
    allocate(dat(m1,m2),theta(m1),w(m1),wp(m1),temp(m2,m1))
    dat=reshape(data,shape(dat))
    call fourrow(dat,isign)
    theta=arth(0,isign,m1)*TWOPI_D/n
    wp=cmplx(-2.0_dp*sin(0.5_dp*theta)**2,sin(theta),kind=dpc)
    w=cmplx(1.0_dp,0.0_dp,kind=dpc)
    do j=2,m2
       w=w*wp+w
       dat(:,j)=dat(:,j)*w
    end do
    temp=transpose(dat)
    call fourrow(temp,isign)
    data=reshape(temp,shape(data))
    deallocate(dat,w,wp,theta,temp)
  END SUBROUTINE four1_dp

!-----------------------------------------------------------------------------
!
! FOURROW.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE fourrow_sp(data,isign)
    IMPLICIT NONE
    COMPLEX(SPC), DIMENSION(:,:), INTENT(INOUT) :: data
    INTEGER(I4B), INTENT(IN) :: isign
    INTEGER(I4B) :: n,i,istep,j,m,mmax,n2
    REAL(DP) :: theta
    COMPLEX(SPC), DIMENSION(size(data,1)) :: temp
    COMPLEX(DPC) :: w,wp
    COMPLEX(SPC) :: ws
    n=size(data,2)
    call assert(iand(n,n-1)==0, 'n must be a power of 2 in fourrow_sp')
    n2=n/2
    j=n2
    do i=1,n-2
       if (j > i) call swap(data(:,j+1),data(:,i+1))
       m=n2
       do
          if (m < 2 .or. j < m) exit
          j=j-m
          m=m/2
       end do
       j=j+m
    end do
    mmax=1
    do
       if (n <= mmax) exit
       istep=2*mmax
       theta=PI_D/(isign*mmax)
       wp=cmplx(-2.0_dp*sin(0.5_dp*theta)**2,sin(theta),kind=dpc)
       w=cmplx(1.0_dp,0.0_dp,kind=dpc)
       do m=1,mmax
          ws=w
          do i=m,n,istep
             j=i+mmax
             temp=ws*data(:,j)
             data(:,j)=data(:,i)-temp
             data(:,i)=data(:,i)+temp
          end do
          w=w*wp+w
       end do
       mmax=istep
    end do
  END SUBROUTINE fourrow_sp

  SUBROUTINE fourrow_dp(data,isign)
    IMPLICIT NONE
    COMPLEX(DPC), DIMENSION(:,:), INTENT(INOUT) :: data
    INTEGER(I4B), INTENT(IN) :: isign
    INTEGER(I4B) :: n,i,istep,j,m,mmax,n2
    REAL(DP) :: theta
    COMPLEX(DPC), DIMENSION(size(data,1)) :: temp
    COMPLEX(DPC) :: w,wp
    COMPLEX(DPC) :: ws
    n=size(data,2)
    call assert(iand(n,n-1)==0, 'n must be a power of 2 in fourrow_dp')
    n2=n/2
    j=n2
    do i=1,n-2
       if (j > i) call swap(data(:,j+1),data(:,i+1))
       m=n2
       do
          if (m < 2 .or. j < m) exit
          j=j-m
          m=m/2
       end do
       j=j+m
    end do
    mmax=1
    do
       if (n <= mmax) exit
       istep=2*mmax
       theta=PI_D/(isign*mmax)
       wp=cmplx(-2.0_dp*sin(0.5_dp*theta)**2,sin(theta),kind=dpc)
       w=cmplx(1.0_dp,0.0_dp,kind=dpc)
       do m=1,mmax
          ws=w
          do i=m,n,istep
             j=i+mmax
             temp=ws*data(:,j)
             data(:,j)=data(:,i)-temp
             data(:,i)=data(:,i)+temp
          end do
          w=w*wp+w
       end do
       mmax=istep
    end do
  END SUBROUTINE fourrow_dp

!-----------------------------------------------------------------------------
!
! SORT2.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE sort2(arr,slave)
    IMPLICIT NONE
    REAL(SP), DIMENSION(:), INTENT(INOUT) :: arr,slave
    INTEGER(I4B) :: ndum
    INTEGER(I4B), DIMENSION(size(arr)) :: index
    ndum=assert_eq(size(arr),size(slave),'sort2')
    call indexx(arr,index)
    arr=arr(index)
    slave=slave(index)
  END SUBROUTINE sort2

!-----------------------------------------------------------------------------
!
! INDEXX.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE indexx_sp(arr,index)
    IMPLICIT NONE
    REAL(SP), DIMENSION(:), INTENT(IN) :: arr
    INTEGER(I4B), DIMENSION(:), INTENT(OUT) :: index
    INTEGER(I4B), PARAMETER :: NN=15, NSTACK=50
    REAL(SP) :: a
    INTEGER(I4B) :: n,k,i,j,indext,jstack,l,r
    INTEGER(I4B), DIMENSION(NSTACK) :: istack
    n=assert_eq(size(index),size(arr),'indexx_sp')
    index=arth(1,1,n)
    jstack=0
    l=1
    r=n
    do
       if (r-l < NN) then
          do j=l+1,r
             indext=index(j)
             a=arr(indext)
             do i=j-1,l,-1
                if (arr(index(i)) <= a) exit
                index(i+1)=index(i)
             end do
             index(i+1)=indext
          end do
          if (jstack == 0) RETURN
          r=istack(jstack)
          l=istack(jstack-1)
          jstack=jstack-2
       else
          k=(l+r)/2
          call swap(index(k),index(l+1))
          call icomp_xchg(index(l),index(r),arr(index(l)),arr(index(r)))
          call icomp_xchg(index(l+1),index(r),arr(index(l+1)),arr(index(r)))
          call icomp_xchg(index(l),index(l+1),arr(index(l)),arr(index(l+1)))
          i=l+1
          j=r
          indext=index(l+1)
          a=arr(indext)
          do
             do
                i=i+1
                if (arr(index(i)) >= a) exit
             end do
             do
                j=j-1
                if (arr(index(j)) <= a) exit
             end do
             if (j < i) exit
             call swap(index(i),index(j))
          end do
          index(l+1)=index(j)
          index(j)=indext
          jstack=jstack+2
          if (jstack > NSTACK) call nrerror('indexx: NSTACK too small')
          if (r-i+1 >= j-l) then
             istack(jstack)=r
             istack(jstack-1)=i
             r=j-1
          else
             istack(jstack)=j-1
             istack(jstack-1)=l
             l=i
          end if
       end if
    end do
  END SUBROUTINE indexx_sp

  SUBROUTINE indexx_i4b(iarr,index)
    IMPLICIT NONE
    INTEGER(I4B), DIMENSION(:), INTENT(IN) :: iarr
    INTEGER(I4B), DIMENSION(:), INTENT(OUT) :: index
    INTEGER(I4B), PARAMETER :: NN=15, NSTACK=50
    INTEGER(I4B) :: a
    INTEGER(I4B) :: n,k,i,j,indext,jstack,l,r
    INTEGER(I4B), DIMENSION(NSTACK) :: istack
    n=assert_eq(size(index),size(iarr),'indexx_sp')
    index=arth(1,1,n)
    jstack=0
    l=1
    r=n
    do
       if (r-l < NN) then
          do j=l+1,r
             indext=index(j)
             a=iarr(indext)
             do i=j-1,l,-1
                if (iarr(index(i)) <= a) exit
                index(i+1)=index(i)
             end do
             index(i+1)=indext
          end do
          if (jstack == 0) RETURN
          r=istack(jstack)
          l=istack(jstack-1)
          jstack=jstack-2
       else
          k=(l+r)/2
          call swap(index(k),index(l+1))
          call icomp_xchg(index(l),index(r),iarr(index(l)),iarr(index(r)))
          call icomp_xchg(index(l+1),index(r),iarr(index(l+1)),iarr(index(r)))
          call icomp_xchg(index(l),index(l+1),iarr(index(l)),iarr(index(l+1)))
          i=l+1
          j=r
          indext=index(l+1)
          a=iarr(indext)
          do
             do
                i=i+1
                if (iarr(index(i)) >= a) exit
             end do
             do
                j=j-1
                if (iarr(index(j)) <= a) exit
             end do
             if (j < i) exit
             call swap(index(i),index(j))
          end do
          index(l+1)=index(j)
          index(j)=indext
          jstack=jstack+2
          if (jstack > NSTACK) call nrerror('indexx: NSTACK too small')
          if (r-i+1 >= j-l) then
             istack(jstack)=r
             istack(jstack-1)=i
             r=j-1
          else
             istack(jstack)=j-1
             istack(jstack-1)=l
             l=i
          end if
       end if
    end do
  END SUBROUTINE indexx_i4b

!-----------------------------------------------------------------------------
! icomp_xchg
!-----------------------------------------------------------------------------

  SUBROUTINE icomp_xchg_sp(i,j,ai,aj)
    REAL(SP), INTENT(IN) :: ai, aj
    INTEGER(I4B), INTENT(INOUT) :: i,j
    INTEGER(I4B) :: swp
    if (aj < ai) then
       swp=i
       i=j
       j=swp
    end if
  END SUBROUTINE icomp_xchg_sp

  SUBROUTINE icomp_xchg_i4b(i,j,ai,aj)
    INTEGER(I4B), INTENT(IN) :: ai, aj
    INTEGER(I4B), INTENT(INOUT) :: i,j
    INTEGER(I4B) :: swp
    if (aj < ai) then
       swp=i
       i=j
       j=swp
    end if
  END SUBROUTINE icomp_xchg_i4b

!-----------------------------------------------------------------------------
!
! TRIDAG.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE tridag(a,b,c,r,u)
    IMPLICIT NONE
    REAL(SP), DIMENSION(:), INTENT(IN) :: a,b,c,r
    REAL(SP), DIMENSION(:), INTENT(OUT) :: u
    REAL(SP), DIMENSION(size(b)) :: gam
    INTEGER(I4B) :: n,j
    REAL(SP) :: bet
    n=ASSERT_EQ((/SIZE(a)+1,SIZE(b),SIZE(c)+1,SIZE(r),SIZE(u)/),'tridag_ser')
    bet=b(1)
    IF (bet == 0.0) CALL NRERROR('tridag_ser: Error at code stage 1')
    u(1)=r(1)/bet
    DO j=2,n
       gam(j)=c(j-1)/bet
       bet=b(j)-a(j-1)*gam(j)
       IF (bet == 0.0) CALL NRERROR('tridag_ser: Error at code stage 2')
       u(j)=(r(j)-a(j-1)*u(j-1))/bet
    END DO
    DO j=n-1,1,-1
       u(j)=u(j)-gam(j+1)*u(j+1)
    END DO
  END SUBROUTINE tridag

!-----------------------------------------------------------------------------
!
! MPROVE.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE mprove(a,alud,indx,b,x)
    IMPLICIT NONE
    REAL(SP), DIMENSION(:,:), INTENT(IN) :: a,alud
    INTEGER(I4B), DIMENSION(:), INTENT(IN) :: indx
    REAL(SP), DIMENSION(:), INTENT(IN) :: b
    REAL(SP), DIMENSION(:), INTENT(INOUT) :: x
    INTEGER(I4B) :: ndum
    REAL(SP), DIMENSION(SIZE(a,1)) :: r
    ndum=ASSERT_EQ((/SIZE(a,1),SIZE(a,2),SIZE(alud,1),SIZE(alud,2),SIZE(b),&
                     SIZE(x),SIZE(indx)/),'mprove')
    r=MATMUL(REAL(a,dp),REAL(x,dp))-REAL(b,dp)
    CALL LUBKSB(alud,indx,r)
    x=x-r
  END SUBROUTINE mprove

!-----------------------------------------------------------------------------
!
! LUBKSB.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE lubksb(a,indx,b)
    IMPLICIT NONE
    REAL(SP), DIMENSION(:,:), INTENT(IN) :: a
    INTEGER(I4B), DIMENSION(:), INTENT(IN) :: indx
    REAL(SP), DIMENSION(:), INTENT(INOUT) :: b
    INTEGER(I4B) :: i,n,ii,ll
    REAL(SP) :: summ
    n=ASSERT_EQ(size(a,1),size(a,2),size(indx),'lubksb')
    ii=0
    DO i=1,n
       ll=indx(i)
       summ=b(ll)
       b(ll)=b(i)
       IF (ii /= 0) THEN
          summ=summ-DOT_PRODUCT(a(i,ii:i-1),b(ii:i-1))
       ELSE IF (summ /= 0.0) THEN
          ii=i
       END IF
       b(i)=summ
    END DO
    DO i=n,1,-1
       b(i) = (b(i)-DOT_PRODUCT(a(i,i+1:n),b(i+1:n)))/a(i,i)
    END DO
  END SUBROUTINE lubksb

!-----------------------------------------------------------------------------
!
! LUDCMP.F
!
!-----------------------------------------------------------------------------

  SUBROUTINE ludcmp(a,indx,d)
    IMPLICIT NONE
    REAL(SP), DIMENSION(:,:), INTENT(INOUT) :: a
    INTEGER(I4B), DIMENSION(:), INTENT(OUT) :: indx
    REAL(SP), INTENT(OUT) :: d
    REAL(SP), DIMENSION(size(a,1)) :: vv
    REAL(SP), PARAMETER :: TINY=1.0e-20_sp
    INTEGER(I4B) :: j,n,imax
    n=ASSERT_EQ(SIZE(a,1),SIZE(a,2),SIZE(indx),'ludcmp')
    d=1.0
    vv=MAXVAL(ABS(a),dim=2)
    IF (ANY(vv == 0.0)) CALL NRERROR('singular matrix in ludcmp')
    vv=1.0_sp/vv
    DO j=1,n
       imax=(j-1)+IMAXLOC(vv(j:n)*ABS(a(j:n,j)))
       IF (j /= imax) THEN
          CALL SWAP(a(imax,:),a(j,:))
          d=-d
          vv(imax)=vv(j)
       END IF
       indx(j)=imax
       IF (a(j,j) == 0.0) a(j,j)=TINY
       a(j+1:n,j)=a(j+1:n,j)/a(j,j)
       a(j+1:n,j+1:n)=a(j+1:n,j+1:n)-OUTERPROD(a(j+1:n,j),a(j,j+1:n))
    END DO
  END SUBROUTINE ludcmp

END MODULE module_nrutils
