\name{flat.read}
\alias{flat.read}
\author{Tim Hoar [\email{thoar@ucar.edu}]~~ \url{http://www.cgd.ucar.edu/~thoar} }
\title{Fortran "flat" read}
\description{
  reads Fortran "flat" unformatted binary records into R objects.
}
\synopsis{
   flat.read(file=NULL, numrow=NULL, numcol=NULL, rec1=1, recN=rec1,
                 ioflag=3, unit=96, type="real")
}
\usage{
   flat.read(file)
   flat.read(file, numrow=1234, numcol=1, ioflag)
   flat.read(file, ioflag, type)
}

\arguments{

   \item{file}{The file name.}

   \item{numrow}{The leading dimension of the Fortran binary record.}

   \item{numcol}{The second dimension of the Fortran binary record.}

   \item{rec1}{[optional] RELATIVE TO THE CURRENT POSITION -- 
               which Fortran record (matrix) to start reading [default is 1].}
   \item{recN}{[optional] last Fortran record to read defaults to value of rec1}

   \item{ioflag}{[optional] integer to describe the read state of the file\cr
   ioflag == 1   open, read and close the file\cr
   ioflag == 2   open, read and leave the file open for further reading\cr
   ioflag == 3   read and leave the file open [default]\cr
   ioflag == 4   read and close the file}

   \item{unit}{[optional] integer to describe the Fortran unit. This is only
   needed if you intend to read/write from multiple files simultaneously. 
   Fortran associates a file with a unique (hopefully) integer and uses 
   the integer for everything.  Stay away from unit == 5, unit == 6, 
   which can have special meaning./cr
   unit == 96 [default]}

   \item{type}{[optional] character string to describe the Fortran variable
   type.\cr
   type == "real" [default]\cr
   type == "real*8"\cr
   type == "integer"}

}
\value{
   \item{An array of dimension numrow,numcol,recN-rec1+1 }
}

\details{
   This works best when you tell it the truth .... numrow and numcol
   should be the DECLARED dimensions of the variable written by the
   Fortran routine ...Cleverly lying will often produce desirble results, poor
   lying will produce poor results.\cr

   Since files can have multiple records, you need a mechanism to
   keep track of whether the file is open or not. Anytime there is a
   Fortran "write" statement -- the thing it writes is a record. A fortran
   unformatted binary record of 4byte REALS contains a 4byte header
   (containing information about the length of the record to be written),
   the thing to be written, and a 4byte "end-of-record". If you hit the
   end-of-record prematurely, the read will choke. If you keep closing the
   file before you read the next record, you will continually read the FIRST
   record.\cr
}

\note{
   \emph{IMPORTANT} If you have multiple files open (for reading or writing),
   YOU had best keep track of which unit is which file. Doing otherwise is
   going to produce undesirable results, guaranteed.\cr

   \emph{REALLY IMPORTANT You} must remember the current position in the file. If you
   have already read a couple of matrices (with ioflag = 1 or 2) and then
   decide to read a block (i.e. by using the rec1, recN parameters); you are
   getting matrices relative to your current position, not relative to the
   beginning of the file.\cr
   \code{rec1, recN, ioflag} work together to determine the absolute record(s).\cr

   Examples 2 and beyond are the complementary examples to flat.write ... and 
   assume the the data file was written with the following Fortran code:

   /code{
      integer dim1,dim2                  ! scalars -- 1x1 "matrices"
      parameter (dim1=2, dim2=3)
      real     a(dim1,dim2)              ! 32bit matrix
      real*8   b(dim1,dim2)              ! 64bit matrix
      integer  d(dim1,dim2)              ! matrix of integers
      ...
      open(9,file='Flat.ieee',form='unformatted')
      ...
      write(9)dim1,dim2
      write(9)a
      write(9)b
      write(9)d
   }
}

\seealso{ \code{\link{flat.write}, \link{direct.read}, \link{direct.write} } }

\examples{
# EXAMPLE 1: WRITING a matrix (of integers)
NDarray <- array(c(111,121,112,122,113,123,
                  211,221,212,222,213,223,
                  311,321,312,322,313,323,
                  411,421,412,422,413,423),c(2,3,4))
datmat <- NDarray[,,1]
A <- flat.write(datmat,"matrix.ieee", ioflag=1, type="integer")
wow <- flat.read("matrix.ieee", 2, 3, ioflag=1,  type="integer"); # open/read/close
#[1] "Read 6 items."
wow
#     [,1] [,2] [,3]
#[1,]  111  112  113
#[2,]  121  122  123
#
# EXAMPLE 2: WRITING a couple of things in a row; then reading them in a bunch
# of different ways.
dims <- dim(NDarray)
nrows <- dims[1]
ncols <- dims[2]
a <- NDarray[,,1]
b <- NDarray[,,2]
d <- NDarray[,,3]
flat.write(c(nrows,ncols), "Flat.ieee",ioflag=2,type="integer");  # open, write, leave open
flat.write(a, "Flat.ieee", ioflag=3);     # write a 'real' matrix, leave open
flat.write(b, "Flat.ieee", ioflag=3, type="real*8");    # write as 'real*8'
flat.write(d, "Flat.ieee", ioflag=4, type="integer");   # write a 'integer', close
#
# now for the read part -- just the first two records of the file
#
z <- flat.read("Flat.ieee",1,2,ioflag=2,type="integer");  # open, read, leave open
A <- flat.read("Flat.ieee",2,3,ioflag=4);     # read a 2x3 matrix, close file
#
# EXAMPLE 3: Alternative way to read the matrix. In this case, the first
# record gives us information about the size of the item to come -- so let's
# use it.
z <- flat.read("Flat.ieee",1,2,ioflag = 2,type="integer");
A <- flat.read("Flat.ieee",z[1,1],z[1,2],ioflag = 4);
#
# EXAMPLE 4: The whole enchilada:
z <- flat.read("Flat.ieee",1,2,ioflag = 2,type="integer");
A <- flat.read("Flat.ieee",z[1,1],z[1,2],ioflag = 3);
B <- flat.read("Flat.ieee",z[1,1],z[1,2],ioflag = 3,type="real*8");
D <- flat.read("Flat.ieee",z[1,1],z[1,2],ioflag = 4,type="integer");
}
\keyword{IO}
