//************************************************************************************//
// Module       :  gtvector.cpp
// Date         :  7/9/01 (DLR)
// Copyright    :  2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  :   Encapsulates the methods and data associated with
//                  an array object
// Derived From :
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include <typeinfo>
#if defined(DO_HPM)
#include "libhpm.h"
#endif
#include "gtvector.hpp"
#include "mtk.hpp"
#include "gcomm.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
template<class T> GTVector<T>::GTVector<T>()
:
n                     (0),
bDistributed          (FALSE),
dtype                 (G_GDOUBLE),
cdtype                (GC_GDOUBLE),
data                  (NULL)
{
  index(n, n, 0, 0, 1,  0);

  if      ( typeid(T) == typeid(GUSHORT ) ) {
   dtype  = G_GUSHORT ;
   cdtype = GC_GUSHORT ; }
  else if ( typeid(T) == typeid(GSHORT ) ) {
    dtype  = G_GSHORT ;
    cdtype = GC_GSHORT ;}
  else if ( typeid(T) == typeid(GINT ) ) {
    dtype  = G_GINT ;
    cdtype = GC_GINT ; }
  else if ( typeid(T) == typeid(GDOUBLE) ) {
    dtype  = G_GDOUBLE;
    cdtype = GC_GDOUBLE;}
  else if ( typeid(T) == typeid(GQUAD) ) {
    dtype  = G_GQUAD;
    cdtype = GC_GQUAD; }

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
template<class T> GTVector<T>::GTVector<T>(GINT  size)
:
n                     (size),
bDistributed          (FALSE),
dtype                 (G_GDOUBLE),
cdtype                (GC_GDOUBLE),
data                  (NULL)
{
  index(n, n, 0, n-1, 1,  0);
  if ( n > 0 ) data = new T [n];
  Zero();

  if      ( typeid(T) == typeid(GUSHORT ) ) {
   dtype  = G_GUSHORT ;
   cdtype = GC_GUSHORT ; }
  else if ( typeid(T) == typeid(GSHORT ) ) {
    dtype  = G_GSHORT ;
    cdtype = GC_GSHORT ;}
  else if ( typeid(T) == typeid(GINT ) ) {
    dtype  = G_GINT ;
    cdtype = GC_GINT ; }
  else if ( typeid(T) == typeid(GDOUBLE) ) {
    dtype  = G_GDOUBLE;
    cdtype = GC_GDOUBLE;}
  else if ( typeid(T) == typeid(GQUAD) ) {
    dtype  = G_GQUAD;
    cdtype = GC_GQUAD; }


} // end of constructor method (2)

//************************************************************************************
//************************************************************************************
// Constructor Method (3)
template<class T> GTVector<T>::GTVector<T>(GIndex si)
:
n                     (si.szlocal()),
bDistributed          (TRUE),
dtype                 (G_GDOUBLE),
cdtype                (GC_GDOUBLE),
data                  (NULL)
{
  index = si;

  GINT  NN=index.end()-index.beg()+1+index.pad();

  if ( NN > 0 ) data = new T [index.end()-index.beg()+1+index.pad()];
  if ( index.end()<0 && index.beg()==0 ) {
    cout << "template<class T> GTVector<T>:: const 3: NULL size" << endl;
    exit(1);
  }
  Zero();
  if ( n == index.szglobal() ) bDistributed = FALSE;

  if      ( typeid(T) == typeid(GUSHORT ) ) {
   dtype  = G_GUSHORT ;
   cdtype = GC_GUSHORT ; }
  else if ( typeid(T) == typeid(GSHORT ) ) {
    dtype  = G_GSHORT ;
    cdtype = GC_GSHORT ;}
  else if ( typeid(T) == typeid(GINT ) ) {
    dtype  = G_GINT ;
    cdtype = GC_GINT ; }
  else if ( typeid(T) == typeid(GDOUBLE) ) {
    dtype  = G_GDOUBLE;
    cdtype = GC_GDOUBLE;}
  else if ( typeid(T) == typeid(GQUAD) ) {
    dtype  = G_GQUAD;
    cdtype = GC_GQUAD; }


} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Constructor Method (4)
template<class T> GTVector<T>::GTVector<T>(GDOUBLE *d, GINT  nn)
:
n                     (nn),
bDistributed          (FALSE),
dtype                 (G_GDOUBLE),
cdtype                (GC_GDOUBLE),
data                  (NULL)
{
  index(n, n, 0, n-1, 1,  0);
  if ( n > 0 ) {
    data = new T [n];
    if ( d != NULL )
      memcpy(data, d, n*sizeof(T));
    else
      memset(data, '\0', n*sizeof(T));
  }

  if      ( typeid(T) == typeid(GUSHORT ) ) {
   dtype  = G_GUSHORT ;
   cdtype = GC_GUSHORT ; }
  else if ( typeid(T) == typeid(GSHORT ) ) {
    dtype  = G_GSHORT ;
    cdtype = GC_GSHORT ;}
  else if ( typeid(T) == typeid(GINT ) ) {
    dtype  = G_GINT ;
    cdtype = GC_GINT ; }
  else if ( typeid(T) == typeid(GDOUBLE) ) {
    dtype  = G_GDOUBLE;
    cdtype = GC_GDOUBLE;}
  else if ( typeid(T) == typeid(GQUAD) ) {
    dtype  = G_GQUAD;
    cdtype = GC_GQUAD; }



} // end of constructor method (4)


//************************************************************************************
//************************************************************************************
// Copy constructor method
template<class T> GTVector<T>::GTVector<T>(const GTVector<T> &a)
{

  GINT  iy=a.GetIndex().beg(), incx=a.GetIndex().stride();

  index        = a.index;
  n            = a.dim() + index.pad();
  dtype        = a.dtype;
  cdtype       = a.cdtype;
  bDistributed = a.bDistributed;
  if ( n > 0 ) {              // make deep copy 
    data  = new T [n];
    memcpy(data, a.data, n*G_TYPESZ[dtype]);
  }

} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// Destructor
template<class T> GTVector<T>::~GTVector<T>()
{
   DeleteDynamic();
}


//************************************************************************************
//************************************************************************************
// Assignment operator method (1)
template<class T> GTVector<T> GTVector<T>::operator=(GTVector<T> a)
{

  if ( a.data == NULL || data == NULL ) return *this;

  GINT  sza, sz;


  if ( this != &a ) {
    sz  = index.end()-index.beg()+1+index.pad();
    sza = a.index.end()-a.index.beg()+1+a.index.pad();
    if ( sz != sza ) {
      cout << " GTVector<T>::operator= (1): incompatbile local dimensions" << endl;
      exit(1);
    }
    n            = sza;   //index.szlocal();
    dtype        = a.dtype;
    cdtype       = a.cdtype;
    bDistributed = a.bDistributed;
    memcpy(data+index.beg(), a.data+a.index.beg(), sza*G_TYPESZ[dtype]);
  }   

  return *this;

} // end = operator (1)


//************************************************************************************
//************************************************************************************
// Assignment operator = method (2) 
template<class T> void GTVector<T>::operator=(T  a)
{
#if 0
  if ( data == NULL ) {
    cout << " GTVector<T>::operator=(T  a): data is NULL" << endl;
    exit(1);
  }
#endif
  GINT  i, nd, ix=0, incx=index.stride();
  T    *px;
  
  if ( n == 0 || data == NULL ) return;

  px = data + index.beg() + index.base();
  nd = index.end() - index.beg() + 1;
  for ( i=0; i<nd; i++ ) {
     px[ix] = a;
     ix    += incx;
  }
} // end = operator (2)


//************************************************************************************
//************************************************************************************
// Assignment operator += method 
template<class T> void GTVector<T>::operator+=(GTVector<T> a)
{ 
#if defined(VEC_LEVEL0_TIMING) || defined(VEC_LEVEL1_TIMING)
  GDOUBLE tstart = STK::Timer();
#endif

  GINT  nt, na; 

  na   = a.GetIndex ().end() - a.GetIndex ().beg() + 1;
  nt   =   GetIndex ().end() -   GetIndex ().beg() + 1;
  if ( na != nt  ) {
    cout << "MTK::fmatvec_prod_srep: incompatible vectors "<< endl;
    exit(1);
  }

  if ( a.dim() == 0 || a.data == NULL ) return;
  if ( a.dim() != n ) { 
    cout << "template<class T> GTVector<T>::+=: incompatible assignment" << endl;
    return;
  }


  switch (dtype) {
    case G_GDOUBLE:
      MTK::fvec_add_rep((GTVector<GDOUBLE>&)*this, (GTVector<GDOUBLE>&)a);
      break;
    case G_GQUAD:
      MTK::qvec_add_rep((GTVector<GQUAD>&)*this, (GTVector<GQUAD>&)a );
      break;
    default:
      cout << " GTVector<T>::operator+=: invalid data type" << endl;
      exit(1);
  }

#if defined(VEC_LEVEL0_TIMING)
  time_result  = MTK::GetTime();
#elif defined(VEC_LEVEL1_TIMING)
  time_result = STK::Timer() - tstart;
#endif

} // end += operator


//************************************************************************************
//************************************************************************************
// Assignment operator -= method
template<class T> void GTVector<T>::operator-=(GTVector<T> a)
{
#if defined(VEC_LEVEL0_TIMING) || defined(VEC_LEVEL1_TIMING)
  GDOUBLE tstart = STK::Timer();
#endif


  if ( a.dim() == 0 || a.data == NULL ) return;
  if ( a.dim() != n ) {
    cout << "template<class T> GTVector<T>::=: incompatible assignment" << endl;
    return;
  }
 
  switch (dtype)
  {
    case G_GDOUBLE:
      MTK::fvec_sub_rep((GTVector<GDOUBLE>&)*this, (GTVector<GDOUBLE>&)a);
      break;
    case G_GQUAD:
      MTK::qvec_sub_rep((GTVector<GQUAD>&)*this, (GTVector<GQUAD>&)a );
      break;
    default:
      cout << " GTVector<T>::operator-=: invalid data type" << endl;
      exit(1); 
  }
  
#if defined(VEC_LEVEL0_TIMING)
  time_result  = MTK::GetTime();
#elif defined(VEC_LEVEL1_TIMING)
  time_result = STK::Timer() - tstart;
#endif

} // end -= operator


#if 0
//************************************************************************************
//************************************************************************************
// multiplication of vector on the RHS by a matrix
template<class T> GTVector<T> GTVector<T>::operator*(SETMatrix<T> &m) 
{
  if ( m.dim(1) != this->n ) {
    cout << "template<class T> GTVector<T>::*: (Matrix) incompatible dimensions"<< endl;
    exit(1);
  }

  GINT       i, j;
  T     sum;
  GTVector<T>  aprod(m.dim(2));
  
  for ( j=m.GetIndex(2)->beg(); j<m.GetIndex(2)->end()+1; j++ ) {
    sum  = 0.0;
    for ( i=m.GetIndex(1)->beg(); i<m.GetIndex(1)->end()+1; i++ ) {
      sum  += (*this)(i) * m(i,j);
    }
    aprod(j)  = sum;
  }
  
  return aprod;

} // end of method oerator *
#endif

//************************************************************************************
//************************************************************************************
// multiplication of vector on the RHS by a const T
template<class T> GTVector<T> GTVector<T>::operator*(T a) 
{
#if defined(VEC_LEVEL0_TIMING) || defined(VEC_LEVEL1_TIMING)
  GDOUBLE tstart = STK::Timer();
#endif

  GTVector<T> aret(index);

  switch ( dtype ) {
    case G_GDOUBLE:
      MTK::fvec_const_prod((GTVector<GDOUBLE>&)*this, a, (GTVector<GDOUBLE>&)aret);
      break;
    case G_GQUAD:
      MTK::qvec_const_prod((GTVector<GQUAD>&)*this, a, (GTVector<GQUAD>&)aret);
      break;
    default:
      cout << "GTVector<T>::operator*(const): invalid data type" << endl;
      exit(1);
  }

#if defined(VEC_LEVEL0_TIMING)
  time_result  = MTK::GetTime();
#elif defined(VEC_LEVEL1_TIMING)
  time_result = STK::Timer() - tstart;
#endif

  return aret;

} // end of * operator (for constant)


//************************************************************************************
//************************************************************************************
// multiplication of vector on the RHS by a const T
template<class T> void  GTVector<T>::operator*=(T a)
{
#if defined(VEC_LEVEL0_TIMING) || defined(VEC_LEVEL1_TIMING)
  GDOUBLE tstart = STK::Timer();
#endif

  switch ( dtype ) {
    case G_GDOUBLE:
      MTK::fvec_const_prod_rep((GTVector<GDOUBLE>&)*this, a);
      break;
    case G_GQUAD:
      MTK::qvec_const_prod_rep((GTVector<GQUAD>&)*this, a);
      break;
    default:
      cout << "GTVector<T>::operator*(const): invalid data type" << endl;
      exit(1);
  }

#if defined(VEC_LEVEL0_TIMING)
  time_result  = MTK::GetTime();
#elif defined(VEC_LEVEL1_TIMING)
  time_result = STK::Timer() - tstart;
#endif

} // end of *= operator (for constant)



//************************************************************************************
//************************************************************************************
// Vector inner product:
template<class T> T GTVector<T>::operator*(GTVector<T> a) 
{
#if defined(VEC_LEVEL0_TIMING) || defined(VEC_LEVEL1_TIMING)
  GDOUBLE tstart = STK::Timer();
#endif

#if 0
  if ( a.GetIndex().szglobal() != index.szglobal() )
  {
    GIndex  ai = a.GetIndex();
    cout << " a_index=" << ai << " this_index=" << index << endl;
    cout << "template<class T> GTVector<T>::*: (Vector) incompatible dimensions"<< endl;
    exit(1);
  }
#endif

  T  dot, glob_dot;

  switch ( dtype ) {
    case G_GDOUBLE:
      dot = MTK::fvec_dot((GTVector<GDOUBLE>&)*this, (GTVector<GDOUBLE>&)a);
      break;
    case G_GQUAD:
      dot = MTK::qvec_dot((GTVector<GQUAD>&)*this, (GTVector<GQUAD>&)a);
      break;
    default:
      cout << "GTVector<T>::operator*(GTVector): invalid data type" << endl;
      exit(1);
  }
#if defined(VEC_LEVEL0_TIMING)
  time_result  = MTK::GetTime();
#elif defined(VEC_LEVEL1_TIMING)
  time_result = STK::Timer() - tstart;
#endif


  if ( a.GetIndex().szglobal() >= index.szlocal() ) return dot;

  GComm::Allreduce(&dot, &glob_dot, 1, cdtype, G_OP_SUM);
    
  return glob_dot;

} // end of * operator (for vector inner prod)


//************************************************************************************
//************************************************************************************
// Vector addition:
template<class T> GTVector<T> GTVector<T>::operator+(GTVector<T> a) 
{
#if defined(VEC_LEVEL0_TIMING) || defined(VEC_LEVEL1_TIMING)
  GDOUBLE tstart = STK::Timer();
#endif


  if ( index.szglobal() != a.GetIndex().szglobal() ) {
    cout << "template<class T> GTVector<T>::+: incompatible vectors "<< endl;
    exit(1);
  }
  GTVector<T>  aret(index);

  switch (dtype) {
    case G_GDOUBLE:
      MTK::fvec_add((GTVector<GDOUBLE>&)*this, (GTVector<GDOUBLE>&)a, (GTVector<GDOUBLE>&)aret);
      break;
    case G_GQUAD:
      MTK::qvec_add((GTVector<GQUAD>&)*this, (GTVector<GQUAD>&)a, (GTVector<GQUAD>&)aret);
      break;
    default:
      cout << " GTVector<T>::operator+: invalid data type" << endl;
      exit(1);
  }

#if defined(VEC_LEVEL0_TIMING)
  time_result  = MTK::GetTime();
#elif defined(VEC_LEVEL1_TIMING)
  time_result = STK::Timer() - tstart;
#endif


  return aret;

} // end of + operator 


//************************************************************************************
//************************************************************************************
// Vector subtraction:
template<class T> GTVector<T> GTVector<T>::operator-(GTVector<T> a) 
{
  if ( index.szglobal() != a.GetIndex().szglobal() ) 
//if ( n != a.dim() )
  {
    cout << "template<class T> GTVector<T>::-: incompatible vectors"<< endl;
    exit(1);
  }

  GTVector<T>  aret(index);

#if defined(VEC_LEVEL0_TIMING) || defined(VEC_LEVEL1_TIMING)
  GDOUBLE tstart = STK::Timer();
#endif

  switch (dtype) {
    case G_GDOUBLE:
      MTK::fvec_sub((GTVector<GDOUBLE>&)*this, (GTVector<GDOUBLE>&)a, (GTVector<GDOUBLE>&)aret);
      break;
    case G_GQUAD:
      MTK::qvec_sub((GTVector<GQUAD>&)*this, (GTVector<GQUAD>&)a, (GTVector<GQUAD>&)aret);
      break;
    default:
      cout << " GTVector<T>::operator+: invalid data type" << endl;
      exit(1);
  }

#if defined(VEC_LEVEL0_TIMING)
  time_result  = MTK::GetTime();
#elif defined(VEC_LEVEL1_TIMING)
  time_result = STK::Timer() - tstart;
#endif



  return aret;

} // end of - operator


#if 0
//************************************************************************************
//************************************************************************************
// () operator method to set slice for consideration
template<class T> void GTVector<T>::operator()(GINT  ib, GINT  ie, GINT  is, GINT  ibase)
{
//index(index.szglobal(),index.szlocal(), ib, ie, is, index.pad());
  index(ib, ie, is, ibase);
} // end of () operator


//************************************************************************************
//************************************************************************************
// () operator method reference 
template<class T> T &GTVector<T>::operator()(const GINT  i)
{
#if defined(GARRAY_BOUNDS)
  //if ( i >= n || i < 0 ) throw "template<class T> GTVector<T>::(): access error";

  if ( i >= index.szglobal() || i < 0 ) 
  {
    cout << "template<class T> GTVector<T>::&(): access error"<< endl;
    exit(1);
  }

  if ( i<index.beg() || i>index.end() ) return (fNULL=0.0);
#endif
  return *(data+i-index.beg()); 
} // end of () operator


//************************************************************************************
//************************************************************************************
// () operator method read-only
template<class T> T GTVector<T>::operator()(const GINT  i) const
{ 
#if defined(GARRAY_BOUNDS)
//if ( i >= n || i < 0 ) throw "template<class T> GTVector<T>::(): access error";
  if ( i >= index.szglobal() || i < 0 ) 
  {
    cout << "GTVector<T>:(): i=" << i << endl;
    cout << "template<class T> GTVector<T>::(): access error"<< endl;
    exit(1);
  }

  if ( i<index.beg() || i>index.end() ) return 0.0;
#endif

  return *(data+i-index.beg()); 
} // end of () operator
#endif
  

//************************************************************************************
//************************************************************************************
// << operator method (1) 
ostream &operator<<(ostream &str, const GTVector<GDOUBLE> &a)
{
  GINT   i;

  str <<  "{ ";
  if ( (a.GetIndex().end()-a.GetIndex().beg()) > 0 ) {
    for ( i=a.GetIndex().beg(); i<a.GetIndex().end(); i++ ) {
      str 
#if 0
        << setw(18)
        << setprecision(15)
        << setiosflags(ios::fixed)
#endif
        << a(i) << ", ";
      }
    str << a(i) ; 
  }
  str << " }";

  return str;
} // end of << operator  (1)


#if defined(GBUFF_DEF_GQUAD)
//************************************************************************************
//************************************************************************************
// << operator method (2)
ostream &operator<<(ostream &str, const GTVector<GQUAD> &a)
{

  GINT   i;

  str <<  "{ ";
  if ( (a.GetIndex().end()-a.GetIndex().beg()) > 0 ) {
    for ( i=a.GetIndex().beg(); i<a.GetIndex().end(); i++ )
      str << (GDOUBLE)a(i) << ", ";
    str << (GDOUBLE)a(i) ; 
  }
  str << " }";

  return str;
} // end of << operator (2)
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : Data
// DESCRIPTION: 
// RETURNS    :  pointer to the array data area
//************************************************************************************
template<class T> T *GTVector<T>::Data() const
{
  return data;
} // end of method Data


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// RETURNS    :  none
//************************************************************************************
template<class T> void GTVector<T>::DeleteDynamic()
{
  if ( data != NULL ) delete [] data ;
  data = NULL;

} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Resize (1)
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
template<class T> GBOOL GTVector<T>::Resize(GINT  newDim, GINT  ib, GINT  ie)
{
  GBOOL bRet = FALSE;

  if ( newDim < (ie-ib+1) ) return FALSE;

  if ( data != NULL ) {
    delete [] data;
    data = NULL;
  }
  if ( newDim > 0 ) {
    data = new T [ie-ib+1+index.pad()];
    bRet = FALSE;
    if ( data != NULL && newDim > 0 ) {
      memset(data, 0   , (ie-ib+1+index.pad())*sizeof(T));
      index(newDim,newDim,0,newDim-1,1,index.pad());
      bRet = TRUE;
    }
  }
  n = newDim;

  return bRet;

} // end of method Resize (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Resize (2)
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
template<class T> GBOOL GTVector<T>::Resize(GINT  newDim)
{
  GBOOL bRet = FALSE;

  if ( newDim == n ) return TRUE;

  if ( data != NULL ) {
    delete [] data;
    data = NULL;
  }
  if ( (newDim+index.pad()) > 0 ) data = new T [newDim+index.pad()];
  bRet = FALSE;
  if ( newDim > 0 ) {
    if ( data != NULL ) {
      memset(data, 0   , (newDim+index.pad())*sizeof(T));
      index(newDim,newDim,0,newDim-1,1,index.pad());
      bRet = TRUE;
    }
  }
  n = newDim;

  return bRet;

} // end of method Resize(2)


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetIndex()
// DESCRIPTION: returns pointer to GIndex member
// RETURNS    : GIndex data member
//************************************************************************************
template<class T> GIndex*  GTVector<T>::GetIndex() const
{ 
  return (&index);
} // end of method GetIndex
#endif

//************************************************************************************
//************************************************************************************
// METHOD     : GetIndex()
// DESCRIPTION: returns pointer to GIndex member
// RETURNS    : GIndex data member
//************************************************************************************
template<class T> GIndex  GTVector<T>::GetIndex() const
{
  GIndex siret = index;
  return (siret);
} // end of method GetIndex


//************************************************************************************
//************************************************************************************
// METHOD     : tsize
// DESCRIPTION: total size of data storage area
//              = n + npad_
// RETURNS    : GINT  size
//************************************************************************************
template<class T> GINT  GTVector<T>::tsize()
{ 
  return (index.end()-index.beg()+1+index.pad());
} // end of mewthod tsize


//************************************************************************************
//************************************************************************************
// METHOD     : dim
// DESCRIPTION: working size of data storage area
// RETURNS    : GINT  n
//************************************************************************************
template<class T> GINT  GTVector<T>::dim() const
{ 
  return n;
} // end of mewthod dim


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : beg
// DESCRIPTION: beginning nozero index
// RETURNS    : GINT   
//************************************************************************************
template<class T> GINT  GTVector<T>::beg() const 
{ 
  return index.beg();
} // end of method beg


//************************************************************************************
//************************************************************************************
// METHOD     : end
// DESCRIPTION: ending nozero index
// RETURNS    : GINT  
//************************************************************************************
template<class T> GINT  GTVector<T>::end() const
{ 
  return index.end();
} // end of method end 


//************************************************************************************
//************************************************************************************
// METHOD     : stride
// DESCRIPTION: stide sizze
// RETURNS    : GINT 
//************************************************************************************
template<class T> GINT  GTVector<T>::stride() const
{
  return index.stride();
} // end of method stride
#endif

//************************************************************************************
//************************************************************************************
// METHOD     : Zero
// DESCRIPTION: Zeros out data elemnts
// ARGUMENTS  : indices: buffer of indices at which to zero vector. If NULL, entire
//                       vector is zeroed.
// RETURNS    : none
//************************************************************************************
template<class T> void  GTVector<T>::Zero(GBuffer<GINT>  *indices)
{
  GINT  i;
  GINT  sz = index.end()-index.beg()+1+index.pad();

  if ( indices == NULL ) {
    if ( data != NULL && sz > 0 ) 
      memset(data,0,sz*sizeof(T));
  }
  else {
    for ( i=0; i<indices->dim(); i++ ) (*this)[(*indices)[i]] = 0.0;
  }
} // end of method Zero


//************************************************************************************
//************************************************************************************
// METHOD     : Set (1)
// DESCRIPTION: Sets specified vector elements to specified value
// ARGUMENTS  : val     : value to set at specified indices
//              iset    : array of indices at which to set val. If NULL, entire
//                        vector is set, starting at index 0, and ending at
//                        index nset-1. 
//              nset    : number of indices in 'iset' array
//
// RETURNS    : none
//************************************************************************************
template<class T> void  GTVector<T>::Set(T val, GINT  *iset, GINT  nset)
{
  GINT  i;

  if ( iset == NULL ) {
    for ( i=0; i<nset; i++ ) (*this)[i] = val;
  }
  else {
    for ( i=0; i<nset; i++ ) (*this)[iset[i]] = val;
  }

} // end of Set(1)


//************************************************************************************
//************************************************************************************
// METHOD     : Set (2)
// DESCRIPTION: Sets specified vector elements to specified value 
// ARGUMENTS  : val     : value to set at specified indices
//              iset    : buffer of indices at which to set val. If NULL, entire
//                        vector is set.
//              iexclude: buffer of indices of vector to exclude from Set
//              
// RETURNS    : none
//************************************************************************************
template<class T> void  GTVector<T>::Set(T val, GBuffer<GINT>  *iset, GBuffer<GINT>  *iexclude)
{
  GINT  i, ind;

  if ( iset == NULL ) {
    if ( iexclude == NULL ) {
      for ( i=0; i<n; i++ ) (*this)[i] = val;
    }
    else {
      for ( i=0; i<n; i++ ) { 
        if ( !iexclude->contains(i,ind) ) (*this)[i] = val;
      }
    }
  }
  else {
    if ( iexclude == NULL ) {
      for ( i=0; i<iset->dim(); i++ ) (*this)[(*iset)[i]] = val;
    }
    else {
      for ( i=0; i<iset->dim(); i++ ) { 
        if ( !iexclude->contains((*iset)[i],ind) ) (*this)[(*iset)[i]] = val;
      }
    }
  }
} // end of method Set (2)


//************************************************************************************
//************************************************************************************
// METHOD     : Min
// DESCRIPTION: Finds min of array
// RETURNS    : min
//************************************************************************************
template<class T> T GTVector<T>::Min()
{
 
  GINT  i;
  T xmin=SEHUGE;

  for ( i=index.beg(); i<index.end()+1; i++ )
     xmin = MIN(xmin, (*this)(i));

  return xmin;
}

//************************************************************************************
//************************************************************************************
// METHOD     : MinA
// DESCRIPTION: Finds min of absolute value of  array
// RETURNS    : min
//************************************************************************************
template<class T> T GTVector<T>::MinA()
{
 
  GINT  i;
  T xmin=SEHUGE;

  for ( i=index.beg(); i<index.end()+1; i++ )
     xmin = MIN(xmin, fabs((*this)(i)));

  return xmin;
}

//************************************************************************************
//************************************************************************************
// METHOD     : Max
// DESCRIPTION: Finds max of array
// RETURNS    : max
//************************************************************************************
template<class T> T GTVector<T>::Max()
{
 
  GINT  i;
  T xmax=-SEHUGE;

  for ( i=index.beg(); i<index.end()+1; i++ )
     xmax = MAX(xmax, (*this)(i));

  return xmax;
}

//************************************************************************************
//************************************************************************************
// METHOD     : MaxA
// DESCRIPTION: Finds absolute max of array
// RETURNS    : max
//************************************************************************************
template<class T> T GTVector<T>::MaxA()
{

  GINT  i;
  T xmax=0.0;

  for ( i=index.beg(); i<index.end()+1; i++ )
     xmax = MAX(xmax, fabs((*this)(i)) );

  return xmax;
}

//************************************************************************************
//************************************************************************************
// METHOD     : EuclidNorm
// DESCRIPTION: Computes Euclidean norm^2 of vector
// RETURNS    : norm
//************************************************************************************
template<class T> T GTVector<T>::EuclidNorm(GDOUBLE *multiplicity)
{

  GINT  i;
  T sum, glob_sum;

  if ( multiplicity == NULL ) {
    for ( i=index.beg(), sum=0.0; i<index.end()+1; i++ )
       sum  += (*this)(i) * (*this)(i);
  }
  else {
    for ( i=index.beg(), sum=0.0; i<index.end()+1; i++ )
       sum  += (*this)(i) * (*this)(i) / multiplicity[i];
  }
  glob_sum = sum;

#if 1
  glob_sum = this->operator*(*this);
#endif
  
  return glob_sum;
}

//************************************************************************************
//************************************************************************************
// METHOD     : isDistributed
// DESCRIPTION: gets the bDistributed flag
// RETURNS    : GBOOL flag
//************************************************************************************
template<class T> GBOOL  GTVector<T>::isDistributed()
{
  return bDistributed ;
} // end of method isDistributed


//************************************************************************************
//************************************************************************************
// METHOD     : GetSection
// DESCRIPTION: gets a section of the array specified by the input indices
// RETURNS    : GTVector<T> reference
//************************************************************************************
template<class T> void GTVector<T>::GetSection(GINT  *indices, GINT  nindices, GTVector<T> &vret)
{
  GINT        i;

  if ( indices == NULL ) {
    vret.Resize(this->dim());
    vret = *this;
    return;
  }

  vret.Resize(nindices);
  for ( i=0; i<nindices; i++ ) {
    vret(i) = (*this)(indices[i]);
  }
} // end of method GetSection



