//************************************************************************************
// Module       : gcomm.cpp
// Date         : 5/14/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Namespace encapsulating general communications
//                utilities.
// Derived From : none.
// Modifications:
//************************************************************************************
#include <string.h>
#include "gcomm.hpp"

#if !defined(GCOMM_GLOBAL_DATA)
#define GCOMM_GLOBAL_DATA
#if defined(MPI_GENERIC_DEFAULT)
MPI_Op       G2MPI_OPTYPE  [] = {MPI_SUM, MPI_PROD, MPI_MAX, MPI_MIN };
#endif

#endif  


//************************************************************************************
//************************************************************************************
// METHOD     : WorldRank
// DESCRIPTION: Gets proc rank from all available procs
// ARGUMENTS  :
// RETURNS    : rank
//************************************************************************************

GSHORT GComm::WorldRank()
{
  GINT  rank=0;

#if defined(MPI_GENERIC_DEFAULT)
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
#endif

  return rank;
} // end of WorldRank

//************************************************************************************
//************************************************************************************
// METHOD     : WorldSize
// DESCRIPTION: Gets number of procs
// ARGUMENTS  :
// RETURNS    : number of procs
//************************************************************************************

GSHORT  GComm::WorldSize()
{
  GINT  NProcs=1;

#if defined(MPI_GENERIC_DEFAULT)
  MPI_Comm_size(MPI_COMM_WORLD,&NProcs);
#endif

  return NProcs;
} // end of WorldSize


//************************************************************************************
//************************************************************************************
// METHOD     : InitComm
// DESCRIPTION: initializes communication layer
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void GComm::InitComm(int *argc, char **argv[])
{
#if defined(MPI_GENERIC_DEFAULT)
  int          ierrMPI, iMPILen;
  char         sMPIErr[GMAX_ERROR_STRING];

  ierrMPI = MPI_Init(argc, argv);
  if ( ierrMPI != MPI_SUCCESS ) {
    MPI_Error_string(ierrMPI, sMPIErr, &iMPILen);
    cout << "GComm::InitComm: MPI_Init failed: MPI_Err: " << sMPIErr << endl;
    exit(1);
  }
#endif

} // end of InitComm


//************************************************************************************
//************************************************************************************
// METHOD     : TermComm
// DESCRIPTION: terminates communication layer
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void GComm::TermComm()
{

#if defined(MPI_GENERIC_DEFAULT)
  MPI_Finalize();
#endif

} // end of TermComm


//************************************************************************************
//************************************************************************************
// METHOD     : ASendRecv
// DESCRIPTION: Performs asynchronous send/recv from any source.
//              The source proc id's are returned in 'source' variable,
//              which must be allocated.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************

GBOOL GComm::ASendRecv(void **RecvBuff, GINT  nRecvBuff, GINT  *isend, GINT  *maxRecvLen, GC_DATATYPE rtype, GSHORT  *source, GBOOL bUseSource,
                       void **SendBuff, GINT  nSendBuff, GINT  *irecv, GINT  *maxSendLen, GC_DATATYPE stype, GSHORT  *dest   )
{
  GINT  i;

#if defined(MPI_GENERIC_DEFAULT)
  MPI_Request *mpi_recv_req=new MPI_Request [nRecvBuff];
  MPI_Status  *r_status    =new MPI_Status  [nRecvBuff];


  if ( mpi_recv_req == NULL || r_status == NULL ) 
  {
     if ( mpi_recv_req != NULL ) delete [] mpi_recv_req;
     if ( r_status     != NULL ) delete [] r_status;
     return FALSE;
  }

  if ( RecvBuff == NULL || SendBuff == NULL ) return FALSE;
  if ( source   == NULL || dest     == NULL ) return FALSE;

  // Post receives:
  if ( !bUseSource )
  {
    if ( irecv == NULL )
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[i], maxRecvLen[i], rtype, MPI_ANY_SOURCE, 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
    else
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[irecv[i]], maxRecvLen[i], rtype, MPI_ANY_SOURCE, 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
  }
  else
  {
    if ( irecv == NULL )
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[i], maxRecvLen[i], rtype, source[i], 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
    else
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[irecv[i]], maxRecvLen[i], rtype, source[i], 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
  }
 

  // Send data:
  if ( isend == NULL )
  {
    for ( i=0; i<nSendBuff; i++ )
        MPI_Send(SendBuff[i], maxSendLen[i], stype, dest[i], 0, MPI_COMM_WORLD);
  }
  else
  {
    for ( i=0; i<nSendBuff; i++ )
        MPI_Send(SendBuff[isend[i]], maxSendLen[i], stype, dest[i], 0, MPI_COMM_WORLD);
  }
    
  MPI_Barrier(MPI_COMM_WORLD);
  MPI_Waitall(nRecvBuff, mpi_recv_req,r_status);

  // if we didn't use the sources, then we return
  // them:
  if ( !bUseSource )
  {
    for ( i=0; i<nRecvBuff; i++ )
      source[i] = r_status[i].MPI_SOURCE;
  }
    
  delete [] mpi_recv_req;
  delete [] r_status;

  return TRUE;
#else
  return FALSE;
#endif

} // end of method ASendRecv


//************************************************************************************
//************************************************************************************
// METHOD     : ASendRecvDB
// DESCRIPTION: Performs asynchronous send/recv from any source.
//              The source proc id's are returned in 'source' variable,
//              which must be allocated. The recv and send data are assumed
//              to be stored in GBuffer<GDWORD> objects.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GComm::ASendRecvDB(GDWBuffer  **RecvBuff, GINT  nRecvBuff, GINT  *irecv, GINT  maxRecvLen, GSHORT  *source, GBOOL bUseSource,
                         GDWBuffer  **SendBuff, GINT  nSendBuff, GINT  *isend, GINT  maxSendLen, GSHORT  *dest  )
{
  GINT  i;

#if defined(MPI_GENERIC_DEFAULT)
  MPI_Request *mpi_recv_req=new MPI_Request [nRecvBuff];
  MPI_Status  *r_status    =new MPI_Status  [nRecvBuff];


  if ( mpi_recv_req == NULL || r_status == NULL ) 
  {
     if ( mpi_recv_req != NULL ) delete [] mpi_recv_req;
     if ( r_status     != NULL ) delete [] r_status;
     return FALSE;
  }

  if ( RecvBuff == NULL || SendBuff == NULL ) return FALSE;
  if ( source   == NULL || dest     == NULL ) return FALSE;


  // Post receives:
  if ( !bUseSource )
  {
    if ( irecv == NULL )
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[i]->Data(), maxRecvLen, GC_GDWORD, MPI_ANY_SOURCE, 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
    else
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[irecv[i]]->Data(), maxRecvLen, GC_GDWORD, MPI_ANY_SOURCE, 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
  }
  else
  {
    if ( irecv == NULL )
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[i]->Data(), maxRecvLen, GC_GDWORD, source[i], 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
    else
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[irecv[i]]->Data(), maxRecvLen, GC_GDWORD, source[i], 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
  }

  // Send data:
    if ( isend == NULL )
    {
      for ( i=0; i<nSendBuff; i++ )
        MPI_Send(SendBuff[i]->Data(), maxSendLen, GC_GDWORD, dest[i], 0, MPI_COMM_WORLD);
    }
    else
    {
      for ( i=0; i<nSendBuff; i++ )
        MPI_Send(SendBuff[isend[i]]->Data(), maxSendLen, GC_GDWORD, dest[i], 0, MPI_COMM_WORLD);
    }
    
  MPI_Barrier(MPI_COMM_WORLD);
  MPI_Waitall(nRecvBuff, mpi_recv_req, r_status);

  // if we didn't use the sources, then we return
  // them:
  if ( !bUseSource )
  {
    for ( i=0; i<nRecvBuff; i++ )
      source[i] = r_status[i].MPI_SOURCE;
  }
    
  delete [] mpi_recv_req;
  delete [] r_status;

  return TRUE;
#else
  return FALSE;
#endif

} // end of method ASendRecvDB


//************************************************************************************
//************************************************************************************
// METHOD     : ASendRecvNB
// DESCRIPTION: Performs asynchronous send/recv from any source.
//              The source proc id's are returned in 'source' variable,
//              which must be allocated. The recv and send data are assumed
//              to be stored in GBuffer<GNODEID> objects.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GComm::ASendRecvNB(GNIDBuffer  **RecvBuff, GINT  nRecvBuff, GINT  *irecv, GINT  maxRecvLen, GSHORT  *source, GBOOL bUseSource,
                         GNIDBuffer  **SendBuff, GINT  nSendBuff, GINT  *isend, GINT  maxSendLen, GSHORT  *dest  )
{
  GINT  i;

#if defined(MPI_GENERIC_DEFAULT)
  MPI_Request *mpi_recv_req=new MPI_Request [nRecvBuff];
  MPI_Status  *r_status    =new MPI_Status  [nRecvBuff];


  if ( mpi_recv_req == NULL || r_status == NULL ) 
  {
     if ( mpi_recv_req != NULL ) delete [] mpi_recv_req;
     if ( r_status     != NULL ) delete [] r_status;
     return FALSE;
  }

  if ( RecvBuff == NULL || SendBuff == NULL ) return FALSE;
  if ( source   == NULL || dest     == NULL ) return FALSE;


  // Post receives:
  if ( !bUseSource )
  {
    if ( irecv == NULL )
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[i]->Data(), maxRecvLen, GC_GNODEID, MPI_ANY_SOURCE, 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
    else
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[irecv[i]]->Data(), maxRecvLen, GC_GNODEID, MPI_ANY_SOURCE, 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
  }
  else
  {
    if ( irecv == NULL )
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[i]->Data(), maxRecvLen, GC_GNODEID, source[i], 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
    else
    {
      for ( i=0; i<nRecvBuff; i++ )
        MPI_Irecv(RecvBuff[irecv[i]]->Data(), maxRecvLen, GC_GNODEID, source[i], 0, MPI_COMM_WORLD, &mpi_recv_req[i]);
    }
  }

  // Send data:
    if ( isend == NULL )
    {
      for ( i=0; i<nSendBuff; i++ )
        MPI_Send(SendBuff[i]->Data(), maxSendLen, GC_GNODEID, dest[i], 0, MPI_COMM_WORLD);
    }
    else
    {
      for ( i=0; i<nSendBuff; i++ )
        MPI_Send(SendBuff[isend[i]]->Data(), maxSendLen, GC_GNODEID, dest[i], 0, MPI_COMM_WORLD);
    }
    
  MPI_Barrier(MPI_COMM_WORLD);
  MPI_Waitall(nRecvBuff, mpi_recv_req, r_status);

  // if we didn't use the sources, then we return
  // them:
  if ( !bUseSource )
  {
    for ( i=0; i<nRecvBuff; i++ )
      source[i] = r_status[i].MPI_SOURCE;
  }
    
  delete [] mpi_recv_req;
  delete [] r_status;

  return TRUE;
#else
  return FALSE;
#endif

} // end of method ASendRecvNB


//************************************************************************************
//************************************************************************************
// METHOD     : ARecv
// DESCRIPTION: Performs posting of asynchronous recv from known source.
// ARGUMENTS  :
// RETURNS    : Valid CHandle on success; else exit
//************************************************************************************

void GComm::ARecv(void **RecvBuff, GINT  nRecvBuff, GINT  *rBuffLen,  GINT  *iBuff, 
                  GC_DATATYPE rtype, GSHORT  *source, CHandle &hret)
{
  GINT    i;

  if ( RecvBuff == NULL || rBuffLen == NULL ) {
    cout << "GComm::ARecv: invalid data" << endl;
    exit(1);
  }

#if defined(MPI_GENERIC_DEFAULT)
  if ( hret.mhandle != NULL ) delete [] hret.mhandle; 
  hret.mhandle = NULL;
  hret.mhandle =  new MPI_Request [nRecvBuff];
  if ( hret.mhandle == NULL ) {
    cout << "GComm::ARecv: invalid mhandle" << endl;
    exit(1);
  }
  // Note:  handle for MPI_GENERIC_DEFAULT ARecv call is just the MPI_Request object....

  hret.nposts = nRecvBuff;
 
//cout << "GComm::ARecv: iHandle=" << iHandle << " handles=" << handles[iHandle] << " nUsedHandles=" << nUsedHandles << endl;
  
  if ( iBuff == NULL )
  {
    for ( i=0; i<nRecvBuff; i++ )
      MPI_Irecv(RecvBuff[i], rBuffLen[i], rtype, source[i], 0, MPI_COMM_WORLD, (MPI_Request*)(hret.mhandle)+i);
  }
  else
  {
    for ( i=0; i<nRecvBuff; i++ )
      MPI_Irecv(RecvBuff[iBuff[i]], rBuffLen[iBuff[i]], rtype, source[i], 0, MPI_COMM_WORLD, (MPI_Request*)(hret.mhandle)+i);
  }
  
#endif

} // end of method ARecv


//************************************************************************************
//************************************************************************************
// METHOD     : BSend
// DESCRIPTION: Performs blocking send of provided buffers
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
  
GBOOL GComm::BSend(void **SendBuff, GINT  nSendBuff, GINT  *sBuffLen, GINT  *iBuff, GC_DATATYPE stype, GSHORT  *dest)
{   
  GINT    i;
  
  if ( SendBuff == NULL || sBuffLen == NULL ) return FALSE;

#if defined(MPI_GENERIC_DEFAULT)
  
  if ( iBuff == NULL )
  {
    for ( i=0; i<nSendBuff; i++ )
      MPI_Send(SendBuff[i], sBuffLen[i], stype, dest[i], 0, MPI_COMM_WORLD);
  }
  else
  {
    for ( i=0; i<nSendBuff; i++ )
      MPI_Send(SendBuff[iBuff[i]], sBuffLen[iBuff[i]], stype, dest[i], 0, MPI_COMM_WORLD);
  }
  
  return TRUE;
#else
  return FALSE;
#endif
    
} // end of method BSend


//************************************************************************************
//************************************************************************************
// METHOD     : AWaitOnRecv
// DESCRIPTION: Performs blocking wait for recv of specified handle
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
 
GBOOL GComm::AWaitOnRecv(CHandle &ch)
{   
  GINT    i;
  

#if defined(MPI_GENERIC_DEFAULT)
  MPI_Status status;
   
  for ( i=0; i<ch.nposts; i++ )
  { 
     MPI_Wait((MPI_Request*)(ch.mhandle)+i, &status);
  }
  delete [] (MPI_Request*)ch.mhandle; ch.mhandle = NULL;

  return TRUE;
#else
  return FALSE;
#endif

    
} // end of method AWaitOnRecv
 

//************************************************************************************
//************************************************************************************
// METHOD     : Allreduce(1)
// DESCRIPTION: Performs reduction
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************

GSHORT  GComm::Allreduce(void *operand, void *result, const GINT  count, GC_DATATYPE itype, G_OP iop)
{
  GINT   i;
  GSHORT  ircount=count;

#if defined(MPI_GENERIC_DEFAULT)
#if defined(GCOMM_DEBUG_OUTPUT)
    cout << "GComm::Allreduce: count       = " << count << endl;
    cout << "GComm::Allreduce: GC_DATATYPE  = " << itype << endl;
    cout << "GComm::Allreduce: MPI_DATATYPE= " <<  itype << endl;
#endif
    ircount = MPI_Allreduce(operand, result, ircount, itype, G2MPI_OPTYPE[iop], MPI_COMM_WORLD);
#else
//  memcpy(result, operand, count*itype);
    for ( i=0; i<count; i++ )
      memcpy((GBYTE*)result+i*itype, (GBYTE*)operand+i*itype, itype);
#endif

  return ircount;

} // end of method Allreduce(1)




//************************************************************************************
//************************************************************************************
// METHOD     : Allreduce(2)
// DESCRIPTION: Performs reduction
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************

GSHORT  GComm::Allreduce(void *operand, void *result, const GINT  count, GC_DATATYPE itype, G_OP iop, void *comm)
{
  GINT   i;
  GSHORT  ircount=count;

#if defined(MPI_GENERIC_DEFAULT)
#if defined(GCOMM_DEBUG_OUTPUT)
    cout << "GComm::Allreduce: count       = " << count << endl;
    cout << "GComm::Allreduce: GC_DATATYPE  = " << itype << endl;
    cout << "GComm::Allreduce: MPI_DATATYPE= " <<  itype << endl;
#endif
    ircount = MPI_Allreduce(operand, result, ircount, itype, G2MPI_OPTYPE[iop], *((MPI_Comm*)comm));
#else
//  memcpy(result, operand, count*itype);
    for ( i=0; i<count; i++ )
      memcpy((GBYTE*)result+i*itype, (GBYTE*)operand+i*itype, itype);
#endif

  return ircount;
} // end of method Allreduce(2)


//************************************************************************************
//************************************************************************************
// METHOD     : Allgather(1)
// DESCRIPTION: Performs all-gather operation
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************

GSHORT  GComm::Allgather(void *operand, GINT  sendcount, GC_DATATYPE stype, void *result, GINT  recvcount, GC_DATATYPE rtype)
{
  GINT   i;
  GSHORT  iRet=sendcount, rank=GComm::WorldRank();

#if defined(MPI_GENERIC_DEFAULT)
    iRet = MPI_Allgather(operand, sendcount, stype, result, recvcount, rtype, MPI_COMM_WORLD);
#else
  if ( recvcount < sendcount ) return 0;
  memcpy((GBYTE*)result+rank*recvcount*rtype, (GBYTE*)operand, sendcount*stype);
#endif

  return iRet;

} // end of method Allgather(1)


//************************************************************************************
//************************************************************************************
// METHOD     : Allgather(2)
// DESCRIPTION: Performs all-gather operation
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************

GSHORT  GComm::Allgather(void *operand, GINT  sendcount, GC_DATATYPE stype, void *result, GINT  recvcount, GC_DATATYPE rtype, void *comm)
{
  GINT   i;
  GSHORT  iRet=sendcount, rank=GComm::WorldRank();

#if defined(MPI_GENERIC_DEFAULT)
    iRet = MPI_Allgather(operand, sendcount, stype, result, recvcount, rtype, *((MPI_Comm*)comm));
#else
  if ( recvcount < sendcount ) return 0;
  memcpy((GBYTE*)result+rank*recvcount*rtype, (GBYTE*)operand, sendcount*stype);
#endif

  return iRet;

} // end of method Allgather(2)


//************************************************************************************
//************************************************************************************
// METHOD     : DataTypeFromStruct
// DESCRIPTION: Builds a derived data type usable by Comm methods, which encapsulates
//              the data associated with a C-type structure, *WHICH CONTAINS
//              CONTIGUOUS DATA*.
// ARGUMENTS  :
//              blk_ptr    : array of pointers to C-type structure members, 
//                           of length 'n-blk'. Note, that for platform-independence,
//                           the blk_ptr array should be formed by calls to
//                           GComm::Address.
//              sz_types   : array of GC_DATATYPE's of length 'n_blk', each of
//                           member of which represents the GC_DATATYPE's of the
//                           c_struct data members in order. The sz_types MUST
//                           NOT be G_STRUCTTYPE, but MUST be a basic type!
//              n_types    : array of length 'n_blk' that gives the corresp. no. of
//                           elements of type sz_types.
//              n_blk      : number of data members in the c_struct structure
//              return_type: the datatype which will be passed in subsequent 
//                           GComm calls, is an MPI_Datatype when MPI is used, else
//                           it is the structure size (an GINT ), in bytes.
// RETURNS    : TRUE on success; else FALSE.
//************************************************************************************
GBOOL GComm::DataTypeFromStruct(AGINT  blk_ptr[], GC_DATATYPE sz_types[], GINT  n_types[], 
                                const GINT  n_blk, GC_DATATYPE *return_type)
{
#if defined(MPI_GENERIC_DEFAULT)
  GINT         i, ierrMPI  ;
  int          *blk_len, iMPILen;
  char         sMPIErr[GMAX_ERROR_STRING];
  MPI_Aint     *offset      ;
  MPI_Aint     pstart       ;
  MPI_Datatype *mpitypes    ;

  blk_len = new int          [n_blk];
  offset  = new MPI_Aint     [n_blk];
  mpitypes= new MPI_Datatype [n_blk];

  // Build data structure for blocks type sz_types[i]:
  pstart = blk_ptr[0];

#if 0
    cout << "GComm::DataTypeFromStruct: n_blk=" << n_blk << endl;
#endif
  for ( i=0; i<n_blk; i++ ) {
    blk_len [i] = n_types [i];
    mpitypes[i] = sz_types[i];
    offset  [i] = blk_ptr[i] - pstart;
#if 0
    cout <<  "GComm::DataTypeFromStruct: "
         <<  "blk_ptr[" << i << "]=" << blk_ptr[i] 
         << " blk_len[" << i << "]=" << blk_len[i] 
         << " mpitypes[" << i << "]=" << mpitypes[i] 
         << " offset[" << i << "]=" << offset[i] << endl;
#endif
  }
#if 0
  cout << "GComm::DataTypeFromStruct: return_type: " << return_type << endl;
#endif

  // Build MPI datatype:
#  if defined(G_MPI1)
  ierrMPI=MPI_Type_struct(n_blk, blk_len, offset, mpitypes, (MPI_Datatype*)return_type);
#  else
  ierrMPI=MPI_Type_create_struct(n_blk, blk_len, offset, mpitypes, (MPI_Datatype*)return_type);
#  endif
  if ( ierrMPI != MPI_SUCCESS ) {
    MPI_Error_string(ierrMPI, sMPIErr, &iMPILen);
    cout << "GComm::DataTypeFromStruct: MPI_Type_create_struct failed: MPI_Err: " << sMPIErr << endl;
    return FALSE;
  }

  // Commit datatype for communication messages
  ierrMPI=MPI_Type_commit((MPI_Datatype*)return_type);
  if ( ierrMPI != MPI_SUCCESS ) {
    MPI_Error_string(ierrMPI, sMPIErr, &iMPILen);
    cout << "GComm::DataTypeFromStruct: MPI_Type_commit failed: MPI_Err: " << sMPIErr << endl;
    return FALSE;
  }
  delete [] blk_len;
  delete [] offset ;
  delete [] mpitypes;

  return TRUE;
#else
  GINT  i, n=0, r;

  for ( i=0,n=0; i<n_blk; i++ ) {
    n += sz_types[i]*n_types [i];
  }
  // Now add the size of padding required to align
  // on integer-byte boundaries:
  r = n % WORDSIZE_BYTES;
  n = (r == 0 ? n :  n + WORDSIZE_BYTES - r); 
  *return_type = (GC_DATATYPE)n; 
  return TRUE;
#endif

} // end of method DataTypeFromStruct



//************************************************************************************
//************************************************************************************
// METHOD     : DataTypeFree
// DESCRIPTION: Frees derived data types created from DataTypeFromStruct method.
// ARGUMENTS  :
//              type: the datatype to be freed. Can be MPI or other
// RETURNS    : TRUE on success; else FALSE.
//************************************************************************************
void GComm::DataTypeFree(GC_DATATYPE *type)
{
#if defined(MPI_GENERIC_DEFAULT)

  // Free MPI datatype:
  MPI_Type_free((MPI_Datatype*)type);
#endif

} // end of method DataTypeFree


//************************************************************************************
//************************************************************************************
// METHOD     : Address
// DESCRIPTION: Gets implementation--independent address of memory location 
// ARGUMENTS  : location: pointer location of interest
//              address : int address or Address Integer, AGINT 
// RETURNS    :
//************************************************************************************
void GComm::Address(void *location, AGINT  *address)
{   
#if defined(MPI_GENERIC_DEFAULT)
//cout << " GComm::Address: location=" << location << " address=" << *address << endl;
#  if defined(G_MPI1)
  MPI_Address(location, address);
#  else
  MPI_Get_address(location, address);
#  endif
//cout << " GComm::Address: location=" << location << " address=" << *address << endl;
#else
  *address = (AGINT )(location);
#endif
} // end of method Address
  


//************************************************************************************
//************************************************************************************
// METHOD     : Synch(1)
// DESCRIPTION: Synchronozises all procs
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GComm::Synch()
{
#if defined(MPI_GENERIC_DEFAULT)
  MPI_Barrier(MPI_COMM_WORLD);
#endif
}// end of Sync(1)


//************************************************************************************
//************************************************************************************
// METHOD     : Synch(2)
// DESCRIPTION: Synchronozises all procs
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GComm::Synch(void *comm)
{
#if defined(MPI_GENERIC_DEFAULT)
  MPI_Barrier(*((MPI_Comm*)comm));
#endif
} // end of method Synch(2)
