//************************************************************************************//
// Module       : gmortar1d.cpp
// Date         : 7/10/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a 'mortar' object, which provides all the information
//                required to maintain connectivity between spectral
//                elements. Mortar in this class represents the true
//                global degrees of freedom.
//                Call sequence might be:
//
//                       SetBasis(new basis);  // sets basis
//                       SetBdyPoints;         // sets spatial endpoints
//                       SetHostGrid;          // sets grid to which to interpolate mortar field
//                      {Host2Mortar;          // sets the field from the host
//                       GetMortarField;       // gets internal mortar field variable 
//                       GetNodeIDs;           // gets mortar node ids (if set)
//                           <...do DSS... with mortar field and IDs, for e.g....>
//                       Mortar2Host;}         // interpolates from mortar field to host field
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "gmortar1d.hpp"
#include "mtk.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GMortar1D::GMortar1D(GNBasis *b)
:
nBdy_            (2),
NN_              (0),
istart_          (0),
bConforming_     (TRUE),
bHostGrid_       (FALSE),
len_             (0.0),
ilen_            (HUGE),
smin_            (0),
smax_            (0),
pBdy_            (NULL),
node_ids_        (NULL),
mtr_basis_       (b),
u_mtr_           (NULL),
xi_mtr_          (NULL),
s_mtr_           (NULL),
u_hst_           (NULL),
xi_hst_          (NULL),
IM2H_            (NULL),
IH2M_            (NULL)
{
  GINT  i, n=0;

  // By convention, the endpoints, which are vertices, are included in the mortar

  if ( b != NULL ) n = b->GetOrder() + 1;  
  iorder_ = new GINT  [nBdy_];
  pBdy_  = new Point3D    [nBdy_];
  node_ids_ = new GIBuffer ();
  u_mtr_    = new GVector();
  xi_mtr_   = new GVector();
  s_mtr_    = new GVector();
  u_hst_    = new GVector();
  xi_hst_   = new GVector();
  IM2H_     = new GMatrix();
  IH2M_     = new GMatrix();
  for ( i=0; i<GDIM; i++ ) {
    x_hst_[i] = new GVector();
    x_mtr_[i] = new GVector();
  }
  Resize(n);
  
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
GMortar1D::~GMortar1D()
{
  DeleteDynamic();
}

//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION:
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void  GMortar1D::DeleteDynamic()
{
  GINT  i;

  if ( node_ids_  != NULL ) delete node_ids_;
  if ( u_mtr_     != NULL ) delete u_mtr_;
  if ( xi_mtr_    != NULL ) delete xi_mtr_;
  if ( s_mtr_     != NULL ) delete s_mtr_;
  if ( u_hst_     != NULL ) delete u_hst_;
  if ( xi_hst_    != NULL ) delete xi_hst_;
  if ( IM2H_      != NULL ) delete IM2H_;
  if ( IH2M_      != NULL ) delete IH2M_;
  for ( i=0; i<GDIM; i++ )
  {
    if ( x_hst_  [i] != NULL ) delete x_hst_[i];
    if ( x_mtr_  [i] != NULL ) delete x_mtr_[i];
  }
  delete [] pBdy_; 
  delete [] iorder_; 

} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void  GMortar1D::Resize(GINT  n)
{
  GINT  i;

  if ( n <= 0 ) return;
  node_ids_->Resize(n);
  u_mtr_   ->Resize(n);
  xi_mtr_  ->Resize(n);
  s_mtr_   ->Resize(n);
  for ( i=0; i<GDIM; i++ ) {
    x_mtr_[i]->Resize(n);
  }
  NN_       = n;
} // end of method Resize


//************************************************************************************
//************************************************************************************
// METHOD     : SetBasis
// DESCRIPTION:
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void GMortar1D::SetBasis(GNBasis *b)
{
  GINT  n;

  if ( b == NULL ) {
    cout << "GMortar1D::SetBasis: NULL basis" << endl;
    exit(1);
  }
  mtr_basis_ = b;

  // By convention, the endpoints, which are vertices, are included in the mortar

  n = b->GetOrder() + 1;
  Resize(n);

} // end of method SetBasis


//************************************************************************************
//************************************************************************************
// METHOD     : GetBasis
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GNBasis *GMortar1D::GetBasis()
{
  return mtr_basis_;
} // end of method GetBasis



//************************************************************************************
//************************************************************************************
// METHOD     : SetBdyPoints
// DESCRIPTION: Sets the bdy points defining geometric bdys for mortar, and
//              sets all relevant quantities associated with this. The
//              points must be s.t. first point has the smallest absolute r-
//              value. Default IDs are assigned to the mortar nodes.
//
//              Note: basis must be set prior to entry
// ARGUMENTS  : p_origin: point wrt which the parent interval, s=[-1,1] (defining basis)
//                        is computed, s.t., s(-1) <-> p_origin.
//              p_end   : second point defining mortar line segment
// RETURNS    :  
//************************************************************************************
void  GMortar1D::SetBdyPoints(Point3D &p_origin, Point3D &p_end )
{
  GINT    i, k;
  GDOUBLE   alpha, sa, ca, glen, dx[GDIM];

  if ( mtr_basis_ == NULL ) {
    cout << "GMortar1D::SetBdyPoints: NULL mortar basis" << endl;
    exit(1);
  }

  pBdy_[0] = p_origin;
  pBdy_[1] = p_end;

  // Set lengths:
  for ( k=0, len_=0.0; k<GDIM; k++ ) {
    dx[k] = pBdy_[1][k] - pBdy_[0][k];
  }

  len_ = PDISTANCE(pBdy_[1],pBdy_[0]);
  if ( len_ <=  0.0 ) {
    cout << "GMortar1D::SetBdyPoints: invalid mortar dimensions" << endl;
    exit(1);
  }
  ilen_ = 1.0/(len_);
  glen  = 0.5*len_;

  // With geometry set, compute spatial location of nodes:
  
  // s = L/2 * ( xi + 1), where xi is the parent
  // (non-dimensional) coordinate location (actually an arc-length):
  mtr_basis_->GetXiNodes(xi_mtr_);
  for ( i=0; i<xi_mtr_->dim(); i++ ) {
    (*s_mtr_ )(i) = glen*((*xi_mtr_)(i)+1.0); 
  }
#if 0
cout << "GMortar1D::SetBdyPoints: xi_mtr = " << *xi_mtr_ << endl;
cout << "GMortar1D::SetBdyPoints: s_mtr = " << *s_mtr_ << endl;
#endif

  // must transform from affine space to 
  // Cartesian coords:
  x_mtr_[0]->Resize(NN_);
  x_mtr_[1]->Resize(NN_); 
  alpha = dx[0]==0 ? 0.5*PI : atan(dx[1]/dx[0]);
  sa = sin(alpha);
  ca = cos(alpha);
  for ( i=0; i<NN_; i++ ) {
    (*x_mtr_[0])(i) = (*s_mtr_)(i)*ca + pBdy_[0].x1 ;
    (*x_mtr_[1])(i) = (*s_mtr_)(i)*sa + pBdy_[0].x2 ;
  }

} // end of method SetBdyPoints


//************************************************************************************
//************************************************************************************
// METHOD     : SetNodeIDs
// DESCRIPTION:  Sets node ids from external source. If it's to be used with
//               SetBdyPoints, then method must be called *after* the call to
//               SetBdyPoints, which sets the ids by default.
// ARGUMENTS  : 
// RETURNS    :  
//************************************************************************************
void  GMortar1D::SetNodeIDs(GINT  n, GINT  *ids)
{
  GINT  i;

  if ( n != node_ids_->dim() ) {
    cout << "GMortar1D::SetNodeIDs: inconsistent number of nodes specified" << endl;
    exit(1);
  }

  for ( i=0; i<n; i++ ) {
    (*node_ids_)(i) = ids[i];
  }

} // end of method SetNodeIDs


//************************************************************************************
//************************************************************************************
// METHOD     : SetHostGrid
// DESCRIPTION: Sets the coordinates for the host element's nodes.
//              NOTE: basis must be set using SetBasis prior to entry
// ARGUMENTS  :  
//               x      : coords for field quantity; one vector for each component, x, y, z
//               nc     : number of coords (rank)
//               indices: indices to use in specifying coords. If NULL, then all
//                        coords are used; else indirect references are made
//                        to set the host's coords.
//               ni     : number of indices
// RETURNS    :  
//************************************************************************************
void  GMortar1D::SetHostGrid(GVector **x, GINT  nc,  GINT  *indices, GINT  ni)
{
  GINT  i, j;
  GDOUBLE dx, s, gilen=2.0*ilen_;
  Point pmtr, phst, tbdy[2];

  if ( nc != GDIM ) {
    cout << "GMortar1D::SetHostGrid: invalid rank" << endl;
    exit(1);
  }

  if ( x == NULL ) {          // use the mortar's grid
    bConforming_ = TRUE;
    bHostGrid_   = TRUE;
    return;
  } 

  if ( mtr_basis_ == NULL  ) {
    cout << "GMortar1D::SetHostGrid: mortar basis is NULL" << endl;
    exit(1);
  }

  bHostGrid_   = FALSE;

  // Set coords:
  if ( indices == NULL ) {     // entire arrays used to specify host coords
    for ( j=0; j<GDIM; j++ ) {
      x_hst_[j]->Resize(x[j]->dim());
      *x_hst_[j] = *x[j];
    }
  }
  else {                       // only part of arrays used to specify host coords
    for ( j=0; j<GDIM; j++ ) {
      x_hst_[j]->Resize(ni);
      for ( i=0; i<ni; i++ ) {
       (*x_hst_[j])(i) = (*x[j])(indices[i]);
      }
    }
  }
#if 0
cout << "GMortar1D: SetHostGrid: pBdy[0]=" << pBdy_[0] << endl;
cout << "GMortar1D: SetHostGrid: pBdy[1]=" << pBdy_[1] << endl;
cout << "GMortar1D: SetHostGrid: len_   =" << len_     << endl;
cout << "GMortar1D: SetHostGrid: ilen_  =" << ilen_    << endl;
#endif

  // Generate the parent domain (on [-1,1]) values 
  // of the host grid's coords:  
  xi_hst_->Resize(x_hst_[0]->dim());
  smin_ =  SEHUGE;
  smax_ = -SEHUGE;
  for ( i=0; i<x_hst_[0]->dim(); i++ ) {
    // First find arclength: s^2 = Sum_1_GDIM (x_i -x_i_0)^2
    for ( j=0,s=0.0; j<GDIM; j++ ) {
      dx = (*x_hst_[j])(i) - pBdy_[0][j];
      s +=  dx*dx;                         
    }
    // Then, find parent coordinate from s = 0.5*L(xi+1);
    // restrict to interval [-1,1], relative to host:
    s             = sqrt(s);
    (*xi_hst_)(i) = MIN( 1.0, MAX(-1.0, s*gilen - 1.0) );
//  (*xi_hst_)(i) = s * gilen - 1.0;
//  sgn = (*xi_hst_)(i) == 0.0 ? 0.0 : (*xi_hst_)(i) / fabs((*xi_hst_)(i));
//  if ( (fabs((*xi_hst_)(i))-1.0) < 10.0*TINY ) (*xi_hst_)(i) = sgn; 
//  if ( fabs((*xi_hst_)(i)) < 10.0*TINY ) (*xi_hst_)(i) = 0.0; 
    smin_ = MIN(smin_,s);   // host's min arclen along mortar
    smax_ = MAX(smax_,s);   // host's max arclen along mortar
  }

  // With host grid and mortar grid, determine if
  // mortar is conforming:
  tbdy[0].x1 = (*x_hst_[0])(0);
  tbdy[0].x2 = (*x_hst_[1])(0);
  tbdy[1].x1 = (*x_hst_[0])(x_hst_[1]->dim()-1);
  tbdy[1].x2 = (*x_hst_[1])(x_hst_[1]->dim()-1);

  bConforming_ = tbdy[0] == pBdy_[0] && tbdy[1] == pBdy_[1]; 

#if defined(GM_DEBUG_OUTPUT)
if ( !bConforming_ ) {
cout << "GMortar1D: SetHostGrid: tbdy0  =" << tbdy[0] << " tbdy1 =" << tbdy[1] << endl;
cout << "GMortar1D: SetHostGrid: pbdy0  =" << pBdy_[0] << " pbdy1 =" << pBdy_[1] << endl;
//cout << "GMortar1D: SetHostGrid: xi_mtr=" << *xi_mtr_ << endl;
cout << "GMortar1D: SetHostGrid: xi_hst=" << *xi_hst_ << endl;
}
#endif

   // If conforming, don't need to allocate interpolation matrices, so:
   if ( bConforming_ ) {
     bHostGrid_   = TRUE;
     return;
   }
  
  // Generate interpolation matrices, using the parent domain, xi values, 
  // for the host element (NOTE: IM2H=J; IH2M=J^T, in the usual (Fischer) notation):
  IM2H_->Resize(xi_hst_->dim(),xi_mtr_->dim());
  IH2M_->Resize(xi_mtr_->dim(),xi_hst_->dim());
  for ( i=0; i<xi_hst_->dim(); i++ ) {
    for ( j=0; j<xi_mtr_->dim(); j++ ) {
      (*IM2H_)(i,j) = mtr_basis_->EvalBasis(j, (*xi_hst_)(i));
    } 
  } 

  if ( !IM2H_->Transpose(*IH2M_) ) {
    cout << "GMortar1D::SetHostGrid: unable to compute interpolation operator" << endl;
    exit(1);
  }
#if 0
  *IH2M_ = 0.0;
  cout << "SetHostGrid: s_mtr=" << *s_mtr_ << " smin=" << smin_ << " smax=" << smax_ << endl;
  for ( i=0; i<xi_mtr_->dim(); i++ ) {
    for ( j=0; j<xi_hst_->dim(); j++ ) {
      if ( (*s_mtr_)[i] >= smin_ && (*s_mtr_)[i] <= smax_ ) {
        eta = smin_ == 0.0 ? 2.0* (*xi_mtr_)(i) + 1 :  2.0*(*xi_mtr_)(i)-1.0; 
        (*IH2M_)(i,j) = mtr_basis_->EvalBasis(j, eta);
      }
    } 
  } 
#endif


#if defined(GM_DEBUG_OUTPUT)
  if ( !bConforming_ )
  cout << "SetHostGrid: J   == iM2H=" << *IM2H_ << endl;
  //cout << "SetHostGrid: J^T == iH2M=" << *IH2M_ << endl;
#endif
  bHostGrid_   = TRUE;

} // end of method SetHostGrid


//************************************************************************************
//************************************************************************************
// METHOD     : GetArcPoints
// DESCRIPTION: Gets the spatial affine points
// ARGUMENTS  :  none
// RETURNS    : GVector  *
//************************************************************************************
GVector *GMortar1D::GetArcPoints()
{
  return s_mtr_;
} // end of method GetArcPoints


//************************************************************************************
//************************************************************************************
// METHOD     : GetSpGrid
// DESCRIPTION: Gets the spatial grid points for mortar nodes, in direction idir
// ARGUMENTS  : idir : coordinate direction
// RETURNS    : GVector  *
//************************************************************************************
GVector *GMortar1D::GetSpGrid(GINT  idir)
{
  if ( idir < 1 && idir > GDIM ) {
    cout << "GMortar1D::GetSpGrid: invalid coordinate direction" << endl;
    exit(1);
  }
  return x_mtr_[idir-1];
} // end of method GetSpGrid


//************************************************************************************
//************************************************************************************
// METHOD     : GetMortarField
// DESCRIPTION: Gets the internal mortar field variable
// ARGUMENTS  : none
// RETURNS    : GVector  *
//************************************************************************************
GVector *GMortar1D::GetMortarField()
{
  return u_mtr_;
} // end of method GetMortarField


//************************************************************************************
//************************************************************************************
// METHOD     : Mortar2Host 
// DESCRIPTION: Gets the mortar field interpolated to host grid, placing
//              results into specified vector. [Equivalent to J application.]
// ARGUMENTS  : uout   : output vector
//              indices: indices for indirection-setting of output vector
//              ni     : number of indirection indices
//              istart : where on uout mortar to start placing values
//              bAdd   : Add 'scattered' mortar data to existing field
// RETURNS    : none
//************************************************************************************
void GMortar1D::Mortar2Host(GVector *uout, GINT  *indices, GINT  ni, GINT  istart, GBOOL bAdd)
{
  if ( uout == NULL ) {
    cout << "GMortar1D::Mortar2Host(1): NULL field" << endl;
    exit(1);
  }

  Mortar2Host(uout->Data(), uout->dim(), indices, ni, istart, bAdd);

} // end of method Mortar2Host


//************************************************************************************
//************************************************************************************
// METHOD     : Mortar2Host (2)
// DESCRIPTION: Gets the mortar field interpolated to host grid, placing
//              results into specified array
// ARGUMENTS  : uout   : output vector array, Fortran-style
//              nu     : number of output array elements
//              indices: indices for indirection-setting of output vector
//              ni     : number of indirection indices
//              istart : where on uout mortar to start placing values
//              bAdd   : Add 'scattered' mortar data to existing field
// RETURNS    : none
//************************************************************************************
void GMortar1D::Mortar2Host(GDOUBLE *uout, GINT  nu, GINT  *indices, GINT  ni, 
                            GINT  istart, GBOOL bAdd)
{
  GINT  i;

  if ( !bHostGrid_ ) {
    cout << "GMortar1D::Mortar2Host(2): host grid not set" << endl;
    exit(1);
  }
  if ( bConforming_ ) {     // mortar and host are conforming
    if ( indices == NULL ) {
      if ( nu != NN_ ) {
        cout << "GMortar1D::Mortar2Host: incompatible dimensions" << endl;
        exit(1);
      }
      if ( !bAdd ) {
        memcpy(uout, u_mtr_, nu*sizeof(GDOUBLE));
      }
      else {
        for ( i=0; i<nu; i++ ) uout[i] += (*u_mtr_)[i];
      }
      return;
    }
    else {
      if ( !bAdd ) {
        for ( i=0; i<ni; i++ ) uout[indices[i]] = (*u_mtr_)(i+istart);
      }
      else {
        for ( i=0; i<ni; i++ ) uout[indices[i]] += (*u_mtr_)(i+istart);
      }
    }
    return;
  }

#if 0
//cout << "GMortar1D::Mortar2Host: IM2H_ (=J) = " << *IM2H_ << endl;
if ( !bConforming_ ) 
cout << "GMortar1D::Mortar2Host: mf         = " << *u_mtr_<< endl;
#endif
  // If not conforming, must interpolate:
  // (J_application)
  if ( indices == NULL ) {  // entire uout changed
    if ( !bAdd ) {
      MTK::afmatvec_prod(*IM2H_, u_mtr_->Data(), u_mtr_->dim(),  uout, nu);
    }
    else {
      u_hst_->Resize(nu);
      for ( i=0; i<nu; i++ ) (*u_hst_)[i] = uout[i];
      MTK::afmatvec_prod(*IM2H_, u_mtr_->Data(), u_mtr_->dim(),  u_hst_->Data(), nu);
      for ( i=0; i<nu; i++ ) uout[i] += (*u_hst_)[i];
    }
  }
  else {                    // use indirection to find output field
    u_hst_->Resize(ni);
    MTK::fmatvec_prod(*IM2H_, *u_mtr_, *u_hst_);  
//  MTK::basic_fmatvec_prod(*IM2H_, u_mtr_->Data(), u_mtr_->dim(),  u_hst_->Data(), ni);
//cout << "GMortar1D::Mortar2Host: u_mtr=" << *u_mtr_ << " u_hst=" << *u_hst_ << endl;
    if ( !bAdd ) {
      for ( i=0; i<ni; i++ ) uout[indices[i]] = (*u_hst_)(i);
    }
    else {
      for ( i=0; i<ni; i++ ) uout[indices[i]] += (*u_hst_)(i);
    }
  }

} // end of method Mortar2Host (2)


//************************************************************************************
//************************************************************************************
// METHOD     : isConforming
// DESCRIPTION: get flag treating mortar as conforming wrt host
// ARGUMENTS  : none
// RETURNS    : GBOOL 
//************************************************************************************
GBOOL &GMortar1D::isConforming()
{
  return bConforming_ ;
} // end of method isConforming


//************************************************************************************
//************************************************************************************
// METHOD     : Host2Mortar (1)
// DESCRIPTION: Sets the mortar variable from host element. 
// ARGUMENTS  :  ehost   : host element
//               uhost   : host's field quantity
//               istart  : start index where mortar is filled
//               iend    : end index where mortar is filled
// RETURNS    :
//************************************************************************************
void GMortar1D::Host2Mortar(Elem2D *ehost, GVector *uhost, GINT  &istart, GINT  &iend)
{
  GINT      i;
  GIBuffer  *iremap=NULL;

  if ( ehost == NULL || uhost == NULL ) {
    cout << "GMortar1D::Host2Mortar(1): NULL field" << endl;
    exit(1);
  }
  
  *u_mtr_ = 0.0;
  istart  = iend = 0;

  // Get points w/in mortar that are also in the element:
  if ( !ehost->Map2NewCoords(uhost, x_mtr_, GDIM, NULL, 0, u_mtr_, iremap) ) {
    cout << "GMortar1D::Host2Mortar(1): remap failed" << endl;
    exit(1);
  }
#if 0
  cout << "GMortar1D::Host2Mortar: mortar points interpolated to element: " << *iremap << endl;
  cout << "GMortar1D::Host2Mortar: mortar field : " << *u_mtr_<< endl;
#endif
  if ( iremap != NULL ) {
    for ( i=0,istart=IHUGE; i<iremap->dim(); i++ ) {
      istart = MIN(istart,(*iremap)[i]);
    }
    for ( i=0,iend=-1; i<iremap->dim(); i++ ) {
      iend = MAX(iend,(*iremap)[i]);
    }
  }

  if ( iremap ) delete iremap;

} // end of method Host2Mortar (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Host2Mortar (1)
// DESCRIPTION: Sets the mortar variable from host element.
// ARGUMENTS  : uin    : input GVector array
//              indices: indices for indirection-setting of input vector
//              ni     : number of indirection indices
// RETURNS    :
//************************************************************************************
void GMortar1D::Host2Mortar(GVector *uin, GINT  *indices, GINT  ni, GINT  istart)
{

  if ( !bHostGrid_ ) {
    cout << "GMortar1D::Host2Mortar(1): host grid not set" << endl;
    exit(1);  
  }

  Host2Mortar(uin->Data(), uin->dim(), indices, ni, istart);
} // end of method Host2Mortar (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Host2Mortar (2)
// DESCRIPTION: Sets the mortar variable from host element. [Equivalent to J^T
//              application]
// ARGUMENTS  : uin    : input vector array, Fortran-style
//              nu     : number of input array elements
//              indices: indices for indirection-setting of input vector
//              ni     : number of indirection indices
// RETURNS    :
//************************************************************************************
void GMortar1D::Host2Mortar(GDOUBLE *uin, GINT  nu, GINT  *indices, GINT  ni, GINT  istart)
{
  GINT  i;

  if ( !bHostGrid_ ) {
    cout << "GMortar1D::Host2Mortar(2): host grid not set" << endl;
    exit(1);  
  }
  if ( bConforming_ ) {     // mortar and host are conforming
    if ( indices == NULL ) {
      if ( nu != NN_ ) { 
        cout << "GMortar1D::Host2Mortar(2): incompatible dimensions" << endl;
        exit(1);
      }
       memcpy(u_mtr_, uin, nu*sizeof(GDOUBLE));
      return;
    }
    else {
      for ( i=0; i<ni; i++ ) (*u_mtr_)(i+istart) = uin[indices[i]];
    }
    return; 
  } 
  
  // If not conforming, must interpolate to compute mtr field:
  // (J^T_ application)
  if ( indices == NULL ) {  // entire uin used to compute mtr field
    MTK::afmatvec_prod(*IH2M_, uin, nu, u_mtr_->Data(), u_mtr_->dim());
  }
  else {                    // use indirection to find contiguous hst field first
    u_hst_->Resize(ni);
    for ( i=0; i<ni; i++ ) (*u_hst_)[i] = uin[indices[i]];

    MTK::fmatvec_prod(*IH2M_, *u_hst_, *u_mtr_);
#if defined(GMORT_DEBUG_OUTPUT)
cout << "GMortar1D:Host2Mortar: u_host_edge=" << *u_hst_ << endl;
cout << "GMortar1D:Host2Mortar: Jt         =" << *IH2M_   << endl;
#endif
  }
#if defined(GMORT_DEBUG_OUTPUT)
cout << "GMortar1D:Host2Mortar: u_mtr=" << *u_mtr_ << endl;
#endif
} // end of method Host2Mortar (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetBdyPoints
// DESCRIPTION: Gets endpoints defining mortar
// ARGUMENTS  : nbdy    : number of bdy points
// RETURNS    :
//************************************************************************************
Point3D *GMortar1D::GetBdyPoints(GINT  &nbdy)
{

  if ( pBdy_ == NULL ) nbdy = 0;
  nbdy = nBdy_;
  return pBdy_;

} // end of method GetBdyPoints


