//************************************************************************************//
// Module       : gpmelemid.cpp
// Date         : 6/9/05 (DLR)
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the access methods and data associated with
//                generating an element/proc mapping for grid partitioning.
//                This class derives the map by using the element ids as
//                a space-filling curve, and dividing them up among the procs.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "gpmelemid.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GPMElemID::GPMElemID(GElemList *gelems)
: GPartMapper(gelems)
{
  char     *serr = "GPMElemID::GPMElemID(1) : ";

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
GPMElemID::~GPMElemID()
{
}

//************************************************************************************
//************************************************************************************
// METHOD     : GenerateMap
// DESCRIPTION: Generates partition map.
// ARGUMENTS  : gpm       : pointer reference to stGPartitionMap structure array.
//                          This will have length nmax * nprocs. This structure  array is
//                          created here. If it is non-NULL, it is first deleted. Caller
//                          is responsible for deleting this array.
//              nmax      : max number of sends for each proc, set here.
// RETURNS    : TRUE if found and deleted; else FALSE 
//************************************************************************************
GBOOL GPMElemID::GenerateMap(stGPartitionMap *&gpm, GINT  &nmax)
{
  char            *serr = "GPMElemID::GenerateMap: ";
  GINT            ind, is, j, k, m, n, nelems, ngpm, npts, nsum, nsib, p2n;
  GKEY            kchk;
  GDOUBLE         fnorm, fsum;
  Point           *center, pavg;
  GKEYBuffer      ktmp, kcurve;
  GIBuffer        isib;
  stGPartitionMap *lpm;

  if ( gpm != NULL ) delete [] gpm; gpm = NULL;

  nelems = gelems_->size();
  GComm::Allreduce(&nelems, &nmax, 1, GC_GINT, G_OP_MAX);
//GComm::Allreduce(&nelems, &nsum, 1, GC_GINT, G_OP_SUM);
  ngpm = nmax*nprocs_;

  is    = this_rank_ * nmax;
  lpm   = new stGPartitionMap [nmax];
  gpm   = new stGPartitionMap [nmax*nprocs_];
  gelems_->start(NULL);
  for ( j=0; j<nelems; j++ ) {
    center = gelems_->member()->GetElemCenter();
    lpm[j].elem_key   = gelems_->member()->GetID(); 
    lpm[j].parent_key = gelems_->member()->GetParentID(); 
    lpm[j].proc_to    = PMDEFAULT_PROC;
    lpm[j].bcoarsened = kcoarsen_!=NULL && kcoarsen_->contains(lpm[j].elem_key,ind) ? 1 : 0;
    for ( k=0; k<GDIM; k++ ) lpm[j].center[k] = (GFLOAT)(*center)[k]; 
    gelems_->next();
  }

  // Set all values beyond the local nelems to defaults. IMPORTANT!!:
  for ( j=nelems; j<nmax; j++ ) {
    lpm[j].elem_key    = PMDEFAULT_KEY;
    lpm[j].parent_key  = PMDEFAULT_KEY;
    lpm[j].proc_to     = PMDEFAULT_PROC;
    lpm[j].bcoarsened  = 0;
    for ( k=0; k<GDIM; k++ ) lpm[j].center[k] = 0.0;
  }
  GComm::Allgather(lpm, nmax, pm_type_, gpm, nmax, pm_type_);
  delete [] lpm;

  // Compute total no. of new elems, after coarsening is done
  // (remember that siblings will be coarsened into a single element):
  for ( j=0,fsum=0; j<ngpm; j++ ) {
    if ( gpm[j].elem_key == PMDEFAULT_KEY ) continue;
    fnorm = gpm[j].bcoarsened ? (GDOUBLE) GetSibCount(isib,gpm[j],gpm,ngpm) : 1.0;
    fsum += (fnorm == 0.0 ? 1.0 : 1.0/fnorm); // don't like infinities....
  }

  // Compute the buffer of element keys for all elements after coarsening (if done).
  // The sorting of this buffer provides the order in which the elements will be
  // divided among processors; hence, it's related to the final space-filling curve:
  nsum = (GINT) fsum;
  kcurve.Resize(nsum); 
  kcurve = -1;
  for ( j=0,n=0; j<ngpm; j++ ) {
    if ( gpm[j].elem_key == PMDEFAULT_KEY ) continue;
    kchk = gpm[j].bcoarsened ? gpm[j].parent_key : gpm[j].elem_key;
    if ( !kcurve.contains(kchk,ind) ) kcurve[n++] = kchk;
  }

  // Sort indices from lowest to highest
  kcurve.sortincreasing();

  // Partition sorted list:
  p2n = nprocs_ / ngpm;
  for ( j=0,n=0; j<ngpm; j++ ) {
    if ( gpm[j].elem_key == PMDEFAULT_KEY ) continue;
    kchk = gpm[j].bcoarsened ? gpm[j].parent_key : gpm[j].elem_key;
    if ( kcurve.contains(kchk,ind) ) gpm[j].proc_to = ind*p2n;
  }

  if ( !bDoCurve_ ) return TRUE;

#if 1
  gnelems_ = nsum;
  if ( curve_ != NULL ) delete [] curve_;
  curve_ = new GDOUBLE [GDIM*gnelems_];
  gelems_->start(NULL);
  ktmp.Resize(ngpm);
  ktmp = -1;
  for ( j=0,n=0,npts=0; j<ngpm; j++ ) {
    if ( gpm[j].elem_key == PMDEFAULT_KEY ) continue;
    for ( k=0; k<GDIM; k++ ) curve_[npts+k] = gpm[j].center[k];   // set default curve point

    // If elem's a sibling to be coarsened that has not been accounted for, 
    // then find the parent's center, and add all siblings to 'accounted for' list:
    if ( gpm[j].bcoarsened && !ktmp.contains(gpm[j].elem_key,ind) ) { 
      nsib = GetSibCount(isib,gpm[j],gpm,ngpm);    
      pavg = 0.0;
      for ( m=0; m<nsib; m++ ) {
        for ( k=0; k<GDIM; k++ ) pavg[k] += gpm[isib[m]].center[k];
        ktmp[n++] =  gpm[isib[m]].elem_key;                       // update 'acctd for' list for all sibs
      }
      for ( k=0; k<GDIM; k++ ) curve_[npts+k] = pavg[k]/nsib;     // avg all sib centers to find parent center
    }
    npts++;
  }
#endif


  return TRUE;

} // end of method GenerateMap


//************************************************************************************
//************************************************************************************
// METHOD     : GetSibCount
// DESCRIPTION: computes siblling count for specified element. If there are no 
//              siblings then count is 0. The returned buffer, isib, is sent back with
//              the indices into the stGPartitionMap structure array that point to the
//              siblings of specified element key (includes this key).
// ARGUMENTS  : 
//              isib      : buffer of indices pointing to array elements in gpm that 
//                          correspond to siblilngs of element key specified. 
//              epm       : stPartitionMap structure for element whose siblings we want
//              gpm       : pointer reference to stGPartitionMap structure array.
//                          This will have length nmax * nprocs. This structure  array is
//                          created here. If it is non-NULL, it is first deleted. Caller
//                          is responsible for deleting this array.
//              ngpm      : number of elements in gpm array.
// RETURNS    : number of siblings of element specified by key arg.
//************************************************************************************
GINT GPMElemID::GetSibCount(GIBuffer &isib, stGPartitionMap &epm, stGPartitionMap *&gpm, GINT  &ngpm)
{   
  GINT j, n, nsibs=0;

  if ( epm.elem_key == PMDEFAULT_KEY ) return 0;

  for ( j=0; j<ngpm; j++ ) {
    if ( gpm[j].elem_key == PMDEFAULT_KEY 
      || epm.parent_key  != gpm[j].parent_key 
      || epm.elem_key    == gpm[j].elem_key  ) continue;
    nsibs++;
  }

  isib.Resize(nsibs);
  for ( j=0, n=0; j<ngpm && nsibs; j++ ) {
    if ( gpm[j].elem_key == PMDEFAULT_KEY 
      || epm.parent_key  != gpm[j].parent_key 
      || epm.elem_key    == gpm[j].elem_key  ) continue;
    isib[n++] = j;
  }
  return nsibs;

} // end of method GetSibCount
