//************************************************************************************//
// Module       : gpmsibpart.hpp
// Date         : 7/5/05 (DLR)
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the access methods and data associated with
//                generating an element/proc mapping for sending sibling elements
//                to the proc where the parent will live when the siblings are
//                coarsened. This is done simply by trying to minimize the 
//                amount of communication for any one group of siblings; this
//                is not intended to act as a global load balancing partition mapper.
//                Note that here, the SetCoarsenables method _must_ be called with a 
//                non-NULL argument, and it is assumed that these coarsenable elements
//                represent only sibling elements that must be coarsened.
// Derived From : GPartMapper.
// Modifications:
//************************************************************************************//
#include "gpmsibpart.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GPMSibPart::GPMSibPart(GElemList *gelems)
: GPartMapper(gelems)
{
  char     *serr = "GPMSibPart::GPMSibPart(1) : ";

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
GPMSibPart::~GPMSibPart()
{
}

//************************************************************************************
//************************************************************************************
// METHOD     : GenerateMap
// DESCRIPTION: Generates partition map.
// ARGUMENTS  : gpm       : pointer reference to stGPartitionMap structure array.
//                          This will have length nmax * nprocs. This structure  array is
//                          created here. If it is non-NULL, it is first deleted. Caller
//                          is responsible for deleting this array.
//              nmax      : max number of sends for each proc, set here.
// RETURNS    : TRUE if found and deleted; else FALSE 
//************************************************************************************
GBOOL GPMSibPart::GenerateMap(stGPartitionMap *&gpm, GINT  &nmax)
{
  char            *serr = "GPMSibPart::GenerateMap: ";
  GINT            i, ind, is, j, k, maxsibs, minelem, ncoarsen, ngpm, ntotal;
  GSHORT          pmax, pmin, pto;
  GKEY            pkey;
  Point           *center;
  GIBuffer        neperproc, ne, ns;
  stGPartitionMap *lpm;
  Elem2D          *elem;

  if ( kcoarsen_ == NULL ) {
    cout << serr << "coarsenables list required for this object" << endl;
    return FALSE;
  }
  if ( gpm != NULL ) delete [] gpm; gpm = NULL;

  ncoarsen = kcoarsen_->dim();
  GComm::Allreduce(&ncoarsen, &nmax, 1, GC_GINT, G_OP_MAX);
  neperproc.Resize(nprocs_);
  ntotal = gelems_->size();
  GComm::Allgather(&ntotal, 1, GC_GINT, neperproc.Data(), 1, GC_GINT);
  ngpm = nmax*nprocs_;

  is    = this_rank_ * nmax;
  lpm   = new stGPartitionMap [nmax];
  gpm   = new stGPartitionMap [ngpm];
  for ( j=0; j<ncoarsen; j++ ) {
    elem   = gelems_->member((*kcoarsen_)[j]);
    center = elem->GetElemCenter();
    lpm[j].elem_key   = elem->GetID(); 
    lpm[j].parent_key = elem->GetParentID(); 
    lpm[j].proc_to    = this_rank_;         // by default, it stays on-proc
    lpm[j].bcoarsened = 1;
    for ( k=0; k<GDIM; k++ ) lpm[j].center[k] = (GFLOAT)(*center)[k]; 
  }

  // Set all values beyond the local ncoarsen to defaults. IMPORTANT!!:
  for ( j=ncoarsen; j<nmax; j++ ) {
    lpm[j].elem_key    = PMDEFAULT_KEY;
    lpm[j].parent_key  = PMDEFAULT_KEY;
    lpm[j].proc_to     = PMDEFAULT_PROC;
    lpm[j].bcoarsened  = 1;
    for ( k=0; k<GDIM; k++ ) lpm[j].center[k] = 0.0;
  }

  // Get global partition map:
  GComm::Allgather(lpm, nmax, pm_type_, gpm, nmax, pm_type_);
  delete [] lpm;

  // Find the processor distribution for each sibling by trying to minimize
  // the amount of communication for each sibling group.
  // Loop over each local sibling to be coarsened, and (1) find proc id where
  // all siblings live; (2) find current number elements on that proc;
  // (3) send sibling to proc that (i) has the most number of siblings or 
  // (ii) otherwise has the fewest number of total elements.
  ne.Resize(nprocs_); 
  ns.Resize(nprocs_); 
  for ( i=0; i<kcoarsen_->dim(); i++ ) {
    ns = 0;
    ne = 0;
    pkey = gelems_->member((*kcoarsen_)[i])->GetParentID();
    for ( j=0; j<nprocs_; j++ ) {
      // Check proc id j and find number of siblings on that proc:
      for ( k=j; k<nmax && gpm[k].elem_key != PMDEFAULT_KEY; k++ ) {
//   cout << serr << "lpm[" << k << "]= " << lpm[k] << endl;
//   cout << serr << "ppm[" << k << "]= " << lpm[k] << endl;
         if ( gpm[k].parent_key == pkey ) { // sibling found on this proc (j)
           ns[j]++;    
           ne[j] = neperproc[j];
         }
      }
    }  // end, proc id loop
    // Find proc that has the max no. of siblings, and the max. no. sibs:
    maxsibs  = 0;
    minelem  = ngpm + 1;
    pmax     = this_rank_;
    pmin     = this_rank_;
    for ( j=0; j<nprocs_; j++ ) {
      if ( ns[j] > maxsibs ) { maxsibs = ns[j]; pmax=j; }
      if ( ne[j] < minelem ) { minelem = ne[j]; pmin=j; }
    }
    pto = maxsibs <= 1 ? pmax : pmin;
    // Set 'send-to proc' for all siblings in partition map:
    for ( j=0; j<ngpm; j++ ) {
      gpm[j].proc_to = gpm[j].parent_key == pkey ? pto : this_rank_;
    }
  }  // end, kcoarsen loop

  return TRUE;

} // end of method GenerateMap


