//************************************************************************************//
// Module       : mesh_reader.cpp
// Date         : 8/17/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the object that reads a mesh file
// Derived From : none.
// Modifications:
//************************************************************************************//
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#if defined(_LINUX)  || defined(__LINUX)
#include <string.h>
#endif
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include "mesh_reader.hpp"
#include "gdd_file.h"

char *serror_[] = {
                  "Normal",
                  "End of file",
                  "Invalid file name",
                  "Invalid file header",
                  "Invalid file type",
                  "Corrupt file/Bad data format",
                  "Bad data dimensionality",
                  "No data",
                  "No file",
                  "Miscellaneous error",
                  "Memory allocation error",
                  "Data not found",
                  "Invalid data item"
                  "Relevant data retrieved"
                   };

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
MeshReader::MeshReader(GLONG maxbuff)
:
ftype_        (FT_BYPROC),
buffsize_     (maxbuff),
bnodes_       (FALSE),
bGetNodeIDs_  (TRUE),
blocated_     (FALSE),
ierror_       (ERRNONE),
procid_       (0),
nd_           (2),
np_           (0),
ne_           (0),
ngverts_      (0),
nrng_         (-1),
ientry_       (0),
elemperproc_  (NULL),
gverts_       (NULL),
buff_         (NULL),
filename_     (NULL),
os_           (NULL)
{
  filename_ = new char [FILE_NAME_MAX];
  buff_     = new char [buffsize_];
  strcpy(delimelem_[0], "(");
  strcpy(delimelem_[1], ")");
  strcpy(delimproc_[0], "{");
  strcpy(delimproc_[1], "}");
  strcpy(delimeol_    , ";");
  strcpy(delimlist_   , ":");

} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Destructor
MeshReader::~MeshReader()
{
  if ( buff_        != NULL ) delete [] buff_       ;  buff_        = NULL;
  if ( elemperproc_ != NULL ) delete [] elemperproc_;  elemperproc_ = NULL;
  if ( gverts_      != NULL ) delete [] gverts_     ;  gverts_      = NULL;
  if ( os_          != NULL ) delete os_            ;  os_          = NULL;
  if ( filename_    != NULL ) delete [] filename_   ;  filename_    = NULL;
}


//************************************************************************************
// METHOD     : SetProc
// DESCRIPTION: Sets processor id
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void MeshReader::SetProc(GSHORT  id)
{
  procid_ = id;
} // end of method SetProc


//************************************************************************************
// METHOD     : bGetNodeIDs
// DESCRIPTION: Sets flag indicating whether or not to retrieve
//              element node ids.
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void MeshReader::bGetNodeIDs(GBOOL bid)
{
  bGetNodeIDs_ = bid;
} // end of method bGetNodeIDs


//************************************************************************************
// METHOD     : Error
// DESCRIPTION: Gets error string
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
const char *MeshReader::Error()
{
  return serror_[ierror_];
 
} // end of method Error


//************************************************************************************
// METHOD     : ErrorID
// DESCRIPTION: Gets error ID
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
GSHORT  MeshReader::ErrorID()
{
  return ierror_;

} // end of method ErrorID

//************************************************************************************
// METHOD     : ReadHeader
// DESCRIPTION: Reads header info from open stream, and 
//              fills in appropriate structures.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else, FALSE
//************************************************************************************
GBOOL MeshReader::ReadHeader()
{
  GLONG    i, j, n;
  GLONG    h[6];
  GDOUBLE  *fv;
  char   delim[DELIM_MAX+2];

  if ( os_ == NULL ) return FALSE;

  // Determine file type:
  sprintf(delim,"%s%c", delimeol_, '\n');
  getline(buff_, buffsize_, delim);
  n = getdata_t(buff_, G_GLONG , " ", delimeol_, h, 6);
  if ( n != 6 )
  {
    for ( i=0; i<6; i++ ) cout << " h[" << i << "]=" << h[i] << endl;
    ierror_ = ERRHEADER;
    return FALSE;
  }
  if ( h[0] != FT_BYPROC && h[0] != FT_BYELEM )
  {
    ierror_ = ERRFILETYPE;
    return FALSE;
  }

  // Get global domain coords:
  getline(buff_, buffsize_, delim);
  n = getdata_t(buff_, G_GSHORT , " ", delimlist_, &ngverts_, 1);
  if ( n != 1 )
  {
    ierror_ = ERRHEADER;
    return FALSE;
  }
  if ( ngverts_ <= 0 )
  {
    ierror_ = ERRHEADER;
    return FALSE;
  }
  fv = new GDOUBLE [3*ngverts_];
  gverts_ = new Point [ngverts_];
  getline(buff_, buffsize_, delim);
  n = getdata_t(buff_, G_GDOUBLE, " ", delimeol_, fv, 3*ngverts_);
  for ( i=0; i<ngverts_; i++ ) {
    for ( j=0; j<3; j++ ) gverts_[i][j] = fv[3*i + j];
  }
  delete [] fv;

  // Set file types, etc.  from header data:
  ftype_  = (GDD_FILE_TYPE) h[0];   // file type
  bnodes_ = (GBOOL)         h[1];   // is there nodal data expected?
  nd_     =                 h[2];   // data dimension
  np_     =                 h[3];   // no. of procs represented in file
  ne_     =                 h[4];   // no. of elements represented in file
  nrng_   =                 h[5];   // node id dynamic range

  if ( nd_ < 1 || nd_ > 3 )
  {
    ierror_ = ERRDIMENSION;
    return FALSE;
  }

  if ( np_ <= 0 || ne_ <= 0 )
  {
    ierror_ = ERRBADDATA;
    return FALSE;
  }
  elemperproc_ = new GSHORT  [np_];  // no. of elems per proc.
  if ( elemperproc_ == NULL )
  {
    ierror_ = ERRMALLOC;
    return FALSE;
  }
  getline(buff_, buffsize_, delim);
  n = getdata_t(buff_, G_GSHORT , " ", delimeol_, elemperproc_, np_);
  if ( n != np_ )
  {
    ierror_ = ERRHEADER;
    return FALSE;
  }

  return TRUE;

} // end of method ReadHeader


//************************************************************************************
// METHOD     : Open
// DESCRIPTION: Opens file; checks header for validity
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else, FALSE
//************************************************************************************
GBOOL MeshReader::Open(const char *fn)
{
  if ( fn == NULL ) 
  {
    ierror_ = ERRFILENAME;
    return FALSE;
  }

  if ( os_ != NULL ) delete os_; os_ = NULL;
  strncpy(filename_,fn,FILE_NAME_MAX);

  os_ = new ifstream(filename_,ios::in);
  if ( os_ == NULL || os_->fail() ) 
  {
    ierror_ = ERRFILENAME;
    return FALSE;
  }

  // Read header, and do some checking (remembering,
  // of course, that this is an ASCII file)

  return ReadHeader();
 
} // end of method Open


//************************************************************************************
// METHOD     : Close
// DESCRIPTION: Closes stream.
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void MeshReader::Close()
{
  if ( os_ != NULL )
  {
    os_->close();
  }
 
} // end of method Close

//************************************************************************************
// METHOD     : GetGlobalVerts
// DESCRIPTION: Gets global vertices obtained from mesh file header
// ARGUMENTS  : verts : array reference to point array. Will be allocated here;
//              reallocated if non-NULL on entry.
//              nverts: number of vertices defining global domain. 
// RETURNS    : 0 onsuccess; -1 on error
//************************************************************************************
GSHORT  MeshReader::GetGlobalVerts(Point *&verts, GINT  &nverts)
{
  GINT  i;

  if ( os_ == NULL || gverts_ == NULL )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }
  if ( verts != NULL ) delete [] verts;
  nverts = ngverts_;
  verts  = new Point [nverts];
  for ( i=0; i<nverts; i++ ) verts[i] = gverts_[i];
  
  return 0;

} // end of method GetGlobalVerts




//************************************************************************************
// METHOD     : GetNumElements
// DESCRIPTION: Gets number of elements corresponding to this proc ID.
//              This is based on header info only; no checking is 
//              done to make sure the information is accurate.
// ARGUMENTS  : 
// RETURNS    : number of elements for iproc_ contained in mesh file; -1 on error
//************************************************************************************
GSHORT  MeshReader::GetNumElements()
{
  if ( os_ == NULL || elemperproc_ == NULL ) 
  {
    ierror_ = ERRNOFILE;
    return -1;
  }
  if ( procid_ >= np_ ) return 0;

  return elemperproc_[procid_];

} // end of method GetNumElements


//************************************************************************************
// METHOD     : GetNumRetrieved
// DESCRIPTION: Gets number of elements retrieved successfully
// ARGUMENTS  : 
// RETURNS    : number of elements retrieved for iproc_ 
//************************************************************************************
GSHORT  MeshReader::GetNumRetrieved()
{
  if ( os_ == NULL || elemperproc_ == NULL )  
  {
    ierror_ = ERRNOFILE;
    return -1;
  }

  return ientry_;

} // end of method GetNumRetrieved


//************************************************************************************
// METHOD     : GetNumProcs
// DESCRIPTION: Gets number of procs represented in file.
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GSHORT  MeshReader::GetNumProcs()
{
  if ( os_ == NULL || elemperproc_ == NULL )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }

  return np_;

} // end of method GetNumProcs


//************************************************************************************
// METHOD     : GetTotNumElements
// DESCRIPTION: Gets total number of elements represented in file.
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
GSHORT  MeshReader::GetTotNumElements()
{
  if ( os_ == NULL || elemperproc_ == NULL )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }

  return ne_;

} // end of method GetTotNumElements


//************************************************************************************
// METHOD     : GetDynRange
// DESCRIPTION: Gets dynam. range of node ids
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GLONG MeshReader::GetDynRange()
{
  if ( os_ == NULL || elemperproc_ == NULL )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }

  return nrng_;

} // end of method GetTotNumElements


//************************************************************************************
// METHOD     : GetDim
// DESCRIPTION: Gets dataset dimension
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GSHORT  MeshReader::GetDim()
{
  if ( os_ == NULL || elemperproc_ == NULL )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }

  return nd_;

} // end of method GetDim


//************************************************************************************
// METHOD     : GetElem
// DESCRIPTION: Gets element data from file. The elements correspond to the
//              set proc id, and repeated calls get subsequent elements in the
//              file. Args are allocated here; callers are responsible for 
//              deletion. 
// ARGUMENTS  :
// RETURNS    : 0 on success; else > 0. ErrorID will be set.
//************************************************************************************
GSHORT  MeshReader::GetElem(ELEMTYPE &etype, Point3D *&vertex, GSHORT  *&bdyn, GLONG  *&xN, 
                           GLBuffer  *&node_ids, GLBuffer  *&bdy_nodes, GBTBuffer  *&bct )
{
  GSHORT  iproc, ielem, iret=0;

  if ( os_ == NULL )
  {
    ierror_ = ERRNOFILE;
    return ierror_;
  }

  if ( procid_ >= np_ || ientry_ >= elemperproc_[procid_] ) return ierror_;

  if ( ftype_ == FT_BYPROC )
  {
    if ( !blocated_ && !LocateProcID() )
    {
      cout << "MeshReader::GetElem: cannot locate proc ID " << procid_ << " in file" << endl; 
      return (ierror_=ERRDATANOTFND);
    }
    if ( ientry_ == 0 )   
      ignore(delimproc_[0]);
    if ( GetElemData(ielem, iproc, etype, vertex, bdyn, xN, node_ids, bdy_nodes, bct, TRUE) > 0 ) 
      return ierror_;
    ignore(delimelem_[1]);
  } 
  else if ( ftype_ == FT_BYELEM )
  {
    while ( (iret=GetElemData(ielem, iproc, etype, vertex, bdyn, xN, node_ids, bdy_nodes, bct, FALSE)) == 0 &&
            iproc != procid_ )
    {
      if ( iret == ERREOF ) ierror_ = ERRDATANOTFND;
      if ( iret > 0 ) return ierror_;
    }
  }
  ientry_++;
  return ierror_;

} // end of method GetElem


//************************************************************************************
// METHOD     : GetElemData
// DESCRIPTION: Gets element data from file. Args are allocated here; callers
//              are responsible for deletion. Args are allocated automatically
//              and filled if bautofill=TRUE; else they are allocated and filled,
//              only if the element's procid is the same as procid_.
//           
//               
// ARGUMENTS  :
// RETURNS    : 0 on success; else > 0. ErrorID will be set.
//************************************************************************************
GSHORT  MeshReader::GetElemData(GSHORT  &elemid, GSHORT  &procid, ELEMTYPE &etype, 
                               Point3D *&vertex, GSHORT  *&bdyn, GLONG  *&xN, GLBuffer  *&node_ids, 
                               GLBuffer  *&bdy_nodes, GBTBuffer  *&bct, const GBOOL bautofill)
{
  GINT      i, n;
  GINT      nbn, nbc, nno, nv;
  ELEMTYPE  id[3];
  GBOOL     doFill=FALSE;
  GDOUBLE   *points;

  if ( os_ == NULL )
  {
    ierror_ = ERRNOFILE;
    return ierror_;
  }

  ierror_ = ERRCORRUPT;
  // Get element header:
  if ( getline(buff_, buffsize_, delimlist_) < 0 ) return FALSE;

  if ( ftype_ == FT_BYPROC )        // expect 2 id members
    n = 2;
  else if ( ftype_ == FT_BYELEM )  // expect 3 id members
    n = 3;

  // If BYPROC, header line is [Elem ID, Elem type]
  // If BYELEM, header line is [Elem_ID, Elem_type, Proc_ID]
  if ( getdata_t(buff_, G_ELEMTYPE, " ", delimlist_, id, n) != n )
    return ierror_;

//if ( !getline(buff_, buffsize_, delimelem_[0]) ) return ierror_;
  ignore(delimelem_[0]);

  // Delete arguments, if necessary:
  if ( vertex    != NULL ) delete [] vertex;   vertex    = NULL;
  if ( bdyn      != NULL ) delete [] bdyn  ;   bdyn      = NULL;
  if ( xN        != NULL ) delete [] xN;       xN        = NULL;
  if ( node_ids  == NULL ) node_ids  = new GLBuffer ; 
  if ( bdy_nodes == NULL ) bdy_nodes = new GLBuffer ;  
  if ( bct       == NULL ) bct       = new GBTBuffer ;    
  node_ids ->Resize(0);
  bdy_nodes->Resize(0);
  bct      ->Resize(0);

  elemid = (GSHORT )id[0];
  etype  = (ELEMTYPE)id[1];
  procid = (GSHORT )-1;
  if ( ftype_ == FT_BYELEM ) procid = id[2];

  doFill = ( bautofill ? TRUE : (procid == procid_) ); 

  // Get element data block:
  // ...Element vertices:
  if ( getline(buff_, buffsize_, delimeol_) < 0 )  return ierror_;
  if ( doFill )
  {
    nv = (GINT)pow(2.0,nd_);
    vertex = new Point3D [nv];
    points = new GDOUBLE [nv*nd_];
    if ( getdata_t(buff_, G_GDOUBLE, " ", NULL, points, nv*nd_) != nv*nd_ )
    {
      delete [] points;
      return ierror_;
    }
    if ( nd_ == 1 )
    {
      for ( i=0; i<nv; i++ )
        vertex[i].x1 = points[i];
    }
    else if ( nd_ == 2 )
    {
      for ( i=0; i<nv; i++ )
      {
        vertex[i].x1 = points  [2*i];
        vertex[i].x2 = points[2*i+1];
      }
    }
    else if ( nd_ == 3 )
    {
      for ( i=0; i<nv; i++ )
      {
        vertex[i].x1 = points  [3*i];
        vertex[i].x2 = points[3*i+1];
        vertex[i].x3 = points[3*i+2];
      }
    }
    delete [] points;
  }

  // ... global bdy indicators:
  if ( getline(buff_, buffsize_, delimeol_) < 0 )  return ierror_;
  if ( doFill )
  {
    bdyn = new GSHORT [2*nd_];
    if ( getdata_t(buff_, G_GSHORT, " ", delimeol_, bdyn, 2*nd_) != 2*nd_ ) return ierror_;
  }

  // ... get expandsion/bdy info:

  // ...expansion order: 
  if ( getline(buff_, buffsize_, delimlist_) < 0 ) return ierror_;
  if ( doFill )
  {
    xN = new GLONG [nd_];
    if ( getdata_t(buff_, G_GLONG , " ", delimlist_, xN, nd_) != nd_ ) return ierror_;
  }


  // If there are node ids, get them:
  if ( bnodes_ ) {
    // ...node ids:
    if ( getline(buff_, buffsize_, delimeol_) < 0 ) return ierror_;
    if ( doFill && bGetNodeIDs_ )
    {
      nno = 1;
      for ( i=0; i<nd_; i++ ) nno *= xN[i];
        node_ids = new GLBuffer (nno);

      if ( getdata_t(buff_, G_GLONG , " ", delimeol_, node_ids->Data(), nno) != nno ) return ierror_;
    }
  }

  // ...no. bdy nodes:
  if ( getline(buff_, buffsize_, delimlist_) < 0 ) return ierror_;
  if ( doFill &&
       getdata_t(buff_, G_GINT , " ", delimeol_, &nbn, 1) != 1) return ierror_;
 
  if ( nbn > 0 )
  {
    // ...bdy nodes ids:
    if ( getline(buff_, buffsize_, delimeol_) < 0 ) return ierror_;
    if ( doFill )
    {
      bdy_nodes ->Resize(nbn); 
      if ( getdata_t(buff_, G_GLONG , " ", delimeol_, bdy_nodes->Data(), nbn) != nbn ) return ierror_;
    }
  
    // ...no. bdy cond nodes:
    if ( getline(buff_, buffsize_, delimlist_) < 0 ) return ierror_;
    if ( doFill && 
         getdata_t(buff_, G_GINT , " ", delimeol_, &nbc, 1) != 1) return ierror_;
 
    if ( nbn != nbc ) return ierror_;
   
    // ... bdy conds :
      if ( getline(buff_, buffsize_, delimeol_) < 0 ) return ierror_;
    if ( doFill )
    {
      bct->Resize(nbc); 
      if ( getdata_t(buff_, G_BDYTYPE, " ", delimeol_, bct->Data(), nbc) != nbc) return ierror_;
    }
  }

  
  return (ierror_=ERRNONE);

} // end of method GetElemData

//************************************************************************************
// METHOD     : LocateProcID
// DESCRIPTION: Locates start of block with index procid_ for
//              cases where ftype_ = FT_BYPROC. 
//    
//    
// ARGUMENTS  :
// RETURNS    : 0 on success; else > 0. ErrorID will be set.
//************************************************************************************
GBOOL MeshReader::LocateProcID()
{
  GSHORT  pid;
  GBOOL  bFound=FALSE;

  if ( os_ == NULL )
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }

  if ( ftype_ != FT_BYPROC ) return FALSE;

  while ( !bFound )
  {
//  ignore(delimproc_[1]);
    if ( getline(buff_, buffsize_, delimlist_) < 0 ) return FALSE;
    if ( getdata_t(buff_, G_GSHORT , delimeol_, delimeol_, &pid, 1) != 1) return FALSE;
    if ( pid == procid_ ) {
       bFound = TRUE;
    }
    else { 
      ignore(delimproc_[1]);
//    if ( !skipblock(delimproc_[0],delimproc_[1]) ) return FALSE;
    }
  }

  return (blocated_=TRUE);

} // end of method LocateProcID


//************************************************************************************
// METHOD     : getdata_t
// DESCRIPTION: Gets ndata elements of data of type dtype from input string.
//              Method skips over all initial white space.
// ARGUMENTS  : 
// RETURNS    : number of data items stored in data array. -1 on failure.
//************************************************************************************
GLONG  MeshReader::getdata_t(char *sdata, G_DATATYPE dtype, const char *datadelim,
                             const char *termdelim,  void *data, const GLONG  ndata)
{ 
  GINT     i, tsize=G_TYPESZ[dtype];
  GSHORT   is;
  GLONG    n=0, l;
  GFPOS    gf;
  BDYTYPE  bt;
  ELEMTYPE et;
  GFLOAT   sf;  
  GQUAD    q; 
  GDOUBLE  df; 
  char     *p, *beg, delim[2*DELIM_MAX+1];

  if ( data  == NULL ) return -1;
  if ( sdata == NULL ) return  0;
  if ( datadelim == NULL ) return  0;
  if ( termdelim != NULL )
    sprintf(delim,"%s%s", datadelim, termdelim);
  else
    strcpy(delim, datadelim);

  // skip all non-numeric data:
  beg = sdata;
  i=0;
  while ( i<strlen(sdata) && !(isdigit(*beg) || *beg=='-') ) { i++; beg++; }
  
  p = strtok(beg,delim);
  while ( p != NULL && n < ndata )
  {
    if ( !isdigit(*p) && *p!='-' ) 
    {
      n = -1;
      break;
    }
    memset(((GBYTE*)data+n*tsize), 0, tsize);
    switch ( dtype )
    {
      case G_GDOUBLE:
        df = atof(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &df, tsize); 
        break;
      case G_GQUAD:
        q = atof(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &q, tsize); 
        break;
      case G_GSHORT:
        is = (GSHORT)atoi(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &is, tsize); 
        break;
      case G_GINT :
        i = (GINT)atoi(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &i, tsize); 
        break;
      case G_GLONG:
        l = (GLONG)atoi(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &l, tsize); 
        break;
      case G_GFLOAT:
        sf = (GFLOAT)atof(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &sf, tsize); 
        break;
      case G_BDYTYPE:
        bt = (BDYTYPE)atoi(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &bt, tsize); 
        break;
      case G_ELEMTYPE:
        et = (ELEMTYPE)atoi(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &et, tsize); 
        break;
      case G_GFPOS:
        gf = (GFPOS)atoi(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &gf, tsize); 
        break;
      default:
        i = atoi(p);
        memcpy((GBYTE*)data+n*tsize, (GBYTE*) &i, tsize); 
    }

    n++;
    p = strtok(NULL,delim);
  }

  return n;

} // end of method getdata_t


//************************************************************************************
// METHOD     : getline
// DESCRIPTION: Gets valid line. Reads os_, filling sbuff, until either
//              num characters have been read, or until a delimiter (in delim)
//              has been reached. 
// ARGUMENTS  :
// RETURNS    : number of characters filling sbuff, if any. Returns -1 on error.
//************************************************************************************
GLONG MeshReader::getline(char *sbuff, const GLONG num, const char *idelim)
{
  GLONG    n=0;
  char     delim[DELIM_MAX+1], c;

  if ( os_ == NULL ) return -1;

  if ( sbuff  == NULL ) return  0;
  if ( idelim == NULL ){
     strcpy(delim,"\n\0"); 
  }
  else
  {
     if ( strlen(idelim) > DELIM_MAX )
       cout << "MeshReader::getline: Warning: input delimiter size too large" << endl;
     strncpy(delim,idelim,DELIM_MAX); 
  }
   
  if ( os_->fail() ) return 0;

 
//ndelim = strlen(delim);
  os_->get(c); 
  while ( !os_->eof() && n < num-1 && strchr(delim,c) ) 
  {
    n++;
    os_->get(c); 
  }

  n = 0;
  while ( !os_->eof() && n < num-1 && !strchr(delim,c) ) 
  {
    buff_[n] = c;
    n++;
    os_->get(c);
  }
  
  if ( n < num-1 && !strchr(delim,c) )  // delimter not found
    return ( n = -1);
  
  sbuff[n] = '\0';
  

  return n;

} // end of method getline


//************************************************************************************
// METHOD     : skipbock
// DESCRIPTION: Skips a data block whose delimeters are d0 and d1. Next
//              getline will start with line/data following block. 
//              Data block need not begin with *d0.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL MeshReader::skipblock(const char *d0, const char *d1)
{

  if ( os_ == NULL )
  {
     ierror_ = ERRNOFILE;
     return FALSE;
  }
  
  if ( getline(buff_, buffsize_, d0) == -1 ||
       getline(buff_, buffsize_, d1) == -1 )
  {
     ierror_ = ERREOF;
     return FALSE;
  }

  return TRUE;

} // end of method skipblock


//************************************************************************************
// METHOD     : skiptoblock
// DESCRIPTION: Skips to a data block whose delimeter is d0. Next
//              getline will start with line/data preceding block, which
//              is the block header, or at the current position, if the 
//              stream pointer is within the header.
// ARGUMENTS  :  bdelim  : start-delimiter of block
//               hdelim  : block header's delimiter
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL MeshReader::skiptoblock(const char *bdelim, const char *hdelim)
{
#if 0
  GSHORT  i
  GBOOL bFound=FALSE, bret=TRUE;
  char  *hterm;
  SIZEI pcurr, p1, p0;

  if ( os_ == NULL )
  {
     ierror_ = ERRNOFILE;
     return FALSE;
  }
  p1 = p0 = os_->tellg();
  while ( !strchr(hdelim,os_->peek()) && !os_->eof() )
     os_->seekg(++p1); 
  if ( os_->eof() )  
  {
    os_->seekg(p0);
    return FALSE ;
  }

  while ( strchr(hdelim,os_->peek()) && !os_->eof() )
     os_->seekg(++p1);

  if ( os_->eof() )  
  {
    os_->seekg(p0);
    return FALSE ;
  }
  
#endif

  return FALSE;

} // end of method skiptoblock


//************************************************************************************
// METHOD     : getblock
// DESCRIPTION: Gets entire string block enclosed by delimeters, up to
//              a maximum number of chars specified by num; place
//              result in sbuff.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL MeshReader::getblock(char *sbuff, GLONG num, char *d0, char *d1)
{

  if ( os_ == NULL )
  {
     ierror_ = ERRNOFILE;
     return FALSE;
  }
  
  if ( getline(sbuff, num, d0) == -1 ||
       getline(sbuff, num, d1) == -1 )
  {
     ierror_ = ERREOF;
     return FALSE;
  }

  return TRUE;

} // end of method getblock


//************************************************************************************
// METHOD     : ignore
// DESCRIPTION: Ignores everything from current position to
//              appearance of first char among delim. If delim
//              is not found, FALSE is returned, and file pointer
//              is unchanged. If delim is found, then file pointer
//              is set to first position beyond the delimiter. 
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL MeshReader::ignore(const char *delim)
{
  GLONG  p0, p1;

  if ( os_ == NULL )
  {
     ierror_ = ERRNOFILE;
     return FALSE;
  }
 
  p1 = p0 = os_->tellg();
 

  while ( !strchr(delim,os_->peek()) && !os_->eof() )
     os_->seekg(++p1);

  if ( os_->eof() ) 
  {
    os_->seekg(p0);
    return FALSE ;
  }

  while ( strchr(delim,os_->peek()) && !os_->eof() )
     os_->seekg(++p1);
  
  if ( os_->eof() ) 
  {
    os_->seekg(p0);
    return FALSE ;
  }

  return TRUE;

} // end of method ignore

