//************************************************************************************//
// Module       : siforce.hpp
// Date         : 12/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                defining the state-independent Einaudi-Villi 
//                forcing function. This function is defined in
//                AIP S1070-664X(99)02711-1, s.t.
//                F_ = Curl ( f z^ ), where
//                  f = A_1(x,y) sin^2(Pi t/(2 t_*) + A_2(x,y) sin^2(Pi t/(2 t_*) + Pi/2),
//                with
//                A_i = Sum_m,n alpha^i_m,n sin(k_n x + k_m y + xi^i_m,n).
// Derived From : SIForce
// Modifications:
//************************************************************************************//
#include "evforce.hpp"
#include "param_reader.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
EVForce::EVForce()
: SIForce(),
bUpdated_     (FALSE),
bRenormalized_(TRUE),
bInitialized_ (FALSE),
last_time_    (0.0),
start_time_   (0.0),
turn_time_    (0.0),
A_avg_        (1.0),
vtmp          (NULL)
{
  GINT  i;

  for ( i=0; i<rank_; i++ ) {
    dA_    [i] = NULL;
    xi_    [i] = NULL;
    alpha_ [i] = NULL;
    k_     [i] = NULL;
    kmin_  [i] = 1.8;
    kmax_  [i] = 4.3;
    ib_ [i][0] = 0;
    ie_ [i][1] = 0;
    norm_  [i] = 1.0;
  }
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
EVForce::EVForce(Elem2D *e)
: SIForce(e),
bUpdated_     (FALSE),
bRenormalized_(TRUE),
bInitialized_ (FALSE),
last_time_    (0.0),
turn_time_    (0.0),
A_avg_        (1.0),
vtmp          (NULL)
{
  GINT  i;
  for ( i=0; i<rank_; i++ ) {
    dA_    [i] = NULL;
    xi_    [i] = NULL;
    alpha_ [i] = NULL;
    k_     [i] = NULL;
    kmin_  [i] = 1.8;
    kmax_  [i] = 4.3;
    ib_ [i][0] = 0;
    ie_ [i][1] = 0;
    norm_  [i] = 1.0;
  }
  for ( i=0; i<rank_; i++ ) {
    dA_[i] = new GVector(NN_);
  }
  vtmp  = new GVector(NN_);
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
EVForce::~EVForce()
{
  GINT  i;
  if ( vtmp  != NULL ) delete vtmp; vtmp = NULL;
  for ( i=0; i<rank_; i++ ) {
    if ( dA_   [i] != NULL ) delete dA_   [i];
    if ( xi_   [i] != NULL ) delete [] xi_   [i];
    if ( alpha_[i] != NULL ) delete [] alpha_[i];
    if ( k_    [i] != NULL ) delete [] k_    [i];
    dA_   [i] = NULL;
    xi_   [i] = NULL;
    alpha_[i] = NULL;
    k_    [i] = NULL;
  }
}


//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets field element
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void  EVForce::SetElem(Elem2D *e)
{

  GINT  i;

  if ( e == NULL ) {
    cout << "EVForce::SetElem: NULL element" << endl;
    exit(1);
  }
  elem  = e;
  rank_ = 2;
  idims_[0] = elem->GetOrder(1)+1;
  idims_[1] = elem->GetOrder(2)+1;
  idims_[2] = 1;
  NN_       = idims_[0]*idims_[1]*idims_[2];
  for ( i=0; i<rank_; i++ ) {
    if ( dA_ [i] != NULL ) delete dA_ [i]; dA_ [i] = NULL;
    dA_[i] = new GVector(NN_);
  }
} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetTurnoverTime
// DESCRIPTION: Sets turn-over time 
// ARGUMENTS  :
// RETURNS    :  
//************************************************************************************
void  EVForce::SetTurnoverTime(GDOUBLE t)
{
  turn_time_ = t;
} // end of method SetTurnoverTime


//************************************************************************************
//************************************************************************************
// METHOD     : SetStartTime
// DESCRIPTION: Sets start time: time at which to 'turn on' force term
// ARGUMENTS  :
// RETURNS    :  
//************************************************************************************
void  EVForce::SetStartTime(GDOUBLE t)
{
  start_time_ = t;
} // end of method SetStartTime


//************************************************************************************
//************************************************************************************
// METHOD     : Accel (1)
// DESCRIPTION: Acceleration term that updates velocity update due to EV force,
//              at time, time:
//               du/dt = Accel_EV = Force_density / rho
// ARGUMENTS  : time:  time at which to evaluate force
//              idir:  acceleration (force) comonent desired
// RETURNS    :  GVector with vector acceleration. Size of vector
//               is determined by total number of nodal points in element.
//************************************************************************************
GVector EVForce::Accel(GDOUBLE time, GINT  idir)
{
  GVector accel(NN_);
  
  if ( !Accel(time, idir, accel) ) {
    cout << "GVector EVForce::Accel (1): acceleration computation failed" << endl;
    exit(1);
  }
  
  return accel;
} // end of method Accel (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Accel (2)
// DESCRIPTION: Acceleration term that updates velocity update due to EV force
//              at time, time:
//               du_idir/dt = Accel_EV = Force_density / rho
// ARGUMENTS  : time:  time at which to evaluate force
//              idir:  acceleration (force) comonent desired
//              ret :  computed vector acceleration. This should have 
//                     a dimension at least = to the no. of nodes in 
//                     the element.
// RETURNS    : TRUE on success; else FALSE if initialization isn't done properly.  
//************************************************************************************
GBOOL EVForce::Accel(GDOUBLE time, GINT  idir, GVector &ret)
{
  GINT  i, n;
  GDOUBLE *data, c2a, cm, cp, *dA1, *dA2, sgn ;

  if ( (time - last_time_) >= turn_time_ || time == start_time_ ) {
    if ( !bUpdated_ ) {
      UpdateRand();
      last_time_ = time;
    }
  }
  else bUpdated_ = FALSE;

  if ( rank_ < 1 || rank_ > 2     ) return FALSE;  // for the time-being at least
  
  for ( i=0; i<rank_; i++ ) {
    if ( !alpha_[i] || !xi_[i] || !k_[i] ) {
      cout << "EVForce::Accel (2): component data incomplete" << endl; 
      return FALSE;
    }
  }
  
 
  ret  = 0.0;
  sgn = pow(-1.0, (GSHORT )idir);
  data = ret.Data();
  c2a  = cos(2.0* PI * time / (2.0*turn_time_) );
  cm   = 0.5*(1.0 - c2a) / rho_;
  cp   = 0.5*(1.0 + c2a) / rho_;
  ComputeDerivs(idir);                              // derivs placed in dA_  
  dA1 = dA_[0]->Data(); 
  dA2 = dA_[1]->Data();
  for ( n=0; n<NN_; n++ ) {
    *(data+n) = sgn*( *(dA1+n) * cm +   
                      *(dA2+n) * cp );
   }
  
  return TRUE;
} // end of method Accel (2)


//************************************************************************************
//************************************************************************************
// METHOD     : SetWaveRange
// DESCRIPTION: Sets wavenumber range
// ARGUMENTS  : idir : coord direction
//              k0   : lowest wavenumber (least energetic mode)
//              k1   : highest wavenumber (most energetic mode)
// RETURNS    :  
//************************************************************************************
void EVForce::SetWaveRange(GINT  idir, GDOUBLE k0, GDOUBLE k1)
{
  if ( idir  < 1 || idir  > rank_ ) {
    cout << "EVForce::SetWaveRange: invalid coordinate direction" << endl;
    exit(1);
  }

  kmin_[idir-1] = k0;
  kmax_[idir-1] = k1;
} // end of method SetWaveRange


//************************************************************************************
//************************************************************************************
// METHOD     : SetIndexRange
// DESCRIPTION: Sets wavenumber index ranges. NOTE: SetWaveRange
//              must be called before this method.
// ARGUMENTS  : icomp: Amplitude component, A1 or A2
//              idir : x (1) or y(2) summation direction for modes
//              ib   : beginning summation index in direction idir, for comp icomp
//              ie   : ending summation index in direction idir, for comp icomp
//                NOTE: values of k at k-indices will depend on total
//                      number of modes specified here, and on the wave
//                      range specified in call to SetWaveRange.
// RETURNS    :
//************************************************************************************
void EVForce::SetIndexRange(GINT  icomp, GINT  idir, GINT  ib, GINT  ie)
{

  if ( idir  < 1 || idir  > rank_ ||
       icomp < 1 || icomp > rank_ ) {
    cout << "EVForce::SetIndexRange: invalid component" << endl;
    exit(1);
  }
  ib_[icomp-1][idir-1] = ib;
  ie_[icomp-1][idir-1] = ie;
  
  bInitialized_ = FALSE;

} // end of method SetIndexRange


//************************************************************************************
//************************************************************************************
// METHOD     : UpdateRand
// DESCRIPTION: Updates the force's random component terms
// ARGUMENTS  :
// RETURNS    : none.  
//************************************************************************************
void EVForce::UpdateRand()
{
  GINT      i, j, ij, k, m, n, mn;
  GDOUBLE    *x, *y, kx, ky, pi2=2.0*PI;
  GVector  *W;

  bUpdated_ = FALSE;
  if ( !bInitialized_ && !Initialize() ){
    cout << "EVForce::UpdateRand: Object not initialized" << endl;
    exit(1);
  }

  // Update the random quantities:
  for ( k=0; k<rank_; k++ ) {
    if ( !alpha_[k] || !xi_[k] || !k_[k] ) {
      cout << "EVForce::UpdateRand: invalid component data" << endl;
      exit(1);
    }
    for ( n=0,mn=0; n<ie_[k][1]-ib_[k][1]+1; n++ ) {
      for ( m=0; m<ie_[k][0]-ib_[k][0]+1; m++,mn++ ) {
        alpha_[k][mn] = MTK::Rand();        // amplitude
        xi_   [k][mn] = MTK::Rand()*pi2;    // constant phase
      }
    }
  }
 
  // Renormalize alpha_, s.t., <A_i> = A_avg:
  // Note: A_i =  Sum_m_n alpha^i_m_n sin(kx_m x + ky_n y + xi^i_m_n).
  if ( bRenormalized_ ) {
    // First, compute normalization for each component:
    x = elem->GetSpNodes(1)->Data();
    y = elem->GetSpNodes(2)->Data();
    W = elem->Get2DWeights();
    if ( !vtmp ) vtmp  = new GVector(NN_);
    vtmp->Resize(NN_);
    for ( k=0; k<rank_; k++ ) {
      for ( n=0,mn=0; n<ie_[k][1]-ib_[k][1]+1; n++ ) {
        ky = k_[1][n] ;
        for ( m=0; m<ie_[k][0]-ib_[k][0]+1; m++,mn++ ) {
          kx = k_[0][m];
         
          // get integrand at nodal points:
          for ( j=0; j<idims_[1]; j++ ) {
            for ( i=0; i<idims_[0]; i++ ) {
              ij = i + j*idims_[0];
              (*vtmp)(ij) = sin(kx*x[ij] + ky*y[ij] + xi_[k][mn]); 
            }
          }
        
          // Compute integral of vtmp over element:
          norm_[k] = (*vtmp) * (*W);   
          norm_[k] = A_avg_/(norm_[k]+TINY);
        }
      }
    }

    // Now, renormalize alpha_:
    for ( k=0; k<rank_; k++ ) {
      for ( n=0,mn=0; n<ie_[k][1]-ib_[k][1]+1; n++ ) {
        for ( m=0; m<ie_[k][0]-ib_[k][0]+1; m++,mn++ ) {
          alpha_[k][mn] *= norm_[k];
        }
      }
    }

  }

  bUpdated_ = TRUE;
} // end of method UpdateRand


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeDerivs
// DESCRIPTION: Computes d_idir (A_1) and d_idir(A_2) ...
// ARGUMENTS  :
// RETURNS    : none. Derivs stored in dA_[] .
//************************************************************************************
void EVForce::ComputeDerivs(GINT  idir)
{
  GINT  i, j, k, m, n, ij, mn;
  GDOUBLE *x, *y, kw[2];

  if ( idir < 1 || idir > rank_ ) {
    cout << "EVForce::ComputeDerivs: invalid derivative component" << endl;
    exit(1);
  }
  if ( !bInitialized_ && !Initialize() ){
    cout << "EVForce::UpdateRand: Object not initialized" << endl;
    exit(1);
  }


  x  = elem->GetSpNodes(1)->Data();
  y  = elem->GetSpNodes(2)->Data();
  for ( k=0; k<rank_; k++ ) {
    if ( dA_[k] == NULL ) continue;
    *dA_[k] = 0.0;
    for ( j=0; j<idims_[1]; j++ ) {
      for ( i=0; i<idims_[0]; i++ ) {
        ij = i + j*idims_[0];
//----------------------------------- begin Sum_m_n    
        for ( n=0,mn=0; n<ie_[k][1]-ib_[k][1]+1; n++ ) {
          kw[1] = k_[1][n];
          for ( m=0;  m<ie_[k][0]-ib_[k][0]+1; m++,mn++ ) {
            kw[0] = k_[0][m];
            (*dA_[k])(ij) += ( alpha_[k][mn] * kw[idir-1] * cos(kw[0]*x[ij] + kw[1]*y[ij] + xi_[k][mn]) );
          }
        }
//------------------------------------- end Sum_m_n    
      }
    }
  }

} // end of method ComputeDerivs


//************************************************************************************
//************************************************************************************
// METHOD     : DoRenormalization
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : none. 
//************************************************************************************
void EVForce::DoRenormalization(GBOOL yn, GDOUBLE avg)
{ 
  bRenormalized_ = yn;
  A_avg_         = avg;
} // end of metho DoRenormalization


//************************************************************************************
//************************************************************************************
// METHOD     : Initialize
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  TRUE on success; else FALSE. 
//************************************************************************************
GBOOL EVForce::Initialize()
{ 
  GINT  i, j, n, *imin, *imax;
  GDOUBLE dk;

  for ( j=0; j<rank_; j++ ) {
    if ( k_    [j] != NULL ) delete [] k_ [j];
    if ( xi_   [j] != NULL ) delete [] xi_   [j];
    if ( alpha_[j] != NULL ) delete [] alpha_[j];
    k_    [j] = NULL;
    xi_   [j] = NULL;
    alpha_[j] = NULL;
  }

  imin = new GINT  [rank_];
  imax = new GINT  [rank_]; 
  for ( i=0; i<rank_; i++ ) {
    imin[i] =  IHUGE;
    imax[i] = -IHUGE;
    for ( j=0,n=1; j<rank_; j++ ) {          // find max/min indices  over A-components
      imin[i] = imin[i] < ib_[j][i] ? imin[i] : ib_[j][i];
      imax[i] = imax[i] > ie_[j][i] ? imax[i] : ie_[j][i];
      n *= (ie_[i][j]-ib_[i][j]+1);
    }
    if ( n <= 0 ) {
      cout << "EVForce::Initialize: invalid index range" << endl;
      return FALSE;
    }
    k_    [i] = new GDOUBLE [imax[i]-imin[i]+1];
    xi_   [i] = new GDOUBLE [n];
    alpha_[i] = new GDOUBLE [n];
  }

  // There is one  max/min index set for each coord direction...

  // Compute wave numbers:
  for ( i=0, bInitialized_=TRUE; i<rank_; i++ ) {
    if ( !k_[i] || !xi_[i] || !alpha_[i] ) bInitialized_ = FALSE;
    if ( imin[i] == imax[i] )
      dk = 0.0;
    else
      dk = (kmax_[i] - kmin_[i]) / ( imax[i] - imin[i] );
    for ( j=0; j<(imax[i]-imin[i]+1) && k_[i]; j++ )
      k_[i][j] = kmin_[i] + j*dk;
  }

  bInitialized_ = TRUE;

  delete [] imin;
  delete [] imax;

  return TRUE;
} // end of metho  Initialize

//************************************************************************************
//************************************************************************************
// METHOD     : Initialize
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  TRUE on success; else FALSE. 
//************************************************************************************
GBOOL EVForce::ParamSet(char *filename, char *blockname)
{ 
  ParamReader pr(TRUE);

  if ( filename == NULL || blockname == NULL ) return TRUE;


  // If config file doesn't exist, this is not considered an error:
  if ( !pr.Access(filename) ) {
//  cout << "EVForce::ParamSet: Command file, '" << filename << "' does not exist; using defaults" << endl;
    return TRUE;
  }

  pr.SetBuffSize(65536);

  // Configure ParamReader object with  all parameters that _may_
  // be changed via param file:
  pr.SetParams("%f %f %f %f %f %f %i %i %i %i %i %i %i %i %i %f"
                                     , "Start_Time"
                                     , "TurnOver_Time"
                                     , "X_kmin"
                                     , "Y_kmin"
                                     , "X_kmax"
                                     , "Y_kmax"
                                     , "IMin_XX"
                                     , "IMin_XY"
                                     , "IMin_YX"
                                     , "IMin_YY"
                                     , "IMax_XX"
                                     , "IMax_XY"
                                     , "IMax_YX"
                                     , "IMax_YY"
                                     , "Renormalize"
                                     , "Renormalization_Amplitude"
       
                   );
  // Get parameters that are in file, and reset the associated
  // values:
  if ( !pr.GetParams(filename, blockname  
                                       , &start_time_
                                       , &turn_time_
                                       , &kmin_[0]
                                       , &kmin_[1]
                                       , &kmax_[0]
                                       , &kmax_[1]
                                       , &ib_[0][0]
                                       , &ib_[0][1]
                                       , &ib_[1][0]
                                       , &ib_[1][1]
                                       , &ie_[0][0]
                                       , &ie_[0][1]
                                       , &ie_[1][0]
                                       , &ie_[1][1]
                                       , &bRenormalized_
                                       , &A_avg_
                                               ) ) {
    cout << "EVForce::ParamSet: Error reading parameter file, " << filename << ": " << blockname <<  endl;
    cout << "ParamReader error: " << pr.Error() << endl;
    return FALSE;
  }

  // Initialize the object with the new parameters:
  return Initialize();

} // end of method ParamSet
