//************************************************************************************//
// Module       : ilaplacian.hpp
// Date         : 2/26/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the inverse of the weak Laplacian operator.
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#include "ilaplacianop.hpp"
#include "laplacianop.hpp"
#include "diagop.hpp"
#include "mtk.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
iLaplacianOp::iLaplacianOp()
:        LinOp(),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
elem     (NULL),
L        (NULL),
iL       (NULL)
{
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
iLaplacianOp::iLaplacianOp(LinOp *AA)
:        LinOp(AA),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
elem     (NULL),
L        (NULL),
iL       (NULL)
{
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
iLaplacianOp::iLaplacianOp(Elem2D *e)
:        LinOp(),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
elem     (NULL),
L        (NULL),
iL       (NULL)
{
  SetElem(e);

} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Destructor
iLaplacianOp::~iLaplacianOp()
{
  if ( L  != NULL ) delete L ; L  = NULL;
  if ( iL != NULL ) delete iL; iL = NULL;
}


//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void iLaplacianOp::SetElem(Elem2D *e)
{

  if ( e == NULL )
  {
    cout << "iLaplacianOp::SetElem: NULL element" << endl;
    exit(1);
  }

  if ( e != elem || N1 != (e->GetOrder(1)+1) ||
                    N2 != (e->GetOrder(2)+1) )
  {
    elem = e;
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    if ( L  != NULL ) delete L ; L  = NULL;
    if ( iL != NULL ) delete iL; iL = NULL;
    if ( !ComputeOp() )
    {
      cout << "iLaplacianOp::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
    return;
  }

} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void iLaplacianOp::SetConst(GDOUBLE clc)
{


  if ( lc1 != clc || lc2 != clc )
  {
    if ( elem == NULL )
    {
      cout << "iLaplacianOp::SetConst: NULL element" << endl;
      exit(1);
    }
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    if ( L  != NULL ) delete L ; L  = NULL;
    if ( iL != NULL ) delete iL; iL = NULL;
    lc1 = lc2 = clc;
    if ( !ComputeOp() )
    {
      cout << "iLaplacianOp::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
  }
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void iLaplacianOp::SetConst(GDOUBLE clc1, GDOUBLE clc2)
{


  if ( lc1 != clc1 || lc2 != clc2 )
  {
    if ( elem == NULL )
    {
      cout << "iLaplacianOp::SetConst: NULL element" << endl;
      exit(1);
    }
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    if ( L  != NULL ) delete L ; L  = NULL;
    if ( iL != NULL ) delete iL; iL = NULL;
    lc1 = clc1;
    lc2 = clc2;
    if ( !ComputeOp() )
    {
      cout << "iLaplacianOp::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
  }
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
GVector  iLaplacianOp::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);

  OpVec_prod(q, newq);

  return newq;

} // end of method \* oeprator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
void iLaplacianOp::OpVec_prod(GVector &q, GVector &ret)
{

  if ( elem == NULL || iL == NULL )
  {
    cout << "iLaplacianOp::operator*: NULL element or operator" << endl;
    exit(1);
  }

  if ( q.dim() != NN  || ret.dim() != NN ) 
  {
    cout << "iLaplacianOp::operator*: incompatible vector(s)" << endl;
    exit(1);
  }

//ret = (*iL) * q;
  MTK::fmatvec_prod(*iL, q, ret);
  if ( mask )  MTK::fvec_point_prod_rep(ret, *mask);

} // end of method OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeOp
// DESCRIPTION: Computes inverse Laplacian operator
// ARGUMENTS  : 
//             
// RETURNS    : TRUE on success; else FALSE
GBOOL iLaplacianOp::ComputeOp()
{

  GBOOL bRet ;

  if ( elem == NULL ) return FALSE;
  if ( L != NULL || iL != NULL ) return FALSE;

  if ( N1 <= 0 || N2 <= 0 || NN <= 0 ) return FALSE;

  L    = new GMatrix(NN,NN);
  iL   = new GMatrix(NN,NN);

  switch (elem->ElemType())
  {
    case DEFORMED_QUAD:
      bRet = DefmQuadH();
      break;
    case RECT_QUAD:
      bRet = RectQuadH();
      break;
    case TRIANGULAR:
      bRet = TriangleH();
      break;
    default:
      cout << "LaplacianOp::Constructor(3): Invalid element type" << endl;
      exit(1);
  }
#if 1
  GMatrix C(NN,NN);
  C = (*L) * (*iL);
  cout.setf(ios::fixed);
  cout << "iLaplacianOp::ComputeOp: L * iL = " << C << endl;
#endif


  // done with L, so delete it:
  if ( L != NULL )
  {
    delete L;
    L = NULL;
  }

  return bRet;
} // end of method ComputeOp



//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadH
// DESCRIPTION: Computes Laplacian^-1 a deformed quad element.
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
GBOOL iLaplacianOp::DefmQuadH()
{ 
  GINT      i, j, k, n, m;
  GVector  *G11, *G12, *G21, *G22, *GA[2];
  GMatrix  *D1 , *D2;

//M   = elem->GetMassMatrix();  // already includes Jacobian
  D1  = elem->Get1DDerivMatrix(1,FALSE);
  D2  = elem->Get1DDerivMatrix(2,FALSE);
  G11 = ((DefQuad2D*)elem)->GetWJMetric(1,1);
  G12 = ((DefQuad2D*)elem)->GetWJMetric(1,2);
  G21 = G12;
  G22 = ((DefQuad2D*)elem)->GetWJMetric(2,2);

  if ( D1  == NULL || D2  == NULL ||
       G11 == NULL || G12 == NULL ||
       G21 == NULL || G22 == NULL ) return FALSE;

 
  GMatrix *d1v, *d2v;

  d1v = new GMatrix(NN,NN);
  d2v = new GMatrix(NN,NN);
  L   = new GMatrix(NN,NN);

  // Do I2 X D1 term:
  for ( k=0; k<N2; k++ )
  {
    for ( i=0; i<N1; i++ )
    {
      for ( j=0; j<N1; j++ )
      {
        n = k * N1 + i;
        m = k * N1 + j;
        (*d1v)(n,m) = (*D1)(i,j);
      }
    }
  }

  // Do D2 X I1 term:
  for ( i=0; i<N2; i++ )
  {
    for ( j=0; j<N2; j++ )
    {
      for ( k=0; k<N1; k++ )
      {
        n = i * N1 + k;
        m = j * N1 + k;
        (*d2v)(n,m) = (*D2)(i,j);
      }
    }
  }

  // Incorporate the metric terms:
  GA[0] = G11;
  GA[1] = G12;
  for ( k=0; k<2; k++ )
    for ( i=0; i<NN; i++ )
      for ( j=0; j<NN; j++ )
        (*d1v)(i,j) = (*d1v)(i,j) * (*GA[k])(j);

  GA[0] = G21;
  GA[1] = G22;
  for ( k=0; k<2; k++ )
    for ( i=0; i<NN; i++ )
      for ( j=0; j<NN; j++ )
        (*d2v)(i,j) = (*d2v)(i,j) * (*GA[k])(j);
  // Multiply by the appropriate transposes, and add up:
  *L   = ( d1v->Transpose() * ( *d1v + *d2v ) )
       + ( d2v->Transpose() * ( *d1v + *d2v ) );

  delete d1v;
  delete d2v;

  // Now, invert to form preconditioner:
  return L->Inverse(*iL);

} // end of method DefmQuadH


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadH
// DESCRIPTION: Computes  Laplacian^-1 for a regular quad element.
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL iLaplacianOp::RectQuadH()
{ 

  GSHORT      i, j;
  GVector     *icol, *rcol;
  LaplacianOp *lop;


  icol = new GVector(NN);
  rcol = new GVector(NN);
  lop  = new LaplacianOp(elem);

  lop->SetConst(lc1, lc2);
  for ( j=0; j<NN; j++ )
  {
    for ( i=0; i<NN; i++ )
    {
      (*icol)(i) = 0.0;
      if ( i == j ) (*icol)(i) = 1.0;
    }
    lop->OpVec_prod(*icol, *rcol);
    for ( i=0; i<NN; i++ )
      (*L)(i,j) = (*rcol)(i);
  }

  delete lop;
  delete icol;
  delete rcol;

  
  // Now, invert to form preconditioner:
  return L->Inverse(*iL);

} // end of method RectQuadH


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleH
// DESCRIPTION: Computes Laplacian^-1 a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL iLaplacianOp::TriangleH()
{
  return FALSE;
} // end of method TriangleH


