//************************************************************************************//
// Module       : massop.cpp
// Date         : 6/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the 2d mass operator class
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include "massop.hpp"
#include "diagop.hpp"
#include "defquad2d.hpp"
#include "rectquad2d.hpp"



//************************************************************************************
//************************************************************************************
// Constructor Method (1)
//************************************************************************************
MassOp::MassOp()
: LinOp(),
cmc_      (1.0),
mask      (NULL),
MM        (NULL),
elem      (NULL) 
{
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
//************************************************************************************
MassOp::MassOp(Elem2D *e)
: LinOp( ), 
cmc_      (1.0),
mask      (NULL),
MM        (NULL),
elem      (e) 
{
  GVector *M;
  if ( (M=elem->GetMassMatrix()) == NULL )
  {
    cout << "MassOp::MassOp(Elem2D *e): NULL Mass Matrix" << endl;
    exit(1);
  }
  MM = new DiagOp(elem);
  MM->SetMask(mask);
  MM->SetDiag(M->Data(), M->dim());
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
MassOp::~MassOp()
{
  if ( MM != NULL ) delete MM;
  MM = NULL;
}

//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void MassOp::SetElem(Elem2D *e)
{
  elem = e;

  if ( elem == NULL ) return;

  elem = e;
  if ( MM != NULL ) delete MM;
  MM = new DiagOp(elem);
  MM->SetMask(mask);
  MM->SetDiag(elem->GetMassMatrix()->Data(), elem->GetMassMatrix()->dim());

} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetMask
// DESCRIPTION: Sets mask
// ARGUMENTS  : GVector *m
// RETURNS    : none
//************************************************************************************
void MassOp::SetMask(GVector *m)
{
  mask = m;
  if ( MM )  MM->SetMask(mask);

} // end of method SetMask


//************************************************************************************
//************************************************************************************
// METHOD     : GetDiag
// DESCRIPTION: Gets diagonal vector
// ARGUMENTS  : none
// RETURNS    : GVector *
//************************************************************************************
GVector *MassOp::GetDiag()
{
  return MM->GetDiag();
} // end of method GetDiag


//************************************************************************************
//************************************************************************************
// METHOD     : Inverse
// DESCRIPTION: Toggles whether the _inverse_ of the Mass operator should
//              be applied.
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void MassOp::Inverse()
{
  MM->Inverse();
} // end of method Inverse


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constant
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void MassOp::SetConst(GDOUBLE c)
{
  cmc_ = c;
} // end of method SetConst

//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
GVector  MassOp::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);
  
  OpVec_prod(q, newq);
  return newq;

} // end of operator \*


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
void MassOp::OpVec_prod(GVector &q, GVector &ret)  
{
  if ( elem == NULL )
  {
    cout << "MassOp::operator*: NULL element" << endl;
    exit(1);
  }

  GBOOL    bRet = TRUE;

  switch (elem->ElemType())
  {
    case DEFORMED_QUAD:
    case RECT_QUAD:
      bRet = QuadOp(&q,&ret);
      break;
    case TRIANGULAR:
      bRet = TriangleOp(&q,&ret);
      break;
    default:
      cout << "MassOp::operator*: Invalid element type" << endl;
      exit(1);
  }
  if ( !bRet )
  {
    cout << "MassOp::operator*: Operator product failed" << endl;
    exit(1);
  }

} // end of method OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : QuadOp
// DESCRIPTION: Computes  Mass operator*vector  for a quad element.
//              Mass operator for quadrilateral element is:
//                M2 X M1
//              where and 'X' is a tensor product. The 
//              result is a computation of the full (incl Jacobian)
//              form.
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL MassOp::QuadOp(GVector *v, GVector *vn)
{
  if ( v  == NULL || vn  == NULL || MM == NULL ) return FALSE;

  GINT       N1, N2, NN;

  N1 = elem->GetOrder(1)+1;
  N2 = elem->GetOrder(2)+1;
  NN = N1 * N2;
  if ( v->dim() != NN  || vn->dim() != NN ) return FALSE; 

  MM->OpVec_prod(*v,*vn);
  if ( cmc_ != 1.0 ) MTK::fvec_const_prod_rep(*vn,cmc_);


  return TRUE;
} // end of method QuadOp 


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleOp
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL MassOp::TriangleOp(GVector *v, GVector *vn)
{
  return FALSE;
} // end of method TriangleOp


