//************************************************************************************//
// Module       : stokes_solver.cpp
// Date         : 8/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the solver for the Stokes problem.  The problem solved is
//                  H _v - G p = f,
//                subject to the constraint that
//                  Div _v = 0;
//                Here, H is the Helmholtz operator,
//                  H =  p L  + q M
//                where p=nu_ and q= c0/dt_ are constants, L is the discrete (weak)
//                Laplacian operator, and M_ is the mass matrix. _v is the
//                velocity vector, and p is the pressure field, and G = 1/rho D^T, the 
//                weak Stokes gradient operator; updated  values of _v and p are 
//                the result.
//                  
//                This class assumes that the pressure and velocity reside on 
//                different grids within the same element, using the P_N-P_(n-2)
//                formalism, in which the pressure grid lies on Gaussian nodes.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "stokes.hpp"
#include "pcpointjac_helm.hpp"
#include "pcblockjac_helm.hpp"
#include "pcblockjac_plap.hpp"
#include "pcblockjac_lap.hpp"
#include "diagop.hpp"
#include "mtk.hpp"
#include "gutils.hpp"

char *sStokesType[] = {"STOKES_SCHUR_DELP", "STOKES_UZAWA"};

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
StokesSolver::StokesSolver(GFieldList *ulist[], GINT nf, GElemList *uelems, GElemList *pelems, STOKES_TYPE t)
:
itype_       (t), 
irank_       (GDIM),
nelems_      (0),
bInitReqd_   (TRUE),
bPInit_      (FALSE),
dt_          (1.0),
dtlast_      (0.0),
c0_          (1.0),
irho_        (1.0),
rho_         (1.0),
hDSOp        (NULL_HANDLE),
uelems_      (uelems),
pelems_      (pelems),
filter_      (NULL),
gsop         (NULL)
{
  GINT  isz, j;

  if ( uelems == NULL || pelems == NULL ) {
    cout << "StokesSolver::StokesSolver: NULL element list" << endl;
    exit(1);
  }

  if ( nf < irank_+1 ) {
    cout << "StokesSolver::StokesSolver: insufficient number of fields computed" << endl;
    exit(1);
  }

  // Last field is assumed to be pressure:
  for ( j=0; j<=irank_; j++ ) {
    if ( ulist[j] == NULL ) {
      cout << "StokesSolver::StokesSolver: NULL input field" << endl;
      exit(1);
    }
    var_list_[j] = ulist[j];
  }
  if ( uelems->size() != pelems->size() ) {
    cout << "StokesSolver::StokesSolver: incompatible grids" << endl;
    exit(1);
  }

  isz = var_list_[0]->size();
  for ( j=1; j<=irank_; j++ ) {
    if ( var_list_[j]->size() != isz ) {
      cout << "StokesSolver::StokesSolver: incompatible input fields" << endl;
      exit(1);
    }
  }
  
  for ( j=0; j<irank_; j++ ) {
    ubdyvals_[j] = NULL;
  }
  for ( j=0; j<=irank_; j++ ) {
    cg_[j] = NULL;
  }
  Init();

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
StokesSolver::~StokesSolver()
{
  

  DeleteDynamic();

}

//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: 
// RETURNS    :

void  StokesSolver::DeleteDynamic()
{
  GINT  j;
  
  for ( j=0; j<=irank_; j++ ) {
    if ( cg_[j] != NULL ) delete cg_[j]   ; 
  }

  H_    .empty();
  E_    .empty();
  Hdiag_.empty();
  for ( j=0; j<irank_; j++ ) {
    D_ [j] .empty();
    DT_[j] .empty();
  }
  pu_[irank_].empty();
  pp_        .empty();
  for ( j=0; j<irank_; j++ ) {
    utmp_    [j].empty();
    ptmp_    [j].empty();
    g_       [j].empty();
    pu_      [j].empty();
    pc_      [j].empty();
    pu_rhs_  [j].empty();
  }


} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Init
// DESCRIPTION: 
// RETURNS    :  

void  StokesSolver::Init()
{ 
  GINT  j;
  
  DeleteDynamic();
  bPrecond_[irank_] = FALSE;
  for ( j=0; j<irank_; j++ ) {
    nu_        [j] = 1.0;
    cg_        [j] = NULL;
    bPrecond_  [j] = FALSE;
  }

  for ( j=0; j<=irank_; j++ ) {
    cg_      [j] = new CG(CG_STANDARD, FALSE);
    pc_type_ [j] = GPC_NONE;
  }
  ResetExpandables();
  ResetErrors();
  
} // end of method Init


//************************************************************************************
//************************************************************************************
// METHOD     : InitPrecond
// DESCRIPTION: Initializes preconditioners
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL StokesSolver::InitPrecond(const GINT iwhich)
{
  GINT  i;
  Elem2D *ev, *ep;

  if ( iwhich < 0 || iwhich > irank_ ) {
    cout << "StokesSolver::InitPrecond: invalid directive" << endl;
    exit(1);
  }

  if ( !bPrecond_[iwhich] ) return TRUE;

  // Delete existing preconditioners, if any, in iwhich list:
  for ( i=0; i<nelems_; i++ ) {
    if ( pc_[iwhich][i] != NULL ) { 
      delete pc_[iwhich][i];
      pc_[iwhich][i] = NULL;
    }
  }

  switch ( pc_type_[iwhich] ) {
      // Pressure preconditioner:
      case GPC_BLOCKJAC_LAP:
        if ( iwhich != irank_ ) {
          cout << "StokesSolver::InitPrecond: PC index invalid (GPC_BLOCKJAC_LAP)" << endl;
          exit(1);
        }
        for ( i=0; i<nelems_; i++ ) {
          ev = (*uelems_)[i];
          ep = (*pelems_)[i];
          if ( pc_[iwhich][i] != NULL ) delete pc_[iwhich][i];
          if ( itype_ ==  STOKES_STEADY_UZAWA ) {
            pc_[iwhich][i] = new MassOp(ep);
            ((MassOp*)(pc_[iwhich][i]))->Inverse();
          }
          else {
            pc_[iwhich][i] = new PCBlockJac_PLap(ev, ep, (SchurLapOp*)E_[i]);
//          pc_[iwhich][i] = new PCBlockJac_Lap(ep);
          }
        }
        break;

      // Velocity preconditioners:
      case GPC_BLOCKJAC_HELM:
        if ( iwhich == irank_ ) {
          cout << "StokesSolver::InitPrecond: PC index invalid (GPC_BLOCKJAC_HELM)" << endl;
          exit(1);
        }
        for ( i=0; i<nelems_; i++ ) {
          ev = (*uelems_)[i];
          pc_[iwhich][i] = new PCBlockJac_Helm(ev,(HelmholtzOp*)H_[i]);
          if ( pc_[iwhich][i] == NULL ) return FALSE;
        }
        break;
      case GPC_POINTJAC_HELM:
        if ( iwhich == irank_ ) {
          cout << "StokesSolver::InitPrecond: PC index invalid (GPC_POINTJAC_HELM)" << endl;
          exit(1);
        }
        for ( i=0; i<nelems_; i++ ) {
          ev = (*uelems_)[i];
          pc_[iwhich][i] = new PCPointJac_Helm(ev);
          if ( pc_[iwhich][i] == NULL ) return FALSE;
        }
        break;
       case GPC_NONE:
         break;
      default:
        return FALSE;
  }

  return TRUE;
  
}  // end of method InitPrecond

//************************************************************************************
//************************************************************************************
// METHOD     : SetComm
// DESCRIPTION: Sets global gather scatter operator
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void StokesSolver::SetComm(GS *ggs)
{
  GINT  j;

  gsop = ggs;

  if ( gsop != NULL ) {
     for ( j=0; j<irank_; j++ ) { 
       if ( cg_[0] == NULL ) {
         cout << "StokesSolver::SetComm: NULL solvers" << endl;
         exit(1);
       }
       cg_[j]->SetComm(gsop);
    }
  }

} // end of method SetComm

//************************************************************************************
//************************************************************************************
// METHOD     : SetCommHandle
// DESCRIPTION: Sets active gather/scatter handle
// ARGUMENTS  :
// RETURNS    : GCHandle
//************************************************************************************
GCHandle StokesSolver::SetCommHandle(GCHandle hIn)
{
  GINT  j;

  if ( !gsop ) return NULL_HANDLE;

  if ( hDSOp != NULL_HANDLE ) return hDSOp;

  hDSOp = hIn;
  for ( j=0; j<irank_; j++ ) {
    if ( cg_[j] == NULL ) {
       cout << "StokesSolver::SetCommHandle: NULL solver, component " << j+1 << endl;
       exit(1);
    }
    cg_[j]->SetCommHandle(hDSOp);
  }

  return hDSOp;
} // end of method SetCommHandle


//************************************************************************************
//************************************************************************************
// METHOD     : SetPreconditioner
// DESCRIPTION: Sets velocity component and pressure preconditioners
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
GBOOL StokesSolver::SetPreconditioner(GINT iwhich, GPC itype)
{   
  if ( iwhich < 1 || iwhich > irank_+1 ) {
    cout << "StokesSolver::SetPreconditioner: invalid field" << endl;
    exit(1);
  }
  if ( cg_[iwhich-1] == NULL ) {
    cout << "StokesSolver::SetPreconditioner: NULL field solver" << endl;
    exit(1);
  }

  pc_type_[iwhich-1] = itype;
  if ( !InitPrecond(iwhich -1) ) return FALSE;
  cg_[iwhich -1]->SetPreconditioner(&pc_[iwhich-1]);
  bPrecond_[iwhich-1] = TRUE;
  

  return TRUE;
} // end of method SetPreconditioner


//************************************************************************************
//************************************************************************************
// METHOD     : GetPreconditioner
// DESCRIPTION: Sets velocity component and pressure  preconditioners
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GLinOpList *StokesSolver::GetPreconditioner(GINT  iwhich)
{  
  if ( iwhich< 1 || iwhich > irank_+1 ) return NULL;
  return &pc_[iwhich-1];

} // end of method GetPreconditioner


//************************************************************************************
//************************************************************************************
// METHOD     : GetDivMax
// DESCRIPTION: Computes Max(Div.V)
//
// ARGUMENTS  :
// RETURNS    : max div.v
//************************************************************************************
GDOUBLE StokesSolver::GetDivMax(GINT  iLevel)
{
  GINT     i, j;
  GDOUBLE  div, divmax=-HUGE;
  GVector  *uj[3]={NULL};

  for ( i=0; i<uelems_->size(); i++ ) {
    for ( j=0; j<irank_; j++ ) {
      uj[j]  = var_list_[j]->member(i)->GetExpCoeffs(iLevel);
    }
    div    = GetDivMax(uj, irank_, i);
    divmax = MAX(divmax,div);
  }

  return divmax;

} // end of method GetDivMax


//************************************************************************************
//************************************************************************************
// METHOD     : GetDivMax
// DESCRIPTION: Computes Max(Div.V) using Stokes derivatives.
//
// ARGUMENTS  : vn[3]: vector whose divergence is to be computed: D1 v1 + D2 v2 + D3 v3
//              ie   : element id of elements on which to compute divergence
// RETURNS    : max div.vn
//************************************************************************************
GDOUBLE StokesSolver::GetDivMax(GVector *vn[], GINT nv, GINT  ie)
{
  GINT    j;
  GDOUBLE divmax;

  if ( ie < 0 || ie > uelems_->size() ) {
    cout << "StokesSolver::GetDiv: element id out of range" << endl;
    exit(1);
  }

  for ( j=0; j<nv; j++ ) {
    if ( vn[j] != NULL ) {
      D_[j][ie]->OpVec_prod(*vn[j], *ptmp_[j][ie]);
    }
  }
  for ( j=1; j<nv; j++ ) {
    if ( vn[j] != NULL ) {
      MTK::fvec_add_rep(*ptmp_[0][ie], *ptmp_[j][ie]);
    }
  }
  divmax = ptmp_[0][ie]->MaxA();

  return divmax;

} // end of method GetDivMax


//************************************************************************************
//************************************************************************************
// METHOD     : DivV
// DESCRIPTION: Computes Div.V using Stokes derivatives.
//
// ARGUMENTS  : divv : vector list on the p-grid, whose number of
//                     members = nelems_
// RETURNS    : void
//************************************************************************************
void StokesSolver::DivV(GFieldList *divv, GFieldList *v[], GINT nv)
{
  GINT    i, j;
  GVector *vc, *dv;

  if ( divv->size() !=  uelems_->size() || nv > irank_ ) {
    cout << "StokesSolver::DivV: incompatible number of list elements" << endl;
    exit(1);
  }
  divv->SetTimeLevel(0);
  *divv = 0.0;
  divv->start(NULL);
  for ( j=0; j<nv; j++ ) v[j]->start(NULL);
  for ( i=0; i<uelems_->size(); i++ ) {
    dv = divv->member()->GetExpCoeffs(0);
    for ( j=0; j<nv; j++ ) {
      vc = v[j]->member()->GetExpCoeffs(0);
      D_[j][i]->OpVec_prod(*vc, *ptmp_[j][i]);
    }
    for ( j=0; j<nv; j++ ) {
      MTK::fvec_add_rep(*dv, *ptmp_[j][i]);
    }
    divv->next();
    for ( j=0; j<nv; j++ ) v[j]->next();
  } 

} // end of method DivV



//************************************************************************************
//************************************************************************************
// METHOD     : SetVisc
// DESCRIPTION: Sets nu_
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void StokesSolver::SetVisc(const GDOUBLE f)
{
  GINT  j;

  for ( j=0; j<irank_; j++ ) nu_[j] = f;
} // end of method SetVisc

//************************************************************************************
//************************************************************************************
// METHOD     : SetVisc
// DESCRIPTION: Sets nu_
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void StokesSolver::SetVisc(const GDOUBLE f, GINT idir)
{
  if ( idir <1 || idir > GDIM ) {
    cout << "StokesSolver::SetVisc: invalid coordinate direction" << endl;
    exit(1);
  }
  nu_[idir-1] = f;
} // end of method SetVisc


//************************************************************************************
//************************************************************************************
// METHOD     : Setc0
// DESCRIPTION: Sets c0_
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void StokesSolver::Setc0(const GDOUBLE f)
{
  c0_ = f;
} // end of method Setc0


//************************************************************************************
//************************************************************************************
// METHOD     : SetDen
// DESCRIPTION: Sets rho_
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void StokesSolver::SetDen(const GDOUBLE f)
{
  rho_ = f;
  irho_ = rho_ == 0.0 ? 0.0 : 1.0/rho_;
} // end of method SetDen


//************************************************************************************
//************************************************************************************
// METHOD     : GetSolver
// DESCRIPTION: Gets pointer to velocity  and pressure solvers s
// ARGUMENTS  :
// RETURNS    :  
//************************************************************************************
CG *StokesSolver::GetSolver(GINT  idir)
{
  if      ( idir < 1 || idir > irank_+1 ) return NULL;
  return cg_[idir-1];
} // end of method GetSolver



//************************************************************************************
//************************************************************************************
// METHOD     : GetNumIterations
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  StokesSolver::GetNumIterations(GINT  idir)
{
  if ( idir < 1 || idir > irank_+1 ) return 0;
  return niter_[idir-1];
} // end of method GetNumIterations

//************************************************************************************
//************************************************************************************
// METHOD     : GetErrorType
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  StokesSolver::GetErrorType(GINT  idir)
{
  if ( idir < 1 || idir > irank_+1 ) return -1;
  return ierror_[idir-1];
} // end of method GetUErrorType


//************************************************************************************
//************************************************************************************
// METHOD     : GetError
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GDOUBLE StokesSolver::GetError(GINT idir)
{
  if ( idir < 1 || idir > irank_+1 ) return 0.0;
  return error_[idir-1];
} // end of method GetError


//************************************************************************************
//************************************************************************************
// METHOD     : GetMinError
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE StokesSolver::GetMinError(GINT  idir)
{
  if ( idir < 1 || idir > irank_+1 ) return 0.0;
  return min_error_[idir-1];

} // end of method GetMinError


//************************************************************************************
//************************************************************************************
// METHOD     : GetMaxError
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE StokesSolver::GetMaxError(GINT  idir)
{
  if ( idir < 1 || idir > irank_+1 ) return 0.0;
  return max_error_[idir-1];
} // end of method GetMaxError


//************************************************************************************
//************************************************************************************
// METHOD     : Solve
// DESCRIPTION: 
// ARGUMENTS  : 
//               iLevelRHS  : Which vector level to use as RHS for each component
//               iLevelFinal: Which vector level to use to store final solution
//
//               indices are wrt var_list_ fields, and they represent either
//               temp space or time level space
// RETURNS    :
//************************************************************************************
GBOOL StokesSolver::Solve(GDOUBLE dt, const GINT  itmpRHS, const GINT  iLevelFinal)
{
  GINT  i, j, isz;
  GBOOL  bRet = TRUE;

  isz = var_list_[0]->size();

  if( isz != nelems_ ) return FALSE;
  
  dt_ = dt;


  // Get RHS fields:
  for ( j=0; j<irank_; j++ ) var_list_[j]->start(NULL); 
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<irank_; j++ ) {
      pu_rhs_[j][i] = var_list_[j]->member()->GetTemp(itmpRHS);
      pu_    [j][i] = var_list_[j]->member()->GetExpCoeffs(iLevelFinal);
    }
    for ( j=0; j<irank_; j++ ) var_list_[j]->next();
  }
  
  switch (itype_) {
    case STOKES_SCHUR_DELP:
      bRet = SchurDelP(iLevelFinal);
      break;
    case STOKES_STEADY_UZAWA:
      bRet = Uzawa(iLevelFinal);
      break;
    default:
      bRet = FALSE;
      cout << "StokesSolver::Solve: invalid solver type" << endl;
      break;
  }

  for ( j=0; j<irank_; j++ ) var_list_[j]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<irank_; j++ ) {
      var_list_[j]->member()->TempUnlock(pu_rhs_[j][i]);
    }
    for ( j=0; j<irank_; j++ ) var_list_[j]->next();
  }
  
  return bRet;

} //end of operator Solve


//************************************************************************************
//************************************************************************************
// METHOD     : Uzawa
// DESCRIPTION: Solves steady Stokes problem using the following
//              Uzawa scheme:
//                  Sum_i D_i L^-1 G_i dp^(n+1) = -Sum D_i L^-1 M f _i
//                  v_i^(n+1) = L^-1 ( M f_i + G_i p^(n+1) );
//               where   H = pL and G = 1/rho * D^T. The
//               operator, Sum D H^-1 G is the  Uzawa operator
//
//               NOTE: it is expected that the RHS function, f, already
//                     has its terms properly multiplied by M prior to entry, 
//                     so that it does not have to be re-done.
// ARGUMENTS  : 
// RETURNS    :  none
//************************************************************************************
GBOOL StokesSolver::Uzawa(const GINT  iLevelFinal)
{
  GINT     i;

  return FALSE;

  // If there are preconditioners, set the timestep:
  UpdatePrecond(dt_);

  // Shift time levels:
  var_list_[1]->start(NULL); var_list_[2]->start(NULL); var_list_[0] ->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    var_list_[1]->member()->ShiftLevels();
    var_list_[2]->member()->ShiftLevels();
    var_list_[0]->member()->ShiftLevels();
    var_list_[1]->next(); var_list_[2]->next(); var_list_[0] ->next();
  }

  // Solve for P, by solving pseudo-Poisson eq:
  //      Sum_i D_i H^-1 G_i p = -Sum_i D_i H^-1 Mf_i
  cout << "StokesSolver::Uzawa: currently unsupported" << endl;  
  return FALSE;

} // end of method Uzawa


//************************************************************************************
//************************************************************************************
// METHOD     : SchurDelP
// DESCRIPTION: Solves steady Stokes problem using the following
//              pressure-correction spitting scheme:
//                  DM^-1 G dp^(n+1) = -c0/dt D H^-1(Mf + G p^n)
//                  H _v^(n+1) = M_ f + G p^n + dt/c0 H M^-1 G dp^(n+1) ;
//               where   H = pL + qM = nu L + c0/dt M
//               and G = 1/rho * D^T. 
//
//               NOTE: it is expected that the RHS function, f, already has
//                     appropriate terms multiplied by M_ prior to entry, so that 
//                     this does not have to be done here. In the Navier-Stokes case, 
//                     this RHS is comprised of the advection term, suitably approximated
//                     on the n+1-st timelevel, and the approximation for the
//                     time derivative, du_/dt.
// ARGUMENTS  : 
// RETURNS    :  none
//************************************************************************************
GBOOL StokesSolver::SchurDelP(const GINT  iLevelFinal)
{
  GINT       i, j;
  GDOUBLE    dtfi, dtf, dtri, dthi, fdt;
  GBOOL      bRet;
  GVector    *pn;
  GVector    *mask=NULL, *iM=NULL;
  Elem2D     *elem;

  dtri  = 1.0/dt_;
  dtfi = -c0_*rho_* dtri; 
  dtf  =  dt_ / ( c0_*rho_);
  fdt  =  c0_*dtri; 
  dthi = -c0_*dtri; 

  // If there are preconditioners, set the timestep:
  UpdatePrecond(dt_);
  ResetErrors();

  // Shift time levels:
  for ( j=0; j<=irank_; j++ ) var_list_[j]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    mask = (*uelems_)[i]->GetMask();
    // Check required time-dependent operators:
    var_list_[irank_]->member()->ShiftLevels();
    for ( j=0; j<irank_; j++ ) {
      var_list_[j]->member()->ShiftLevels();
      ((HelmholtzOp*)H_[i])->SetConst(nu_[0],nu_[1],fdt);
//    ((SchurLapOp*)E_  [i])->SetVBdyData(j+1,cg_[j]->GetBdyValues()[i]);
    }
//  E_  [i]->SetConst(dtf);  // set if we put the time const on the LHS
    ((SchurLapOp*)E_  [i])->SetMask(mask);
    for ( j=0; j<=irank_; j++ ) var_list_[j]->next();
  }
  // Stokes solver procedes by (1) solving for viscous contribution
  // in construction of the RHS for the pressure solve; (2) solving
  // for the pressure change, dP; (3) computing the divergence-free
  // (in the Stokes sense) solution based on the pressure correction

  // First, compute RHS's; remember that incoming RHS (pu_rhs) contains mass already.
  // To compute RHS for pressure solve, must solve system H x_i = Mf_i + G_i p^n:
  if ( !bPInit_ ) {
    var_list_[irank_]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      pn         = var_list_[irank_]->member()->GetExpCoeffs(0);
      for ( j=0; j<irank_; j++ ) {
        DT_[j][i]->OpVec_prod(*pn, *g_[j][i]);
        MTK::fvec_const_prod_sum_rep(*g_[j][i], *pu_rhs_[j][i], irho_, 1.0);   // M f_j + G_j p^n
      }
      var_list_[irank_]->next();
    }
    for ( j=0; j<irank_; j++ ) {
//    (*cg_[j])(uelems_, &H_, NULL, &(utmp_[j]), &(g_[j]), NULL);
      bRet          = cg_[j]->SolveCG();
      niter_    [j] = cg_[j]->GetNumIterations();
      ierror_   [j] = cg_[j]->ErrNo();
      error_    [j] = cg_[j]->GetError();
      min_error_[j] = cg_[j]->GetMinError();
      max_error_[j] = cg_[j]->GetMaxError();
      if ( !bRet ) {
        cout << "StokesSolver::SchurDelP: u[" << j << "]-Helmholtz solve failed" << endl;
        cout << "              CG iterations     : " << niter_    [j] << endl;
        cout << "              CG error condition: " << ierror_   [j] << endl;
        cout << "              CG min error      : " << min_error_[j] << endl;
        cout << "              CG max error      : " << max_error_[j] << endl;
        return FALSE;
      }
    }
  } else {
    for ( j=0; j<irank_; j++ ) var_list_[j]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {        // set u* = u
      for ( j=0; j<irank_; j++ ) {
        *utmp_[j][i] = *(var_list_[j]->member()->GetExpCoeffs(0));
        var_list_[j]->next();
      }
    }
  }

  // Compute RHS for pressure solve,
  // E (dP) = -gamma/dt Sum_i D_i * utmp_i: 
  for ( j=0; j<irank_; j++ ) D_[j].start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<irank_; j++ ) {
      D_[j].member()->OpVec_prod(*utmp_[j][i],*ptmp_[j][i]);
    }
    for ( j=1; j<irank_; j++ ) {
      MTK::fvec_add_rep(*ptmp_[0][i],*ptmp_[j][i]);
    }
    MTK::fvec_const_prod_rep(*ptmp_[0][i],dthi);
    for ( j=0; j<irank_; j++ ) D_[j].next();
  }

  // Solve for pressure correction, dP:
//(*cg_[irank_])(pelems_, &E_, NULL, &pp_, &ptmp_[0], NULL); // pp_ is, here, actually, del P = p^(n+1) - p^n
  bRet               = cg_[irank_]->SolveCG();
  niter_    [irank_] = cg_[irank_]->GetNumIterations();
  ierror_   [irank_] = cg_[irank_]->ErrNo();
  error_    [irank_] = cg_[irank_]->GetError();
  min_error_[irank_] = cg_[irank_]->GetMinError();
  max_error_[irank_] = cg_[irank_]->GetMaxError();
  if ( !bRet ) {
    cout << "StokesSolver::SchurDelP: pressure solve failed" << endl;
    cout << "              CG iterations     : " << niter_    [irank_] << endl;
    cout << "              CG error condition: " << ierror_   [irank_] << endl;
    cout << "              CG min error      : " << min_error_[irank_] << endl;
    cout << "              CG max error      : " << max_error_[irank_] << endl;
    return FALSE;
  }

//SmoothPressure(pp_);

  // Now, use dP to find new velocity; also compute new pressure: 
  // u_i_0 = H^-1 ( B f_i + G_i p -H_bar u_b_i)  + dt/gamma B^-1 G_i dP 
  //       = utmp + dt/(gamma*rho) B^-1 DT_i dP
  // Note: Dirichlet bdys must be included in the utmp solve, if the 
  //       following masking is to have any meaning...
  if ( !bPInit_ ) {
    var_list_[irank_] ->start(NULL);
    uelems_->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      elem = uelems_->member();
      mask = elem->GetMask();
      iM   = elem->GetiMass();
      for ( j=0; j<irank_; j++ ) {
        DT_[j][i]->OpVec_prod(*pp_[i], *g_[j][i]);
        MTK::fvec_point_prod_rep(*g_[j][i], *iM );
        MTK::fvec_const_prod_rep(*g_[j][i], dtf);
        MTK::fvec_point_prod_rep(*g_[j][i], *mask );
        MTK::fvec_add(*utmp_[j][i], *g_[j][i], *pu_[j][i]);
      }
      pn = var_list_[irank_]->member()->GetExpCoeffs(0);
      MTK::fvec_add_rep(*pn, *pp_[i]);  // new pressure, points to field
      var_list_[irank_] ->next();
      uelems_->next();
    }
    DoFilter();
  } else {
    var_list_[irank_] ->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      pn = var_list_[irank_]->member()->GetExpCoeffs(0);
      MTK::fvec_add_rep(*pn, *pp_[i]);  // new pressure, points to field
      var_list_[irank_] ->next();
    }
  }
   
  return TRUE;

} // end of method SchurDelP


//************************************************************************************
//************************************************************************************
// METHOD     : UpdatePrecond
// DESCRIPTION: Updates preconditioners in time, as required
// ARGUMENTS  : 
// RETURNS    :  none
//************************************************************************************
void StokesSolver::UpdatePrecond(GDOUBLE dt)
{
  GINT  i, j;
  GDOUBLE fdt;
  
  if ( dt == dtlast_ ) return;  // don't update if we don't need to

  dtlast_ = dt;

  fdt  =  c0_/dt ;
  for ( j=0; j<=irank_; j++ ) {
    if (  !bPrecond_ [j] ) continue;
    switch ( pc_type_[j] ) {
      case GPC_BLOCKJAC_HELM:
        for ( i=0; i<nelems_; i++ ) {
          ((PCBlockJac_Helm*)(pc_[j][i]))->SetConst(nu_[0], nu_[1], fdt);
        }
        break;
      case GPC_POINTJAC_HELM:
        // Note: update here if nu1, nu2 or dt don't change is dangerous!
        for ( i=0; i<nelems_; i++ ) {
          Hdiag_[i] = ((PCPointJac_Helm*)(pc_[j][i]))->SetConst(nu_[0], nu_[1], fdt);
        }
        // Must now do a direct stiffness summation (global gather/scatter) on
        // unassembled H-diagonal:
        if ( !gsop->DSOp(Hdiag_, G_OP_SUM, hDSOp) ) {
          cout << "NS::UpdatePrecond: Unable to assemble operator" << endl;
          exit(1);
        }
        for ( i=0; i<nelems_; i++ ) {
          if ( ((PCPointJac_Helm*)(pc_[j][i]))->Invert() == NULL ) {
            cout << "NS::UpdatePrecond: Unable to form preconditioner" << endl;
            exit(1);
          }
        }
        break;
      case GPC_BLOCKJAC_LAP:
        for ( i=0; i<nelems_; i++ ) {
          if ( pc_[j][i] != NULL )  ((PCBlockJac_PLap*)pc_[j][i])->SetConst(1.0);
        }
        break;
       case GPC_NONE:
         break;
      default:
        cout << "StokesSolver::UpdatePrecond: invalid preconditioner type" << endl ;
        exit(1);
    }
  }

#if 0
  if (  pc_[0] != NULL ) {
    switch ( pc_type_[0] ) {
      case GPC_BLOCKJAC_HELM:
        for ( i=0; i<nelems_; i++ ) {
          if ( pc_[0][i] != NULL )  ((PCBlockJac_Lap*)pc_[0][i])->SetConst(nu_[0], nu_[1], fdt);
        }
        break;
       case GPC_NONE:
         break;
      default:
        cout << "StokesSolver::UpdatePrecond: invalid preconditioner type" << endl ;
        exit(1);
    }
  }
#endif


} // end of method UpdatePrecond


//************************************************************************************
//************************************************************************************
// METHOD     : SetVBdyData
// DESCRIPTION: Set boundary condition data
// ARGUEMENTS : idir       : v-component whose bdy vals are being set
//              bdy_vals   : if present, gives the boundary values
// RETURNS    :
//************************************************************************************
void StokesSolver::SetVBdyData(GINT  idir, GVecList *bdy_vals)
{
  if ( idir < 1 || idir > irank_ ) {
    cout << "StokesSolver::SetVBdyData: invalid component direction" << endl;
    exit(1);
  }

  // Pressure doesn't have a boundary condition:
  if ( idir > irank_ ) return;

  ubdyvals_[idir-1] = bdy_vals;
  bInitReqd_ = TRUE;


} // end of method SetVBdyData


//************************************************************************************
//************************************************************************************
// METHOD     : ResetErrors
// DESCRIPTION: Resets error quantities
//
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void StokesSolver::ResetErrors()
{
  
  GINT  j;

  for ( j=0; j<=irank_; j++ ) {
    niter_    [j]      = -1; 
    ierror_   [j]      = -1; 
    error_    [j]      = 0.0;
    min_error_[j]      = 0.0;
    max_error_[j]      = 0.0;
  }

} // end of method ResetErrors


//************************************************************************************
//************************************************************************************
// METHOD     : UpdateSolver
// DESCRIPTION: updates solver due to grid changes
//
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
void StokesSolver::UpdateSolver()
{
  GINT     j;

  for ( j=0; j<irank_;  j++ ) {
    if ( bPrecond_[j] ) {
      (*cg_[j]).SetPreconditioner(&pc_[j]);
    }
    (*cg_[j])(uelems_, &H_, NULL, utmp_+j, g_+j, NULL);
    if ( !bPInit_ )
    cg_[j]->SetBdyValues(ubdyvals_[j]);
  }
  if ( bPrecond_[irank_] ) {
    (*cg_[irank_]).SetPreconditioner(&pc_[irank_]);
  }
  (*cg_[irank_])(pelems_, &E_, NULL, &pp_, &ptmp_[0], NULL); 
//cg_[irank_]->SetBdyValues(ubdyvals_[irank_]);

} // end of method UpdateSolver


//************************************************************************************
//************************************************************************************
// METHOD     : GetVisc
// DESCRIPTION: 
// ARGUMENTS  : idir: coordinate direction 
// RETURNS    :
//************************************************************************************
GDOUBLE StokesSolver::GetVisc(GINT idir)
{
  if ( idir <1 || idir > irank_ ) {
    cout << "StokesSolver::GetVisc: invalid coordinate direction" << endl;
    exit(1);
  }

  return nu_[idir-1];

} // end of method GetVisc


//************************************************************************************
//************************************************************************************
// METHOD     : GetStokesOp
// DESCRIPTION: 
// ARGUMENTS  : idir: coordinate direction 
// RETURNS    :
//************************************************************************************
GLinOpList *StokesSolver::GetStokesOp(GINT idir, GBOOL btranspose)
{
  char *serr = "StokesSolver::GetStokesOp: ";
  if ( idir <1 || idir > irank_ ) {
    cout << serr << "invalid coordinate direction" << endl;
    exit(1);
  } 
  
  if ( btranspose ) return &DT_[idir-1];
  
  return &D_[idir-1];
  
} // end of method GetStokesOp


//************************************************************************************
//************************************************************************************
// METHOD     : GetPseudoLap
// DESCRIPTION: 
// ARGUMENTS  : idir: coordinate direction 
// RETURNS    :
//************************************************************************************
GLinOpList *StokesSolver::GetPseudoLap()
{
  char *serr = "StokesSolver::GetPseudoLap: ";
  
  return &E_;
  
} // end of method GetPseudoLap


//************************************************************************************
//************************************************************************************
// METHOD     : InitPress
// DESCRIPTION: 
// ARGUMENTS  : bInit = TRUE or FALSE
// RETURNS    :
//************************************************************************************
void StokesSolver::InitPress(GBOOL bInit)
{
  char *serr = "StokesSolver::InitPress: ";
  
  bPInit_ = bInit;
  
} // end of method InitPress



//************************************************************************************
//************************************************************************************
// METHOD     : ResetExpandables
// DESCRIPTION: Set quantities that depend on the number and size of elements in 
//              field lists. The member data, nelems_, repres. the current
//              number of elements *is* set here.
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GBOOL StokesSolver::ResetExpandables()
{
  GINT     i, j, num, ncurr, pNN, vNN;
  Elem2D   *uelem, *pelem;

  num   = uelems_->size();
  H_    .empty();
  E_    .empty();
  Hdiag_.empty();
  for ( j=0; j<irank_; j++ ) {
    D_ [j] .empty();
    DT_[j] .empty();
  }
  ncurr = H_.size();
  pu_[irank_].empty();
  pp_        .empty();
  pc_[irank_].empty();
  for ( j=0; j<irank_; j++ ) {
    utmp_    [j].empty();
    ptmp_    [j].empty();
    g_       [j].empty();
    pu_      [j].empty();
    pc_      [j].empty();
    pu_rhs_  [j].empty();
  }

  // Add elements as required:
  for ( i=ncurr; i<num; i++ ) {
    uelem = (*uelems_)[i];
    pelem = (*pelems_)[i];
    vNN   = var_list_     [0]->member(i)->GetExpCoeffs(0)->dim();
    pNN   = var_list_[irank_]->member(i)->GetExpCoeffs(0)->dim();

    // Operators:
    H_     .add(NULL,TRUE); H_    [i] = new HelmholtzOp(uelem) ;
    E_     .add(NULL,TRUE); 
    if      ( itype_ == STOKES_SCHUR_DELP ) 
      E_[i]     =  new SchurLapOp(uelem,pelem); 
    else if ( itype_ == STOKES_STEADY_UZAWA ) 
      E_[i]     =  new UzawaLapOp(uelem,pelem); 
    pc_     [irank_].add(NULL,TRUE);
    for ( j=0; j<irank_; j++ ) {
      D_      [j].add(NULL,TRUE); D_ [j][i] = new StokesOp(uelem,pelem,j+1) ;
      DT_     [j].add(NULL,TRUE); DT_[j][i] = new StokesOp(uelem,pelem,j+1) ;
      ((StokesOp*)DT_     [j][i])->Transpose();
      pc_     [j].add(NULL,TRUE);
    }

    // Pointers
//  pp_.add(NULL,FALSE);
    for ( j=0; j<irank_; j++ ) {
      pu_     [j].add(NULL,FALSE);
      pu_rhs_ [j].add(NULL,FALSE);
    }
    Hdiag_.add(NULL,FALSE); // Hdiag_[i] = new GVector(NN);

    // Fully-instantiated data:
    pp_.add(NULL,TRUE); pp_[i] = new GVector(pNN);
    for ( j=0; j<irank_; j++ ) {
      utmp_   [j].add(NULL,TRUE); utmp_[j][i] = new GVector(vNN);
      ptmp_   [j].add(NULL,TRUE); ptmp_[j][i] = new GVector(pNN);
      g_      [j].add(NULL,TRUE); g_   [j][i] = new GVector(vNN);
    }
  }

  // Delete unnecessary elements from lists:
  for ( i=num; i<ncurr; i++ ) {
    H_     .del(i);
    E_     .del(i);
    Hdiag_ .del(i);
    pp_    .del(i);
    pc_[irank_].del(i);
    for ( j=0; j<irank_; j++ ) {
      D_      [j].del(i);
      DT_     [j].del(i);
      pc_     [j].del(i);
      pu_     [j].del(i);
      pu_rhs_ [j].del(i);
      utmp_   [j].del(i);
      ptmp_   [j].del(i);
      g_      [j].del(i);
    }
  }

  // Must now renumber/re-order the reference ids for lists:
  H_     .renumber();
  E_     .renumber();
  Hdiag_ .renumber();
  pp_    .renumber();
  pc_[irank_].renumber();
  for ( j=0; j<irank_; j++ ) {
    D_      [j].renumber();
    DT_     [j].renumber();
    pc_     [j].renumber();
    pu_     [j].renumber();
    pu_rhs_ [j].renumber();
    utmp_   [j].renumber();
    ptmp_   [j].renumber();
    g_      [j].renumber();
  }

  nelems_ = num;
  dtlast_ = 0.0;

  for ( j=0; j<=irank_; j++ ) {
    if  ( !InitPrecond(j) ) {
      cout << "BurgersSolver::ResetExpandables: InitPrecond failed" << endl;
      exit(1);
    }
  }
  bInitReqd_ = TRUE;
  return TRUE;

} // end of method ResetExpandables


//************************************************************************************
//************************************************************************************
// METHOD     : SetFilter
// DESCRIPTION: sets filter
//
// ARGUMENTS  : GLinOpList *
// RETURNS    : none.
//************************************************************************************
void StokesSolver::SetFilter(GLinOpList *filter)
{   
  filter_ = filter;
} // end of method SetFilter
    


//************************************************************************************
//************************************************************************************
// METHOD     : DoFilter
// DESCRIPTION: Carries out filter operation
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL StokesSolver::DoFilter()
{
  char    *serr = "StokesSolver::DoFilter: ";
  GINT    i, j;
  GVector *vtmp=NULL;


  if ( filter_ == NULL ) return TRUE;

  for ( j=0; j<GDIM; j++ ) var_list_[j]->start(NULL); uelems_->start(NULL); filter_->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    vtmp = uelems_->member()->GetTemp(0);
//  if ( !uelems_->member()->GetTemp(vtmp) ) {
//    cout << serr << "temp vector unavailable" << endl;
//    return FALSE;
//  }
    for ( j=0; j<GDIM; j++ ) {
      *vtmp = *(var_list_[j]->member()->GetExpCoeffs(0));
      filter_->member()->OpVec_prod(*vtmp,*var_list_[j]->member()->GetExpCoeffs(0));
    }
    uelems_->member()->TempUnlock(vtmp);
    for ( j=0; j<GDIM; j++ ) var_list_[j]->next(); uelems_->next(); filter_->next();
  }

  return TRUE;
} // end of method DoFilter

