//************************************************************************************//
// Module       : stokes_solver.hpp
// Date         : 8/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the solver for the Stokes problem. The problem solved is
//                  H _v - G p = f,
//                subject to the constraint that
//                  Div _v = 0;
//                Here, H is the Helmholtz operator,
//                  H =  p L  + q M
//                where p=nu and q=c0/dt are constants, L is the discrete (weak)
//                Laplacian operator, and M is the mass matrix. _v is the
//                velocity vector, and p is the pressure field and G is the
//                weak Stokes gradient operator; updated  values of _v and p are 
//                the result. 
//
//                This class assumes that the pressure and velocity reside on 
//                different grids within the same element, using the P_N-P_(n-2)
//                formalism, in which the pressure grid lies on Gaussian nodes.
// Derived From : none.
// Modifications:
//************************************************************************************//
#if !defined(STOKES_SOLVER_HPP)
#define STOKES_SOLVER_HPP 

#include "gs.hpp"
#include "cg.hpp"
#include "gttlist.hpp"
#include "gfieldlist.hpp"
#include "stokesop.hpp"
#include "schurlapop.hpp"
#include "uzawalapop.hpp"
#include "helmholtzop.hpp"
#include "pcblockjac_helm.hpp"
#include "ntree_adapt.hpp"

enum STOKES_TYPE {STOKES_SCHUR_DELP, STOKES_STEADY_UZAWA};
extern char *sStokesType[];
class StokesSolver
{
public:
                            StokesSolver(GFieldList *ulist[], GINT nf, GElemList *uelems, GElemList *pelems, STOKES_TYPE t);
                           ~StokesSolver();

         void               SetComm(GS *ggs);                                                // Set global gather scatter operator
         GCHandle           SetCommHandle(GCHandle hIn) ;                                    // Sets active gather/scatter handle 
         GBOOL              SetPreconditioner(GINT  iwhich, GPC itype);                      // Set v-solve preconditioners
         void               SetVBdyData(GINT  idir, GVecList *bdy_vals);                     // Set bdy condition data
         void               SetDen(const GDOUBLE rho);                                       // Set const. density, rho
         void               SetVisc(const GDOUBLE nu);                                       // Set kinemtaic viscosity, nu
         void               SetVisc(const GDOUBLE nu, GINT idir);                            // Set kinemtaic viscosity, nu
         void               Setc0  (const GDOUBLE c0);                                       // Helmholtz op st H = nuL + c0/dt M 
         void               SetFilter(GLinOpList *filter);                                   // Set filter object
         void               InitPress(GBOOL bInit);                                          // Set flag to initialize P
         CG                 *GetSolver(GINT iwhich);                                         // Get Vel. solver for component idir
         GINT                GetNumIterations(GINT iwhich);                                  // Get num U_idir iterations
         GINT                GetErrorType(GINT  iwhich);                                     // Get U_idir error condition
         GDOUBLE             GetError(GINT iwhich);                                          // Get U_idir error magnitude
         GDOUBLE             GetMinError(GINT iwhich);                                       // Get min U_idir error
         GDOUBLE             GetMaxError(GINT iwhich);                                       // Get max U_idir error
         GDOUBLE             GetDivMax(GINT  iLevel);                                        // Get max(divergence V ) for tim

         GLinOpList         *GetPreconditioner(GINT  iwhich);                                // Get u-solve precond for comp. idir
         GDOUBLE             GetDivMax(GVector *vn[], GINT nv, GINT  ie);                    // Get extremal(Div vn) on element ie
         GDOUBLE             GetVisc(GINT  idir);                                            // Get viscosity in direction idir

         GLinOpList         *GetStokesOp(GINT idir, GBOOL btranspose=FALSE);                 // Get Stokes operators
         GLinOpList         *GetPseudoLap();                                                 // Get pseudo Laplacian operator


         void                DivV(GFieldList *divv, GFieldList *v[], GINT nv );              // Compute Div.V 
         GBOOL               Solve(GDOUBLE dt, 
                                   const GINT  itmpRHS, const GINT  iLevelFinal);            // Solve with RHS itmpRHS within var_list
         GBOOL               ResetExpandables();                                             // Reset expandable data after mesh change
         void                UpdateSolver();                                                 // Update solvers, if any due to mesh change

private:
         // Methods:
         void              Init();
         GBOOL             InitPrecond(const GINT  iwhich);
         GBOOL             Classical(const GINT  iStoreLevel);
         GBOOL             Uzawa(const GINT  iStoreLevel);
         GBOOL             SchurDelP(const GINT  iStoreLevel);
         GBOOL             ComputeBdyVectors();
         GBOOL             DoFilter();
         void              UpdatePrecond(GDOUBLE dt);
         void              DeleteDynamic();
         void              ResetErrors();

         // Field and other data:
         STOKES_TYPE       itype_;
         GPC               pc_type_   [GDIM+1];
         GINT              irank_;
         GINT              nelems_;
         GBOOL             bInitReqd_;
         GBOOL             bPrecond_   [GDIM+1];
         GBOOL             bPInit_;
         GDOUBLE           dt_;
         GDOUBLE           dtlast_;
         GDOUBLE           c0_;
         GDOUBLE           nu_         [GDIM];
         GDOUBLE           rho_;
         GDOUBLE           irho_;
         GFieldList        *var_list_  [GDIM+1];
         GVecList          pu_         [GDIM];
         GVecList          Hdiag_;
         GVecList          utmp_       [GDIM];
         GVecList          ptmp_       [GDIM];
         GVecList          g_          [GDIM];
         GVecList          pu_rhs_    [GDIM];
         GVecList          pp_;
         GVecList          *ubdyvals_  [GDIM];
         CG                *cg_        [GDIM+1];
         GCHandle          hDSOp;
         GElemList         *uelems_;
         GElemList         *pelems_;

         // Math operators
         GLinOpList      H_       ;   // Helmholtz operator acting on v_
         GLinOpList      D_ [GDIM];   // i-derivative operator
         GLinOpList      DT_[GDIM];   // (i-derivative operator) ^T
         GLinOpList      E_;          // Laplacian op. arising from Uzawa or Schur decomposition 
         GLinOpList      pc_[GDIM+1]; // p, and v-preconditioners
         GLinOpList      *filter_;    // filter operator
 
         // Solver return vals:
         GINT            niter_    [GDIM+1];
         GINT            ierror_   [GDIM+1];
         GDOUBLE         error_    [GDIM+1];
         GDOUBLE         min_error_[GDIM+1];
         GDOUBLE         max_error_[GDIM+1];

         // Communication:
         GS              *gsop;       // communication operator

};

#endif
