//************************************************************************************//
// Module       : cg.cpp
// Date         : 9/14/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a conjugate-gradient object.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include <math.h>
#include "cg.hpp"
#include "gcomm.hpp"
#include "mtk.hpp"
#include "glop.hpp"
#include "iconn_amr.hpp"
#include "gutils.hpp"

char *s_cgerror_[] = {
  "Normal",
  "NULL ops",
  "Incompatible list size",
  "Bad iteration number",
  "Solution not converging",
  "Solve-Op/Vec",
  };
                     

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
CG::CG(CGType itype, GBOOL isNested)
:
MaxIterations       (512),
nResidualChk        (50),
numIterations       (-1),
nIntermedProd       (0),
iErrType            (CGERR_NONE),
nTotalNodes         (0),
bad_iter_number     (10),
nops                (0),
nprecon             (0),
bQuiet              (TRUE),
bNested             (isNested),
iSolveType          (itype),
tolerance           (1.0e-8),
minErr              (-1.0), 
maxErr              (-1.0), 
finErr              (-1.0),
glob_ids            (NULL),
hDSOp         (NULL_HANDLE),
hDSOp_Intermed(NULL_HANDLE),
gsop                (NULL),
gelems_             (NULL),
A                   (NULL),
x                   (NULL),
b                   (NULL),
cg_intermed         (NULL)
{
  EucResidual_.Resize(MaxIterations);
  L2Residual_ .Resize(MaxIterations);
} // end of constructor (1) method


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
CG::CG(GElemList *e, GLinOpList *AA, GLinOpList *inROp, GVecList *xx, GVecList *bb, GS *comm=NULL, CGType itype, GBOOL isNested)
:
MaxIterations       (512),
nResidualChk        (50),
numIterations       (-1),
nIntermedProd       (0),
iErrType            (CGERR_NONE),
nTotalNodes         (0),
bad_iter_number     (10),
nops                (0),
nprecon            (0),
bQuiet              (TRUE),
bNested             (isNested),
iSolveType          (itype),
tolerance           (1.0e-6),
minErr              (-1.0),
maxErr              (-1.0), 
finErr              (-1.0),
glob_ids            (NULL),
hDSOp         (NULL_HANDLE),
hDSOp_Intermed(NULL_HANDLE),
gsop                (comm),
gelems_             (e),
A                   (NULL),
x                   (NULL),
b                   (NULL),
cg_intermed         (NULL)
{
  GINT  i;

  if ( gelems_ == NULL ) {
    cout << "CG::CG: NULL element list not allowed" << endl;
    exit(1);
  }
  ResetExpandables(AA, inROp);
  EucResidual_.Resize(MaxIterations);
  L2Residual_ .Resize(MaxIterations);

} // end of constructor (2) method


//************************************************************************************
//************************************************************************************
// Destructor
CG::~CG()
{
  GINT  i;

  if ( cg_intermed != NULL && A ) {
    for ( i=0; i<nIntermedProd; i++ ) {
      delete cg_intermed[i];
    }
    delete [] cg_intermed; cg_intermed = NULL;
  }
}


//************************************************************************************
//************************************************************************************
// Copy constructor method
CG::CG(const CG &a)
{
} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// METHOD     : =
// DESCRIPTION: Assgnment operator
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::operator=(const CG &elem)
{
} // end of = operator


//************************************************************************************
//************************************************************************************
// METHOD     : ()
// DESCRIPTION: Method sets the operators
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::operator()(GElemList *e, GLinOpList *in_A, GLinOpList *in_ROp, GVecList *in_x, GVecList *in_b, GS *in_comm)
{  
  gelems_= e;

  if ( gelems_ == NULL ) {
    cout << "CG::operator(): NULL element list not allowed" << endl;
    exit(1);
  }
  A      = in_A;
  x      = in_x;
  b      = in_b;
  ROp    = in_ROp;
  ResetExpandables(in_A, in_ROp);

  if ( in_comm != NULL ) {  // so that SetComm makes sense:
    gsop = in_comm;
  }

} // end of operator () 


//************************************************************************************
//************************************************************************************
// METHOD     : SetNoDSS
// DESCRIPTION: Method sets contrib to RHS which should _NOT_ be DSS'd
// ARGUEMENTS : 
// RETURNS    :
//************************************************************************************
void CG::SetNoDSS(GVecList *rb)
{
  GINT  i;

  if ( rb == NULL ) return;

  for ( i=0; i<bnodss.size(); i++ ) {
    bnodss.del(i);
  }
  for ( i=0; i<rb->size(); i++ ) {
    bnodss.add((*rb)[i],FALSE);
  }
} // end of SetNoDSS


//************************************************************************************
//************************************************************************************
// METHOD     : Quiet
// DESCRIPTION: Method sets bQuiet flag
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::Quiet(GBOOL b)
{  
  bQuiet = b;
} // end of Quiet


//************************************************************************************
//************************************************************************************
// METHOD     : SolveCG
// DESCRIPTION: Public method to solve systems using one of the CG algorithms
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GBOOL CG::SolveCG()
{

  if ( iSolveType == CG_STANDARD )
    iErrType = SolveStandard();
  else if ( iSolveType == CG_REDUCED_VAR1 ||
            iSolveType == CG_REDUCED_VAR2  )
    iErrType = SolveReduced(iSolveType);
  else
    return FALSE;

  if ( iErrType != CGERR_NONE ) {
    cout << "CG::SolveCG: error[" << iErrType << "]: " << s_cgerror_[iErrType] << endl;
    return FALSE;
  }

  return TRUE;

} //end of method SolveCG


//************************************************************************************
//************************************************************************************
// METHOD     : SolveStandard
// DESCRIPTION: Method to solve system using standard CG. This form of this algorithm
//              was adapted from P. Fischer's code, which was more efficient than
//              the one that was used previously.
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GINT  CG::SolveStandard()
{
  GINT       i, j, n, nbad=0;
  GDOUBLE    esq0, wtq1, alpha, alpham, beta, dot_local, err;
  GDOUBLE    rho0, rho1;

  if ( A == NULL || b == NULL || x == NULL ) {
    return CGERR_NULLOP;
  }
  if ( precon.size() && nprecon != nops ) {
    return CGERR_LISTSZERR;
  }

  if ( MaxIterations <= 0 || nResidualChk <=0 ) {
    return CGERR_BADITERNUM;
  }
  EucResidual_=0.0;
  L2Residual_ =0.0;

  vk = 0.0;
  if ( xb.size() ) vk = xb;
#if defined(CG_DEBUG_OUTPUT)
  cout << endl << "CG::SolveSt: Projecting ub..." << endl;
#endif
  GUtils::H1_Project(vk, *gelems_, (NTreeAdapt*)gsop, hDSOp);
#if defined(CG_DEBUG_OUTPUT)
  cout << endl << "CG::SolveSt: Projecting ub done." << endl;
#endif
  *x = 0.0;
  sk = 0.0;
  wk = 0.0;
  for ( i=0; i<nops; i++ ) {
    if ( ROp!=NULL && (*ROp)[i]!=NULL )  (*ROp)[i]->OpVec_prod(*(*b)[i], *rk[i]);
    else                                 *rk[i] = *(*b)[i];
//#if defined(CG_DEBUG_OUTPUT)
#if 0
    cout << "CG::SolveSt:" << " f   [" << i << "]=" << *rk[i] << endl;
    cout << "CG::SolveSt:" << " ub  [" << i << "]=" << *xb[i] << endl;
    cout << "CG::SolveSt:" << " ub_p[" << i << "]=" << *vk[i] << endl;
    cout << "CG::SolveSt:" << "mask [" << i << "]=" << *((*gelems_)[i]->GetMask()) << endl;
#endif
  }
  if ( xb.size() && !OpVec_prod(vk, sk) ) {
    return CGERR_SOLVE;
  }
  for ( i=0; i<nops; i++ ) {
    MTK::fvec_sub_rep(*rk[i], *sk[i]);
#if defined(CG_DEBUG_OUTPUT)
    cout << "CG::SolveSt:" << " (f - Hub)[" << i << "]=" << *rk[i] << endl;
#endif
  }
#if defined(CG_DEBUG_OUTPUT)
  cout << endl << "CG::SolveSt: DSS'ing initial rk..." << endl;
#endif
  if ( gsop && !gsop->DSOp(rk, G_OP_SUM, hDSOp) ) {
    return CGERR_SOLVE;
  }
  if ( gsop ) ((NTreeAdapt*)gsop)->PreMask(FALSE);
#if defined(CG_DEBUG_OUTPUT)
   cout << "CG::SolveSt: end, DSS'ing initial rk." << endl << endl;
#endif
  
  rho1   = 1.0;
  esq0   = 1.0;
  err    = SEHUGE;   
  minErr = SEHUGE;
  maxErr = TINY;
  n      =  0;
  while ( (n < MaxIterations) && (err > tolerance) ) {   // Iteration loop...
    EucResidual_[n] = GUtils::ComputeEucNorm(rk, *gelems_);
    L2Residual_ [n] = GUtils::ComputeL2Norm (rk, *gelems_);
    if ( precon.size() ) {
      for ( i=0; i<nops; i++ ) precon[i]->OpVec_prod(*rk[i], *zk[i]);
      GUtils::H1_ProjectM(zk, *gelems_, (NTreeAdapt*)gsop, hDSOp);
    }
    else {
      zk = rk;
#if defined(CG_DEBUG_OUTPUT)
      cout << endl << "CG::SolveSt: projecting zk=rk:..." << endl;
#endif
      GUtils::H1_ProjectM(zk, *gelems_, (NTreeAdapt*)gsop, hDSOp);
#if defined(CG_DEBUG_OUTPUT)
      cout << "CG::SolveSt: end, projecting zk=rk." << endl << endl;
#endif
    }
#if defined(CG_DEBUG_OUTPUT)
    for ( i=0; i<nops; i++ ) {
      cout << "CG::SolveSt: iter=" << n << " rk0[" << i << "]=" << *rk[i] << endl;
      cout << "CG::SolveSt: iter=" << n << " zk0[" << i << "]=" << *zk[i] << endl;
      cout << "CG::SolveSt: iter=" << n << " wk0[" << i << "]=" << *wk[i] << endl;
    }
#endif
    rho0 = rho1;
    GUtils::DoDotProducts(*gelems_, rk, &zk, &dot_local, &rho1, 1);
    beta = n == 0 ? 0.0 : rho1 / rho0;
    esq0 = n == 0 ? rho1 : esq0;
#if defined(CG_DEBUG_OUTPUT)
    cout << "CG:SolveStand: iter=" << n << " rho0=" << rho0 << " rho1=" << wtq1 
     << " beta=" << beta  << endl;
#endif
    if ( esq0 == 0.0 ) {
      err = 0.0;
      break;
    }
    for ( i=0; i<nops; i++ ) {
      MTK::fvec_const_prod_sum_rep(*wk[i], *zk[i], beta, 1.0);
#if defined(CG_DEBUG_OUTPUT)
      cout << "CG::SolveSt: iter=" << n << " rk1[" << i << "]=" << *rk[i] << endl;
      cout << "CG::SolveSt: iter=" << n << " zk1[" << i << "]=" << *zk[i] << endl;
      cout << "CG::SolveSt: iter=" << n << " wk1[" << i << "]=" << *wk[i] << endl;
      cout << "CG::SolveSt: iter=" << n << " qk1[" << i << "]=" << *qk[i] << endl;
#endif
    }
  
    if ( !OpVec_prod(wk, qk) ) return CGERR_SOLVE;

#if defined(CG_DEBUG_OUTPUT)
    cout << endl << "CG::SolveSt: DSS'ing qk..." << endl;
#endif
    if ( gsop ) ((NTreeAdapt*)gsop)->PreMask(TRUE);
    if ( gsop && !gsop->DSOp(qk, G_OP_SUM, hDSOp) ) {
      return CGERR_SOLVE;
    }
    if ( gsop ) ((NTreeAdapt*)gsop)->PreMask(FALSE);
#if defined(CG_DEBUG_OUTPUT)
    cout << "CG::SolveSt: end, DSS of qk." << endl << endl;
#endif

    GUtils::DoDotProducts(*gelems_, wk, &qk, &dot_local, &wtq1, 1);
    alpha  = rho1 / wtq1;
    alpham = -alpha;
#if defined(CG_DEBUG_OUTPUT)
      cout << "CG::SolveSt: iter=" << n << " rho1=" << rho1 << " wtq1=" << wtq1 << endl;
#endif

    for ( i=0; i<nops; i++ ) {
      MTK::fvec_const_prod_sum_rep(*((*x)[i]), *wk[i], 1.0, alpha );
      MTK::fvec_const_prod_sum_rep(*rk    [i], *qk[i], 1.0, alpham);
//    *(*x)[i]  = *(*x)[i] + ( (*wk[i]) * alpha  );
//    *rk  [i]  = *rk  [i] + ( (*qk[i]) * alpham );
#if defined(CG_DEBUG_OUTPUT)
      cout << "CG::SolveSt: iter=" << n << " rk3[" << i << "]=" << *rk[i] << endl;
      cout << "CG::SolveSt: iter=" << n << " xk3[" << i << "]=" << *((*x)[i]) << endl;
#endif
    }
    n++;
//  err     = sqrt(fabs(wtq1)) / sqrt(fabs(esq0));
    err     = sqrt(fabs(rho1/esq0));
#if defined(CG_DEBUG_OUTPUT)
    cout << "CG:SolveStand: iter=" << n << " err=" << err << endl;
#endif
    minErr  = MIN(err,minErr);
    maxErr  = MAX(err,maxErr);
    nbad = (err>1.0 ? nbad+1 : nbad);
#if defined(CG_DEBUG_OUTPUT)
    cout << "CG::SolveStandard: iteration number completed: " << n << endl;
#endif
  }  // end main iteration loop 

  finErr = err;

  // add in inhomogeneous solution
  for ( i=0; i<nops && xb.size(); i++ ) {
//  MTK::fvec_point_prod_rep(*(*x)[i], *((*gelems_)[i]->GetMask()));
    MTK::fvec_add_rep(*(*x)[i], *xb[i]);
  }
#if defined(CG_DEBUG_OUTPUT)
  cout << "CG:SolveSt: before final project..." << endl;
  for ( i=0; i<nops; i++ ) {
    cout << "x[" << i << "]=" << *(*x)[i] << endl;
  }
#endif
#if defined(CG_DEBUG_OUTPUT)
  cout << "CG:SolveSt: final project..." << endl;
#endif
  GUtils::H1_Project(*x, *gelems_, (NTreeAdapt*)gsop, hDSOp);
#if defined(CG_DEBUG_OUTPUT)
  cout << "CG:SolveSt: final project completed." << endl;
#endif
#if defined(CG_DEBUG_OUTPUT)
  cout << "CG:SolveSt: after final project..." << endl;
  for ( i=0; i<nops; i++ ) {
    cout << "x[" << i << "]=" << *(*x)[i] << endl;
  }
#endif

  numIterations = n;
  if ( n >= MaxIterations && err > tolerance ) return CGERR_CONVERGE;
  
#if 0
  GIBuffer   *bi;
  Elem2D     *elem;
  GVector    vb;

  cout << "_____________________________________________________________________________" << endl;
  for ( i=0; i<nops; i++ ) {
    elem= (*gelems_)[i];
    bi  = elem->GetBdyIndices();
    (*x)[i]->GetSection(bi->Data(),bi->dim(),vb);
    cout << "CG::SolveStandard: u_on_bdy_2 [" << i << "]=" << vb << endl;
    xb[i]->GetSection(bi->Data(),bi->dim(),vb);
    cout << "CG::SolveStandard: u_bdy_cond [" << i << "]=" << vb << endl;
//  elem->GetMask()->GetSection(bi->Data(),bi->dim(),vb);
//  cout << "CG::SolveStandard: mask        [" << i << "]=" << vb << endl;
  }
  cout << "_____________________________________________________________________________" << endl;
#endif

  return CGERR_NONE;
} // end of function SolveStandard


//************************************************************************************
//************************************************************************************
// METHOD     : OpVec_prod
// DESCRIPTION: Method to carry out operator-vector products
//              yy = A * xx
// ARGUEMENTS : xx: operand vector pointer array, of length nops 
//              yy: result vector pointer array, of length nops
// RETURNS    : TRUE on success.; else FALSE
//************************************************************************************
GBOOL CG::OpVec_prod(GVecList &xx, GVecList &yy)
{
  GINT     i, j ;

  if ( A->size() == 0 ) return FALSE;

  if ( !bNested ) {
    for ( i=0; i<nops && yy.size() && xx.size(); i++ ) {
      (*A)[i]->OpVec_prod(*xx[i], *yy[i]);
    }
    return TRUE;
  } 

  // If there is an intermediate product, then must solve
  // a linear system Intermed_op y = Intermed_Product, before computing full product 
  // Req'd quantities should have been set up prior to entry.
  // Note that the intermediate solver must be instantiated as
  // _not_ being nested, else there will be a recursion error.

  // The LinOp interface is s.t. the operator to be solved
  // is, e.g.,  Sum_i D_i L^-1 D^T_i.
  // This requires products of this operator and a vector, xx. 
  // So the intermed product is D^T_i * xx, for each i, in [0, nIntermedProd-1].
  // We need to solve the intermed lin. system
  //   L y = D^T_i * xx, for y, and then multiply y by
  // D_i, for each i in order to arrive at the final operator-vector product...

  for ( i=0; i<nops; i++ ) {
    *yy[i] = 0.0;
  }
  for ( j=0; j<nIntermedProd; j++ ) {
    if ( cg_intermed == NULL || cg_intermed[j] == NULL ) return FALSE;

    // Compute intermediate products:
    for ( i=0; i<nops; i++ ) {
      intermed_vecp[i] = (*A)[i]->GetIntermedProd(j);
    }

    // Solve intermediate linear system:
    (*cg_intermed[j])(gelems_, &intermed_op, NULL, &intermed_soln, &intermed_vecp, NULL);
    if ( !cg_intermed[j]->SolveCG() ) return FALSE;

    // Use intermed. solution to compute final product:
    for ( i=0; i<nops; i++ ) {
      (*A)[i]->OpVec_prod(*intermed_soln[i],*ytmp[i]);
      MTK::fvec_add_rep(*yy[i], *ytmp[i]);
    }
  }

  return TRUE;
} // end of function OpVec_prod


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : DSS_Intermediate
// DESCRIPTION: Method to compute the DSS'd interpemediate product of
//              an operator, whose intermediate product must be DSS'd.
//              For example, the pseudo Laplacian,
//              E = Sum_i D_i [B^-1 G_i dp] 
//              must be computed _after_ DSSumming the intermediate
//              local product, 
//                B_L^-1 [(G_i) dp]_L
//
// ARGUEMENTS : in_A : vector of length nops of LinOp operator pointers
//                     The operators must have a method that computes
//                     intermediate products, called IntermedOpVec_prod.
//              in_x : input vectors [GVector * array of length nops]
//              out_x: output vectors [GVector * array of length nops],
//                     values are overwritten by DSS'd product
// RETURNS    :
//************************************************************************************
GBOOL CG::DSS_Intermediate(GLinOpList &in_A, GVecList &in_x, GVecList &out_x)
{
  GINT  i, j;

  if ( in_x  == NULL ) return TRUE;
  if ( in_A  == NULL ) return FALSE;
  if ( out_x == NULL ) return FALSE;
  if ( gsop  == NULL ) return FALSE;
  for ( j=0; j<in_A[0]->NumIntermedProds(); j++ ) {
    // for each term of intermediate product, do a DSSum:
    // Note: all elements must provide same number of 
    // intermediate products
    for ( i=0; i<nops; i++ ) {
      vk[i] = in_A[i]->Intermediate_OpVec_prod(*in_x[i], j+1); // in_x must be non-NULL
    }

#if 1
    // ...Recall that after doing a DSSum, the result is 
    // scattered to each element/operator (contained in data
    // pointer, vk)
    if ( !gsop->DSOp(vk, G_OP_SUM, hDSOp_Intermed) ) { 
      cout << "CG::DSS_Intermediate: Intermediate DSS failed" << endl;
      return FALSE;
    }
#endif
#if 0
    for ( i=0; i<nops; i++ ) {
      if ( j == 1 )
      cout << "CG::DSS_Intermediate: vk[" << i << "]=" << *vk[i] << endl; 
    }
#endif
  }

  // Now, use intermediate producs to do full operator-vector product:
  for ( i=0; i<nops; i++ ) {
    if ( mask_Intermed[i]  != NULL ) mask_Intermed[i]->OpVec_prod(*vk[i], *vk[i]);
    in_A[i]->OpVec_prod(*out_x[i]); 
 
  }
   
  return TRUE;

} // end of method DSS_Intermediate
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : SetIntermedMask
// DESCRIPTION: Method to set mask for intermediate products
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::SetIntermedMask(GLinOpList *im)
{
  GINT  i;

  if ( im == NULL ) return;

  for ( i=0; i<mask_Intermed.size(); i++ ) {
    mask_Intermed.del(i);
  }
  for ( i=0; i<im->size(); i++ ) {
    mask_Intermed.add((*im)[i],FALSE);
  }
  
} // end of method SetIntermedMask


//************************************************************************************
//************************************************************************************
// METHOD     : SolveReduced
// DESCRIPTION: Method to solve system using one of the reduced
//              methods, which are derived from the d'Azevedo et. al. paper
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GINT  CG::SolveReduced(CGType itype)
{

#if 0
  if ( A == NULL || b == NULL || x == NULL ) return CGERR_NULLOP ;
//if ( A->GetIndex(1)->szglobal() != A->GetIndex(2)->szglobal() ||
//     A->GetIndex(2)->szglobal() != b->GetIndex ()->szglobal() ||
//     A->GetIndex(1)->szglobal() != x->GetIndex ()->szglobal() ) return CGERR_LISTSZERR;
  if ( MaxIterations <= 0 || nResidualChk <=0 ) return CGERR_BADITERNUM;

  GINT    n, NNg, n_blk_size[2];
  CGERRNO iRet=CGERR_NONE;
  GDOUBLE   gam0, gamkm1, gam, alpha, beta, eps,  err;
  GDOUBLE   sig, eta, Reductions[3];
  GVector *aVec[3];

  NNg = b->dim();

  
  n       =  0;
  *rk     = (*mask)*DSOp( (*ROp)*(*b) ); 
  *zk     = (*precon)*(*rk); 
  *wk     = *zk;
  *qk     = (*mask)*DSOp( (*A) * (*wk) ); 
  gam     = gamkm1 = (*zk)*(*rk); //pow(rk->EuclidNorm(),2.0) ;
  gam0    = gam;
  sig     = (*wk) * (*qk);
  *x      = (*wk) * (gam/(sig+TINY));
//*rk     = (*b) - (*A)*(*x);
  *rk     = (*mask)*DSOp((*ROp)*(*b) - (*A)*(*x) );
  if ( xb.size() ) *rk -= (*A)(*xb);
  err     = gam / (gam0 +TINY);
  minErr = SEHUGE;
  maxErr = TINY;
  if ( itype == CG_REDUCED_VAR1 ) {
    aVec[0] = zk;
    aVec[1] = msk;
    while ( n < MaxIterations && err > tolerance ) {
       *zk   = (*precon) * (*rk);
       *sk   = (*mask)*DSOp( (*A)*(*zk) );
       *msk  = (*precon) * (*sk);
       if ( !GUtils::DoDotProducts(*gelems_, rk, aVec, Reductions, 2) ) {
         iRet = 5;
         break;
       }
       gam     = Reductions[0];
       eta     = Reductions[1];
       beta    = gam / (gamkm1 + TINY);
       eps     = -beta * sig;
       *wk     = (*zk) + (*wk) * beta;
       *qk     = (*sk) + (*qk) * beta;
       sig     = eta + beta*eps;
       alpha   = gam / (sig + TINY);
       *x      = (*x ) + (*wk) * alpha;
       *rk     = (*rk) - (*qk) * alpha;

//     cout << "****> SolveCG:: n=" << n << " alpha=" << alpha << " q=" << *q << " A=" << *A << " d=" << *d << endl << endl;

       n++;
       err     = sqrt(gam) / sqrt(gam0 +TINY);
       minErr  = MIN(err,minErr);
       maxErr  = MAX(err,maxErr);
       gamkm1  = gam;
    }
    numIterations = n;
    if ( n >= MaxIterations && err > tolerance ) iRet = 4;
  }
  else if  ( itype == CG_REDUCED_VAR2 ) {
    aVec[0] = zk;
    aVec[1] = msk;
    aVec[2] = mqk;
    while ( n < MaxIterations && err > tolerance ) {
       *zk   = (*precon) * (*rk);
       *sk   = (*mask)*DSOp( (*A)*(*zk) );
       *msk  = (*precon) * (*sk);
       *mqk  = (*precon) * (*qk);
       if ( !GUtils::DoDotProducts(*gelems_, rk, aVec, Reductions, 3) ) {
         iRet = 5;
         break;
       }
       gam     = Reductions[0];
       eta     = Reductions[1];
       eps     = Reductions[2];
       beta    = gam / (gamkm1 + TINY);
       *wk     = (*zk) + (*wk) * beta;
       *qk     = (*sk) + (*qk) * beta;
       sig     = eta + beta*eps;
       alpha   = gam / (sig + TINY);
       *x      = (*x ) + (*wk) * alpha;
       *rk     = (*rk) - (*qk) * alpha;

//     cout << "****> SolveCG:: n=" << n << " alpha=" << alpha << " q=" << *q << " A=" << *A << " d=" << *d << endl << endl;

       n++;
       err     = sqrt(gam) / sqrt(gam0 +TINY);
       minErr  = MIN(err,minErr);
       maxErr  = MAX(err,maxErr);
       gamkm1  = gam;
    }
    numIterations = n;
    if ( n >= MaxIterations && err > tolerance ) iRet = 4;
  }
  else 
  {
     iRet = 6;
  }


  return iRet;
#endif
  return CGERR_NONE;

} // end of method SolveReduced

//************************************************************************************
//************************************************************************************
// METHOD     : SetTolerance
// DESCRIPTION: Sets CG tolerance
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::SetTolerance(GDOUBLE etol)
{
  tolerance = etol;
} // end of method SetTolerance


//************************************************************************************
//************************************************************************************
// METHOD     : SetMaxIterations
// DESCRIPTION: Sets max number of iterations
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::SetMaxIterations(GINT  imax)
{
  MaxIterations = imax;
  EucResidual_.Resize(MaxIterations);
  L2Residual_ .Resize(MaxIterations);
} // end of method  SetMaxIterations


//************************************************************************************
//************************************************************************************
// METHOD     : SetResidualChkNum
// DESCRIPTION: Sets number of iterations before a residual check is performed
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::SetResidualChkNum(GINT  imax)
{
  nResidualChk = imax;
} // end of method  SetMaxIterations


//************************************************************************************
//************************************************************************************
// METHOD     : SetSolveType
// DESCRIPTION: Sets solve method thype
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::SetSolveType(CGType itype)
{
  iSolveType = itype;

} // end of method  SetSolveType


//************************************************************************************
//************************************************************************************
// METHOD     : SetPrecond
// DESCRIPTION: Sets preconditioner
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::SetPreconditioner(GLinOpList *p)
{
  GINT  i;

  if ( p == NULL ) {
    cout << "CG::SetPreconditioner: NULL GLinOpList" << endl;
    exit(1);
  }

  precon.empty();
  for ( i=0; i<nops; i++ ) {
    precon.del(i);
  }
  for ( i=0; i<p->size(); i++ ) {
    precon.add(NULL,FALSE);  precon[i] = (*p)[i];
  }
  nprecon = p!=NULL ? p->size() : 0;
} // end of method  SetPreconditioner


//************************************************************************************
//************************************************************************************
// METHOD     : SetComm
// DESCRIPTION: Set gather/scatter object
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::SetComm(GS *op)
{
  gsop = op;
} // end of method  SetComm


//************************************************************************************
//************************************************************************************
// METHOD     : ResetExpandables
// DESCRIPTION: Initialize operators
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::ResetExpandables(GLinOpList *AA, GLinOpList *inROp)
{
  GINT  i, j, NN, num;

  if ( AA == NULL ) {
    cout << "CG::ResetExpandables: NULL operators" << endl;
    exit(1);
  }

  num = AA->size();
  if ( bNested ) {
    intermed_soln.empty();
    ytmp         .empty();
    mask_Intermed.empty();
  }
  xb           .empty();
  rk           .empty();
  wk           .empty();
  qk           .empty();
  sk           .empty();
  zk           .empty();
  msk          .empty();
  mqk          .empty();
  vk           .empty();

#if 0
  // Resize the member data for each existing element:
  for ( i=0; i<nops; i++ ) {
    NN    = (*xx)[i]->dim();
    if ( bNested ) {
      intermed_soln[i]->Resize(NN);
      ytmp         [i]->Resize(NN);
      ((DiagOp*)mask_Intermed[i])->Resize(NN);
    }
    xb  [i]->Resize(NN);
    rk [i]->Resize(NN);
    wk [i]->Resize(NN);
    qk [i]->Resize(NN);
    sk [i]->Resize(NN);
    zk [i]->Resize(NN);
    msk[i]->Resize(NN);
    mqk[i]->Resize(NN);
    vk [i]->Resize(NN);
  }
#endif

  // Add elements as required:
  for ( i=0; i<num; i++ ) {
    for ( j=0, NN=1; j<GDIM; j++ ) NN *= ((*gelems_)[i]->GetOrder(j+1)+1);
    if ( bNested ) {
      intermed_op  .add((*A)[i]->GetIntermedOp(),FALSE);     // pointer only; don't delete
      intermed_soln.add(NULL,FALSE);                      // pointer only; don't delete
      ytmp         .add(NULL,TRUE); ytmp[i] = new GVector(NN);
      mask_Intermed.add(NULL,TRUE); mask_Intermed[i] = new DiagOp(NN);
    }

    // Fully-instantiated data:
    xb  .add(NULL,TRUE); xb  [i] = new GVector(NN);
    rk  .add(NULL,TRUE); rk  [i] = new GVector(NN);
    wk  .add(NULL,TRUE); wk  [i] = new GVector(NN);
    qk  .add(NULL,TRUE); qk  [i] = new GVector(NN);
    sk  .add(NULL,TRUE); sk  [i] = new GVector(NN);
    zk  .add(NULL,TRUE); zk  [i] = new GVector(NN);
    msk .add(NULL,TRUE); msk [i] = new GVector(NN);
    mqk .add(NULL,TRUE); mqk [i] = new GVector(NN);
    vk  .add(NULL,TRUE); vk  [i] = new GVector(NN);
  }

  //  Delete unnecessary elements from lists:
  for ( i=num; i<nops; i++ ) {
    if ( bNested ) {
      intermed_op  .del(i);
      intermed_soln.del(i);
      ytmp         .del(i);
      mask_Intermed.del(i);
    }
    xb  .del(i);
    rk .del(i);
    wk .del(i);
    qk .del(i);
    sk .del(i);
    zk .del(i);
    msk.del(i);
    mqk.del(i);
    vk .del(i);
  }

  // Must now renumber/re-order the reference ids for lists:
  if ( bNested ) {
    intermed_op  .renumber();
    intermed_soln.renumber();
    ytmp         .renumber();
  }
  xb  .renumber();
  rk  .renumber();
  wk  .renumber();
  qk  .renumber();
  sk  .renumber();
  zk  .renumber();
  msk .renumber();
  mqk .renumber();
  vk  .renumber();

  // Set up solvers for intermediate solves, if doing a nested solve:
  if ( bNested ) {
    if ( cg_intermed != NULL ) {
      for ( i=0; i<nIntermedProd; i++ ) delete cg_intermed[i];
      delete [] cg_intermed; cg_intermed = NULL;
    }
    nIntermedProd = (*AA)[0]->GetNumIntermedProd(); // Assume that all operators are nested
    cg_intermed = new CG * [nIntermedProd];
    for ( i=0; i<nIntermedProd; i++ ) cg_intermed[i] = new CG(CG_STANDARD,FALSE);
  }
  nops = num;

} // end of method ResetExpandables


//************************************************************************************
//************************************************************************************
// METHOD     : InitComm (1)
// DESCRIPTION: Method to initialize the gather/scatter operator:
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GCHandle CG::InitComm(GDWBuffer  *ids, GINT  maxid)
{ 
#if 0
  GINT  i;

  if ( gsop == NULL ) {
    cout << "CG::InitComm: Comm operator not set" << endl;
    exit(1);
  }

  glob_ids = ids; 
  nTotalNodes = maxid;
  hDSOp = ((GlOp*)gsop)->Init(glob_ids, nTotalNodes);
  if ( hDSOp == NULL_HANDLE ) {
    cout << "CG::InitComm: NULL operator handle; initialization failed" << endl;
    exit(1);
  }

  return hDSOp;
#endif
  return NULL_HANDLE;
} // end of method InitComm (1)


//************************************************************************************
//************************************************************************************
// METHOD     : SetCommHandle
// DESCRIPTION: Method to initialize required CG quantities from a 
//              gather/scatter operator  that has already been
//              initialized.
// ARGUEMENTS : GCHandle: handle of previously-called GS::Init
// RETURNS    : input handle
//************************************************************************************
GCHandle CG::SetCommHandle(GCHandle hIn)
{

  if ( gsop == NULL ) {
    cout << "CG::SetCommHandle: Comm operator not set" << endl;
    exit(1);
  }
  if ( hIn == NULL_HANDLE ) {
    cout << "CG::SetCommHandle: NULL input handle; initialization failed" << endl;
    exit(1);
  }
  hDSOp = hIn;

  return hDSOp;
} // end of method SetCommHandle


//************************************************************************************
//************************************************************************************
// METHOD     : SetIntermedCommHandle
// DESCRIPTION: Method to initialize required CG quantities from a
//              gather/scatter operator  that has already been
//              initialized. This handle is to be applied only
//              to intermediate products.
// ARGUEMENTS : GCHandle: handle of previously-called GS::Init
// RETURNS    : input handle
//************************************************************************************
GCHandle CG::SetIntermedCommHandle(GCHandle hIn)
{

  if ( gsop == NULL ) {
    cout << "CG::SetIntermedCommHandle: Comm operator not set" << endl;
    exit(1);
  }
  if ( hIn == NULL_HANDLE ) {
    cout << "CG::SetIntermedCommHandle: NULL input handle; initialization failed" << endl;
    exit(1);
  }
  hDSOp_Intermed = hIn;

  return hDSOp_Intermed;
} // end of method SetIntermedCommHandle



//************************************************************************************
//************************************************************************************
// METHOD     : SetBdyValues
// DESCRIPTION: Set boundary values based on bdy condition type. 
//              NOTE: if the DSS operation isn't wrapped, this method 
//                    should be called only after gsop (GS operator) 
//                    is initialized, since mask makes use of it.
//              NOTE: the member data, nops, must be set prior to entry ==>
//                    the fields must have been set, with a call to ().
// ARGUEMENTS :
// RETURNS    :  
//************************************************************************************
void CG::SetBdyValues(GVecList *bdy_vals)
{ 
  GINT       i, j, index;
  GVector    *mask, *bv;
  GIBuffer   *bdy_indices;
  GBTBuffer  *bdy_types;
  Elem2D     *elem;

  if ( gelems_ == NULL ) {
    cout << "CG::SetBdyValues: NULL element list" << endl;
    exit(1);
  }

#if 0
  if ( hDSOp == NULL_HANDLE ) {
    cout << "CG::SetBdyValues: gather/scatter operator is not itialized" << endl;
    exit(1);
  }
#endif

  if ( nops == 0 || nops != gelems_->size() ) {
    cout << "CG::SetBdyValues: invalid number of elements; fields not initialized." << endl; 
    exit(1);
  }

  // If there are bc's, set the values
  // in the boundary vector based on bc type:
  // NOTE: assume that PERIODIC bcs are already handled 
  // by setting the glob_ids properly in InitComm. If we want to
  // handle this here, we'll need a periodic boundary
  // id vector as an argument to this method.

  // Require quantities on entry:
  if ( bdy_vals == NULL ) {
    cout << "CG::SetBdyValues: NULL boundary values" << endl; 
    exit(1);
  }

  for ( i=0; i<nops; i++ ) {
    elem  = (*gelems_)[i];
    mask  = elem->GetMask();  
    bdy_indices = elem->GetBdyIndices();
    bdy_types   = elem->GetBdyTypes  ();
    if ( bdy_indices == NULL || bdy_types == NULL ) {
      cout << "CG::SetBdyValues: NULL boundary type structures" << endl; 
      exit(1);
    }
    for ( j=0; j<bdy_indices->dim(); j++ ) {
      index = (*bdy_indices)(j);
      if ( (*bdy_types)(j) == DIRICHLET || (*bdy_types)(j) == NOSLIP ) {  // mask solution 
        (*mask)(index) = 0.0;
      }
    }
  }
  
#if 0
  // Perform Direct Stiffness Multiply to determine _all_ DIRICHLET nodes:
  gsop->DSOp(mask, G_OP_PROD, hDSOp);
#endif

  for ( i=0; i<nops; i++ ) {
    elem  = (*gelems_)[i];
    bdy_indices = elem->GetBdyIndices();
    bdy_types   = elem->GetBdyTypes  ();
    bv          = (*bdy_vals)[i];
    *xb[i] = 0.0;
    for ( j=0; j<bdy_indices->dim() && bv && bv->dim(); j++ ) {
      index = (*bdy_indices)(j);
      if      ( (*bdy_types)(j) == DIRICHLET ) (*xb[i])[index] = (*bv)[j];  
      else if ( (*bdy_types)(j) == NOSLIP    ) (*xb[i])[index] = 0.0;
    }
#if defined(CG_DEBUG_OUTPUT)
    cout << "CG::SetBdyValues: xb[" << i << "]=" << *xb[i] << endl;
#endif
  }
 
} // end of method SetBdyValues


//************************************************************************************
//************************************************************************************
// METHOD     : SetElems
// DESCRIPTION: Sets element list
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
void CG::SetElems(GElemList *e)
{
  if ( e == NULL ) {
    cout << "CG::SetElems: NULL element list not allowed" << endl;
    exit(1);
  }
  gelems_ = e;
} // end of method SetElems


//************************************************************************************
//************************************************************************************
// METHOD     : GetTolerance
// DESCRIPTION: 
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GDOUBLE CG::GetTolerance()
{
  return tolerance ;
} // end of method GetTolerance


//************************************************************************************
//************************************************************************************
// METHOD     : GetMaxIterations
// DESCRIPTION: 
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GINT  CG::GetMaxIterations()
{
  return MaxIterations ;
} // end of method  GetMaxIterations


//************************************************************************************
//************************************************************************************
// METHOD     : GetResidualChkNum
// DESCRIPTION: 
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GINT  CG::GetResidualChkNum()
{
  return nResidualChk ;
} // end of method  GetResidualChkNum


//************************************************************************************
//************************************************************************************
// METHOD     : GetNumIterations
// DESCRIPTION: Gets no. iterations taken
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GINT  CG::GetNumIterations()
{
  return numIterations;
} // end of method  GetMaxIterations


//************************************************************************************
//************************************************************************************
// METHOD     : GetError
// DESCRIPTION: Gets final error
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GDOUBLE CG::GetError()
{
  return finErr;
} // end of method  GetError

//************************************************************************************
//************************************************************************************
// METHOD     : GetMinError
// DESCRIPTION: Gets min error encountered during iteration
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GDOUBLE CG::GetMinError()
{
  return minErr;
} // end of method  GetMinError


//************************************************************************************
//************************************************************************************
// METHOD     : GetMaxError
// DESCRIPTION: Gets max error ecountered during iteration
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GDOUBLE CG::GetMaxError()
{
  return maxErr;
} // end of method  GetMaxError


#if 0
//************************************************************************************
//************************************************************************************
LinOp *CG::GetPreconditioner()
{
  return precon;
} // end of method  GetPreconditioner
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : DoDotProducs
// DESCRIPTION: Performs dot products required in CG
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GBOOL CG::DoDotProducts(GVecList &a1, GVecList a2Vec[], GDOUBLE lProds[], GDOUBLE prods[], const GINT  n_prods)
{
  GINT      i, j, k, m, nlocal;
  GDOUBLE     *a1Data,  *a2Data, *mData;
  GVector   *imult=NULL;
  GBOOL     bRet = TRUE;
#if 0
  GIBuffer  *ii, *ei;
  GMortar1D *mortar;
  GVector   *mf1, *mf2;
#endif
  Elem2D    *elem;

  nlocal = a1.size();

  for ( j=0; j<n_prods && bRet; j++ ) {   // loop over required dot prods
    lProds[j] = 0.0;
    for ( k=0; k<nlocal && bRet; k++ ) {  // loop over local elements
      elem   = (*gelems_)[k];
      a1Data = a1[k]->Data();
//    imult = gsop->iMultiplicity(k, hDSOp);
      imult = elem->GetNodalMultiplicity();
      if ( a1[k]->dim() != a2Vec[j][k]->dim() ) bRet = FALSE;
      a2Data =  a2Vec[j][k]->Data();

      if ( imult == NULL ) {
        lProds[j] += MTK::fvec_dot(*a1[k],*a2Vec[j][k]);
      }
      else {
        mData  =  imult->Data();
#if 1
cout << "CG::DoDotProds: a1[" << k << "]=" << *a1[k] << endl;
cout << "CG::DoDotProds: a2[" << k << "]=" << *a2Vec[j][k] << endl;
cout << "CG::DoDotProds: W [" << k << "]=" << *imult << endl;
        for ( i=0; i<a1[k]->dim(); i++ ) { // if there is a counting vector specified
          lProds[j] += ( a1Data[i] * a2Data[i] * mData[i] );
        }
#else
        ii     = elem->GetInteriorIndices();
        mortar = elem->GetEdgeMortar();
        for ( i=0; i<ii->dim(); i++ ) {                    // interior nodes 
          lProds[j] += ( a1Data[(*ii)[i]] * a2Data[(*ii)[i]] * mData[(*ii)[i]] );
        }
        for ( m=0; m<elem->GetNumEdges(); m++ ) { // edge/face nodes
          is = m % 2;            // good only for 2d
          ei = elem->GetEdgeIndices(m);
          mortar[m].Host2Mortar(a1[k]      , ei->Data(), ei->dim(), 0, 0);
          mortar[m].Host2Mortar(a2Vec[j][k], ei->Data(), ei->dim(), 0, 1);
          mf1 = mortar[m].GetMortarField(0);
          mf2 = mortar[m].GetMortarField(1);
cout << "CG::DoDotProds: mf1[" << k << "][" << m << "]=" << *mf1 << endl;
cout << "CG::DoDotProds: mf2[" << k << "][" << m << "]=" << *mf2 << endl;
          for ( i=is; i<ei->dim()-is; i++ ) { 
            lProds[j] += ( (*mf1)[i] * (*mf2)[i] * mData[(*ei)[i]] );
          }
        }

#endif
      }
cout << "CG::DoDotProds: dot_local[" << k << "]=" << lProds[j] << endl;
    }
  }

  if ( bRet ) {
    GComm::Allreduce(lProds, prods, n_prods, GC_GDOUBLE, G_OP_SUM);
  }

  return bRet;

} // end of method DoDotProducts 


//************************************************************************************
//************************************************************************************
// METHOD     : ErrNo
// DESCRIPTION: Gets error condition number
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GINT  CG::ErrNo()
{
  return iErrType;
} // end of method ErrNo


//************************************************************************************
//************************************************************************************
// METHOD     : ErrString
// DESCRIPTION: Gets error string
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
char *CG::ErrString()
{
  return s_cgerror_[iErrType];
} // end of method ErrString

//************************************************************************************
//************************************************************************************
// METHOD     : GetBdyValues
// DESCRIPTION: Get boundary values list
// ARGUEMENTS : none
// RETURNS    : xb, GVecList pointer
//************************************************************************************
GVecList &CG::GetBdyValues()
{ 
 return xb;
} // end of method GetBdyValues


//************************************************************************************
//************************************************************************************
// METHOD     : GetEucResidualHistory
// DESCRIPTION: Gets history vector of Euclidean-norms of residuals
// ARGUEMENTS : none
// RETURNS    : GVector pointer
//************************************************************************************
GVector *CG::GetEucResidualHistory()
{
 return &EucResidual_;
} // end of method GetEucResidualHistory


//************************************************************************************
//************************************************************************************
// METHOD     : GetL2ResidualHistory
// DESCRIPTION: Gets history vector of L2-norms of residuals
// ARGUEMENTS : none
// RETURNS    : GVector pointer
//************************************************************************************
GVector *CG::GetL2ResidualHistory()
{
 return &L2Residual_;
} // end of method GetL2ResidualHistory


#if 0
GBOOL CG::CheckPrecond()
{
  GINT    i, j, NN;
  GVector tmp1, tmp2, dH, *iH;

  for ( i=0; i<H.size(); i++ ) {
    NN = ((PCPointJac_Helm*)pc[i])->GetDiag()->dim();
    tmp1.Resize(NN);
    tmp2.Resize(NN);
    dH  .Resize(NN);
    tmp1 = 0.0;
    iH = ((PCPointJac_Helm*)pc[i])->GetiDiag();
    for ( j=0; j<NN; j++ ) {
      tmp1[j] = 1.0;
      H[i]->OpVec_prod(tmp1, tmp2);
      dH[j] = tmp2[j];
      tmp1[j] = 0.0;
    }
    MTK::fvec_point_prod_rep(dH,*iH);
    cout << "CheckPrecond: [diag(H) * PC][" << i << "]=" << dH << endl;
  }

  return TRUE;
} // end of method CheckPrecond
#endif
