//************************************************************************************//
// Module       : pcblockjac_lap.hpp
// Date         : 7/17/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a block Jacobi preconditioner object to be used
//                for preconditioning the Laplacian operator
// Derived From : none.
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#include "pcblockjac_lap.hpp"
#include "rectquad2d.hpp"
#include "defquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
PCBlockJac_Lap::PCBlockJac_Lap()
:        LinOp(),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
elem     (NULL),
vtmpv0   (NULL),
vtmpv1   (NULL),
mask     (NULL),
L        (NULL),
iL       (NULL),
I1       (NULL),
I2       (NULL),
I1T      (NULL),
I2T      (NULL)
{
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
PCBlockJac_Lap::PCBlockJac_Lap(LinOp *AA)
:        LinOp(AA),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
elem     (NULL),
vtmpv0   (NULL),
vtmpv1   (NULL),
mask     (NULL),
L        (NULL),
iL       (NULL),
I1       (NULL),
I2       (NULL),
I1T      (NULL),
I2T      (NULL)
{
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
PCBlockJac_Lap::PCBlockJac_Lap(Elem2D *e)
:        LinOp(e),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
elem     (NULL),
vtmpv0   (NULL),
vtmpv1   (NULL),
mask     (NULL),
L        (NULL),
iL       (NULL),
I1       (NULL),
I2       (NULL),
I1T      (NULL),
I2T      (NULL)
{
  SetElem(e); 
} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Destructor
PCBlockJac_Lap::~PCBlockJac_Lap()
{
  if ( L   != NULL ) delete L  ; L   = NULL;
  if ( iL  != NULL ) delete iL ; iL  = NULL;
  if ( I1  != NULL ) delete I1 ; I1  = NULL;
  if ( I2  != NULL ) delete I2 ; I2  = NULL;
  if ( I1T != NULL ) delete I1T; I1T = NULL;
  if ( I2T != NULL ) delete I2T; I2T = NULL;
  if ( vtmpv0 != NULL ) delete vtmpv0; vtmpv0 = NULL;
  if ( vtmpv1 != NULL ) delete vtmpv1; vtmpv1 = NULL;
}


//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Lap::SetElem(Elem2D *e)
{

  if ( e == NULL )
  {
    cout << "PCBlockJac_Lap::SetElem: NULL element" << endl;
    exit(1);
  }

  if ( e != elem || N1 != (e->GetOrder(1)+1) ||
                    N2 != (e->GetOrder(2)+1) )
  {
    elem = e;
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    if ( L  != NULL ) delete L ; L  = NULL;
    if ( iL != NULL ) delete iL; iL = NULL;
    if ( !ComputeOp() )
    {
      cout << "PCBlockJac_Lap::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
    return;
  }

} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetMask
// DESCRIPTION: Sets mask
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Lap::SetMask(GVector *m)
{
  mask = m;
} // end of method SetMask


//************************************************************************************
// METHOD     : ComputeOp
// DESCRIPTION: Carries out computation of operator
// ARGUMENTS  :  
// RETURNS    : none
//************************************************************************************
GBOOL PCBlockJac_Lap::ComputeOp()
{
  GBOOL bRet ;

  if ( elem == NULL ) return FALSE;
  if ( L != NULL || iL != NULL ) return FALSE;

  if ( N1 <= 0 || N2 <= 0 || NN <= 0 ) return FALSE;


  L     = new GMatrix(NN,NN);
  iL    = new GMatrix(NN,NN);
  switch (elem->ElemType())
  {
    case DEFORMED_QUAD:
      bRet = DefmQuadL();
      break;
    case RECT_QUAD:
      bRet = RectQuadL();
      break;
    case TRIANGULAR:
      bRet = TriangleL();
      break;
    default:
      cout << "LaplacianOp::SetElem: Invalid element type" << endl;
      exit(1);
  }

  if ( !bRet )
  {
    cout << "LaplacianOp::SetElem: Operator computation failed" << endl;
    exit(1);
  }
  if ( L != NULL )
  {
    delete L;
    L = NULL;
  }
#if 0
  cout << " PCBlockJac_Lap::SetElem: L=" << *L << endl;
  cout << " PCBlockJac_Lap::SetElem: iL=" << *iL << endl;
  GMatrix *I = new GMatrix(NN,NN);
  *I = (*L) * (*iL);
  cout << " PCBlockJac_Lap:Setelem: iL * L = " << *I << endl;
  delete I;
#endif
  return bRet;

} // end of method ComputeOp


//************************************************************************************
// METHOD     : SetInterpBasis
// DESCRIPTION: Sets basis to which to restrict Laplacian
// ARGUMENTS  : GNBasis *
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Lap::SetInterpBasis(GNBasis *b1, GNBasis *b2)
{
  GSHORT  pN1, pN2, vN1, vN2, vNN;
  GVector *pnodes1, *pnodes2;

  if ( b1 == NULL || b2 == NULL ) return;
  if ( elem == NULL )
  {
    cout << "PCBlockJacLap::SetInterpBasis: Attempting to use NULL element" << endl;
    exit(1);
  }
  if ( I1  != NULL ) delete I1 ; I1  = NULL;
  if ( I2  != NULL ) delete I2 ; I2  = NULL;
  if ( I1T != NULL ) delete I1T; I1T = NULL;
  if ( I2T != NULL ) delete I2T; I2T = NULL;

  pN1 = b1->GetOrder()+1;
  pN2 = b2->GetOrder()+1;
//pNN = pN1 * pN2;
  vN1 = elem->GetOrder(1)+1;
  vN2 = elem->GetOrder(2)+1;
  vNN = vN1 * vN2;

  I1     = new GMatrix(pN1, vN1);
  I2     = new GMatrix(pN2, vN2);
  I1T    = new GMatrix(vN1, pN1);
  I2T    = new GMatrix(vN2, pN2);
  pnodes1= new GVector(pN1);
  pnodes2= new GVector(pN2);
  vtmpv0 = new GVector(vNN);
  vtmpv1 = new GVector(vNN);
  if ( I1 == NULL || I2 == NULL || I1T == NULL || I2T == NULL ||
       vtmpv0  == NULL || vtmpv1  == NULL || 
       pnodes1 == NULL || pnodes1 == NULL )
  {
    cout << "PCBlockJacLap::SetInterpBasis: allocation error" << endl;
    exit(1);
  }
  b1->GetXiNodes(pnodes1);
  b2->GetXiNodes(pnodes2);

  if ( elem->GetBasisObj(1)->EvalBasis(pnodes1,I1) == NULL )
  {
    cout << "PCBlockJacLap::SetInterpBasis: 1: EvalBasis failed" << endl;
    exit(1);
  }
  if ( elem->GetBasisObj(2)->EvalBasis(pnodes2,I2) == NULL )
  {
    cout << "PCBlockJacLap::SetInterpBasis: 2: EvalBasis failed" << endl;
    exit(1);
  }
  delete pnodes1;
  delete pnodes2;
   
  I1->Transpose(*I1T);
  I2->Transpose(*I2T);

} // end of method SetInterpBasis


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Lap::SetConst(GDOUBLE clc)
{
  if ( lc1 != clc || lc2 != clc )
  {
    if ( elem == NULL )
    {
      cout << "PCBlockJac_Lap::SetConst: NULL element" << endl;
      exit(1);
    }
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    lc1 = lc2 = clc;
    if ( L  != NULL ) delete L ; L  = NULL;
    if ( iL != NULL ) delete iL; iL = NULL;
    if ( !ComputeOp() )
    {
      cout << "PCBlockJac_Lap::SetElem: ComputeOp failed" << endl;
      exit(1);
    }    
  }

} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void PCBlockJac_Lap::SetConst(GDOUBLE clc1, GDOUBLE clc2)
{
  if ( lc1 != clc1 || lc2 != clc2 )
  {
    if ( elem == NULL )
    {
      cout << "PCBlockJac_Lap::SetConst: NULL element" << endl;
      exit(1);
    }
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    lc1 = clc1;
    lc2 = clc2;
    if ( L  != NULL ) delete L ; L  = NULL;
    if ( iL != NULL ) delete iL; iL = NULL;
    if ( !ComputeOp() )
    {
      cout << "PCBlockJac_Lap::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
  }

} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
GVector  PCBlockJac_Lap::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);

  OpVec_prod(q, newq);

  return newq;

} // end of method \* oeprator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
void PCBlockJac_Lap::OpVec_prod(GVector &q, GVector &ret)
{

  GINT  iN1, iN2, iNN;

  if ( elem == NULL || iL == NULL )
  {
    cout << "PCBlockJac_Lap::operator*: NULL element or operator" << endl;
    exit(1);
  }

  if ( I1 != NULL && I2 != NULL )  // restrict inv. Laplacian using interp. operators
  {
    iN1 = I1->dim(1);
    iN2 = I2->dim(1);
    iNN = iN1 * iN2;
  }
  else
  {
    iN1 = N1;
    iN2 = N2;
    iNN = NN ;
  }

  if ( q.dim() != iNN  || ret.dim() != iNN ) 
  {
    cout << "PCBlockJac_Lap::operator*: incompatible vector(s)" << endl;
    exit(1);
  }

  if ( I1 != NULL && I2 != NULL )  // restrict inv. Laplacian using interp. operators
  {
    // restriction is s.t.: ret = ( I2 X I1 ) L^-1 ( I2T X I1T ) q
    MTK::D2_X_D1(*I1T, *I2T, q, iN1, iN2, ret, iN1, iN2, *vtmpv1,iN1, iN2);
//cout << "PCBlockJac_Lap::OpVec_prod: vtmp0=" << *vtmpv0 << endl;
    MTK::fmatvec_prod(*iL, *vtmpv1, *vtmpv0);
//cout << "PCBlockJac_Lap::OpVec_prod: vtmp1=" << *vtmpv1 << endl;
    MTK::D2_X_D1(*I1, *I2, *vtmpv0, iN1, iN2, *vtmpv1, iN1, iN2, ret, iN1, iN2);
//cout << "PCBlockJac_Lap::OpVec_prod: ret=" << ret << endl;
  }
  else                             // simply apply
  {
//  ret = (*iL) * q;
    MTK::fmatvec_prod(*iL, q, ret);
  }

  if ( mask )  MTK::fvec_point_prod_rep(ret, *mask);



} // end of method OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadL
// DESCRIPTION: Computes Laplacian operator*vector  for a deformed quad element.
//              Only diagonal^1 elements of operator are considered.
//              Diag elements are:
//                K_ij = Sum_k[D1_ki ^2  G11_pj  + D2_pj^2 G22_ip]
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
GBOOL PCBlockJac_Lap::DefmQuadL()
{ 

  GINT      i, j, k, n, m;
  GVector  *G11, *G12, *G21, *G22, *GA[2];
  GMatrix  *D1 , *D2;

  D1  = elem->Get1DDerivMatrix(1,FALSE);
  D2  = elem->Get1DDerivMatrix(2,FALSE);
  G11 = ((DefQuad2D*)elem)->GetWJMetric(1,1);
  G12 = ((DefQuad2D*)elem)->GetWJMetric(1,2);
  G21 = G12;
  G22 = ((DefQuad2D*)elem)->GetWJMetric(2,2);

  if ( D1  == NULL || D2  == NULL ||
       G11 == NULL || G12 == NULL ||
       G21 == NULL || G22 == NULL ) return FALSE;

 
  GMatrix *d1v, *d2v;

  d1v = new GMatrix(NN,NN);
  d2v = new GMatrix(NN,NN);
  L   = new GMatrix(NN,NN);

  // Do I2 X D1 term:
  for ( k=0; k<N2; k++ )
  {
    for ( i=0; i<N1; i++ )
    {
      for ( j=0; j<N1; j++ )
      {
        n = k * N1 + i;
        m = k * N1 + j;
        (*d1v)(n,m) = (*D1)(i,j);
      }
    }
  }

  // Do D2 X I1 term:
  for ( i=0; i<N2; i++ )
  {
    for ( j=0; j<N2; j++ )
    {
      for ( k=0; k<N1; k++ )
      {
        n = i * N1 + k;
        m = j * N1 + k;
        (*d2v)(n,m) = (*D2)(i,j);
      }
    }
  }

  // Incorporate the metric terms:
  GA[0] = G11;
  GA[1] = G12;
  for ( k=0; k<2; k++ )
    for ( i=0; i<NN; i++ )
      for ( j=0; j<NN; j++ )
        (*d1v)(i,j) = (*d1v)(i,j) * (*GA[k])(j);

  GA[0] = G21;
  GA[1] = G22;
  for ( k=0; k<2; k++ )
    for ( i=0; i<NN; i++ )
      for ( j=0; j<NN; j++ )
        (*d2v)(i,j) = (*d2v)(i,j) * (*GA[k])(j);
  // Multiply by the appropriate transposes, and add up:
  *L   = ( d1v->Transpose() * ( *d1v + *d2v ) )
       + ( d2v->Transpose() * ( *d1v + *d2v ) );

  delete d1v;
  delete d2v;

  // Now, invert to form preconditioner:
  return L->Inverse(*iL);

} // end of method DefmQuadL


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadL
// DESCRIPTION: Computes Laplacian operator*vector  for a deformed quad element.
//              Only diagonal^1 elements of operator are considered.
//              Diag elements are:
//                K_ij = 
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL PCBlockJac_Lap::RectQuadL()
{ 

  GINT      i, j, k, n, m;
  GDOUBLE     L1, L2, R1, R2;
  Point3D  *vert;
  GVector  *M1, *M2;  
  GMatrix  *K1 , *K2 ;

  M1   = elem->Get1DWeights(1);
  M2   = elem->Get1DWeights(2);
  K1   = elem->Get1DStiffMatrix(1,FALSE);
  K2   = elem->Get1DStiffMatrix(2,FALSE);
  vert = elem->GetSpVertices();
  L1   = fabs((vert+1)->x1 - (vert)->x1);
  L2   = fabs((vert+3)->x2 - (vert)->x2);

  if ( L1 == 0 || L2 == 0 ) return FALSE;
  
  R1  = lc1 * L2 / L1;
  R2  = lc2 * L1 / L2;

 
  if ( K1  == NULL || K2  == NULL ||
       M1  == NULL || M2  == NULL 
     ) return FALSE;
  
  *L = 0.0;
  // Do M2 X K1 term:
  for ( k=0; k<N2; k++ )
  {
    for ( i=0; i<N1; i++ )
    {
      for ( j=0; j<N1; j++ )
      {
        n = k * N1 + i;
        m = k * N1 + j;
        (*L)(n,m) = R1*(*M2)(k) * (*K1)(i,j);
      }
    }
  }

  // Do K2 X M1 term:
  for ( i=0; i<N2; i++ )
  {
    for ( j=0; j<N2; j++ )
    {
      for ( k=0; k<N1; k++ )
      {
        n = i * N1 + k;
        m = j * N1 + k;
        (*L)(n,m) += ( R2*(*M1)(k) * (*K2)(i,j) );
      }
    }
  }

  
  // Now, invert to form preconditioner:
  return L->Inverse(*iL);

} // end of method RectQuadL


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleL
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL PCBlockJac_Lap::TriangleL()
{
  return FALSE;
} // end of method TriangleL

