//************************************************************************************//
// Module       : pcpointjac_helm.cpp
// Date         : 7/17/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a point Jacobi preconditioner object to be used
//                for preconditioning the Helmholtz operator
// Derived From : none.
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#include "pcpointjac_helm.hpp"
#include "diagop.hpp"
#include "mtk.hpp"
#include "helmholtzop.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
PCPointJac_Helm::PCPointJac_Helm()
:        LinOp(),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
mc       (1.0),
elem     (NULL),
H        (NULL),
iH       (NULL)
{
  H  = new GVector(1);
  iH = new GVector(1);
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
PCPointJac_Helm::PCPointJac_Helm(LinOp *AA)
:        LinOp(AA),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
mc       (1.0),
elem     (NULL),
H        (NULL),
iH       (NULL)
{
  H  = new GVector(1);
  iH = new GVector(1);
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
PCPointJac_Helm::PCPointJac_Helm(Elem2D *e)
:        LinOp(e),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
mc       (1.0),
elem     (NULL),
H        (NULL),
iH       (NULL)
{
  H  = new GVector(1);
  iH = new GVector(1);
  SetElem(e);
} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Destructor
PCPointJac_Helm::~PCPointJac_Helm()
{
  if ( H  != NULL ) delete H ; H  = NULL;
  if ( iH != NULL ) delete iH; iH = NULL;
}


//************************************************************************************
//************************************************************************************
// METHOD     : SetElem 
// DESCRIPTION: Sets associated element 
// ARGUMENTS  : Elem2D *
// RETURNS    : unassembled H diagonal
//************************************************************************************
GVector *PCPointJac_Helm::SetElem(Elem2D *e)
{

  if ( e == NULL ) {
    cout << "PCPointJac_Helm::SetElem: NULL element" << endl;
    exit(1);
  }

  if ( e != elem || N1 != (e->GetOrder(1)+1) ||
                    N2 != (e->GetOrder(2)+1)  ) {
    elem = e;
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    H ->Resize(NN); 
    iH->Resize(NN); 
    if ( !ComputeOp() ) {
      cout << "PCPointJac_Helm::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
    return H;
  }
  return NULL;
} // end of method SetElem 


//************************************************************************************
//************************************************************************************
// METHOD     : GetDiag
// DESCRIPTION: Gets diagonal preconditioner for global assembly
// ARGUMENTS  :
// RETURNS    : GVector *
//************************************************************************************
GVector *PCPointJac_Helm::GetDiag()
{
  return H;
} // end of method GetDiag


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeOp
// DESCRIPTION: Carries out computation of the local operator
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCPointJac_Helm::ComputeOp()
{
  GBOOL bRet ;

  if ( elem == NULL ) return FALSE;

  switch (elem->ElemType()) {
    case DEFORMED_QUAD:
      bRet = DefmQuadH();
      break;
    case RECT_QUAD:
      bRet = RectQuadH();
      break;
    case TRIANGULAR:
      bRet = TriangleH();
      break;
    default:
      cout << "LaplacianOp::ComputeOp: Invalid element type" << endl;
      exit(1);
  }
  return bRet;
} // end of method ComputeOp


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : unassembled Helmholtz diagonal
//************************************************************************************
GVector *PCPointJac_Helm::SetConst(GDOUBLE clc, GDOUBLE cmc)
{
  if ( lc1 != clc || lc2 != clc || mc != cmc ) {
    if ( elem == NULL ) {
      cout << "PCPointJac_Helm::SetConst: NULL element" << endl;
      exit(1);
    }
    lc1 = lc2 = clc;
    mc = cmc;
    if ( !ComputeOp() ) {
      cout << "PCPointJac_Helm::SetConst: ComputeOp failed" << endl;
      exit(1);
    }
  }
  return H;
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : unassembled Helholtz diagonal
//************************************************************************************
GVector *PCPointJac_Helm::SetConst(GDOUBLE clc1, GDOUBLE clc2, GDOUBLE cmc)
{

  if ( lc1 != clc1 || lc2 != clc2 || mc != cmc ) {
    if ( elem == NULL ) {
      cout << "PCPointJac_Helm::SetConst: NULL element" << endl;
      exit(1);
    }
    lc1 = clc1;
    lc2 = clc2;
    mc  = cmc;
    if ( !ComputeOp() ) {
      cout << "PCPointJac_Helm::SetConst: ComputeOp failed" << endl;
      exit(1);
    }
  }
  return H;

} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
GVector PCPointJac_Helm::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);

  OpVec_prod(q, newq);

  return newq;

} // end of method \* oeprator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
void PCPointJac_Helm::OpVec_prod(GVector &q, GVector &ret)
{

  if ( elem == NULL || iH == NULL ) {
    cout << "PCPointJac_Helm::operator*: NULL element or operator; check that H has been DSS'd!" << endl;
    exit(1);
  }

  if ( q.dim() != NN  || ret.dim() != NN ) {
    cout << "PCPointJac_Helm::operator*: incompatible vector(s)" << endl;
    exit(1);
  }

//ret = (*iH) * q;
  MTK::fvec_point_prod(*iH, q, ret);


} // end of method OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadH
// DESCRIPTION: Computes Helmholtz operator*vector  for a deformed quad element.
//              Only diagonal^1 elements of operator are considered.
//              Diag elements are:
//                K_ij = Sum_k[D1_ki ^2  G11_pj  + D2_pj^2 G22_ip]
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCPointJac_Helm::DefmQuadH()
{ 

  GINT      i, j, k;
  GVector  *M;  
  GVector  *G11, *G22;
  GMatrix  *D1 , *D2 ;
    
  if ( elem == NULL ) return FALSE;

  D1  = elem->Get1DDerivMatrix(1,FALSE);
  D2  = elem->Get1DDerivMatrix(2,FALSE);
  G11 = ((DefQuad2D*) elem)->GetWJMetric(1,1);
  G22 = ((DefQuad2D*) elem)->GetWJMetric(2,2);
  M   = elem->GetMassMatrix();   // includes J

  if ( D1  == NULL || D2  == NULL ||
       G11 == NULL || G22 == NULL || 
       M   == NULL                 ) return FALSE;
  
  *H = 0.0;
  
  // Compute diagonal elements of Helmholtz operator
  // ... first term:
  for ( j=0; j<N2; j++ ) { 
    for ( i=0; i<N1; i++ ) { 
      for ( k=0; k<N1; k++ )
        (*H)(i+j*N1) += lc1 * ((*D1)(k,i)*(*D1)(k,i) * (*G11)(k+j*N1));
    }
  }

  // ... second term:
  for ( i=0; i<N1; i++ ) { 
    for ( j=0; j<N2; j++ ) { 
      for ( k=0; k<N2; k++ )
        (*H)(i+j*N1) += lc2*((*D2)(k,j)*(*D2)(k,j) * (*G22)(i+k*N1));
    }
  }

  // Include the mass matrix:
  *H +=   ( (*M) * mc) ;

  return TRUE;

} // end of method DefmQuadH


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadH
// DESCRIPTION: Computes Helmholtz operator*vector  for a deformed quad element.
//              Only diagonal^1 elements of operator are considered.
//              Diag elements are:
//                K_ij = 
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

//************************************************************************************
GBOOL PCPointJac_Helm::RectQuadH()
{ 
#if 1
  GINT      i, j;
  GDOUBLE     L1, L2, R1, R2 ;
  Point3D  *vert;
  GVector  *M  , *M1, *M2;  
  GMatrix  *K1 , *K2 ;

  if ( elem == NULL ) return FALSE;

  M   = elem->GetMassMatrix();  // contains Jacobian
  M1  = elem->Get1DWeights(1);
  M2  = elem->Get1DWeights(2);
  K1   = elem->Get1DStiffMatrix(1,FALSE);
  K2   = elem->Get1DStiffMatrix(2,FALSE);
  vert = elem->GetSpVertices();
  L1   = fabs((vert+1)->x1 - (vert)->x1);
  L2   = fabs((vert+3)->x2 - (vert)->x2);

  if ( L1 == 0 || L2 == 0 ) return FALSE;
  
  R1  = lc1 * L2 / L1;
  R2  = lc2 * L1 / L2;

 
  if ( K1  == NULL || K2  == NULL ||
       M1  == NULL || M2  == NULL ||
       M   == NULL                 ) return FALSE;
  
  // Compute diagonal elements of Helmholtz operator
  // ... first + second term of the stiffness matrices,
  // add in mass matrix component:
  for ( i=0; i<N1; i++ ) {
    for ( j=0; j<N2; j++ ) { 
      (*H)(i+j*N1) = 
                     ( R1*(*M2)     (j) * (*K1)(i,i)
                   +   R2*(*K2)   (j,j) * (*M1)  (i) ) 
                   +   mc*(*M )(i+j*N1)   ;
    }
  }

#else
  GINT        j;
  HelmholtzOp *Hf;
  GVector      tmp;

  Hf = new HelmholtzOp(elem);
  Hf->SetConst(lc1,lc2,mc);
  tmp.Resize(NN);
  *iH = 0.0;
  for ( j=0; j<NN; j++ ) {
    (*iH)[j] = 1.0;
    Hf->OpVec_prod(*iH, tmp);
    (*H) [j] = tmp[j];
    (*iH)[j] = 0.0;
  }
  delete Hf;
#endif  
  return TRUE;
} // end of method RectQuadH


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleH
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCPointJac_Helm::TriangleH()
{
  return FALSE;
} // end of method TriangleH

//************************************************************************************
//************************************************************************************
// METHOD     : Invert
// DESCRIPTION: Computes operator inverse, which forms preconditioner. The
//              operator must be globally assembled before this function is
//              called. This method must be called; else the operator*vector
//              product will fail.
// ARGUMENTS  : 
// RETURNS    : iH pointer on success; else NULL
//************************************************************************************
GVector *PCPointJac_Helm::Invert()
{ 
  GINT  i;

  for ( i=0; i<NN; i++ ) (*iH)(i) = 1.0/(*H)(i); 

  return iH;

} // end of method Invert
  

//************************************************************************************
//************************************************************************************
// METHOD     : GetiDiag
// DESCRIPTION: Gets inverse of diagonal--the actual preconditioner
//              'matrix'.
// ARGUMENTS  : 
// RETURNS    : iH pointer on success; else NULL
//************************************************************************************
GVector *PCPointJac_Helm::GetiDiag()
{ 
  return iH;
} // end of method GetiDiag
  

