//************************************************************************************//
// Module       : guser_heat_test.cpp
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
//************************************************************************************//
#include "gtypes.h"
#include "gaspar_t.h"
#include "guser.h"

//############################################################################################
//############################################################################################
//                                  Problem initialization:
//############################################################################################
//############################################################################################

//************************************************************************************
//************************************************************************************
// METHOD     : GUserConfig
// DESCRIPTION: User-furnished method to configure the code. This method is
//              called during setup, before GUserInit.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserConfig()
{

  // Where appropriate, set the number of quantities to solve for:
  nEvolvedFields_ = 1;
  bLinAdvection_  = TRUE;
  bDoAdvection_   = FALSE;

  return TRUE;
} // end of method GUserConfig



//************************************************************************************
//************************************************************************************
// METHOD     : GUserInit
// DESCRIPTION: User-furnished method to initialize bi-periodic Lorentzian 
//              and Gaussian 'lump' for solving Heat Eqn.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserInit(int argc, char *argv[])
{
  GINT       i, j, k, m, n, nlevels, N;
  GDOUBLE    xtheta, ytheta, x, y, *xv, *yv, *u, t0, del[GDIM];
  GBOOL      bRefining;
  GIBuffer   *bdyindices=NULL;
  GBTBuffer  *bdytype=NULL;
  Elem2D     *e;

  SetGlobals();

  t0 = 0.0;
  if ( bFixedTimeStep_ )
    dt_       /= (pow(2.0,(GINT)aux_));
  else
    Courant_  /= (pow(2.0,(GINT)aux_));

  nlevels             = ntimelevels_;
  bAutoTimeLevelFill_ = FALSE;
  bRefining           = TRUE;
  bDoCoarse_          = FALSE;
  while ( bRefining ) {
    time_ = t0;
    time_ -= dt_;
    for ( i=0; i<nelems_; i++ ) {  // reset all levels:
      for ( k=0; k<nlevels; k++ ) *(u1[i]->GetExpCoeffs(k)) = 0.0;
    }
    for ( k=nlevels-1; k>=0; k-- ) {
      time_ += dt_;
      for ( i=0; i<nelems_; i++ ) {
        u = u1[i]->GetExpCoeffs(k)->Data();
        xv = u1[i]->GetElement()->GetSpNodes(1)->Data();
        yv = u1[i]->GetElement()->GetSpNodes(2)->Data();
        N  = u1[i]->GetExpCoeffs(k)->dim();
        for ( n=0; n<nlumps; n++ ) {
          PLump(u, xv, yv, N, xp[n], yp[n],  time_, a1[n], TRUE); 
        }
        u1[i]->SetTime    (k,time_);
      } // end, element loop
      if ( k == (nlevels-1) ) GTimeStep(k);
      dthist_[k] = dt_;
    } // end, time level loop


//   Set DIRICHLET BC vector
    for ( i=0; i<nelems_; i++ ) {
      e = uelems[i];
      bdyindices = e->GetBdyIndices();
      bdytype    = e->GetBdyTypes  ();
      u1bdyvals[i]->Resize(bdyindices->dim());  // Remember, ubdyvals points to member data in Field

      for ( j=0; j<bdyindices->dim(); j++ ) {
        m = (*bdyindices)(j);
        if ( (*bdytype)(j) == NOSLIP ) {
          cout << "GUserInit: problem requires periodic boundaries" << endl;
          exit(1);
          u1 (0,i,m) = 0.0;
        }
        if ( (*bdytype)(j) == DIRICHLET ) {
          cout << "GUserInit: problem requires periodic boundaries" << endl;
          exit(1);
          u1bdyvals(i,j) = 0.0;
          u1     (0,i,m) = u1bdyvals(i,j);
        }
      }
    }

#if 1
    if ( !GAdaptGrid() ) {
      cout << "GUserInit:: adaption failed" << endl;
      exit(1);
    }
    bRefining = bElemListChange_;
    bElemListChange_ = FALSE;
    if ( bRefining ) {
      cout << "GUserInit: resampling on refined grid; nelems=" << nelems_ << endl;
    }
#endif
  }
  bElemListChange_ = TRUE;
  bDoCoarse_       = TRUE;
  GUserTimeDep(t0, 0.0);

  return TRUE;

} // end of method GUserInit
//############################################################################################
//############################################################################################


//************************************************************************************
//************************************************************************************
// METHOD     : GUserStart
// DESCRIPTION: User-furnished method to provide any information required after
//              a start or restart. For example, the method may set an analytic
//              solution, or set bdy conditions from an analytic solution, etc.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserStart()
{ 

  SetGlobals();

  return TRUE;
} // end of method GUserStart


//************************************************************************************
//************************************************************************************
// METHOD     : GUserTimeDep
// DESCRIPTION: User-furnished method to provide any information required 
//              during time stepping. This method is called at the start 
//              of the time step, before the solver takes a step. 
//              For example, time dependent boundary conditions, 
//              or a non-steady analytic solution might be computed here.
// ARGUMENTS  : ptime: current time
//              pdt  : current time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserTimeDep(GDOUBLE ptime, GDOUBLE pdt)
{
  GINT       i, j, n, N;
  GDOUBLE    *xv, *yv, tt;
  GVector    *uv, *uva, *duv;
  Elem2D     *e;

  tt = ptime + pdt;
  for ( i=0; i<nelems_; i++ ) {
    uva = u1a [i]->GetExpCoeffs(0);
    uv  = u1  [i]->GetExpCoeffs(0);
    duv = delu[i]->GetExpCoeffs(0);
    *uva = 0.0;
    xv = u1a[i]->GetElement()->GetSpNodes(1)->Data();
    yv = u1a[i]->GetElement()->GetSpNodes(2)->Data();
    N  = uva->dim();
    for ( n=0; n<nlumps; n++ ) {
      PLump(uva->Data(), xv, yv, N, xp[n], yp[n], tt, a1[n], TRUE);
    }
    *duv = (*uva) - (*uv);
  }

  return TRUE;
} // end of method GUserTimeDep


//************************************************************************************
//************************************************************************************
// METHOD     : GUserLogConfig
// DESCRIPTION: Provides user with a place to configure static and dynamic components
//              of log file.  Data variables set in the SetStaticParams call must be 
//              globalized, and this may be done by placing them in the 'guser.h' header.

// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserLogConfig()
{
  GBOOL bOk;
  
  bOk = glogger_.SetDynamicParamDesc("%f"
                                   , "#L2_ERR"
                                    );
  bOk = bOk &
        glogger_.SetDynamicData(1 , &errL2
                                 );

  return bOk;

} // end of method GUserLogConfig


//************************************************************************************
//************************************************************************************
// METHOD     : GUserLogUpdate
// DESCRIPTION: Provides user with a place to update variables that are
//              printed in the dynamic log component of the log file.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserLogUpdate()
{
  errL2 = ComputeL2Error();
  return TRUE;
} // end of method GUserLogUpdate


//************************************************************************************
//************************************************************************************
// METHOD     : GUserTerm
// DESCRIPTION: Provids user with a place to perform some activity at the
//              end of time stepping
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserTerm()
{
  GINT      pe;
  ofstream  os;
  os.open("../heat.conv",ios::app);

  GUserLogUpdate(); // compute new error

  pe = uelems[0]->GetOrder(1);

  // write error to file:
  os << pe << " " << errL2 << endl;
  os.close();

  return TRUE;
} // end of method GUserTerm


//************************************************************************************
//************************************************************************************
// METHOD     : PLump
// DESCRIPTION: Produces periodic lump solution
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void PLump(GDOUBLE u[], GDOUBLE x[], GDOUBLE y[], GINT nxy, GDOUBLE xp, GDOUBLE yp, 
           GDOUBLE t, GDOUBLE sig_0, GBOOL bAdd)
{
  GINT    j, k, n;
  GDOUBLE wsum, prod, argxp, argxm, da, eps=1e-18;
  GDOUBLE xx[2], si[2], sig[2], ufact=1.0;
  GBOOL   bContin;

  sig[0] = sqrt(sig_0*sig_0 + 2.0*t*nu_[0]);
  sig[1] = sqrt(sig_0*sig_0 + 2.0*t*nu_[1]);
  si [0] = 0.5/(sig[0]*sig[0]);
  si [1] = 0.5/(sig[1]*sig[1]);
  ufact = bAdd ? 1.0 : 0.0;

  for ( j=0; j<nxy; j++ ) {
    xx [0] = x[j]-xp;
    xx [1] = y[j]-yp;

    prod = 1.0;
    for ( k=0; k<2; k++ ) {
      wsum     = exp(-xx[k]*xx[k]*si[k]);
      n       = 1;
      bContin = TRUE;
      while ( bContin ) {
        argxp = -pow((xx[k]+n*gL_[k]),2.0)*si[k];
        argxm = -pow((xx[k]-n*gL_[k]),2.0)*si[k];
        da    =  exp(argxp) + exp(argxm);
        wsum  += da;
        bContin = da/wsum > eps;
        n++;
      }
      prod *= wsum;
    }
    u[j] = ufact*u[j] + pow(sig_0,2)/pow(sig[0],2)*prod;
  }

}  // end of method PLump


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeL2Error
// DESCRIPTION: Computes L2 error
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE ComputeL2Error()
{

  GINT       i, j, m;
  GDOUBLE    xtheta, ytheta, errnum, errden;

  GUserTimeDep(time_, 0.0);  
  errden = GUtils::ComputeL2Norm(u1a , 0);
  errnum = GUtils::ComputeL2Norm(delu, 0);

  return (errnum / errden);

}  // end of method ComputeL2Error

//************************************************************************************
// METHOD     : SetGlobals
// DESCRIPTION: Method set user-global variables
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL SetGlobals()
{
  GINT       i, j, n;
  GDOUBLE    del[GDIM];

  bDoAdvection_       = FALSE;
  gL_[0]     = PDISTANCE(gd_[1],gd_[0]);
  gL_[1]     = PDISTANCE(gd_[2],gd_[1]);

  nlumps = 1;
  a1.Resize(nlumps); a2.Resize(nlumps);
  xp.Resize(nlumps); yp.Resize(nlumps);
//a1[0]     = -log(0.80); a2[0]     = -log(0.80);
//a1[1]     = -log(0.80); a2[1]     = -log(0.80);
  for ( j=0; j<GDIM; j++ ) {
    del[j]    = gL_[j] / (nlumps+1);
    nu_[j]    = 0.1;
  }
  for ( n=0; n<nlumps; n++ ) {
    a1[n]     = 0.05      ;
    xp[n]     = (n+1)*del[0];
    yp[n]     = (n+1)*del[1];
  }

  for ( i=0; i<nelems_; i++ ) {
    u1a .add(1,uelems.member(i));
    delu.add(1,uelems.member(i));
  }
  if ( GRegisterUserFields(2, &u1a, "V1A", &delu, "DelV") == 0 ) {
   cout << "SetGlobals: user field registration failed" << endl;
   exit(1);
  }

//sig_mult_ = 0.01 ;
//sig_tol_ =  20.0;

  return TRUE;
} // end of method SetGlobals
