//************************************************************************************//
// Module       : guser_nwave_test.cpp
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
//************************************************************************************//
#include "gtypes.h"
#include "gaspar_t.h"
#include "guser.h"

//############################################################################################
//############################################################################################
//                                  Problem initialization:
//############################################################################################
//############################################################################################

#undef DO_DIRICHLET_PLANARNWAVE
#define NWAVE_STD

//************************************************************************************
//************************************************************************************
// METHOD     : GUserConfig
// DESCRIPTION: User-furnished method to configure the code. This method is
//              called during setup, before GUserInit.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserConfig()
{

  // Where appropriate, set the number of quantities to solve for:
  nEvolvedFields_ = 2;

  return TRUE;
} // end of method GUserConfig


//************************************************************************************
//************************************************************************************
// METHOD     : GUserInit
// DESCRIPTION: User-furnished method to initialize Burgers
//              problem, using N-wave solution.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserInit(int argc, char *argv[])
{
  GINT       i, j, k, m, nlevels, nrefine=0;
  GDOUBLE    x, y, time0;
  GBOOL      bRefining=TRUE;
  GIBuffer  *bdyindices=NULL;
  GBTBuffer *bdytype=NULL;
  Elem2D     *e;
  
  // Set up user fields:
  RegisterMyFields();
 
  for ( i=0; i<nelems_; i++ ) {
    u1a.add(1,uelems.member(i));
    u2a.add(1,uelems.member(i));
    kea.add(1,uelems.member(i));
    ke .add(1,uelems.member(i),2);
  }

#if 0
  if ( GSetAPostFields(1, &ke) == 0 ) {
    cout << "GUserInit: Error in setting a-posteriori fields" << endl;
    exit(1);
  }
#endif

  rho0 = 0.0;
  // Set NWave parameters. Two types are provided (see Whitham's book):
  // those accommodating a Reynolds number (R0, t0, nu) and those
  // that don't (A, nu). Which is used is determined by whether
  // NWAVE_STD is set, in which case the (A,nu) params are used; 
  // else the (R0, t0, nu) params are used. See NWave method.
  SetMyParams();
  time0 = time_;

  if ( bFixedTimeStep_ ) {
    dt_       /= (pow(2.0,(GINT)aux_));
  }
  else {
    Courant_  /= (pow(2.0,(GINT)aux_));
  }

 
  bAutoTimeLevelFill_ = FALSE; 
  nlevels = ntimelevels_;
  if ( bAutoTimeLevelFill_ ) nlevels = 1;
  bDoCoarse_          = TRUE;
  while ( bRefining ) {
cout << "____________________________________________________________________________________" << endl;
cout << "                                  refine_cycle=" << nrefine << " nelems=" << nelems_ << endl;

    time_ -= dt_;
    for ( k=nlevels-1; k>=0; k-- ) {
      time_ += dt_;
      for ( i=0; i<nelems_; i++ ) {
        for ( j=0; j<u1[i]->GetExpCoeffs(0)->dim(); j++ ) {
          x = u1.X(i,1,j) - rho0[0];
          y = u1.X(i,2,j) - rho0[1];
          if ( nEvolvedFields_ == 1 ) { 
            PlNWave(u1 (k,i,j),x,y,time_);
          }
          else {
            NWave(u1 (k,i,j),u2 (k,i,j),x,y,time_,FALSE);
          }
        } // end of node loop
        u1(i)->SetTime    (k,time_); 
        if ( nEvolvedFields_ > 1 )
        u2(i)->SetTime    (k,time_);
      } // end of element loop
      if ( k == (nlevels-1) ) GTimeStep(k);
      dthist_[k] = dt_;
    }   // end of time-level loop

    bTimeDepBdy_   = TRUE;
    GUserTimeDep(time_, 0.0);
#if 1
    if ( !GAdaptGrid() ) {
      cout << "GUser:: adaption failed" << endl;
      exit(1);
    }
#else
    bElemListChange_ = FALSE; 
#endif
    bRefining = bElemListChange_ && nrefine < nrefinelevels_ ;
    bElemListChange_ = FALSE;
cout << "                                                       refine_cycle=" << nrefine << endl;
cout << "____________________________________________________________________________________" << endl;
    nrefine++;
  }
  bDoCoarse_  = TRUE;

  return TRUE;

} // end of method GUserInit
//############################################################################################
//############################################################################################


//************************************************************************************
//************************************************************************************
// METHOD     : GUserStart
// DESCRIPTION: User-furnished method to provide any information required after
//              a start or restart. For example, the method may set an analytic
//              solution, or set bdy conditions from an analytic solution, etc.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserStart()
{ 
  GINT       i, j, k, m;
  GDOUBLE    x, y ;
  GIBuffer   *bdyindices=NULL;
  GBTBuffer  *bdytype=NULL;
  Elem2D     *e;

  // Set up user fields:
  RegisterMyFields();
  for ( i=0; i<nelems_; i++ ) {
    u1a.add(1,uelems.member(i));
    u2a.add(1,uelems.member(i));
    kea.add(1,uelems.member(i));
    ke.add(1,uelems.member(i));
  }

  SetMyParams();

  bTimeDepBdy_ = TRUE;
  GUserTimeDep(time_, 0.0);
  return TRUE;
} // end of method GUserStart


//************************************************************************************
//************************************************************************************
// METHOD     : GUserTimeDep
// DESCRIPTION: User-furnished method to provide any information required 
//              during time stepping. This method is called at the start 
//              of the time step, before the solver takes a step. 
//              For example, time dependent boundary conditions, 
//              or a non-steady analytic solution might be computed here.
// ARGUMENTS  : ptime: current time
//              pdt  : current timestep
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserTimeDep(GDOUBLE ptime, GDOUBLE pdt)
{ 
  GINT        i, j, k, m;
  GDOUBLE     x, y, tt;
  GIBuffer   *bdyindices=NULL;
  GBTBuffer  *bdytype=NULL;
  GVector    *evec, **vec, **tmp;
  Elem2D     *e;


  tt = ptime + pdt ;

  // Update KE:
  vec = new GVector * [nEvolvedFields_];
  tmp = new GVector * [nEvolvedFields_];
  ke.start(NULL);
  for ( k=0; k<nEvolvedFields_; k++ ) pgfields_[k]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    evec   = ke.member()->GetExpCoeffs(0);
    vec[0] = pgfields_[0]->member()->GetExpCoeffs(0);
    MTK::fvec_point_prod(*vec[0], *vec[0], *evec);      // initialize ke
    for ( k=1; k<nEvolvedFields_; k++ ) {
      m = k - 1;
      tmp[m] = pgfields_[k]->member()->GetTemp();      // locks memory area
      vec[m] = pgfields_[k]->member()->GetExpCoeffs(0);
      MTK::fvec_point_prod(*vec[m], *vec[m], *tmp[m]);
      MTK::fvec_add_rep(*evec, *tmp[m]);                // update ke
      pgfields_[k]->member()->TempUnlock(tmp[m]);
    }
    ke.next();
    for ( k=0; k<nEvolvedFields_; k++ ) pgfields_[k]->next();
  }
  delete [] vec;
  delete [] tmp;

  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<u1a[i]->GetExpCoeffs(0)->dim(); j++ ) {
     x = u1.X(i,1,j) - rho0[0];
     y = u1.X(i,2,j) - rho0[1];
     if ( nEvolvedFields_ == 1 ) { 
      PlNWave(u1a(0,i,j),x,y,tt);
      kea(0,i,j) = pow(u1a(0,i,j),2);
     }
     else {
      NWave(u1a(0,i,j),u2a(0,i,j),x,y,tt,FALSE);
      kea(0,i,j) = pow(u1a(0,i,j),2) + pow(u2a(0,i,j),2);
     }
    }
  }

  // Update boundary vectors:
  if ( !bTimeDepBdy_ ) return TRUE;
  for ( i=0; i<nelems_; i++ ) {
//   Set DIRICHLET BC vector
    e = uelems[i];
    bdyindices = e->GetBdyIndices();
    bdytype    = e->GetBdyTypes  ();
    u1bdyvals[i]->Resize(bdyindices->dim());  // Remember, ubdyvals points to member data in Field
    if ( nEvolvedFields_ > 1 ) 
    u2bdyvals[i]->Resize(bdyindices->dim());  

    for ( j=0; j<bdyindices->dim(); j++ ) {
      m = (*bdyindices)(j);
      if ( (*bdytype)(j) == PERIODIC ) {
        cout << "GUserInit: Periodic boundaries not allowed." << endl;
        exit(1);
      }
      else if ( (*bdytype)(j) == DIRICHLET ) {
        x = u1.X(i,1,m) - rho0[0];
        y = u1.X(i,2,m) - rho0[1];
        if ( nEvolvedFields_ == 1 ) 
          PlNWave(u1bdyvals(i,j),x,y,tt);
        else
          NWave(u1bdyvals(i,j),u2bdyvals(i,j),x,y,tt,FALSE);
      }
    }
//cout << "GUserTimeDep: u1bdyvals[" << i << "]=" << *(u1bdyvals[i]) << endl;
  }
//return DoCompare("nwave_conv.dat");

  return TRUE;
} // end of method GUserTimeDep


//************************************************************************************
//************************************************************************************
// METHOD     : GUserLogConfig
// DESCRIPTION: Provides user with a place to configure static and dynamic components
//              of log file.  Data variables set in the SetStaticParams call must be 
//              globalized, and this may be done by placing them in the 'guser.h' header.

// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserLogConfig()
{
  GBOOL bOk;

  bOk = glogger_.SetDynamicParamDesc("%f"
                                   , "#L2_ERR"
                                    );
  bOk = bOk &
        glogger_.SetDynamicData(1 
                              , &gdUL2
                               );

  return bOk;

} // end of method GUserLogConfig


//************************************************************************************
//************************************************************************************
// METHOD     : GUserLogUpdate
// DESCRIPTION: Provides user with a place to update variables that are
//              printed in the dynamic log component of the log file.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserLogUpdate()
{
  GBOOL bOk;
  bOk = GUserTimeDep(time_, 0.0);
  bOk = bOk && DoCompare(NULL);
  return bOk;
} // end of method GUserLogUpdate


//************************************************************************************
//************************************************************************************
// METHOD     : GUserTerm
// DESCRIPTION: Provids user with a place to perform some activity at the
//              end of time stepping
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserTerm()
{
  if ( bFileWrite ) {
    if ( !GUserTimeDep(time_, 0.0) ) return FALSE;
    return DoCompare("../nwave_conv.dat");
  }

  return TRUE;
} // end of method GUserTerm


//************************************************************************************
//************************************************************************************
// METHOD     : NWave
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void NWave(GDOUBLE &ux, GDOUBLE &uy, GDOUBLE x, GDOUBLE y, GDOUBLE t, GBOOL bplanar)
{
  GDOUBLE r2, denom, tdenom, efact, xfact, tfact, trt;
  
  if ( bplanar ) r2 = x*x ;
  else           r2 = x*x + y*y;
#if !defined(NWAVE_STD)
  // NOTE: not tested in 2D!
  tdenom = 1.0/(4.0*nu_[0]*t);
  trt    = bplanar ? sqrt(t/t0): t/t0;
  denom  = 1.0 / ( exp(R0) - 1.0 );
  xfact  = 1.0 /( t * ( 1.0+(trt*denom*exp(r2*tdenom)) ) );

  if ( bplanar ) {
    ux     = x * xfact ;
    uy     = 0.0 ;
  } else {
    ux     = x * xfact ;
    uy     = y * xfact ;
  }
#else
  tdenom = 1.0/(4.0*nu_[0]*t);
  tfact  = bplanar ? sqrt(A/t) : A/t;
  efact  = tfact * exp(-r2*tdenom);
  xfact  = efact/( t * (  1.0 + efact ) );

  if ( bplanar ) {
    ux     = x * xfact ;
    uy     = 0.0 ;
  } else {
    ux     = x * xfact ;
    uy     = y * xfact ;
  }
#endif
}  // end of method NWave


//************************************************************************************
//************************************************************************************
// METHOD     : PlNWave
// DESCRIPTION: solutions for planar N-wave. If DO_DIRICHLET_PLANARNWAVE preproc 
//              variable is set, then Dirichlet version is used; else
//              periodic solution is generated.
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void PlNWave(GDOUBLE &ux, GDOUBLE x, GDOUBLE y, GDOUBLE t)
{
#if defined(DO_DIRICHLET_PLANARNWAVE)
  GDOUBLE uy;
  
  NWave(ux, uy, x, y, t,TRUE);
#else

  // This section computes a planar periodic solution.
  // Must adjust the boudary condition checks so that 
  // periodic conditions are allowed./
  GINT    i=1, j=1;
  GDOUBLE sum, uy, dux, utpx, utpy, utmx, utmy, xpp, ypp, xpm, ypm, eps=1e-18;
  GDOUBLE gfactp, gfactm, nufact, num, den, dnum, dden , tinv;
  GBOOL   bContin=TRUE;

  xpp = x;
  ypp = 0.0;
  ypm = 0.0;
  nufact = 1.0/(4.0*nu_[0]*t);
  tinv   = 1.0 / t;
  gfactp = exp(-xpp*xpp*nufact);
  num    = xpp*tinv * gfactp;
  den    = gfactp;
  while ( bContin ) {
    xpp     = x + i*0.5*gL_[0];
    xpm     = x - i*0.5*gL_[0];
    gfactp  = exp(-xpp*xpp*nufact);
    gfactm  = exp(-xpm*xpm*nufact);
    dnum    = xpp*tinv * gfactp + xpm*tinv * gfactm;
    dden    =            gfactp +            gfactm;
    num    += dnum;
    den    += dden;
    bContin = fabs(dnum) > eps*fabs(num) ||
              fabs(dden) > eps*fabs(den) ;
    i++;
  }
  ux = num / den;
#endif
}  // end of method PlNWave


//************************************************************************************
//************************************************************************************
// METHOD     : DoCompare
// DESCRIPTION: Compare analytic solution to user solution; place 
//              diff norm in specified file
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL DoCompare(const char *filename)
{
  GINT       i, j, m, N;
  GDOUBLE    dU, gdULinf, norm[6]={0.0}, gnorm[6]={0.0};
  GVector   *v1, *v1a, *v2, *v2a, *vsq, *vasq, *dvsq, *dv1, *dv2;
  Elem2D     *e;
  ofstream   os;

  dv1  = new GVector ();
  dv2  = new GVector ();
  vsq  = new GVector ();
  vasq = new GVector ();
  dvsq = new GVector ();

  for ( j=0; j<6; j++ ) norm[j] = 0.0;

  for ( i=0; i<nelems_; i++ ) {
    N   = u1 .member(i)->GetExpCoeffs(0)->dim();
    v1  = u1 .member(i)->GetExpCoeffs(0);
    v1a = u1a.member(i)->GetExpCoeffs(0);
    dv1 ->Resize(N);
    if ( nEvolvedFields_ > 1 ) {
      v2a = u2a.member(i)->GetExpCoeffs(0);
      v2  = u2 .member(i)->GetExpCoeffs(0);
      dv2 ->Resize(N);
    }
    vsq ->Resize(N);
    vasq->Resize(N);
    dvsq->Resize(N);
    MTK::fvec_sub(*v1, *v1a, *dv1);
    if ( nEvolvedFields_ > 1 ) {
      MTK::fvec_sub(*v2, *v2a, *dv2);
    }
    for ( j=0; j<N; j++ ) {
      (*vsq )(j) = (*v1 )(j)*(*v1 )(j) ;
      (*vasq)(j) = (*v1a)(j)*(*v1a)(j) ;
      (*dvsq)(j) = (*dv1)(j)*(*dv1)(j) ;
      if ( nEvolvedFields_ > 1 ) {
        (*vsq )(j) += (*v2 )(j)*(*v2 )(j) ;
        (*vasq)(j) += (*v2a)(j)*(*v2a)(j) ;
        (*dvsq)(j) += (*dv2)(j)*(*dv2)(j) ;
      }
    }
    norm[0] += u1.member(i)->GetElement()->Integrate(dvsq, NULL);
    norm[1] += u1.member(i)->GetElement()->Integrate(vasq, NULL);
    norm[2] += u1.member(i)->GetElement()->Integrate(vsq , NULL);
  }

  GComm::Allreduce(norm, gnorm, 3, GC_GDOUBLE, G_OP_SUM);
  gdUL2   = sqrt( gnorm[0]/ ( gnorm[1] ) );
//gdULinf =     ( gnorm[4]/ ( gnorm[5] + TINY ) );

  if ( GComm::WorldRank() == 0 && filename != NULL && bFileWrite ) {
    os.open(filename,ios::app);
  //os.width(22);
  //os.precision(15);
    os       << rank
             << " " << icycle_
             << " " << time_
             << " " << dt_
             << " " << uelems[0]->GetOrder(1)
             << " " << uelems[0]->GetOrder(2)
             << " " << ngelems_
             << " " << gdUL2
             << " " << sqrt(gnorm[0])
             << " " << sqrt(gnorm[1])
             << " " << sqrt(gnorm[2])
//           << " " << gdULinf
             << endl;
    os.close();
  }

  delete dv1 ;
  delete dv2 ;
  delete vsq ;
  delete vasq ;
  delete dvsq;

  return TRUE;

} // end of method DoCompare

//************************************************************************************
//************************************************************************************
// METHOD     : RegisterMyFields
// DESCRIPTION: Register user fields
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void RegisterMyFields()
{   
  if ( GRegisterUserFields(1, &ke, "KE") == 0 ) {
    cout << "RegisterMyFields: user field V1A registration failed" << endl;
    exit(1);
  }
  if ( GRegisterUserFields(1, &u1a, "V1A") == 0 ) {
    cout << "RegisterMyFields: user field V1A registration failed" << endl;
    exit(1);
  }
  if ( nEvolvedFields_ > 1 && GRegisterUserFields(1, &u2a, "V2A") == 0 ) {
    cout << "RegisterMyFields: user field V1A registration failed" << endl;
    exit(1);
  }
  if ( GRegisterUserFields(1, &kea, "KEA") == 0 ) {
    cout << "RegisterMyFields: user field KEA registration failed" << endl;
    exit(1);
  }
} // end of method RegisterMyFields


//************************************************************************************
//************************************************************************************
// METHOD     : SetMyParams
// DESCRIPTION: Set problem params
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void SetMyParams()
{   
  if ( nEvolvedFields_ == 1 ) {
    nu_[0] = nu_[1] = 1.0; 
    A    = 1.0;
    R0   = 0.1;
    t0   = 1.0;
    if ( time_ == 0.0 ) time_= 1.0e-2;
  }
  else {
    nu_[0] = nu_[1] = 5.0e-3;
    A    = 1.0e4;
    R0   = 5.0;
    t0   = 5e-2;
    if ( time_ == 0.0 ) time_= t0; 
  }
  gL_[0] = PDISTANCE(gd_[1],gd_[0]);
  gL_[1] = PDISTANCE(gd_[2],gd_[1]);
  kx   = 2.0*PI / gL_[0];
  ky   = 2.0*PI / gL_[1];
  rho0[0] =  gd_[0].x1 + 0.5*gL_[0];
  rho0[1] =  gd_[0].x2 + 0.5*gL_[1];
  bTimeDepBdy_ = TRUE;
    

} // end of method SetMyParams

