//************************************************************************************//
// Module       : gtmatbuffer.cpp
// Date         : 6/5/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a template matrix-buffer object, whose data is composed of
//                variable length buffers.
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include <typeinfo>
#include "gtmatbuffer.hpp"


//************************************************************************************
//************************************************************************************
// constructor (1)
template<class T> GTMatBuffer<T>::GTMatBuffer<T>()
:
n1                    (0),
n2                    (0),
dtype                 (G_GDOUBLE),
data                  (NULL)
{

  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;


} // end of constructor 1 



//************************************************************************************
//************************************************************************************
// constructor (2)
template<class T>  GTMatBuffer<T>::GTMatBuffer<T>(const GINT   size1, const GINT   size2)
:
n1                    (size1),
n2                    (size2),
dtype                 (G_GDOUBLE),
data                  (NULL)
{
  GINT   i;

  data = new GBuffer<T> * [n1];
  for ( i=0; i<n1; i++ ) {
    data[i] = new GBuffer<T> (n2);
  }

  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;

} // end of constructor 2


//************************************************************************************
//************************************************************************************
// constructor (3)
template<class T>  GTMatBuffer<T>::GTMatBuffer<T>(T *array, GINT   m1, GINT   m2)
:
n1                    (m1),
n2                    (m2),
dtype                 (G_GDOUBLE),
data                  (NULL)
{
  GINT   i,j;


  if      ( typeid(T) == typeid(GUSHORT ) )
   dtype = G_GUSHORT ;
  else if ( typeid(T) == typeid(GSHORT ) )
    dtype = G_GSHORT ;
  else if ( typeid(T) == typeid(GINT ) )
    dtype = G_GINT ;
  else if ( typeid(T) == typeid(GDOUBLE) )
    dtype = G_GDOUBLE;
  else if ( typeid(T) == typeid(GQUAD) )
    dtype = G_GQUAD;

  data = new GBuffer<T> * [n1];
  for ( i=0; i<n1; i++ )
  {
    data[i] = new  GBuffer<T> (n2);
    for ( j=0; j<n2; j++ )
      memcpy(data[i]->Data(), array+i*n1, n2*G_TYPESZ[dtype]);
  }


} // end of constructor 3


//************************************************************************************
//************************************************************************************
// Copy constructor:
template<class T> GTMatBuffer<T>::GTMatBuffer<T>(const GTMatBuffer<T> &m)
{
} // end of copy constructor


//************************************************************************************
//************************************************************************************
// Destructor
template<class T> GTMatBuffer<T>::~GTMatBuffer<T>()
{
   DeleteDynamic();
}

//************************************************************************************
//************************************************************************************
// Assignment operator method
template<class T> GTMatBuffer<T> GTMatBuffer<T>::operator=(const GTMatBuffer<T> &m)
{
  if ( data == NULL ) return *this;


  GINT   i;

  if ( &m != this ) {
    if ( m.n1 != n1 || m.n2 != n2 ) {
      cout << "GTMatBuffer<T>::=: incompatible matrices" << endl;
      exit(1);
    }
    // copy member data:
    n1      = m.n1;
    n2      = m.n2;
    dtype        = m.dtype;


    //  copy array data:
    if ( m.data != NULL ) {
      for ( i=0; i<n1; i++ ) {
        *data[i] = *(m.data[i]);
      }
    }
  
  }
  return *this;

} // end = operator

//************************************************************************************
//************************************************************************************
// Assignment operator method
template<class T> void  GTMatBuffer<T>::operator=(T m)
{
  GINT   i;

  if ( data == NULL ) {
     cout << "GTMatBuffer<T>::operator=(T m): illegal assignment" << endl;
     exit(1);
  }

  for ( i=0; i<n1; i++ ) {
      (*data[i]) = m;
  }

} // end = operator


//************************************************************************************
//************************************************************************************
// << operator method 
template<class T> ostream &operator<<(ostream &str, GTMatBuffer<T> a)
{

  str << "{ ";

  GINT   i, j;
  for ( i=0; i<a.a.dim(1); i++ ) {
    str << "{ ";
    for ( j=0; j<a.dim(2)-1; j++ )
      str  
//        << setiosflags(ios::scientific)
//        << setw(11)
//        << setprecision(4)
//        << setiosflags(ios::fixed)
          << a(i,j)
//        << setw(1)
          << ", ";

      str
//        << setiosflags(ios::scientific)
//        << setw(11)
//        << setprecision(4)
//        << setiosflags(ios::fixed)
          << a(i,j) ;
//        << setw(1) ;
    if ( i < a.dim(1)-1 ) str << " }, ";
    else str << " }";
  }
    str << " } \n ";

  return str;
} // end of << operator


//************************************************************************************
//************************************************************************************
// METHOD     : Data
// DESCRIPTION: 
// RETURNS    :  pointer to the data area
template<class T> GBuffer<T> **GTMatBuffer<T>::Data() const
{
  return data;
} // end of method Data


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// RETURNS    :  none
template<class T> void GTMatBuffer<T>::DeleteDynamic()
{
  GINT   i;

  for ( i=0; i<n1; i++ )
  {
    delete data[i];
    data[i] = NULL;
  }
  delete [] data;
  data = NULL;

} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// RETURNS    :  TRUE on success, else FALSE
template<class T> GBOOL GTMatBuffer<T>::Resize(GINT   new1, GINT   new2)
{

  GINT   i;
  GBOOL bRet = TRUE;

  if ( new1 == n1 && new2 == n2 ) return TRUE;

  for ( i=0; i<n1; i++ ) {
    delete data[i];
    data[i] = NULL;
  }
  data = NULL;

  n1 = new1;
  n2 = new2;

  data = new GBuffer<T> * [n1];
  if ( data == NULL ) return FALSE;
  for ( i=0; i<n1; i++ ) {
    data[i] = new GBuffer<T>(n2);
    if ( data[i] == NULL ) {  
      bRet = FALSE;
      break;
    }
  }

  return bRet;

} // end of method Resize


//************************************************************************************
//************************************************************************************
// METHOD     : tsize
// DESCRIPTION: total size of data storage area
//              in direction idir (= n + npad_)
// RETURNS    : GINT   size
template<class T> GINT   GTMatBuffer<T>::tsize(GINT   idir)
{

  if      ( idir == 1 )
    return (n1 );
  else if ( idir == 2 )
    return ( n2 ); 
  else
    return 0;
} // end of method tsize


//************************************************************************************
//************************************************************************************
// METHOD     : dim
// DESCRIPTION: array dimension (usable)
//              in direction idir 
// RETURNS    : GINT   size
template<class T> GINT   GTMatBuffer<T>::dim(GINT   idir) const
{

  if      ( idir == 1 )
    return (n1);
  else if ( idir == 2 )
    return (n2);
  else
    return 0;
} // end of method dim



