//************************************************************************************
// Module       : gutils.cpp
// Date         : 2003/10/08 (AF)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Namespace encapsulating miscellaneous C-style utilities
//************************************************************************************


#include <string>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include "gutils.hpp"
#include "gcomm.hpp"
#include "timer.h"

GBOOL bLocal_=FALSE;

//************************************************************************************
//************************************************************************************
// METHOD     : InitMesh
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
// MODIFIC'NS : Paste from helm_t.cpp r1.30
//************************************************************************************

GBOOL GUtils::InitMesh(char *fnmesh, GSHORT  rank, GTBasisListGLL &gllpool,
               GElemList &gelems, GINT  &gnelems_, GNIDBuffer  *&vnode_ids, GNODEID &vmaxid, 
               Point *&gd, GINT  &ngd, GINT ntmp, GINT  nadapt)
{
  char       *serr = "GUtils::InitMesh: ";
  GSHORT     *bgbdy=NULL; 
  GINT       i, j, k, m, ind;
  GINT       n=0, ne, nv, nelems_, gne_max_;
  GLONG      *xxN=NULL;
  GBOOL      bVertex;
  GKEY       rootid;
  ELEMTYPE   etype;
  GIBuffer   *edge_ind, *vert_ind, *ebi;
  GLBuffer   *bi=NULL, **ni=NULL;
  GBTBuffer  *bt=NULL, *ebt=NULL;
  Point3D    *vert3=NULL;
  MeshReader mr;
 
  // Read mesh file, and get relevant v-mesh parameters
  if ( !mr.Open(fnmesh)  ) { 
    cout << serr << "Open failed: " << mr.Error() << " (file " << fnmesh << ")" << endl;
    return FALSE;
  }

  mr.SetProc(rank);
  mr.bGetNodeIDs(FALSE);
  nelems_ = mr.GetNumElements();
  GComm::Allreduce(&nelems_, &gnelems_, 1, GC_GINT , G_OP_SUM);
  GComm::Allreduce(&nelems_, &gne_max_, 1, GC_GINT , G_OP_MAX);
  mr.GetGlobalVerts(gd, ngd);
  ni      = new GLBuffer  * [nelems_];
  for ( i=0; i<nelems_; i++ ) ni[i] = NULL;
  while ( n<nelems_ && mr.GetElem(etype, vert3, bgbdy, xxN, ni[n], bi, bt)==0 ) { 
    gelems.add(etype, ntmp);
    for ( j=0; j<GDIM; j++ ) { 
      if ( !gllpool.findorder(xxN[j]-1) ) gllpool.add(xxN[j]-1);
      gllpool.member()->Solve();
      gelems.member()->SetBasis(gllpool.member(), j+1);
    }
    gelems.member()->SetVertices(vert3,4);
    gelems.member()->SolveFE();
//  ubdyvals .add(NULL      ,bi->dim());
    
    // Set boundary conditions:
    ebi      = gelems.member()->GetBdyIndices();
    ebt      = gelems.member()->GetBdyTypes  ();
    ne       = gelems.member()->GetNumEdges();   
    nv       = gelems.member()->GetNumVertices();   
    edge_ind = gelems.member()->GetEdgeIndices();
    vert_ind = gelems.member()->GetVertexIndices();
    for ( j=0; j<ne; j++ ) {
      k = (j+1)%ne;
      gelems.member()->bGlobalBdyEdge(j) = 0;
      for ( m=0; m<edge_ind[j].dim() && !gelems.member()->bGlobalBdyEdge(j); m++ ) {
        bVertex = edge_ind[j][m] == vert_ind[j][0] || edge_ind[j][m] == vert_ind[k][0];
        if ( bi->contains(edge_ind[j][m],ind) ) {
          if ( !bVertex && bgbdy[j] ) {
            gelems.member()->bGlobalBdyEdge(j) = 1;
            gelems.member()->GetEdgeType   (j) = (*bt)(ind);
          }
        }
      }
    }
    for ( j=0; j<nv; j++ ) {
      if ( bi->contains(vert_ind[j][0],ind) ) {
        gelems.member()->GetVertType(j) = (*bt)(ind);
      }
    }
    gelems.member()->ComputeBdyInfo();

    // Compute element id and its root id. Initially
    // these are equal:
    rootid = GUtils::RootID(rank*gne_max_+n,nadapt);
    gelems.member()->SetRootID  (rootid);
    gelems.member()->SetParentID(rootid);
    gelems.member()->SetID      (rootid);

#if defined(GU_DEBUG_OUTPUT)
    cout << serr << "root_key[" << n << "]=" << rootid << endl;
    cout << serr << "elem[" << n << "]: " << *gelems[n] << endl;
//  cout << serr << "elem[" << n << "]; bdy_indices=" << *ebi
//       << endl << " bdy_types=" << *ebt << endl;
#endif
    n++;
    if ( bi ) delete bi; bi = NULL;
    if ( bt ) delete bt; bt = NULL;
  }
  if ( n != nelems_ ) {
    cout << serr << "inconsistent number of elements" << endl;
    return FALSE;
  }
  if ( mr.ErrorID() != 0 ) {
    cout << serr << "GetElemBlock failed: Error: " << mr.Error() << endl;
    return FALSE;
  }
  mr.Close();


  // Find dynamic range for v-mesh node-ids
#if 0
  vmaxid = 0;
  for ( i=0,sum=0; i<n; i++ ) {
    sum += ni[i]->dim();
    for ( j=0; j<ni[i]->dim(); j++ ) vmaxid = MAX(vmaxid,(*ni[i])(j));
  }
  // make 'flat' node id list from structured list retrieved from mesh file:
  if ( vnode_ids != NULL ) delete vnode_ids; vnode_ids = NULL;
  vnode_ids = new GNIDBuffer  (sum);
  for ( i=0,m=0; i<n; i++ ) {
    for ( j=0; j<ni[i]->dim(); j++,m++ ) {
      nid = (*ni[i])(j);
//    eid = (GINT )gelems.member(i)->GetID();
//    SET_DW(vnode_ids->Data()+m, &eid, &nid );
      (*vnode_ids)[m] = nid;
#if 0
  //  cout << " vnodeid[" << i << "][" << j << "]=" << (*vnode_ids)(m) << endl;
      cout << " elemid[" << i << "][" << j << "]=" << (gelems.member(i))->GetID()
           << " HIWORD[" << i << "][" << j << "]=" << HIWORD(vnode_ids->Data()+m) << endl;
      cout << " nodeid[" << i << "][" << j << "]=" << (*ni[i])(j)
           << " LOWORD[" << i << "][" << j << "]=" << LOWORD(vnode_ids->Data()+m) << endl;
#endif
    }

  }
#endif
#if defined(GU_DEBUG_OUTPUT)
    cout << serr << "done." << endl;
#endif

  // cleanup:
  for ( i=0; i<nelems_; i++ ) if ( ni[i] ) delete ni[i];

  if ( xxN   ) delete [] xxN;
  if ( vert3 ) delete [] vert3;
  if ( ni    ) delete [] ni;
  if ( bi    ) delete    bi;
  if ( bt    ) delete    bt;
  if ( bgbdy ) delete [] bgbdy;

  return TRUE;
} // end of method InitMesh


//************************************************************************************
//************************************************************************************
// METHOD     : InitPMesh
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
// MODIFIC'NS : Paste from helm_t.cpp r1.30
//************************************************************************************

GBOOL GUtils::InitPMesh2d(GINT Nx, GINT Ny, GINT xNx, GINT xNy, GTBasisListGLL &gllpool, 
                        GElemList &gelems, GINT  &gnelems_, GNIDBuffer *&vnode_ids,
                        GNODEID &vmaxid, Point *&gd, GINT  &ngd, GINT ntmp, GINT  nadapt)
{
  GSHORT     rank=GComm::WorldRank(), rem, nprocs=GComm::WorldSize();
  GINT       I0, ie, j, J0, je;
  GINT       n, ne, nelems_, gne_max_;
  GLONG      xxN[GDIM];
  GKEY       rootid;
  GDOUBLE    Lx, Ly;
  ELEMTYPE   etype=RECT_QUAD;
  GIBuffer   *ebi;
  GBTBuffer  *ebt=NULL;
  Point      vert[4];
  
    
  nelems_ = Nx * Ny / nprocs ; 
  rem     = nelems_ % nprocs;
  if ( rank < rem ) nelems_++;
  GComm::Allreduce(&nelems_, &gnelems_, 1, GC_GINT , G_OP_SUM);
  GComm::Allreduce(&nelems_, &gne_max_, 1, GC_GINT , G_OP_MAX);
  ngd     = (GINT)pow(2.0,GDIM);
  gd      = new Point [ngd];
  gd[0].x1 = 0.0;  gd[0].x2 = 0.0;
  gd[1].x1 = 1.0;  gd[1].x2 = 0.0;
  gd[2].x1 = 1.0;  gd[2].x2 = 1.0;
  gd[3].x1 = 0.0;  gd[3].x2 = 1.0;
  xxN[0]   = xNx;  xxN[1]   = xNy;
  Lx       = PDISTANCE(gd[0],gd[1]) / Nx;
  Ly       = PDISTANCE(gd[2],gd[1]) / Ny;
//while ( n<nelems_ && mr.GetElem(etype, vert3, bgbdy, xxN, ni[n], bi, bt)==0 ) { 
  RankIJ(I0, J0, Nx, Ny, rank);
  for ( je=J0,n=0; je<Ny && n<nelems_; je++ ) {
    for ( ie=I0; ie<Nx && n<nelems_; ie++,n++ ) {
      vert[0][0] = ie*Lx    ;  vert[0][1] = je*Ly;
      vert[1][0] = (ie+1)*Lx;  vert[1][1] = je*Ly;
      vert[2][0] = (ie+1)*Lx;  vert[2][1] = (je+1)*Ly;
      vert[3][0] = ie*Lx    ;  vert[3][1] = (je+1)*Ly;
      gelems.add(etype,ntmp);
      for ( j=0; j<GDIM; j++ ) { 
        if ( !gllpool.findorder(xxN[j]-1) ) gllpool.add(xxN[j]-1);
        gllpool.member()->Solve();
        gelems.member()->SetBasis(gllpool.member(), j+1);
      }
      gelems.member()->SetVertices(vert,4);
      gelems.member()->SolveFE();
      
      // Set boundary conditions:
      ebi      = gelems.member()->GetBdyIndices();
      ebt      = gelems.member()->GetBdyTypes  ();
      ne       = gelems.member()->GetNumEdges();   
       
      for ( j=0; j<ne; j++ ) {
        gelems.member()->bGlobalBdyEdge(j) = 0;
        if ( (je==0 && j==0) || (je==(Ny-1) && j==2) ||
             (ie==0 && j==3) || (ie==(Nx-1) && j==1) ) {
          gelems.member()->bGlobalBdyEdge(j) = 1;
          gelems.member()->GetEdgeType   (j) = PERIODIC;
        }
      }
      if ( ie==0    && je == 0    ) gelems.member()->GetVertType(0) = PERIODIC;
      if ( ie==Nx-1 && je == 0    ) gelems.member()->GetVertType(1) = PERIODIC;
      if ( ie==Nx-1 && je == Ny-1 ) gelems.member()->GetVertType(2) = PERIODIC;
      if ( ie==0    && je == Ny-1 ) gelems.member()->GetVertType(3) = PERIODIC;
      gelems.member()->ComputeBdyInfo();
  //  cout << "GUtils::InitPMesh: bdy_ind[" << n << "]=" << *ebi << endl;
  
      // Compute element id and its root id. Initially
      // these are equal:
      rootid = GUtils::RootID(rank*gne_max_+n,nadapt);
      gelems.member()->SetRootID  (rootid);
      gelems.member()->SetParentID(rootid);
      gelems.member()->SetID      (rootid);
#if defined(GU_DEBUG_OUTPUT)
      cout << "GUtils::InitPMesh: elem[" << n << "] rootid=" << rootid 
           << " elem_RootID= " << gelems.member()->GetRootID()
           << " parent_key = " << gelems.member()->GetParentID() << endl;
//         << " elem_key   = " << gelems.member()->GetID() << endl;
//    cout << "GUtils::InitPMesh: elem[" << n << "] =" << *(gelems.member()) << endl;
#endif
  
#if 0
      cout << "GUtils::InitPMesh: elem[" << n << "]; bdy_indices=" << *ebi
           << endl << " bdy_types=" << *ebt << endl;
#endif
    }
  }
  if ( n != nelems_ ) {
    cout << "GUtils::InitPMesh: inconsistent number of elements" << endl;
    return FALSE;
  }

  return TRUE;
} // end of method InitPMesh


//************************************************************************************
//************************************************************************************
// METHOD     : RankIJ
// DESCRIPTION: Returns starting indices for Nx x Ny grid for rank irank, if
//              elems are distributed evenly and in order with x varying most
//              rapidly.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
// MODIFIC'NS : 
//************************************************************************************

void GUtils::RankIJ(GINT &I, GINT &J, GINT Nx, GINT Ny, GSHORT irank)
{
  GINT i, j, n, Ne=Nx*Ny/GComm::WorldSize();
 
#if 0
  for ( J=J0, n=0; J<Ny && n<nelems; J++ ) 
    for ( I=I0; I<Nx && n<nelems; I++, n++ ) 
#endif

  for ( j=0, n=0; j<Ny; j++ ) { 
    for ( i=0; i<Nx; i++, n++ ) { 
      if ( n == MAX(irank,0)*Ne ) {
        I = i; J = j;
        return;
      }
    }
  }

} // end of method RankIJ


//************************************************************************************
//************************************************************************************
// METHOD     : RootID
// DESCRIPTION: Computes root id for tree, indicated by the tree index, i_tree,
//              given the max no. of refinement levels allowed
// ARGUEMENTS : i_tree    : index of the tree root of interest; must be globally
//                          unique
//              max_levels: max no of refinement levels; must the same globally
// RETURNS    : GKEY root 
//************************************************************************************
GKEY GUtils::RootID(GINT  i_tree, GINT  max_levels)
{ 
  GKEY    rkey=0;
  GWORD   ihi, ilo;
  GDOUBLE fact = pow(2.0,GDIM);
  
  if ( i_tree == 0 ) return 1;
//rkey = (GKEY)( pow(fact,max_levels) * i_tree  );
//rkey = (GKEY)( pow(pow(fact,max_levels),i_tree) );
  // hi-word is h = log_(2^d) R_j = j * l_max, where
  // j = i_tree and l_max = max_levels:
  ihi = i_tree * max_levels;
  ilo = 0;
  SET_HW(&rkey,&ihi);
  SET_LW(&rkey,&ilo);
  
#if defined(GU_DEBUG_OUTPUT)
  cout << "GUtils::RootID: itree=" << i_tree << " max_levels=" << max_levels << " root_id=" << rkey << endl;
#endif
  return rkey;
} // end of method RootID


//************************************************************************************
//************************************************************************************
// METHOD     : in_neighborlist
// DESCRIPTION: determines if element with specified input params is
//              contained within the specified list
// ARGUEMENTS :
// RETURNS    : TRUE if yes; else FALSE
//************************************************************************************
GBOOL GUtils::in_neighborlist(GNeighborList &elist, GSHORT  iproc, GINT  ielem, GINT  compid)
{

  GBOOL                     bfound=FALSE;
  BasicLinkElemT<GNeighbor> *pkeep=elist.curr(), *p;

  elist.start(NULL);
  while ( (p=elist.curr()) != NULL && !bfound )
  {
    bfound = ( p->member->proc() == iproc && p->member->elemid() == ielem);
    elist.next();
  }
  elist.start(pkeep);

  return bfound;


} // end of method in_neighborlist


//************************************************************************************
//************************************************************************************
// METHOD     : DoDotProducs
// DESCRIPTION: Performs dot products:
//                dot_i = a1 . a2[i]
// ARGUEMENTS : gelems :  element list
//              a1     :  GVector list, number of list entries must = that in gelems
//              a2Vec  :  array of GVector lists (n_prods of them)
//              lProds :  array holding locally computed inner prods (temp array)
//              prods  :  array holding solutions. There are n_prods solutions
//              n_prods:  number of inner prods; also the number of lProds, and number
//                        of lists in the a2Vec list array.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUtils::DoDotProducts(GElemList &gelems, GVecList &a1, GVecList a2Vec[],
                            GDOUBLE lProds[], GDOUBLE prods[], const GINT  n_prods)
{
  GINT      i, j, k, m, nlocal;
  GDOUBLE   *a1Data,  *a2Data, *mData;
  GVector   *imult=NULL;
  GBOOL     bRet = TRUE;

  nlocal = a1.size();

  for ( j=0; j<n_prods && bRet; j++ ) {   // loop over required dot prods
    lProds[j] = 0.0;
    for ( k=0; k<nlocal && bRet; k++ ) {  // loop over local elements
      a1Data = a1[k]->Data();
      imult = gelems[k]->GetNodalMultiplicity();
      if ( a1[k]->dim() != a2Vec[j][k]->dim() ) bRet = FALSE;
      a2Data =  a2Vec[j][k]->Data();

      if ( imult == NULL ) {
        lProds[j] += MTK::fvec_dot( *a1[k],*a2Vec[j][k] );
      }
      else {
        mData  =  imult->Data();
        for ( i=0; i<a1[k]->dim(); i++ ) { // if there is a counting vector specified
          lProds[j] += ( a1Data[i] * a2Data[i] * mData[i] );
        }
      }
    }
  }

  if ( bRet ) {
    GComm::Allreduce(lProds, prods, n_prods, GC_GDOUBLE, G_OP_SUM);
  }
  return bRet;

} // end of method DoDotProducts


//************************************************************************************
//************************************************************************************
// METHOD     : Rand
// DESCRIPTION: returns pseudo-random number unif. distributed
//              in range [0,1].
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE GUtils::Rand()
{
  GINT  irand=rand();
  GDOUBLE  rr = irand / ( RAND_MAX + 1.0 );
  return rr;
} // end of method Rand()

//************************************************************************************
//************************************************************************************
// METHOD     : TSeedRand
// DESCRIPTION: seeds random number generator using system clock
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GUtils::TSeedRand()
{
  GDOUBLE   itime = STK::Timer();
  GUSHORT  iseed = (GUSHORT )(itime/1000.0); 
  srand(iseed);
} // end of method TSeedRand()


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeMasks
// DESCRIPTION: Computes mask and inverse of multiplicity vectors
// ARGUMENTS  : gelems:  element list
//              gsop  :  gather-scatter operator
//              hDSOp :  handle (not used, set to NULL_HANDLE)
//              ub    :  GVecList of lenght = gelems.size. Acts as tmp space. 
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::ComputeMasks(GElemList &gelems, NTreeAdapt *gsop, GCHandle hDSOp, GVecList &ub)
{
  GINT      i, j, k, nelems, ice[2], icp[2];
  GVector   *mask, *imult;
  GIBuffer  *ie, *iv;
  Elem2D    *elem;
  GMortar1D *mortar;

  // Initialize a vector list for masks:
  nelems = gelems.size();
  for ( i=0; i<nelems; i++ ) {
    elem     = gelems[i];
    mortar   = elem->GetEdgeMortar();
    ie       = elem->GetEdgeIndices();
    iv       = elem->GetVertexIndices();
    *ub[i]   = 1.0;
#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::ComputeMasks: ub_pre-DSS[" << i << "]=" << *ub[i] << endl;
#endif
    for ( j=0; j<elem->GetNumEdges(); j++ ) {   
      if ( !mortar[j].isConforming() ) {
        ub[i]->Set(0.0, &ie[j]);
      }
    }
  }
  if ( gsop && !gsop->DSOp(ub, G_OP_SUM, hDSOp) ) {
    cout << "GUtils::ComputeMasks: Unable to assemble operator" << endl;
    exit(1);
  }

  // Invert multiplicity vector to form counting vector:
  for ( i=0; i<nelems; i++ ) {
    elem     = gelems[i];
    imult    = elem->GetNodalMultiplicity();
    *imult   = 1.0;
#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::ComputeMasks: ub_post-DSS[" << i << "]=" << *ub[i] << endl;
#endif
    for ( k=0; k<imult->dim(); k++ ) {  
//    (*ub[i])[k] == 0.0 ? 1.0 : (*ub[i])[k];
      if ( (*ub[i])[k] != 1.0 ) (*imult)[k] = (*ub[i])[k]==0.0 ? 1.0 : 1.0/(*ub[i])[k];
    }
#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::ComputeMasks: imult[" << i << "]=" << *imult << endl;
#endif
  }

  // Compute masking matrix, and reset imult vector to 0 on
  // child faces:
  for ( i=0; i<nelems; i++ ) {
    elem    = gelems[i];
    mask     = elem->GetMask();
    imult    = elem->GetNodalMultiplicity();
    mortar   = elem->GetEdgeMortar();
    ie       = elem->GetEdgeIndices();
    iv       = elem->GetVertexIndices();
    *mask    = 1.0;

    // Set imult=0 on child edge, again:
    for ( j=0; j<elem->GetNumEdges(); j++ ) { 
      if      ( !mortar[j].isConforming() ) {
        imult->Set(0.0, &ie[j]);
#if defined(GU_DEBUG_OUTPUT)
        cout << "GUtils::ComputeMasks: imult[" << i << "][" << j << "]=" << *imult << endl;
#endif
      }
    }
//  for ( k=0; k<mask->dim(); k++ ) if ( (*imult)[k] == 0.0 ) (*mask)[k] = 0.0;

    // At intersecion of real vertex and mortar, set imult=0
    for ( j=0; j<elem->GetNumVertices(); j++ ) {
      elem->Vertex2EdgePoint(ice, icp, j);
      if    (!elem->isVirtualVertex(ice[0],icp[0]) &&  elem->isVirtualVertex(ice[1],icp[1]) 
         || elem->isVirtualVertex(ice[0],icp[0]) && !elem->isVirtualVertex(ice[1],icp[1]) ) {
        imult->Set(0.0, &iv[j]);
      }
    }

#if 0
    cout << "GUtils::ComputeMasks: mask [" << i << "]=" << *mask   << endl;
    cout << "GUtils::ComputeMasks: imult[" << i << "]=" << *imult << endl;
#endif
  }

  return TRUE;
} // end of method ComputeMasks


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeGlobalDOFs
// DESCRIPTION: Computes number of global DOFs and related quantities
//              NOTE: GUtils::ComputeMasks must have been computed prior to entry
// ARGUMENTS  :  gelems: elem. list
//               gndof : global number of DOFs
//               minLen: min element length
//               maxLen: max element length
// RETURNS    : none.
//************************************************************************************
void GUtils::ComputeGlobalDOFs(GElemList &gelems, GINT  &gndof, GDOUBLE &minLen, GDOUBLE &maxLen)
{
  GINT      i, j;
  GDOUBLE   fdof, gfdof, flocal[2], fglobal[2];
  Elem2D    *elem;
  GVector   *imult;

  flocal[0] = 0.0;
  flocal[1] = 0.0;
  for ( i=0, fdof=0.0; i<gelems.size(); i++ ) {
    elem     = gelems[i];
    imult    = elem->GetNodalMultiplicity();
    for ( j=0; j<imult->dim(); j++ ) fdof += (*imult)[j];
    flocal[0] = MAX(flocal[0], 1.0 / elem->GetMinEdgeLength());
    flocal[1] = MAX(flocal[1], elem->GetMaxEdgeLength());
  }
  GComm::Allreduce(&fdof  , &gfdof , 1, GC_GDOUBLE, G_OP_SUM);
  GComm::Allreduce( flocal, fglobal, 2, GC_GDOUBLE, G_OP_MAX);
  gndof  = (GINT )gfdof;
  minLen = 1.0/fglobal[0];
  maxLen = fglobal[1];

} // end of method ComputeGlobalDOFs


//************************************************************************************
//************************************************************************************
// METHOD     : SetMorton
// DESCRIPTION: Computes integral length and sets origin for Morton type key
//              generator

// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::SetMorton(Morton_KeyGen *keygen_, GElemList *gelems, 
      Point *p_origin, Point *dX, Point *globdomain_, GINT  ng, GINT  max_levels_)
{
  GINT     i, j, nv, iP, nBits, Levmax;
  GDOUBLE  gfact, hfact, safety, *Ltg, *Lte, gLe[GDIM];
  GDOUBLE  Lg[GDIM], Le[GDIM], Lgmax, Lemin;
  GBOOL    bOK=TRUE;
  Point    *vert;
  Elem2D   *elem;
  
  if ( gelems == NULL ) {
    cout << "SetMorton: initial element list required" << endl;
    exit(1);
  }
  if ( !dX  || !p_origin) {
    cout << "SetMorton: global domain not set" << endl;
    exit(1);
  }
#if 0
  nv  = (GINT )pow(2.0,GDIM);    // rectangular domain only
  if ( nv != ng ) {
    cout << "SetMorton: invalid number of points specifying global domain" << endl;
    return FALSE; 
  }
#endif
  Ltg = new GDOUBLE [ng];
  Lte = new GDOUBLE [ng];
  for ( j=0; j<ng; j++ ) {
    Ltg[j] = PDISTANCE(globdomain_[j], p_origin[0]);
  }
  for ( j=0; j<GDIM; j++ ) {
    Lg[j] = -SEHUGE;
    Le[j] = SEHUGE;
  }

  // Find Max[global domain length (relative to origin)], Lg:
  for ( j=0; j<ng; j++ ) {
    Lg[j%GDIM] = MAX(Ltg[j],Lg[j%GDIM]);
  }
  
  // Find Min[local edge nodal distance] over grid, Le:
  for ( i=0; i<gelems->size(); i++ ) {
    elem = (*gelems)[i];
    nv   = elem->GetNumVertices();
    vert = elem->GetSpVertices();
    for ( j=0, iP=0; j<GDIM; j++ ) {
      iP      = MAX(iP, elem->GetOrder(j+1));
    }
    hfact = 1.0/(iP*iP);
    for ( j=0; j<nv; j++ ) {
      Lte[j]  = PDISTANCE(vert[(j+1)%nv], vert[j]);
      Lte[j] *= hfact;
      Le[j%GDIM] = MIN(Lte[j],Le[j%GDIM]);
    }
  }
  GComm::Allreduce(Le, gLe, GDIM, GC_GDOUBLE, G_OP_MIN);
  Lgmax = -SEHUGE;
  Lemin =  SEHUGE;
  for ( j=0; j<GDIM; j++ ) {
    Lgmax = MAX(Lgmax,Lg[j]);
    Lemin = MIN(Lemin,Le[j]);
  }
// cout << "GUtils::SetMorton: Lemin=" << Lemin << " Lgmax=" << Lgmax << endl;
  // Use 2 pieces of information:
  // (1)   dx >= Lg_max / 2^(B_i==number of bits for one dimen. representation in GKEY)
  // (2)   dx <= Le_min / (2^M P^2), where M is the max. no levels
  //                                 and P is the polynomial order
  // to set a limit on the max no. levels:
  // Lev_max <= log10(2)^-1 * log10(2^B_i Lemin/Lgmax):
  safety = 2.0;  // factor > 1 by which (1) and (2) are met 
  nBits  = (GINT )( (GDOUBLE)(BITSPERBYTE*sizeof(GKEY))/((GDOUBLE)GDIM) + 0.5 );
  gfact  = safety/pow(2.0,max_levels_);  
  hfact  = pow(2.0,nBits);

  Levmax = (GINT )(3.3219281 * log10(hfact * Lemin /(safety*Lgmax)));
//cout << "GUtils::SetMorton: nBits=" << nBits << " gfact=" << gfact << " hfact=" << hfact 
//   << " Levmax=" << Levmax << endl;
  bOK    = max_levels_ >=0 ? max_levels_ <= Levmax: TRUE; 
  for ( j=0; j<GDIM; j++ ) {
    (*dX)[j] = gfact * gLe[j];
  }
 
  keygen_->SetOrigin     (*p_origin);
  keygen_->SetIntegralLen(*dX);
  if ( !bOK ) {
    cout << "GUtils::SetMorton: too many refinement levels."       << endl;
    cout << "                   max refine levels=" << Levmax      << endl;
    cout << "                    number specified=" << max_levels_ << endl;
    cout << "                                  dX=" << *dX         << endl; 
    cout << "                                  P0=" << *p_origin   << endl; 
  }
#if defined(GU_DEBUG_OUTPUT)
  cout << "SetMorton: dX = " << *dX  << endl; 
  cout << "SetMorton: P0 = " << *p_origin  << endl; 
#endif

  delete [] Ltg;
  delete [] Lte;

  return bOK;
} // end of method SetMorton


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeGMass
// DESCRIPTION: Computes global mass matrix, and its inverse, stored locally
//              within the elements
// ARGUMENTS  : gelems:  element list
//              gsop  :  gather-scatter operator
//              hDSOp :  handle (not used, set to NULL_HANDLE)
//              ub    :  GVecList of length = gelems.size. Acts as tmp space. 
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::ComputeGMass(GElemList &gelems, NTreeAdapt *gsop, GCHandle hDSOp, GVecList &ub)
{
  GINT      i, k, nelems;
  GVector   *B_L, *gB, *iB;
  Elem2D    *elem;


  // Initialize a vector list
  nelems = gelems.size();
  for ( i=0; i<nelems; i++ ) {
    *ub[i]   = 1.0;
  }

  // Compute Q (1_global), essentially:
#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::ComputeGMass: doing projection of 1_global..." << endl;
#endif
  ((NTreeAdapt*)gsop)->DoProjection(TRUE);
  if ( gsop && !gsop->DSOp(ub, G_OP_SUM, hDSOp) ) {
    cout << "GUtils::ComputeGMass: Unable to assemble operator" << endl;
    exit(1);
  }
  ((NTreeAdapt*)gsop)->DoProjection(FALSE);
#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::ComputeGMass: projection of 1_global done." << endl;
#endif


  // Compute : B_L * Q (1_global):
  for ( i=0; i<nelems; i++ ) {
    elem     = gelems[i];
    B_L      = elem->GetMassMatrix();
    MTK::fvec_point_prod_rep(*ub[i],*B_L);
  }

#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::ComputeGMass: doing projection of  Q Q^T  B_L * Q (1_global)..." << endl;
#endif
  // Compute B_mass-lumped = Q Q^T  B_L * Q (1_global):
  ((NTreeAdapt*)gsop)->DoProjection(TRUE);
  if ( gsop && !gsop->DSOp(ub, G_OP_SUM, hDSOp) ) {
    cout << "GUtils::ComputeGMass: Unable to assemble operator" << endl;
    exit(1);
  }
  ((NTreeAdapt*)gsop)->DoProjection(FALSE);
#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::ComputeGMass: projection of  Q Q^T  B_L * Q (1_global) done." << endl;
#endif

  for ( i=0; i<nelems; i++ ) {
    elem   = gelems[i];
    iB     = elem->GetiMass();
    gB     = elem->GetgMass();
    *gB    = *ub[i];
//  for ( k=0; k<gB->dim(); k++ ) if ( (*imult)[k] != 0.0 ) (*gB)[k] = (*gB)[k] * (*imult)[k];
    for ( k=0; k<gB->dim(); k++ ) (*iB)[k] = 1.0 / (*gB)[k];
  }

  return TRUE;
} // end of method ComputeGMass


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeGOp
// DESCRIPTION: Computes global version of (diagnonal) operator, op
// ARGUMENTS  : gelems:  element list
//              gsop  :  gather-scatter operator
//              hDSOp :  handle (not used, set to NULL_HANDLE)
//              opvec :  GVecList of operator vectors. Is modified to be 
//                       global operator on exit
//              ub    : GVecList of lenght the number elements; will act as temp space
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::ComputeGOp(GElemList &gelems, NTreeAdapt *gsop, GCHandle hDSOp, GVecList &opvec, GVecList &ub)
{
  GINT      i, nelems;

  // Initialize vector lists
  nelems = gelems.size();
  for ( i=0; i<nelems; i++ ) {
    *ub   [i]   = *opvec[i];
    *opvec[i]   = 1.0;
  }

  // Compute Q (1_global), essentially:
  ((NTreeAdapt*)gsop)->DoProjection(TRUE);
  if ( gsop && !gsop->DSOp(opvec, G_OP_SUM, hDSOp) ) {
    cout << "GUtils::ComputeGOp: Unable to assemble operator" << endl;
    exit(1);
  }
  ((NTreeAdapt*)gsop)->DoProjection(FALSE);


  // Compute : Op_L * Q (1_global):
  for ( i=0; i<nelems; i++ ) {
    MTK::fvec_point_prod_rep(*opvec[i],*ub[i]);
  }

  // Compute Op-lumped = Q Q^T  Op_L * Q (1_global):
  ((NTreeAdapt*)gsop)->DoProjection(TRUE);
  if ( gsop && !gsop->DSOp(opvec, G_OP_SUM, hDSOp) ) {
    cout << "GUtils::ComputeGOp: Unable to assemble operator" << endl;
    exit(1);
  }
  ((NTreeAdapt*)gsop)->DoProjection(FALSE);

  return TRUE;

} // end of method ComputeGOp


//************************************************************************************
//************************************************************************************
// METHOD     : Smooth
// DESCRIPTION: Computes bcs by taking a weighted average. NOTE: global (assembled)
//              mass matrix (actually, its inverse) must be calculated prior to entry
//                Calculates:
//                  u_b -> M^-1 DSS(M_L u_b),
//              where u_b is the bdy conditions expressed in local format; M^-1 is
//              the inverse of the assembled mass matrix; M_L is the local mass matrix 
//              (unassembled); DSS is the application of Q Q^T, or the direct stiffness
//              operator.
// ARGUMENTS  : ub    :  GVecList of length = gelems.size, represents locally-stored
//                       bcs. Contents will be altered upon exit.
//              gelems:  element list
//              gsop  :  gather-scatter operator
//              hDSOp :  handle (not used, set to NULL_HANDLE)
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::Smooth(GVecList &ub, GElemList &gelems, NTreeAdapt *gsop, GCHandle hDSOp)
{
  GINT      i, nelems;
  GVector   *B_L, *iB;
  Elem2D    *elem;

  // Compute M_L u_b:
  nelems = ub.size();
  for ( i=0; i<nelems; i++ ) {
    elem   = gelems[i];
    B_L    = elem->GetMassMatrix();
    MTK::fvec_point_prod_rep(*ub[i],*B_L);
  }

  // Compute DSS(M_L u_b):
  if ( gsop && !gsop->DSOp(ub, G_OP_SUM, hDSOp) ) {
    cout << "GUtils::Smooth: Unable to assemble operator" << endl;
    exit(1);
  }

  // Compute M^-1 DSS(M_L u_b), and store back into u_b:
  // NOTE: M^-1 must be assembled prior to entry...
  for ( i=0; i<nelems; i++ ) {
    elem   = gelems[i];
    iB     = elem->GetiMass();
#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::Smooth: M^-1[" << i << "]=" << *iB << endl;
#endif
    MTK::fvec_point_prod_rep(*ub[i],*iB);
#if defined(GU_DEBUG_OUTPUT)
    cout << "GUtils::Smooth: ub_smth[" << i << "]=" << *ub[i] << endl;
#endif
  }
   
  return TRUE;
} // end of method Smooth


//************************************************************************************
//************************************************************************************
// METHOD     : PointOnSeg
// DESCRIPTION: Determines of point, r, lies on segment specified P0-P1.
// ARGUMENTS  : P0    
//              P1   : points specifying segment
//              r    : test point
// RETURNS    : TRUE if r lies on segment; else FALSE
//************************************************************************************
GBOOL GUtils::PointOnSeg(Point &P0, Point &P1, Point &r)
{
  GINT    i , j ; 
  GDOUBLE xP, xR, alpha, dot;
  Point   dP, dR;

  // Compute difference position vectors:
  for ( i=0; i<GDIM; i++ ) {
    dP[i] = P1[i] - P0[i];
    dR[i] = r [i] - P0[i];
  }

  // Compute magnitudes of dP and dR vectors:
  xP = xR = 0.0;
  dot     = 0.0;
  for ( i=0; i<GDIM; i++ ) {
    xP  += dP[i] * dP[i];
    xR  += dR[i] * dR[i];
    dot += dP[i] * dR[i];
  }
  xP = sqrt(xP);
  xR = sqrt(xR);
 
  if ( xP <= TINY ) {
    cout << "GUtils::PointOnSeg: invalid segment specification" << endl;
    exit(1);
  }
 
  if ( xR <= TINY ) return TRUE;
 
  alpha = acos(dot / (xP * xR));

  return ( fabs(alpha) < TINY * 100.0 && xR <= xP );
  
} // end of PointOnSeg

//************************************************************************************
//************************************************************************************
// METHOD     : isPGlobalBdyElem
// DESCRIPTION: Determine if element is on global bdy. If so, provide global face 
//              index on which it resides, and provide element face (local) index
//              which actually lies on bdy. This is valid only for polygonal
//              global domains.
// ARGUMENTS  :
//              iGlobalFace  : buffer of global bdy face indices. Is filled only to 
//                             the number of global faces for which elems have faces
//                             that correspond with global bdy. Default values are -1.
//              iElemFace    : buffer of element (local) face index. Is filled only 
//                             to number of local faces that corresp. with global bdy.
//                             Default values are -1.
//              nBdy         : number of elemental (local) faces that corresp. to 
//                             global bdy
//              elem         : element pointer
//              pGlobalDomain: Point array specifying global domain
//              nGlobalDomain: number of points in pGlobDomain array
// RETURNS    : TRUE if element is on global bdy; else FALSE
//************************************************************************************
GBOOL GUtils::isPGlobalBdyElem(GIBuffer  &iGlobalFace, GIBuffer  &iElemFace, GINT  &nBdy,
                               Elem2D *elem, Point pGlobalDomain[], GINT  nGlobalDomain)
{
  GINT  i, j, k, ne;
  GBOOL bglobalelem=FALSE;
  Point *vertices;

#if defined(IS3D)
  cout << "GUtils::isPGlobalBdyElem: 3D currently unsupported" << endl;
  exit(1);
#endif

  if ( elem == NULL ) {
    cout << "GUtils::isPGlobalBdyElem: NULL element specified" << endl;
    exit(1);
  }

  vertices   = elem->GetSpVertices ();
  ne         = elem->GetNumEdges   ();
  iGlobalFace.Resize(ne);  iGlobalFace = -1;
  iElemFace  .Resize(ne);  iElemFace   = -1;

  // Find if element has edge on bdy: 
  for ( j=0,nBdy=0; j<ne; j++ ) {
    k = (j+1+ne);
    for ( i=0; i<nGlobalDomain-1; i++ ) {
      if ( PointOnSeg(pGlobalDomain[i],pGlobalDomain[i+1],vertices[j]) 
        && PointOnSeg(pGlobalDomain[i],pGlobalDomain[i+1],vertices[k]) ) {
        bglobalelem = TRUE;
        iGlobalFace(nBdy) = i;
        iElemFace  (nBdy) = j;
        nBdy++;
      } 
    }
  } 
  
  return bglobalelem;
} // end of method isPGlobalBdyElem


//************************************************************************************
//************************************************************************************
// METHOD     : H1_Project (1)
// DESCRIPTION: Computes a 'projection' (actually, interpolation from 
//              parent edge to child edges.
//              Method computes:
//                       ~ ~T
//               u ->  J Q Q  u,
//              where J is the interpolation matrix from parent to child,
//              Q~ is the Boolean conectivity matrix, and u is the 
//              quantity desired.  On nonconforming faces, the 'projected' quantity
//              is continuous, which means that the value is scaled by its 
//              multiplicity. This really only affects the face edges (or 
//              edge endpoints)
// ARGUMENTS  : u     :  GVecList of length = gelems.size, represents locally-stored
//                       quantity. Contents will be altered upon exit.
//              gelems:  element list
//              gsop  :  gather-scatter operator
//              hDSOp :  handle (not used, set to NULL_HANDLE)
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::H1_Project(GVecList &u, GElemList &gelems, NTreeAdapt *gsop, GCHandle hDSOp)
{
  if ( !gsop ) return TRUE;

  ((NTreeAdapt*)gsop)->DoProjection(TRUE);
  if ( !gsop->DSOp(u, G_OP_SUM, hDSOp) ) {
    return FALSE;
  }
  ((NTreeAdapt*)gsop)->DoProjection(FALSE);


  return TRUE;
} // end of method H1_Project (1)


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : H1_Project (2)
// DESCRIPTION: Computes a 'projection' (actually, interpolation from 
//              parent edge to child edges.
//              Method computes:
//                       ~ ~T
//               u ->  J Q Q  u,
//              where J is the interpolation matrix from parent to child,
//              Q~ is the Boolean conectivity matrix, and u is the 
//              quantity desired.  On nonconforming faces, the 'projected' quantity
//              is continuous, which means that the value is scaled by its 
//              multiplicity. This really only affects the face edges (or 
//              edge endpoints)
// ARGUMENTS  : u     :  GFieldList of length = gelems.size, represents locally-stored
//                       quantity. Contents will be altered upon exit.
//              gsop  :  gather-scatter operator
//              hDSOp :  handle (not used, set to NULL_HANDLE)
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::H1_Project(GFieldList &u, NTreeAdapt *gsop, GCHandle hDSOp)
{ 
  if ( !gsop ) return TRUE;
  ((NTreeAdapt*)gsop)->DoProjection(TRUE);
  if ( !gsop->DSOp(u, G_OP_SUM, hDSOp) ) {
    return FALSE;
  }
  ((NTreeAdapt*)gsop)->DoProjection(FALSE);


  return TRUE;
} // end of method H1_Project (2)
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : H1_ProjectM
// DESCRIPTION: Computes a 'projection' (actually, interpolation from 
//              parent edge to child edges.
//              Method computes:
//                            ~ ~T
//               u ->  J Mask Q Q  u,
//              where J is the interpolation matrix from parent to child,
//              Q~ is the Boolean conectivity matrix, Mask is the mask (used
//              if provided, and it is a diagonal matrix, and u is the 
//              quantity desired.
// ARGUMENTS  : u     :  GVecList of length = gelems.size, represents locally-stored
//                       quantity. Contents will be altered upon exit.
//              gelems:  element list
//              gsop  :  gather-scatter operator
//              hDSOp :  handle (not used, set to NULL_HANDLE)
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::H1_ProjectM(GVecList &u, GElemList &gelems, NTreeAdapt *gsop, GCHandle hDSOp)
{
  if ( !gsop ) return TRUE;
  ((NTreeAdapt*)gsop)->DoProjection(TRUE);
  ((NTreeAdapt*)gsop)->PreMask     (TRUE);
//((NTreeAdapt*)gsop)->DoMult      (TRUE);
  if ( !gsop->DSOp(u, G_OP_SUM, hDSOp) ) {
    return FALSE;
  }
  ((NTreeAdapt*)gsop)->DoProjection(FALSE);
  ((NTreeAdapt*)gsop)->PreMask     (FALSE);
//((NTreeAdapt*)gsop)->DoMult      (FALSE);


  return TRUE;
} // end of method H1_ProjectM


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeGDiag
// DESCRIPTION: Computes global diagonal matrix, as given by vector list.
//              On entry, input vector list is local, but on exit it is
//              global.
// ARGUMENTS  : vlist :  input vector list representing (local) diag(Matrix)
//              gelems:  element list
//              gsop  :  gather-scatter operator
//              hDSOp :  handle (not used, set to NULL_HANDLE)
//              ub    :  GVecList of length = gelems.size. Acts as tmp space. 
// RETURNS    : none.
//************************************************************************************
GBOOL GUtils::ComputeGDiag(GVecList &vlist, GElemList &gelems, NTreeAdapt *gsop, GCHandle hDSOp, GVecList &ub)
{
  GINT      i, j, k, nelems;
  GVector   *V_L, *gV;
  GIBuffer  *ie;
  Elem2D    *elem;
  GMortar1D *mortar;

  if ( !gsop ) return TRUE;
  if ( !gsop->DSOp(vlist, G_OP_SUM, hDSOp) ) {
    cout << "GUtils::ComputeGDiag: Unable to assemble operator" << endl;
    exit(1);
  }
//GUtils::H1_ProjectM(vlist, gelems, gsop, hDSOp);

  return TRUE;

  // Initialize a vector list; compute Q ( 1_global ):
  nelems = gelems.size();
  for ( i=0; i<nelems; i++ ) {
    elem     = gelems[i];
    mortar   = elem->GetEdgeMortar();
    ie       = elem->GetEdgeIndices();
    *ub[i]   = 1.0;
    for ( j=0; j<elem->GetNumEdges(); j++ ) { // set field=0 on edges; set mortars=1
      ub[i]->Set(0.0, &ie[j]);
      *(mortar[j].GetMortarField()) = 1.0;
    }
    for ( j=0; j<elem->GetNumEdges(); j++ ) { // Apply J:
      mortar[j].Mortar2Host(ub[i],ie[j].Data(),ie[j].dim(),0,FALSE);
    }
//  cout << "GUtils::ComputeGDiag: Q(1)_L[" << i << "]=" << *ub [i] << endl;
  }

  // Compute : V_L * Q (1_global):
  for ( i=0; i<nelems; i++ ) {
    elem   = gelems[i];
    V_L    = vlist[i];
//  cout << "GUtils::ComputeGDiag: V_L [" << i << "]=" << *V_L    << endl;
    MTK::fvec_point_prod_rep(*ub[i],*V_L);
//  cout << "GUtils::ComputeGDiag: V_lumped[" << i << "]=" << *ub[i] << endl;
  }
   
  // Compute V_mass-lumped = Q Q^T  V_L * Q (1_global):
#if 1
  if ( !gsop->DSOp(ub, G_OP_SUM, hDSOp) ) {
    cout << "GUtils::ComputeGDiag: Unable to assemble operator" << endl;
    exit(1);
  }
#endif

#if 0
  GUtils::H1_Project(ub, gelems, (NTreeAdapt*)gsop, hDSOp);
#endif

#if 1
  for ( i=0; i<nelems; i++ ) {
    elem   = gelems[i];
//  mask   = elem->GetMask();
//  imult  = elem->GetNodalMultiplicity();
//  iV     = elem->GetiMass();
    gV     = vlist[i];
    *gV    = *ub[i];
    cout << "GUtils::ComputeGDiag: V_assembled [" << i << "]=" << *ub[i] << endl;
//  MTK::fvec_point_prod_rep(*gV,*mask);
//  MTK::fvec_point_prod_rep(*iV,*mask);
//  for ( k=0; k<gV->dim(); k++ ) if ( (*imult)[k] != 0.0 ) (*gV)[k] = (*gV)[k] * (*imult)[k];
  }
#endif

  return TRUE;
} // end of method ComputeGDiag


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeGlobalArea
// DESCRIPTION: Computes global grid area by adding areas of individual elements
// ARGUMENTS  : gelems:  element list
// RETURNS    : Global area
//************************************************************************************
GDOUBLE  GUtils::ComputeGlobalArea(GElemList &gelems)
{
  GDOUBLE  garea, larea=0.0;
  Elem2D *elem;

  gelems.start(NULL);
  while ( (elem=gelems.member()) != NULL ) {
    larea += elem->GetArea();
    gelems.next();
  }

  if ( !bLocal_ )
    GComm::Allreduce(&larea, &garea, 1, GC_GDOUBLE, G_OP_SUM);
  else
    garea = larea;

  return garea;
} // end of method ComputeGlobalArea


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeMinLength
// DESCRIPTION: Computes global minimum element length
// ARGUMENTS  : gelems:  element list
// RETURNS    : Global min. length
//************************************************************************************
GDOUBLE  GUtils::ComputeMinLength(GElemList &gelems)
{ 
  GDOUBLE  gmin, lmin=SEHUGE;
  Elem2D *elem;
  
  gelems.start(NULL); 
  while ( (elem=gelems.member()) != NULL ) {
    lmin = MIN(lmin,elem->GetMinEdgeLength());
    gelems.next();
  }
  
  if ( !bLocal_ )
    GComm::Allreduce(&lmin, &gmin, 1, GC_GDOUBLE, G_OP_MIN);
  else
    gmin = lmin;

  return gmin;
} // end of method ComputeMinLength



//************************************************************************************
//************************************************************************************
// METHOD     : ComputeMaxDerivA
// DESCRIPTION: Computes global |max| of the derivative in specified direction of 
//              input field.
// ARGUMENTS  : gelems:  element list
//              gfield:  field list
//              idir  :  derivative coordinate direction
// RETURNS    : MAx derivative
//************************************************************************************
GDOUBLE GUtils::ComputeMaxDerivA(GElemList &gelems, GFieldList &gfield, GINT  idir )
{
  
  GDOUBLE    gmax, lmax=0.0;
  Elem2D    *elem;
  Field2D   *field;
  GVector    *u, *du;

  if ( idir < 1 || idir > GDIM ) {
    cout << "GUtils::ComputeMaxDeriv: invalid coordinate direction" << endl;
    exit(1);
  }

  gelems.start(NULL);
  gfield.start(NULL);
  while ( (elem=gelems.member()) != NULL ) {
    field =  gfield.member();
    u     = field->GetExpCoeffs(0);
    du    = elem->GetTemp();
    elem->Differentiate(du, u, idir);
    lmax = MAX(lmax, du->MaxA());
    elem->TempUnlock(du);
    gelems.next();
    gfield.next();
  }

  if ( !bLocal_ )
    GComm::Allreduce(&lmax, &gmax, 1, GC_GDOUBLE, G_OP_MAX);
  else
    gmax = lmax;

  return gmax;
} // end of method ComputeMaxDerivA


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeGlobalIntegralV
// DESCRIPTION: Computes global integral of specified vector list
// ARGUMENTS  : uv    : input vector list to integrate
//              gelems:  element list
// RETURNS    : global integral of uv
//************************************************************************************
GDOUBLE GUtils::ComputeGlobalIntegralV(GVecList &uv, GElemList &gelems)
{
  GDOUBLE    gsum, lsum=0.0;
  Elem2D    *elem;
  GVector   *u;

  gelems.start(NULL);
  uv    .start(NULL);
  while ( (elem=gelems.member()) != NULL ) {
    u     = uv.member();
    lsum += elem->Integrate(u);
    gelems.next();
    uv    .next();
  }

  if ( !bLocal_ )
    GComm::Allreduce(&lsum, &gsum, 1, GC_GDOUBLE, G_OP_SUM);
  else
    gsum = lsum;
  
  uv.start(NULL);
  return gsum;
} // end of method ComputeGlobalIntegralV


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeGlobalIntegralF
// DESCRIPTION: Computes global integral of specified field list
// ARGUMENTS  : uf    : input field list to integrate
//              gelems:  element list
// RETURNS    : global integral of uf
//************************************************************************************
GDOUBLE GUtils::ComputeGlobalIntegralF(GFieldList &uf, GElemList &gelems)
{
  GDOUBLE    gsum, lsum=0.0;
  Elem2D    *elem;
  GVector   *u;

  gelems.start(NULL);
  uf    .start(NULL);
  while ( (elem=gelems.member()) != NULL ) {
    u     = uf.member()->GetExpCoeffs(0);
    lsum += elem->Integrate(u);
    gelems.next();
    uf    .next();
  }

  if ( !bLocal_ )
    GComm::Allreduce(&lsum, &gsum, 1, GC_GDOUBLE, G_OP_SUM);
  else  
    gsum = lsum;
  uf.start(NULL);

  return gsum;
} // end of method ComputeGlobalIntegralF


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeL2Norm (1)
// DESCRIPTION: Computes L2 norm (global) of specified quantity:
//                L2norm = [Int_globalgrid (q^2 dV) ] ^1/2
// ARGUMENTS  : q     : input vector list to integrate
//              gelems:  element list
// RETURNS    : global L2 norm of q
//************************************************************************************
GDOUBLE GUtils::ComputeL2Norm(GVecList &q, GElemList &gelems)
{
  GINT     j;
  GDOUBLE  gsum, lsum=0.0;
  Elem2D  *elem;
  GVector  *u, *q2;

  gelems.start(NULL);
  q     .start(NULL);
  while ( (elem=gelems.member()) != NULL ) {
    u  = q.member();
    q2 = elem->GetTemp();
    for ( j=0; j<u->dim(); j++ ) (*q2)[j] = (*u)[j] * (*u)[j];
    lsum += elem->Integrate(q2);
    elem->TempUnlock(q2);
    gelems.next();
    q     .next();
  }
  if ( !bLocal_ )
    GComm::Allreduce(&lsum, &gsum, 1, GC_GDOUBLE, G_OP_SUM);
  else
    gsum = lsum;
  q.start(NULL);

  return sqrt(gsum);
} // end of method ComputeL2Norm (1)



//************************************************************************************
//************************************************************************************
// METHOD     : ComputeInfNorm (1)
// DESCRIPTION: Computes inf norm (global) of specified quantity:
//                inf_norm = MAX_globalgrid (|q|)
// ARGUMENTS  : q     : input vector list quantity to find norm of
//              gelems:  element list
// RETURNS    : global infinity norm of q
//************************************************************************************
GDOUBLE GUtils::ComputeInfNorm(GVecList &q, GElemList &gelems)
{
  GINT     i=0;
  GDOUBLE  gmax, lmax=0.0;
  Elem2D  *elem;

  gelems.start(NULL);
  q     .start(NULL);
  while ( (elem=gelems.member()) != NULL ) {
    lmax  = MAX(lmax,q[i]->MaxA());
    gelems.next();
    q     .next();
    i++;
  }
  if ( !bLocal_ )
    GComm::Allreduce(&lmax, &gmax, 1, GC_GDOUBLE, G_OP_MAX);
  else
    gmax = lmax;
  q.start(NULL);

  return gmax;
} // end of method ComputeInfNorm (1)



//************************************************************************************
//************************************************************************************
// METHOD     : ComputeEucNorm (1)
// DESCRIPTION: Computes Euclidean norm (global) of specified quantity:
//                inf_norm = [Sum_globalgrid (q_i^2)]^1/2
// ARGUMENTS  : q     : input vector list quantity to find norm of
//              gelems:  element list
// RETURNS    : global Euclidean norm of q
//************************************************************************************
GDOUBLE GUtils::ComputeEucNorm(GVecList &q, GElemList &gelems)
{
  GINT     j;
  GDOUBLE  gsum, lsum=0.0;
  GVector  *u;
  Elem2D   *elem;

  gelems.start(NULL);
  q     .start(NULL);
  while ( (elem=gelems.member()) != NULL ) {
    u = q.member();
    for ( j=0; j<u->dim(); j++ ) lsum += (*u)[j]*(*u)[j];
    gelems.next();
    q     .next();
  }
  if ( !bLocal_ )
    GComm::Allreduce(&lsum, &gsum, 1, GC_GDOUBLE, G_OP_SUM);
  else
    gsum = lsum;
  q.start(NULL);

  return sqrt(gsum);
} // end of method ComputeEucNorm (1)


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeL2Norm (2)
// DESCRIPTION: Computes L2 norm (global) of specified quantity:
//                L2norm = [Int_globalgrid (q^2 dV) ] ^1/2
// ARGUMENTS  : q     : input field list to integrate
//              ilev  : time level of field for which to compute norm
// RETURNS    : global L2 norm of q
//************************************************************************************
GDOUBLE GUtils::ComputeL2Norm(GFieldList &q, GINT ilev)
{
  GINT     i, j;
  GDOUBLE  gsum, lsum=0.0;
  Elem2D  *elem;
  GVector  *u, *q2;

  q     .start(NULL);
  for ( i=0; i<q.size(); i++ ) {
  //while ( q.member() != NULL && (elem=q.member()->GetElement()) != NULL ) {
    if ( (elem = q.member()->GetElement()) == NULL  ) {
      cout << "GUtils::ComputeL2Norm: NULL element: i=" << i << endl;
      exit(1);
    }
    u  = q.member()->GetExpCoeffs(ilev);
    q2 = elem->GetTemp();
    for ( j=0; j<u->dim(); j++ ) (*q2)[j] = (*u)[j] * (*u)[j];
    lsum += elem->Integrate(q2,NULL);
    elem->TempUnlock(q2);
    q     .next();
  }
  if ( !bLocal_ )
    GComm::Allreduce(&lsum, &gsum, 1, GC_GDOUBLE, G_OP_SUM);
  else
    gsum = lsum;

  q.start(NULL);

  return sqrt(gsum);
} // end of method ComputeL2Norm (2)



//************************************************************************************
//************************************************************************************
// METHOD     : ComputeInfNorm (2)
// DESCRIPTION: Computes inf norm (global) of specified quantity:
//                inf_norm = MAX_globalgrid (|q|)
// ARGUMENTS  : q     : input field list quantity to find norm of
//              ilev  : time level of field for which to compute norm
// RETURNS    : global infinity norm of q
//************************************************************************************
GDOUBLE GUtils::ComputeInfNorm(GFieldList &q, GINT ilev)
{
  GDOUBLE  gmax, lmax=0.0;
  GVector  *u;
  Elem2D   *elem;

  q     .start(NULL);
  while ( q.member() != NULL && (elem=q.member()->GetElement()) != NULL ) {
    u = q.member()->GetExpCoeffs(ilev);
    lmax  = MAX(lmax,u->MaxA());
    q     .next();
  }
  if ( !bLocal_ )
    GComm::Allreduce(&lmax, &gmax, 1, GC_GDOUBLE, G_OP_MAX);
  else
    gmax = lmax;
  q.start(NULL);

  return gmax;
} // end of method ComputeInfNorm (2)



//************************************************************************************
//************************************************************************************
// METHOD     : ComputeEucNorm (2)
// DESCRIPTION: Computes Euclidean norm (global) of specified quantity:
//                inf_norm = [Sum_globalgrid (q_i^2)]^1/2
// ARGUMENTS  : q     : input field list quantity to find norm of
//              ilev  : time level of field for which to compute norm
// RETURNS    : global Euclidean norm of q
//************************************************************************************
GDOUBLE GUtils::ComputeEucNorm(GFieldList &q, GINT ilev)
{
  GINT     j;
  GDOUBLE  gsum, lsum=0.0;
  GVector  *u;
  Elem2D   *elem;

  q     .start(NULL);
  while ( q.member() != NULL && (elem=q.member()->GetElement()) != NULL ) {
    u = q.member()->GetExpCoeffs(ilev);
    for ( j=0; j<u->dim(); j++ ) lsum += (*u)[j]*(*u)[j];
    q     .next();
  }
  if ( !bLocal_ )
    GComm::Allreduce(&lsum, &gsum, 1, GC_GDOUBLE, G_OP_SUM);
  else
    gsum = lsum;
  q.start(NULL);

  return sqrt(gsum);
} // end of method ComputeEucNorm (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetGlobalPeriodicBdy
// DESCRIPTION: Determines whether or not global bdy is periodic.
//              Currently good only for 2D.
//
// ARGUMENTS  : giPeriodic : for each global periodic bdy, the number of elems with 
//                           global periodic bdy that lines up with it
//              gd         : array of points comprising global vertices
//              nvert      : number of elements in 'gd'
//              elems      : ElemList of all local elements.
// RETURNS    : none
//************************************************************************************
void GUtils::GetGlobalPeriodicity(GIBuffer &giPeriodic, Point *&gd, GINT nvert, GElemList &elems) 
{
  char serrmsg_[] = "GUtils::GetGlobalPeriodicity: ";
  GINT      i, j, jj, jk, k, ndv=(GINT)(pow(2.0,GDIM)), ne;
  GBOOL     bBdyEdge, isPeriodic, bOnSeg;
  GIBuffer  iPeriodic;
  Point     *everts;
  Elem2D    *elem;

  giPeriodic.Resize(ndv); giPeriodic = 0;
  iPeriodic .Resize(ndv); iPeriodic  = 0;
  elems.start(NULL);
  for ( i=0; i<elems.size(); i++ ) {
    elem   = elems[i];
    everts = elem->GetSpVertices();
    ne     = elem->GetNumEdges();
    for ( j=0; j<ne; j++ ) {
      jj = j;
      jk = (j+1)%ne;
      bBdyEdge    = elem->bGlobalBdyEdge(j);
      isPeriodic  = elem->GetEdgeType(j) == PERIODIC;
      for ( k=0; k<nvert; k++ ) {
        bOnSeg = PointOnSeg(gd[k], gd[(k+1)%ndv], everts[jj])
              && PointOnSeg(gd[k], gd[(k+1)%ndv], everts[jk]);
        iPeriodic[k] += (bBdyEdge && isPeriodic && bOnSeg ? 1 : 0);
      }
    }
    elems.next();   
  }

  GComm::Allreduce(iPeriodic.Data(), giPeriodic.Data(), ndv, GC_GINT, G_OP_SUM);

} // end of method GetGlobalPeriodicity
    

//************************************************************************************
//************************************************************************************
// METHOD     : GetLoadStatistics
// DESCRIPTION: Gets global load statistics based on element list.
//
// ARGUMENTS  : gWork: float buffer containing the local work measure for each
//                      processor. The work measure is given as 
//                                number_local_nodes / avg(number_local_nodes)
//              
//              favg       : average over all procs of the work measure.
// RETURNS    : none
//************************************************************************************
void GUtils::GetLoadStatistics(GDBuffer &gWork, GDOUBLE &favg, GElemList &gelems)
{
  char     serrmsg_[] = "GUtils::GetLoadStatistics: ";
  GINT     i, j, NN;
  GDOUBLE  lWork, giNorm;

  gWork.Resize(GComm::WorldSize());
  for ( i=0, lWork=0.0; i<gelems.size(); i++ ) {
    for ( j=0, NN=1; j<GDIM; j++ ) NN *= gelems[i]->GetOrder(j+1); 
    lWork += (GDOUBLE)NN;
  }

  GComm::Allgather(&lWork, 1, GC_GDOUBLE, gWork.Data(), 1, GC_GDOUBLE);

  for ( j=0, favg=0.0; j<gWork.dim(); j++ ) {
    favg += gWork[j];
  }
  favg /= gWork.dim();
  giNorm= 1.0/favg;

  for ( j=0, favg=0.0; j<gWork.dim(); j++ ) {
    gWork[j] *= giNorm;
  }
  
} // end of method GetLoadStatistics


//************************************************************************************
//************************************************************************************
// METHOD     : GetGBinVarV
// DESCRIPTION: Gets variable, var, from open GBinReader object 
// ARGUMENTS  : var   : list of vectors representing variable. This is added to as
//                      needed, but no elements are deleted.
//              vlabel: label for variable, given exactly as in GBin file.
//              rgbin : GBinReader object. File should already be opened on entry.
//
//                      NOTE: It is user's responsibility to ensure that the size of 
//                            new list is consistent with the number of field vectors
//                            expected.
// RETURNS    : TRUE on success; else FALSE. Error message issued.
//************************************************************************************
GBOOL GUtils::GetGBinVarV(GVecList &var, char *vlabel, GBinReader &rgbin)
{
  GINT     i, j, *idims, n, NN;
  GBOOL    bRet=TRUE;
  
  for ( n=0,i=0; n<rgbin.GetNumDataSets(); n++ ) {
    if ( strcmp(vlabel,rgbin.GetLabel(n)) != 0  ) continue;
    idims   = rgbin.GetDims(n); 
    for ( j=0,NN=1; j<rgbin.GetRank(n); j++ ) NN *= idims[j];
    var.add(); var[i]->Resize(NN);
    if ( rgbin.GetFieldData(n, var[i]->Data(), var[i]->dim() )==NULL ) {
      cout << "GUtils::GetGBinVarV: Cannot retrieve field data from file. "
           << " Reader error: " << rgbin.Error() << endl;
      bRet = FALSE;
      break;
    }
    i++;
  }

  return bRet;

} // end of method GetGBinVarV


//************************************************************************************
//************************************************************************************
// METHOD     : GetGBinVarF
// DESCRIPTION: Gets variable, var, from open GBinReader object 
// ARGUMENTS  : var    : list of fields representing variable. This is added to as
//                       needed, but no elements are deleted.
//              vlabel : label for variable, given exactly as in GBin file.
//              ilevel : time level in which to store read-in data.
//              nlevels: number of levels with which to instantiate field.
//              ntmp   : number of temp levels with which to instantiate field.
//              gelems : element list underlying field. Should be filled on entry. It
//                       is user's responsibility to ensure that the size of gelems is
//                       the same as the number of fields that will be read in. An error
//                       will result if gelems.size < number of fields in file.
//              rgbin  : GBinReader object. File should already be opened on entry.
//
//                        NOTE: It is user's responsibility to ensure that the size of 
//                              new list is consistent with the number of field vectors
//                              expected.
// RETURNS    : TRUE on success; else FALSE. Error message issued.
//************************************************************************************
GBOOL GUtils::GetGBinVarF(GFieldList &var, char *vlabel, GINT ilevel, GINT nlevels, 
                          GINT ntmp, GElemList &gelems, GBinReader &rgbin)
{
  GINT     i, j, *idims, n, NN;
  GBOOL    bRet=TRUE;
  
  for ( n=0,i=0; n<rgbin.GetNumDataSets(); n++ ) {
    if ( strcmp(vlabel,rgbin.GetLabel(n)) != 0  ) continue;
    idims   = rgbin.GetDims(n); 
    for ( j=0,NN=1; j<rgbin.GetRank(n); j++ ) NN *= idims[j];
    var.add(nlevels,gelems[i],ntmp);
    if ( rgbin.GetFieldData(n, var[i]->GetExpCoeffs(ilevel)->Data(), 
                               var[i]->GetExpCoeffs(ilevel)->dim()  )==NULL ) {
      cout << "GUtils::GetGBinVarF: Cannot retrieve field data from file. "
           << " Reader error: " << rgbin.Error() << endl;
      bRet = FALSE;
      break;
    }
    i++;
  }

  return bRet;

} // end of method GetGBinVarF


//************************************************************************************
//************************************************************************************
// METHOD     : DoLocal
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
// MODIFIC'NS : Paste from helm_t.cpp r1.30
//************************************************************************************

void GUtils::DoLocal(GBOOL blocal)
{
  bLocal_ = blocal;
}  // end of method DoLocal



//************************************************************************************
//************************************************************************************
// METHOD     : InterpLL
// DESCRIPTION: interpolates quantity from one grid to another. Both qantities (new and old)
//              are represented as lists here.
// ARGUMENTS  : pto   : GVecList of quantity interpolated to new grid, gto
//              gto   : grid (GElemList) representing new grid
//              pfrom : GVecList of quantity on old grid, gfrom
//              gfrom : grid (GElemList) representing old grid
// RETURNS    : TRUE on success; else FALSE. 
//************************************************************************************
GBOOL GUtils::InterpLL(GVecList &pto, GElemList &gto, GVecList &pfrom, GElemList &gfrom) 
{
  char     *serr = "GUtils::InterpLL: ";
  GINT     i, j, k, nelems, nold=0, tNN;
  GBOOL    bRet=TRUE;
  Point    *xto=NULL;
  GVector  *xp[GDIM];


  if ( gto.size() != gfrom.size() ) {
    cout << serr << "incompatible grids" << endl;
    exit(1);
  }
  nelems = gfrom.size();

  // Interpolate pressure to GL grid:
  for ( i=0; i<nelems && bRet; i++ ) {
    tNN = gto[i]->GetNumNodes();
    if ( xto==NULL || tNN != nold ) {
      delete [] xto; xto = new Point [tNN];
    }
    for ( k=0; k<GDIM; k++ ) {
      xp[k] = gto[i]->GetSpNodes(k+1);
      for ( j=0; j<tNN; j++ ) xto[j][k] = (*xp[k])[j];
    }
    bRet = gfrom[i]->Interp(pfrom[i],xto,NULL,tNN,pto[i]);
    nold = tNN;
  }
  if ( !bRet ) {
    cout << serr << "Interp failure" << endl;
    return FALSE;
  }

  return TRUE;

} // end of method InterpLL


//************************************************************************************
//************************************************************************************
// METHOD     : InterpFL
// DESCRIPTION: interpolates quantity from one grid to another. Old quantity is
//              a field; new quantity is a list.
//              are represented as lists here.
// ARGUMENTS  : pto   : GVecList of quantity interpolated to new grid, gto
//              gto   : grid (GElemList) representing new grid
//              pfrom : GVecList of quantity on old grid, gfrom
//              gfrom : grid (GElemList) representing old grid
// RETURNS    : TRUE on success; else FALSE. 
//************************************************************************************
GBOOL GUtils::InterpFL(GVecList &pto, GElemList &gto, GFieldList &pfrom, GElemList &gfrom)
{
  char     *serr = "GUtils::InterpFL: ";
  GINT     i, j, k, nelems, nold=0, tNN;
  GBOOL    bRet=TRUE;
  Point    *xto=NULL;
  GVector  *xp[GDIM];


  if ( gto.size() != gfrom.size() ) {
    cout << serr << "incompatible grids" << endl;
    exit(1);
  }
  nelems = gfrom.size();

  // Interpolate pressure to GL grid:
  for ( i=0; i<nelems && bRet; i++ ) {
    tNN = gto[i]->GetNumNodes();
    if ( xto==NULL || tNN != nold ) {
      delete [] xto; xto = new Point [tNN];
    }
    for ( k=0; k<GDIM; k++ ) {
      xp[k] = gto[i]->GetSpNodes(k+1);
      for ( j=0; j<tNN; j++ ) xto[j][k] = (*xp[k])[j];
    }
    bRet = gfrom[i]->Interp(pfrom[i]->GetExpCoeffs(0),xto,NULL,tNN,pto[i]);
    nold = tNN;
  }
  if ( !bRet ) {
    cout << serr << "Interp failure" << endl;
    return FALSE;
  }

  return TRUE;

} // end of method InterpFL




//************************************************************************************
//************************************************************************************
// METHOD     : SmoothPGrid
// DESCRIPTION: Smooths pressure by interpolating (extrapolating) pressure grid
//              quantity to GL grid,doing a DSS, then interpolating back to G grid.  
// ARGUMENTS  : pp    : GVecList of quantity residing on the G (pressure) grid.
//                      pp is modified on exit to contain the 'smoothed' quantity on 
//                      the G grid.
//              utmp  : temp list on u-grid
//              uelems: u-grid
//              pelems: p-grid
//              gsop  : NTreeAdapt object
// RETURNS    : TRUE on success; else FALSE. 
//************************************************************************************
GBOOL GUtils::SmoothPGrid(GVecList &pp, GVecList &utmp, GElemList &uelems, 
                           GElemList &pelems, NTreeAdapt *gsop)
{
  char     *serr = "GUtils::SmoothPGrid: ";
  GINT     i, j, k, nelems, nold=0, pNN, vNN;
  GBOOL    bRet=TRUE;
  Point    *xto=NULL;
  GVector  *xp[GDIM];

  if ( gsop == NULL ) {
    cout << serr << "NULL comm object" << endl;
    exit(1);
  }
  nelems = uelems.size();

  // Interpolate pressure to GL grid:
  for ( i=0; i<nelems&&bRet; i++ ) {
    vNN = uelems[i]->GetNumNodes();
    if ( xto==NULL || vNN != nold ) {
      delete [] xto; xto = new Point [vNN];
    }
    for ( k=0; k<GDIM; k++ ) {
      xp[k] = uelems[i]->GetSpNodes(k+1);
      for ( j=0; j<vNN; j++ ) xto[j][k] = (*xp[k])[j];
    }
    bRet = pelems[i]->Interp(pp[i],xto,NULL,vNN,utmp[i]);
    nold = vNN;
  }
  if ( !bRet ) {
    cout << serr << "Interp failure" << endl;
    return FALSE;
  }

#if 1
  // Do smoothing 'projection':
  if ( !GUtils::H1_Project(utmp, uelems, (NTreeAdapt*)gsop, NULL_HANDLE) ) {
    cout << serr << "H1_Project failure" << endl;
    return FALSE;
  }
#endif

  // Interpolate smoothed pressure back to G grid:
  for ( i=0,nold=0; i<nelems&&bRet; i++ ) {
    pNN = pelems[i]->GetNumNodes();
    if ( xto==NULL || pNN != nold ) {
      delete [] xto; xto = new Point [pNN];
    }
    for ( k=0; k<GDIM; k++ ) {
      xp[k] = pelems[i]->GetSpNodes(k+1);
      for ( j=0; j<pNN; j++ ) xto[j][k] = (*xp[k])[j];
    }
    bRet = uelems[i]->Interp(utmp[i],xto,NULL,pNN,pp[i]);
    nold = vNN;
  }

  if ( !bRet ) {
    cout << serr << "Interp failure" << endl;
    return FALSE;
  }

  if ( xto != NULL ) delete [] xto;

  return TRUE;

} // end of method SmoothPGrid

