library(splines)
library(BMS)
source("bc-techniques.R")

##############
#
# Generates a figure illustrating how distribution mapping works
#
# Uses data for one particular gridcell, stored in tabular text files
# obs.txt, cur.txt, and fut.txt


var <- "tmin"

width <- 7   # half-width of selection window (7 ~= 2 weeks)
t <- 280     # center of selection window

data <- list()
x <- list()   # data values falling within 2-week window across 30 years
d <- list()   # density(x)

color <- c(obs="black", cur="red", fut="blue", proj="darkgreen")

for(i in c("obs", "cur", "fut")){
    raw <- read.table(paste0(i, ".txt"))
    data[[i]] <- raw[[var]]
    time <- as.numeric(as.Date(row.names(raw)))
    # normalize time to deal w/ calendars
    time <- time * 360.0 / ifelse(i=="obs", 365.25, 365)
    x[[i]] <- data[[i]][((time - time[1] - (t - width)) %% 360) <= (width*2+1)]
}

# need same number of points in each dataset for Q-Q plot
n <- min(sapply(x, length))

for(i in c("obs", "cur", "fut")){
    x[[i]] <- x[[i]][1:n]
    mu     <- mean(x[[i]])
    x[[i]] <- x[[i]] - mu
    d[[i]] <- density(x[[i]], bw="SJ")
}    

x[["proj"]] <- kddm(x[["cur"]], x[["fut"]], x[["obs"]], bw="SJ")
d[["proj"]] <- density(x[["proj"]], bw="SJ")


xr <- range(sapply(lapply(d, "[[", "x"), range))
yr <- range(sapply(lapply(d, "[[", "y"), range))


# sort data for ease in plotting

x = lapply(x,sort)



#####
# Two-panel plot explaining bias correction via xfer f'n

x11(height=4.5, width=8)

par(mfrow=c(1,2), las=1)

### first panel: Q-Q current/obs, with transfer function

plot(x[["cur"]], x[["obs"]],
     xlim = xr, ylim = xr, asp=1, pty="s", cex=1, lwd=0.5, mgp=c(2.5,0.75,0),
     main = "Q-Q plot + transfer fn",
     xlab = expression("Current"~T[min]~~~(degree~C)),
     ylab = expression("Observed"~T[min]~~~(degree~C))
     )

mtext("a)", adj=0, line=1.6)

# add rug plots of data values; transparency used to handle overlap
tcolor = adjustcolor("black",alpha=0.3)
rug(x[["cur"]], side=1, col=tcolor)
rug(x[["fut"]], side=2, col=tcolor)

# overlay PDF curves for each dataset
scale  <- diff(xr) * 0.2  # vertical scaling
offset <- min(xr)         # offset from rug

lines(d[["cur"]]$x, (d[["cur"]]$y / max(d[["cur"]]$y) * scale + offset))
lines((d[["obs"]]$y / max(d[["obs"]]$y) * scale + offset), d[["obs"]]$x)


# fit curve (equivalent to KDDM transfer function)
dq <- 0.001
q <- seq(0+dq, 1-dq, dq)
lines(quantile(d[["cur"]], q), quantile(d[["obs"]], q), col="red", lwd=2)




### second panel: bias-correct future using xfer fn

plot(NA, type="n",
     xlim = xr, ylim = xr, asp=1, pty="s", cex=1, mgp=c(2.5,0.75,0),
     main = "Bias correction",
     xlab = expression("Future"~T[min]~~~(degree~C)),
     ylab = expression("Bias-corrected"~T[min]~~~(degree~C))
     )

abline(0, 1, lty=3)

mtext("b)", adj=0, line=1.6)

lines(quantile(d[["cur"]], q), quantile(d[["obs"]], q), col="red", lwd=2)

# add rug plots
rug(x[["fut"]], side=1, col=tcolor)
rug(x[["proj"]], side=2, col=tcolor)

# overlay PDF curves
lines(d[["fut"]]$x, (d[["fut"]]$y / max(d[["fut"]]$y) * scale + offset))
lines((d[["proj"]]$y / max(d[["proj"]]$y) * scale + offset), d[["proj"]]$x)


# illustrative transfer lines

peak1 = 330
peak2 = 70

for (i in c(1,peak1,peak2,n)){
    lines(c(x[["fut"]][i], x[["fut"]][i]), c(-999, x[["proj"]][i]), lty=2, lwd=0.5)
    lines(c(x[["fut"]][i], -999), c(x[["proj"]][i],x[["proj"]][i]), lty=2, lwd=0.5)
}



# save to PDF and png

dev.copy(pdf, width=8, height=4.5, file="figures/fig.1.bc-example.pdf")
dev.off()

dev.copy(png, width=8, height=4.5, units="in", res=120, file="figures/fig.1.bc-example.png")
dev.off()


# Copyright 2014 Univ. Corp for Atmos. Research
# Author: Seth McGinnis, mcginnis@ucar.edu
